#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import os
import yaml
import sys

def check_file_format(file_path):
    """
    Check if the file is in valid yaml format
    """
    try:
        with open(file_path, 'r') as config_file:
            yaml.safe_load(config_file)
        return True
    except (yaml.parser.ParserError, yaml.scanner.ScannerError) as e:
        print(f"Error: {file_path} is not in valid yaml format. {e}")
        return False

def get_keys(config):
    """
    Returns the keys of a valid yaml file, or None if the file is not found or not in yaml format
    """
    return list(config.keys()) if config else None

def check_and_ask(config):
    """
    Recursively check for undefined keys and ask for input
    """
    if isinstance(config, dict):
        for key in config:
            if config[key] is None:
                config[key] = input(f"Please enter a value for {key}: ")
            elif isinstance(config[key], dict):
                check_and_ask(config[key])

def parse_config():
    config_files = [
        os.path.join(os.path.expanduser("/opt/robin/config/sd-upload"), "sd-upload-config.yml"),
        os.path.join(os.path.expanduser("/opt/robin/config/sd-upload"), "sd-upload-config.yaml"),
        os.path.join(os.path.expanduser("~"), ".sd-upload-config.yml"),
        os.path.join(os.path.expanduser("~"), ".sd-upload-config.yaml"),
        "sd-upload-config.yml",
        "sd-upload-config.yaml"
    ]

    config = None
    file = None

    # Set variable config_file to the first config file found
    for cfg_file in config_files:
        if os.path.isfile(cfg_file) and check_file_format(cfg_file):
            with open(cfg_file, 'r') as config_file:
                config = yaml.safe_load(config_file)
                file = cfg_file
                # print(f"Config {cfg_file} found at full path {os.path.abspath(cfg_file)}")
                break
        # else:
            # print(f"Config file not found in {cfg_file} directory.")
            # print(f"Checking next directory...")
    # if found config file logger which one was found and path
    if not config:
        admin_interface = "https://software-api-admin.robinradar.systems"
        doc_pages = "https://robinradar.atlassian.net/wiki/spaces/DEV/pages/284033095/Software+Deployment+API"
        print(f"Error: no config file found.")
        print(f"Please browse to {admin_interface} to create a config file.")
        print(f"Please browse to {doc_pages} to learn how to create a config file.")
        sys.exit(0)

    check_and_ask(config)
    # print(f"Config {file} found at full path {os.path.abspath(file)}")
    # write the config file with the new values
    try:
        with open(file, "w") as config_file:
            yaml.dump(config, config_file, allow_unicode=True, width=float("inf"))
    except (OSError, yaml.parser.ParserError, yaml.scanner.ScannerError) as e:
        print(f"Error: Failed to write config file. {e}")

    return config
