"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FunctionConstruct = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
const constructs_1 = require("constructs");
class FunctionConstruct extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const functionCodePath = path.resolve(__dirname, `../lambdas/cross-account-parameter-store`);
        this.scope = scope;
        this.props = props;
        this.handler = 'main.on_event';
        this.region = aws_cdk_lib_1.Stack.of(this).region;
        this.runtime = lambda.Runtime.PYTHON_3_8;
        this.functionRole = this.createFunctionRole(id);
        this.functionCode = this.loadFunctionCode(functionCodePath);
        this._function = this.createFunction(id);
    }
    /**
     * Try and load the Lambda function code assets
     */
    loadFunctionCode(sourceCodePath) {
        try {
            return lambda.Code.fromAsset(sourceCodePath);
        }
        catch (error) {
            console.error('failed to load the function code');
            throw error;
        }
    }
    /**
     * Create the IAM role and its required policies to attach the the Lambda function
     */
    createFunctionRole(id) {
        const { roleArn } = this.props;
        const role = new iam.Role(this.scope, `${id}FunctionRole`, {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
        });
        role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['sts:AssumeRole'],
            effect: iam.Effect.ALLOW,
            resources: [roleArn],
        }));
        role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'));
        return role;
    }
    /**
     * Create the Lambda function resource
     */
    createFunction(id) {
        const { handler, runtime, region, functionCode: code, functionRole: role, } = this;
        const { roleArn, roleExternalId, roleSessionName } = this.props;
        const fn = new lambda.Function(this.scope, `${id}LambdaFunction`, {
            code,
            role,
            handler,
            runtime,
            environment: {
                REGION: region,
                ROLE_ARN: roleArn,
                ROLE_EXTERNAL_ID: roleExternalId || '',
                ROLE_SESSION_NAME: roleSessionName || 'halloumi_cross_account_parameter_store',
            },
            timeout: aws_cdk_lib_1.Duration.seconds(900),
            logRetention: logs.RetentionDays.FIVE_MONTHS,
        });
        return fn;
    }
    /**
     * Get the IAM Role attached to the function
     */
    gFunctionRole() {
        return this.functionRole;
    }
    /**
     * Get the function
     */
    gFunction() {
        return this._function;
    }
}
exports.FunctionConstruct = FunctionConstruct;
_a = JSII_RTTI_SYMBOL_1;
FunctionConstruct[_a] = { fqn: "halloumi-cross-account-parameter-store.FunctionConstruct", version: "1.0.1" };
//# sourceMappingURL=data:application/json;base64,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