# Automated Valuation Model

#### Automated Valuation Model (AVM) is a term for a service that uses mathematical modeling combined with databases of existing properties and transactions to calculate real estate values. The majority of automated valuation models (AVMs) compare the values of similar properties at the same point in time. Many appraisers, and even Wall Street institutions, use this type of model to value residential properties. (see [What is an AVM](https://www.investopedia.com/terms/a/automated-valuation-model.asp) Investopedia.com)


#### Why do real estate business use AVMs?
- Real estate companies often use Real Estate AVMs to capture the contact information of potential home-sellers.  Home sellers are valuable leads in the real estate industry and the assumption is that people looking to learn the value of a home might want to sell that home.  Many solution providers sell versions of AVM (the most well-known example is the Zillow Zestimate).  Companies take those AVMs, put them on their website or page and try to identify people in their area who will likely enter the real estate market.


#### Popular Commercial AVM's
* [Zestimate](https://www.zillow.com/zestimate/) - The Zestimate® home valuation model is Zillow’s estimate of a home's market value. The Zestimate incorporates public and user-submitted data, taking into account home facts, location and market conditions.

* [Core Logic](https://www.corelogic.com/landing-pages/automated-valuation-models.aspx) -  CoreLogic® is the chosen AVM provider for 8 of the top 10 U.S. mortgage lenders. 

* [House Canary](https://www.housecanary.com/products/data-points/) - We compute instant valuations spanning property and land values, home equity, and more, and report on the data density behind our conclusions. Gain speed and reduce errors with values and context exactly when and where you need them most.

* [Attom Data](https://www.attomdata.com/data/analytics-derived-data/avm-property-valuations/) - Utilizing more than 80 million homes in our property database across all 50 states and representing 99% of the US population, and valuation software developed by Automated Valuation Model Analytics

### Valuation Process
<img src="valuation_process.png" height="120" >

### Interesting GitHub Repositories related to AVM's
* [Zillow-Kaggle](https://github.com/junjiedong/Zillow-Kaggle/blob/master/README.md) - This repo tackles the first round of Zillow’s Home Value Prediction Competition, which challenges competitors to predict the log error between Zestimate and the actual sale price of houses. And the submissions are evaluated based on Mean Absolute Error between the predicted log error and the actual log error. The competition was hosted from May 2017 to October 2017 on Kaggle, and the final private leaderboard was revealed after the evaluation period ended in January 2018.

* [AutomatedValuationModel](https://github.com/jayshah5696/AutomaticValuationModel/blob/master/notebooks/Final_notebook.ipynb) -  Automated valuation model (AVM) is the name given to a service that can provide real estate property valuations using mathematical modelling combined with a database. Most AVMs calculate a property’s value at a specific point in time by analyzing values of comparable properties. Some also take into account previous surveyor valuations, historica…

* [Lots more on Kaggle](https://www.kaggle.com/c/zillow-prize-1/notebooks) - The Zillow Prize contest competition, sponsored by Zillow, Inc. (“Sponsor”) is open to all individuals over the age of 18 at the time of entry. The competition will contain two rounds, one public and one private.. Each round will have separate datasets, submission deadlines and instructions on how to participate. The instructions on how to participate in each round are listed below. Capitalized terms used but not defined herein have the meanings assigned to them in the Zillow Prize competition Official Rules.

### Other interesting articles
* [towardsdatascience.com](https://towardsdatascience.com/automated-valuation-model-how-it-works-in-real-estate-industry-8d082757e1ed) - Automated Valuation Model — How It Works in Real Estate Industry?

### How does it relate to BCI Risk Modeling?
<img src="bci_flowchart_2.png" height="120" >

### Interesting cross-over companies using AVM technology in the context of climate risk modeling
* [Jupiter Intelligence](https://jupiterintel.com) - Predicting Risk in a Changing Climate: Jupiter’s AI and Scientific Models Deliver Unrivaled Power

### Development Plan
- EDA on golden dataset (due by July 1st) - volunteers? email mike.casale@blockchainclimate.org
- Basic machine learning studies of different models (due by July 8th) - volunteers? email mike.casale@blockchainclimate.org
- Hypertuning & final analysis of machine learning studies (due by July 8th) - volunteers? email mike.casale@blockchainclimate.org
- Completed AVM modeling and pipeline + integrate with AWS microservice (due by Aug 1st) - volunteers? email mike.casale@blockchainclimate.org

# AutoML in EvalML


```python
import evalml, glob, os
from evalml.preprocessing import load_data
import pandas as pd
import urllib
import requests, json
import numpy as np
import pandas as pd
```

# LOAD EPC_PRICE SAMPLE


```python
datapath = '../../data/processed/sample_EPC_Price_merged.csv'
data = pd.read_csv(datapath)
data = data.reset_index()
data
```




<div>
<style scoped>
    .dataframe tbody tr th:only-of-type {
        vertical-align: middle;
    }

    .dataframe tbody tr th {
        vertical-align: top;
    }

    .dataframe thead th {
        text-align: right;
    }
</style>
<table border="1" class="dataframe">
  <thead>
    <tr style="text-align: right;">
      <th></th>
      <th>index</th>
      <th>Postcode</th>
      <th>PriceAddress</th>
      <th>EpcAddress</th>
      <th>JaroDistance</th>
      <th>Price</th>
      <th>PurchaseDate</th>
      <th>PropertyType</th>
      <th>New</th>
      <th>Duration</th>
      <th>...</th>
      <th>PotentialEnergyRating</th>
      <th>CurrentEnergyEfficiency</th>
      <th>PotentialEnergyEfficiency</th>
      <th>EpcInspectionDate</th>
      <th>GlazedArea</th>
      <th>HabitableRooms</th>
      <th>HeatedRooms</th>
      <th>FlatStoreyCount</th>
      <th>TotalFloorArea</th>
      <th>FloorLevel</th>
    </tr>
  </thead>
  <tbody>
    <tr>
      <th>0</th>
      <td>0</td>
      <td>WV69QH</td>
      <td>142, CODSALL ROAD,</td>
      <td>142, Codsall Road,</td>
      <td>1.00</td>
      <td>437500</td>
      <td>5/20/20 0:00</td>
      <td>D</td>
      <td>N</td>
      <td>F</td>
      <td>...</td>
      <td>C</td>
      <td>55</td>
      <td>75</td>
      <td>6/24/13</td>
      <td>Normal</td>
      <td>8.0</td>
      <td>8.0</td>
      <td>NaN</td>
      <td>348.0</td>
      <td>NODATA!</td>
    </tr>
    <tr>
      <th>1</th>
      <td>1</td>
      <td>WV108AB</td>
      <td>49, CARISBROOKE ROAD,</td>
      <td>49, Carisbrooke Road, Bushbury,</td>
      <td>0.94</td>
      <td>140000</td>
      <td>2/7/20 0:00</td>
      <td>S</td>
      <td>N</td>
      <td>F</td>
      <td>...</td>
      <td>B</td>
      <td>81</td>
      <td>85</td>
      <td>6/17/16</td>
      <td>Normal</td>
      <td>5.0</td>
      <td>4.0</td>
      <td>NaN</td>
      <td>84.0</td>
      <td>NODATA!</td>
    </tr>
    <tr>
      <th>2</th>
      <td>2</td>
      <td>WV46BJ</td>
      <td>56, GREENOCK CRESCENT,</td>
      <td>56, Greenock Crescent,</td>
      <td>1.00</td>
      <td>102000</td>
      <td>3/11/20 0:00</td>
      <td>F</td>
      <td>N</td>
      <td>L</td>
      <td>...</td>
      <td>C</td>
      <td>78</td>
      <td>79</td>
      <td>5/2/12</td>
      <td>Normal</td>
      <td>3.0</td>
      <td>3.0</td>
      <td>NaN</td>
      <td>77.0</td>
      <td>2nd</td>
    </tr>
    <tr>
      <th>3</th>
      <td>3</td>
      <td>WV112QQ</td>
      <td>21, RYAN AVENUE,</td>
      <td>21, Ryan Avenue,</td>
      <td>1.00</td>
      <td>141000</td>
      <td>5/18/20 0:00</td>
      <td>S</td>
      <td>N</td>
      <td>F</td>
      <td>...</td>
      <td>C</td>
      <td>62</td>
      <td>79</td>
      <td>10/30/19</td>
      <td>Normal</td>
      <td>5.0</td>
      <td>5.0</td>
      <td>NaN</td>
      <td>77.0</td>
      <td>NODATA!</td>
    </tr>
    <tr>
      <th>4</th>
      <td>4</td>
      <td>WV147AP</td>
      <td>34, MARBURY DRIVE,</td>
      <td>34, Marbury Drive,</td>
      <td>1.00</td>
      <td>122000</td>
      <td>3/13/20 0:00</td>
      <td>S</td>
      <td>N</td>
      <td>L</td>
      <td>...</td>
      <td>C</td>
      <td>75</td>
      <td>75</td>
      <td>9/29/11</td>
      <td>Normal</td>
      <td>3.0</td>
      <td>3.0</td>
      <td>NaN</td>
      <td>56.5</td>
      <td>NODATA!</td>
    </tr>
    <tr>
      <th>...</th>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
    </tr>
    <tr>
      <th>1116</th>
      <td>1116</td>
      <td>WV22AW</td>
      <td>116 - 126, STEELHOUSE LANE,</td>
      <td>100, Steelhouse Lane,</td>
      <td>0.75</td>
      <td>280000</td>
      <td>5/26/20 0:00</td>
      <td>O</td>
      <td>N</td>
      <td>F</td>
      <td>...</td>
      <td>A</td>
      <td>96</td>
      <td>97</td>
      <td>1/31/20</td>
      <td>NO DATA!</td>
      <td>NaN</td>
      <td>NaN</td>
      <td>NaN</td>
      <td>107.0</td>
      <td>NO DATA!</td>
    </tr>
    <tr>
      <th>1117</th>
      <td>1117</td>
      <td>WV108RP</td>
      <td>55, PRIMROSE LANE,</td>
      <td>55, Primrose Lane,</td>
      <td>1.00</td>
      <td>162000</td>
      <td>2/7/20 0:00</td>
      <td>S</td>
      <td>N</td>
      <td>F</td>
      <td>...</td>
      <td>B</td>
      <td>72</td>
      <td>85</td>
      <td>9/16/19</td>
      <td>Normal</td>
      <td>5.0</td>
      <td>5.0</td>
      <td>NaN</td>
      <td>102.0</td>
      <td>NODATA!</td>
    </tr>
    <tr>
      <th>1118</th>
      <td>1118</td>
      <td>WV106BA</td>
      <td>12, THREE TUNS PARADE,</td>
      <td>Flat, 12a, Three Tuns Parade,</td>
      <td>0.76</td>
      <td>150000</td>
      <td>6/24/20 0:00</td>
      <td>O</td>
      <td>N</td>
      <td>F</td>
      <td>...</td>
      <td>C</td>
      <td>57</td>
      <td>72</td>
      <td>4/11/19</td>
      <td>Normal</td>
      <td>3.0</td>
      <td>3.0</td>
      <td>NaN</td>
      <td>67.0</td>
      <td>Ground</td>
    </tr>
    <tr>
      <th>1119</th>
      <td>1119</td>
      <td>WV38NA</td>
      <td>201, CASTLECROFT ROAD,</td>
      <td>201, Castlecroft Road,</td>
      <td>1.00</td>
      <td>470000</td>
      <td>4/6/20 0:00</td>
      <td>D</td>
      <td>N</td>
      <td>F</td>
      <td>...</td>
      <td>C</td>
      <td>56</td>
      <td>75</td>
      <td>5/22/14</td>
      <td>Normal</td>
      <td>7.0</td>
      <td>7.0</td>
      <td>NaN</td>
      <td>229.0</td>
      <td>NODATA!</td>
    </tr>
    <tr>
      <th>1120</th>
      <td>1120</td>
      <td>WV11HN</td>
      <td>COACH HOUSE, 6, WHEELERS FOLD,</td>
      <td>The Coach House, 6 Wheelers Fold,</td>
      <td>0.79</td>
      <td>220000</td>
      <td>6/15/20 0:00</td>
      <td>O</td>
      <td>N</td>
      <td>F</td>
      <td>...</td>
      <td>C</td>
      <td>56</td>
      <td>77</td>
      <td>1/21/20</td>
      <td>Normal</td>
      <td>7.0</td>
      <td>7.0</td>
      <td>NaN</td>
      <td>97.0</td>
      <td>NODATA!</td>
    </tr>
  </tbody>
</table>
<p>1121 rows × 21 columns</p>
</div>



# Handle Date Features 


```python
#handle datefields
from evalml.pipelines.components.transformers import DateTimeFeaturizer

dtf = DateTimeFeaturizer(features_to_extract = ["year", "month", "day_of_week", "hour"])
datefields = ['PurchaseDate','EpcInspectionDate']
data[datefields[0]] = pd.to_datetime(data[datefields[0]])
data[datefields[1]] = pd.to_datetime(data[datefields[1]])
Xdates = data[datefields]
dtf.fit(Xdates)
Xdates = dtf.transform(Xdates)
Xdates = Xdates.reset_index()
Xdates

```




<div>
<style scoped>
    .dataframe tbody tr th:only-of-type {
        vertical-align: middle;
    }

    .dataframe tbody tr th {
        vertical-align: top;
    }

    .dataframe thead th {
        text-align: right;
    }
</style>
<table border="1" class="dataframe">
  <thead>
    <tr style="text-align: right;">
      <th></th>
      <th>index</th>
      <th>PurchaseDate_year</th>
      <th>PurchaseDate_month</th>
      <th>PurchaseDate_day_of_week</th>
      <th>PurchaseDate_hour</th>
      <th>EpcInspectionDate_year</th>
      <th>EpcInspectionDate_month</th>
      <th>EpcInspectionDate_day_of_week</th>
      <th>EpcInspectionDate_hour</th>
    </tr>
  </thead>
  <tbody>
    <tr>
      <th>0</th>
      <td>0</td>
      <td>2020</td>
      <td>May</td>
      <td>Wednesday</td>
      <td>0</td>
      <td>2013</td>
      <td>June</td>
      <td>Monday</td>
      <td>0</td>
    </tr>
    <tr>
      <th>1</th>
      <td>1</td>
      <td>2020</td>
      <td>February</td>
      <td>Friday</td>
      <td>0</td>
      <td>2016</td>
      <td>June</td>
      <td>Friday</td>
      <td>0</td>
    </tr>
    <tr>
      <th>2</th>
      <td>2</td>
      <td>2020</td>
      <td>March</td>
      <td>Wednesday</td>
      <td>0</td>
      <td>2012</td>
      <td>May</td>
      <td>Wednesday</td>
      <td>0</td>
    </tr>
    <tr>
      <th>3</th>
      <td>3</td>
      <td>2020</td>
      <td>May</td>
      <td>Monday</td>
      <td>0</td>
      <td>2019</td>
      <td>October</td>
      <td>Wednesday</td>
      <td>0</td>
    </tr>
    <tr>
      <th>4</th>
      <td>4</td>
      <td>2020</td>
      <td>March</td>
      <td>Friday</td>
      <td>0</td>
      <td>2011</td>
      <td>September</td>
      <td>Thursday</td>
      <td>0</td>
    </tr>
    <tr>
      <th>...</th>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
    </tr>
    <tr>
      <th>1116</th>
      <td>1116</td>
      <td>2020</td>
      <td>May</td>
      <td>Tuesday</td>
      <td>0</td>
      <td>2020</td>
      <td>January</td>
      <td>Friday</td>
      <td>0</td>
    </tr>
    <tr>
      <th>1117</th>
      <td>1117</td>
      <td>2020</td>
      <td>February</td>
      <td>Friday</td>
      <td>0</td>
      <td>2019</td>
      <td>September</td>
      <td>Monday</td>
      <td>0</td>
    </tr>
    <tr>
      <th>1118</th>
      <td>1118</td>
      <td>2020</td>
      <td>June</td>
      <td>Wednesday</td>
      <td>0</td>
      <td>2019</td>
      <td>April</td>
      <td>Thursday</td>
      <td>0</td>
    </tr>
    <tr>
      <th>1119</th>
      <td>1119</td>
      <td>2020</td>
      <td>April</td>
      <td>Monday</td>
      <td>0</td>
      <td>2014</td>
      <td>May</td>
      <td>Thursday</td>
      <td>0</td>
    </tr>
    <tr>
      <th>1120</th>
      <td>1120</td>
      <td>2020</td>
      <td>June</td>
      <td>Monday</td>
      <td>0</td>
      <td>2020</td>
      <td>January</td>
      <td>Tuesday</td>
      <td>0</td>
    </tr>
  </tbody>
</table>
<p>1121 rows × 9 columns</p>
</div>




```python
data_t = pd.merge(data, Xdates, on='index')
data_t = data_t.drop(datefields,axis=1)
data_t = data_t.drop('index',axis=1)
t_datapath = datapath.replace('.csv','_t.csv')
data_t.to_csv(t_datapath,index=False)

```


```python
from evalml.preprocessing import load_data
index_fields = ['Postcode','PriceAddress','EpcAddress','JaroDistance']
X, y = load_data(t_datapath,index=index_fields,target='Price')
X, y
```

                 Number of Features
    Categorical                  11
    Numeric                      10
    
    Number of training examples: 1121
    Targets
    130000    2.14%
    140000    1.96%
    125000    1.69%
    90000     1.61%
    155000    1.61%
              ...  
    234500    0.09%
    277500    0.09%
    257000    0.09%
    236500    0.09%
    256000    0.09%
    Name: Price, Length: 380, dtype: object





    (                                                                                         PropertyType  \
     Postcode PriceAddress                    EpcAddress                         JaroDistance                
     WV69QH   142, CODSALL ROAD,              142, Codsall Road,                 1.00                    D   
     WV108AB  49, CARISBROOKE ROAD,           49, Carisbrooke Road, Bushbury,    0.94                    S   
     WV46BJ   56, GREENOCK CRESCENT,          56, Greenock Crescent,             1.00                    F   
     WV112QQ  21, RYAN AVENUE,                21, Ryan Avenue,                   1.00                    S   
     WV147AP  34, MARBURY DRIVE,              34, Marbury Drive,                 1.00                    S   
     ...                                                                                               ...   
     WV22AW   116 - 126, STEELHOUSE LANE,     100, Steelhouse Lane,              0.75                    O   
     WV108RP  55, PRIMROSE LANE,              55, Primrose Lane,                 1.00                    S   
     WV106BA  12, THREE TUNS PARADE,          Flat, 12a, Three Tuns Parade,      0.76                    O   
     WV38NA   201, CASTLECROFT ROAD,          201, Castlecroft Road,             1.00                    D   
     WV11HN   COACH HOUSE, 6, WHEELERS FOLD,  The Coach House, 6 Wheelers Fold,  0.79                    O   
     
                                                                                              New  \
     Postcode PriceAddress                    EpcAddress                         JaroDistance       
     WV69QH   142, CODSALL ROAD,              142, Codsall Road,                 1.00           N   
     WV108AB  49, CARISBROOKE ROAD,           49, Carisbrooke Road, Bushbury,    0.94           N   
     WV46BJ   56, GREENOCK CRESCENT,          56, Greenock Crescent,             1.00           N   
     WV112QQ  21, RYAN AVENUE,                21, Ryan Avenue,                   1.00           N   
     WV147AP  34, MARBURY DRIVE,              34, Marbury Drive,                 1.00           N   
     ...                                                                                       ..   
     WV22AW   116 - 126, STEELHOUSE LANE,     100, Steelhouse Lane,              0.75           N   
     WV108RP  55, PRIMROSE LANE,              55, Primrose Lane,                 1.00           N   
     WV106BA  12, THREE TUNS PARADE,          Flat, 12a, Three Tuns Parade,      0.76           N   
     WV38NA   201, CASTLECROFT ROAD,          201, Castlecroft Road,             1.00           N   
     WV11HN   COACH HOUSE, 6, WHEELERS FOLD,  The Coach House, 6 Wheelers Fold,  0.79           N   
     
                                                                                              Duration  \
     Postcode PriceAddress                    EpcAddress                         JaroDistance            
     WV69QH   142, CODSALL ROAD,              142, Codsall Road,                 1.00                F   
     WV108AB  49, CARISBROOKE ROAD,           49, Carisbrooke Road, Bushbury,    0.94                F   
     WV46BJ   56, GREENOCK CRESCENT,          56, Greenock Crescent,             1.00                L   
     WV112QQ  21, RYAN AVENUE,                21, Ryan Avenue,                   1.00                F   
     WV147AP  34, MARBURY DRIVE,              34, Marbury Drive,                 1.00                L   
     ...                                                                                           ...   
     WV22AW   116 - 126, STEELHOUSE LANE,     100, Steelhouse Lane,              0.75                F   
     WV108RP  55, PRIMROSE LANE,              55, Primrose Lane,                 1.00                F   
     WV106BA  12, THREE TUNS PARADE,          Flat, 12a, Three Tuns Parade,      0.76                F   
     WV38NA   201, CASTLECROFT ROAD,          201, Castlecroft Road,             1.00                F   
     WV11HN   COACH HOUSE, 6, WHEELERS FOLD,  The Coach House, 6 Wheelers Fold,  0.79                F   
     
                                                                                              CurrentEnergyRating  \
     Postcode PriceAddress                    EpcAddress                         JaroDistance                       
     WV69QH   142, CODSALL ROAD,              142, Codsall Road,                 1.00                           D   
     WV108AB  49, CARISBROOKE ROAD,           49, Carisbrooke Road, Bushbury,    0.94                           B   
     WV46BJ   56, GREENOCK CRESCENT,          56, Greenock Crescent,             1.00                           C   
     WV112QQ  21, RYAN AVENUE,                21, Ryan Avenue,                   1.00                           D   
     WV147AP  34, MARBURY DRIVE,              34, Marbury Drive,                 1.00                           C   
     ...                                                                                                      ...   
     WV22AW   116 - 126, STEELHOUSE LANE,     100, Steelhouse Lane,              0.75                           A   
     WV108RP  55, PRIMROSE LANE,              55, Primrose Lane,                 1.00                           C   
     WV106BA  12, THREE TUNS PARADE,          Flat, 12a, Three Tuns Parade,      0.76                           D   
     WV38NA   201, CASTLECROFT ROAD,          201, Castlecroft Road,             1.00                           D   
     WV11HN   COACH HOUSE, 6, WHEELERS FOLD,  The Coach House, 6 Wheelers Fold,  0.79                           D   
     
                                                                                              PotentialEnergyRating  \
     Postcode PriceAddress                    EpcAddress                         JaroDistance                         
     WV69QH   142, CODSALL ROAD,              142, Codsall Road,                 1.00                             C   
     WV108AB  49, CARISBROOKE ROAD,           49, Carisbrooke Road, Bushbury,    0.94                             B   
     WV46BJ   56, GREENOCK CRESCENT,          56, Greenock Crescent,             1.00                             C   
     WV112QQ  21, RYAN AVENUE,                21, Ryan Avenue,                   1.00                             C   
     WV147AP  34, MARBURY DRIVE,              34, Marbury Drive,                 1.00                             C   
     ...                                                                                                        ...   
     WV22AW   116 - 126, STEELHOUSE LANE,     100, Steelhouse Lane,              0.75                             A   
     WV108RP  55, PRIMROSE LANE,              55, Primrose Lane,                 1.00                             B   
     WV106BA  12, THREE TUNS PARADE,          Flat, 12a, Three Tuns Parade,      0.76                             C   
     WV38NA   201, CASTLECROFT ROAD,          201, Castlecroft Road,             1.00                             C   
     WV11HN   COACH HOUSE, 6, WHEELERS FOLD,  The Coach House, 6 Wheelers Fold,  0.79                             C   
     
                                                                                               CurrentEnergyEfficiency  \
     Postcode PriceAddress                    EpcAddress                         JaroDistance                            
     WV69QH   142, CODSALL ROAD,              142, Codsall Road,                 1.00                               55   
     WV108AB  49, CARISBROOKE ROAD,           49, Carisbrooke Road, Bushbury,    0.94                               81   
     WV46BJ   56, GREENOCK CRESCENT,          56, Greenock Crescent,             1.00                               78   
     WV112QQ  21, RYAN AVENUE,                21, Ryan Avenue,                   1.00                               62   
     WV147AP  34, MARBURY DRIVE,              34, Marbury Drive,                 1.00                               75   
     ...                                                                                                           ...   
     WV22AW   116 - 126, STEELHOUSE LANE,     100, Steelhouse Lane,              0.75                               96   
     WV108RP  55, PRIMROSE LANE,              55, Primrose Lane,                 1.00                               72   
     WV106BA  12, THREE TUNS PARADE,          Flat, 12a, Three Tuns Parade,      0.76                               57   
     WV38NA   201, CASTLECROFT ROAD,          201, Castlecroft Road,             1.00                               56   
     WV11HN   COACH HOUSE, 6, WHEELERS FOLD,  The Coach House, 6 Wheelers Fold,  0.79                               56   
     
                                                                                               PotentialEnergyEfficiency  \
     Postcode PriceAddress                    EpcAddress                         JaroDistance                              
     WV69QH   142, CODSALL ROAD,              142, Codsall Road,                 1.00                                 75   
     WV108AB  49, CARISBROOKE ROAD,           49, Carisbrooke Road, Bushbury,    0.94                                 85   
     WV46BJ   56, GREENOCK CRESCENT,          56, Greenock Crescent,             1.00                                 79   
     WV112QQ  21, RYAN AVENUE,                21, Ryan Avenue,                   1.00                                 79   
     WV147AP  34, MARBURY DRIVE,              34, Marbury Drive,                 1.00                                 75   
     ...                                                                                                             ...   
     WV22AW   116 - 126, STEELHOUSE LANE,     100, Steelhouse Lane,              0.75                                 97   
     WV108RP  55, PRIMROSE LANE,              55, Primrose Lane,                 1.00                                 85   
     WV106BA  12, THREE TUNS PARADE,          Flat, 12a, Three Tuns Parade,      0.76                                 72   
     WV38NA   201, CASTLECROFT ROAD,          201, Castlecroft Road,             1.00                                 75   
     WV11HN   COACH HOUSE, 6, WHEELERS FOLD,  The Coach House, 6 Wheelers Fold,  0.79                                 77   
     
                                                                                              GlazedArea  \
     Postcode PriceAddress                    EpcAddress                         JaroDistance              
     WV69QH   142, CODSALL ROAD,              142, Codsall Road,                 1.00             Normal   
     WV108AB  49, CARISBROOKE ROAD,           49, Carisbrooke Road, Bushbury,    0.94             Normal   
     WV46BJ   56, GREENOCK CRESCENT,          56, Greenock Crescent,             1.00             Normal   
     WV112QQ  21, RYAN AVENUE,                21, Ryan Avenue,                   1.00             Normal   
     WV147AP  34, MARBURY DRIVE,              34, Marbury Drive,                 1.00             Normal   
     ...                                                                                             ...   
     WV22AW   116 - 126, STEELHOUSE LANE,     100, Steelhouse Lane,              0.75           NO DATA!   
     WV108RP  55, PRIMROSE LANE,              55, Primrose Lane,                 1.00             Normal   
     WV106BA  12, THREE TUNS PARADE,          Flat, 12a, Three Tuns Parade,      0.76             Normal   
     WV38NA   201, CASTLECROFT ROAD,          201, Castlecroft Road,             1.00             Normal   
     WV11HN   COACH HOUSE, 6, WHEELERS FOLD,  The Coach House, 6 Wheelers Fold,  0.79             Normal   
     
                                                                                               HabitableRooms  \
     Postcode PriceAddress                    EpcAddress                         JaroDistance                   
     WV69QH   142, CODSALL ROAD,              142, Codsall Road,                 1.00                     8.0   
     WV108AB  49, CARISBROOKE ROAD,           49, Carisbrooke Road, Bushbury,    0.94                     5.0   
     WV46BJ   56, GREENOCK CRESCENT,          56, Greenock Crescent,             1.00                     3.0   
     WV112QQ  21, RYAN AVENUE,                21, Ryan Avenue,                   1.00                     5.0   
     WV147AP  34, MARBURY DRIVE,              34, Marbury Drive,                 1.00                     3.0   
     ...                                                                                                  ...   
     WV22AW   116 - 126, STEELHOUSE LANE,     100, Steelhouse Lane,              0.75                     NaN   
     WV108RP  55, PRIMROSE LANE,              55, Primrose Lane,                 1.00                     5.0   
     WV106BA  12, THREE TUNS PARADE,          Flat, 12a, Three Tuns Parade,      0.76                     3.0   
     WV38NA   201, CASTLECROFT ROAD,          201, Castlecroft Road,             1.00                     7.0   
     WV11HN   COACH HOUSE, 6, WHEELERS FOLD,  The Coach House, 6 Wheelers Fold,  0.79                     7.0   
     
                                                                                               HeatedRooms  \
     Postcode PriceAddress                    EpcAddress                         JaroDistance                
     WV69QH   142, CODSALL ROAD,              142, Codsall Road,                 1.00                  8.0   
     WV108AB  49, CARISBROOKE ROAD,           49, Carisbrooke Road, Bushbury,    0.94                  4.0   
     WV46BJ   56, GREENOCK CRESCENT,          56, Greenock Crescent,             1.00                  3.0   
     WV112QQ  21, RYAN AVENUE,                21, Ryan Avenue,                   1.00                  5.0   
     WV147AP  34, MARBURY DRIVE,              34, Marbury Drive,                 1.00                  3.0   
     ...                                                                                               ...   
     WV22AW   116 - 126, STEELHOUSE LANE,     100, Steelhouse Lane,              0.75                  NaN   
     WV108RP  55, PRIMROSE LANE,              55, Primrose Lane,                 1.00                  5.0   
     WV106BA  12, THREE TUNS PARADE,          Flat, 12a, Three Tuns Parade,      0.76                  3.0   
     WV38NA   201, CASTLECROFT ROAD,          201, Castlecroft Road,             1.00                  7.0   
     WV11HN   COACH HOUSE, 6, WHEELERS FOLD,  The Coach House, 6 Wheelers Fold,  0.79                  7.0   
     
                                                                                               ...  \
     Postcode PriceAddress                    EpcAddress                         JaroDistance  ...   
     WV69QH   142, CODSALL ROAD,              142, Codsall Road,                 1.00          ...   
     WV108AB  49, CARISBROOKE ROAD,           49, Carisbrooke Road, Bushbury,    0.94          ...   
     WV46BJ   56, GREENOCK CRESCENT,          56, Greenock Crescent,             1.00          ...   
     WV112QQ  21, RYAN AVENUE,                21, Ryan Avenue,                   1.00          ...   
     WV147AP  34, MARBURY DRIVE,              34, Marbury Drive,                 1.00          ...   
     ...                                                                                       ...   
     WV22AW   116 - 126, STEELHOUSE LANE,     100, Steelhouse Lane,              0.75          ...   
     WV108RP  55, PRIMROSE LANE,              55, Primrose Lane,                 1.00          ...   
     WV106BA  12, THREE TUNS PARADE,          Flat, 12a, Three Tuns Parade,      0.76          ...   
     WV38NA   201, CASTLECROFT ROAD,          201, Castlecroft Road,             1.00          ...   
     WV11HN   COACH HOUSE, 6, WHEELERS FOLD,  The Coach House, 6 Wheelers Fold,  0.79          ...   
     
                                                                                               TotalFloorArea  \
     Postcode PriceAddress                    EpcAddress                         JaroDistance                   
     WV69QH   142, CODSALL ROAD,              142, Codsall Road,                 1.00                   348.0   
     WV108AB  49, CARISBROOKE ROAD,           49, Carisbrooke Road, Bushbury,    0.94                    84.0   
     WV46BJ   56, GREENOCK CRESCENT,          56, Greenock Crescent,             1.00                    77.0   
     WV112QQ  21, RYAN AVENUE,                21, Ryan Avenue,                   1.00                    77.0   
     WV147AP  34, MARBURY DRIVE,              34, Marbury Drive,                 1.00                    56.5   
     ...                                                                                                  ...   
     WV22AW   116 - 126, STEELHOUSE LANE,     100, Steelhouse Lane,              0.75                   107.0   
     WV108RP  55, PRIMROSE LANE,              55, Primrose Lane,                 1.00                   102.0   
     WV106BA  12, THREE TUNS PARADE,          Flat, 12a, Three Tuns Parade,      0.76                    67.0   
     WV38NA   201, CASTLECROFT ROAD,          201, Castlecroft Road,             1.00                   229.0   
     WV11HN   COACH HOUSE, 6, WHEELERS FOLD,  The Coach House, 6 Wheelers Fold,  0.79                    97.0   
     
                                                                                               FloorLevel  \
     Postcode PriceAddress                    EpcAddress                         JaroDistance               
     WV69QH   142, CODSALL ROAD,              142, Codsall Road,                 1.00             NODATA!   
     WV108AB  49, CARISBROOKE ROAD,           49, Carisbrooke Road, Bushbury,    0.94             NODATA!   
     WV46BJ   56, GREENOCK CRESCENT,          56, Greenock Crescent,             1.00                 2nd   
     WV112QQ  21, RYAN AVENUE,                21, Ryan Avenue,                   1.00             NODATA!   
     WV147AP  34, MARBURY DRIVE,              34, Marbury Drive,                 1.00             NODATA!   
     ...                                                                                              ...   
     WV22AW   116 - 126, STEELHOUSE LANE,     100, Steelhouse Lane,              0.75            NO DATA!   
     WV108RP  55, PRIMROSE LANE,              55, Primrose Lane,                 1.00             NODATA!   
     WV106BA  12, THREE TUNS PARADE,          Flat, 12a, Three Tuns Parade,      0.76              Ground   
     WV38NA   201, CASTLECROFT ROAD,          201, Castlecroft Road,             1.00             NODATA!   
     WV11HN   COACH HOUSE, 6, WHEELERS FOLD,  The Coach House, 6 Wheelers Fold,  0.79             NODATA!   
     
                                                                                              PurchaseDate_year  \
     Postcode PriceAddress                    EpcAddress                         JaroDistance                     
     WV69QH   142, CODSALL ROAD,              142, Codsall Road,                 1.00                      2020   
     WV108AB  49, CARISBROOKE ROAD,           49, Carisbrooke Road, Bushbury,    0.94                      2020   
     WV46BJ   56, GREENOCK CRESCENT,          56, Greenock Crescent,             1.00                      2020   
     WV112QQ  21, RYAN AVENUE,                21, Ryan Avenue,                   1.00                      2020   
     WV147AP  34, MARBURY DRIVE,              34, Marbury Drive,                 1.00                      2020   
     ...                                                                                                    ...   
     WV22AW   116 - 126, STEELHOUSE LANE,     100, Steelhouse Lane,              0.75                      2020   
     WV108RP  55, PRIMROSE LANE,              55, Primrose Lane,                 1.00                      2020   
     WV106BA  12, THREE TUNS PARADE,          Flat, 12a, Three Tuns Parade,      0.76                      2020   
     WV38NA   201, CASTLECROFT ROAD,          201, Castlecroft Road,             1.00                      2020   
     WV11HN   COACH HOUSE, 6, WHEELERS FOLD,  The Coach House, 6 Wheelers Fold,  0.79                      2020   
     
                                                                                               PurchaseDate_month  \
     Postcode PriceAddress                    EpcAddress                         JaroDistance                       
     WV69QH   142, CODSALL ROAD,              142, Codsall Road,                 1.00                         May   
     WV108AB  49, CARISBROOKE ROAD,           49, Carisbrooke Road, Bushbury,    0.94                    February   
     WV46BJ   56, GREENOCK CRESCENT,          56, Greenock Crescent,             1.00                       March   
     WV112QQ  21, RYAN AVENUE,                21, Ryan Avenue,                   1.00                         May   
     WV147AP  34, MARBURY DRIVE,              34, Marbury Drive,                 1.00                       March   
     ...                                                                                                      ...   
     WV22AW   116 - 126, STEELHOUSE LANE,     100, Steelhouse Lane,              0.75                         May   
     WV108RP  55, PRIMROSE LANE,              55, Primrose Lane,                 1.00                    February   
     WV106BA  12, THREE TUNS PARADE,          Flat, 12a, Three Tuns Parade,      0.76                        June   
     WV38NA   201, CASTLECROFT ROAD,          201, Castlecroft Road,             1.00                       April   
     WV11HN   COACH HOUSE, 6, WHEELERS FOLD,  The Coach House, 6 Wheelers Fold,  0.79                        June   
     
                                                                                              PurchaseDate_day_of_week  \
     Postcode PriceAddress                    EpcAddress                         JaroDistance                            
     WV69QH   142, CODSALL ROAD,              142, Codsall Road,                 1.00                        Wednesday   
     WV108AB  49, CARISBROOKE ROAD,           49, Carisbrooke Road, Bushbury,    0.94                           Friday   
     WV46BJ   56, GREENOCK CRESCENT,          56, Greenock Crescent,             1.00                        Wednesday   
     WV112QQ  21, RYAN AVENUE,                21, Ryan Avenue,                   1.00                           Monday   
     WV147AP  34, MARBURY DRIVE,              34, Marbury Drive,                 1.00                           Friday   
     ...                                                                                                           ...   
     WV22AW   116 - 126, STEELHOUSE LANE,     100, Steelhouse Lane,              0.75                          Tuesday   
     WV108RP  55, PRIMROSE LANE,              55, Primrose Lane,                 1.00                           Friday   
     WV106BA  12, THREE TUNS PARADE,          Flat, 12a, Three Tuns Parade,      0.76                        Wednesday   
     WV38NA   201, CASTLECROFT ROAD,          201, Castlecroft Road,             1.00                           Monday   
     WV11HN   COACH HOUSE, 6, WHEELERS FOLD,  The Coach House, 6 Wheelers Fold,  0.79                           Monday   
     
                                                                                              PurchaseDate_hour  \
     Postcode PriceAddress                    EpcAddress                         JaroDistance                     
     WV69QH   142, CODSALL ROAD,              142, Codsall Road,                 1.00                         0   
     WV108AB  49, CARISBROOKE ROAD,           49, Carisbrooke Road, Bushbury,    0.94                         0   
     WV46BJ   56, GREENOCK CRESCENT,          56, Greenock Crescent,             1.00                         0   
     WV112QQ  21, RYAN AVENUE,                21, Ryan Avenue,                   1.00                         0   
     WV147AP  34, MARBURY DRIVE,              34, Marbury Drive,                 1.00                         0   
     ...                                                                                                    ...   
     WV22AW   116 - 126, STEELHOUSE LANE,     100, Steelhouse Lane,              0.75                         0   
     WV108RP  55, PRIMROSE LANE,              55, Primrose Lane,                 1.00                         0   
     WV106BA  12, THREE TUNS PARADE,          Flat, 12a, Three Tuns Parade,      0.76                         0   
     WV38NA   201, CASTLECROFT ROAD,          201, Castlecroft Road,             1.00                         0   
     WV11HN   COACH HOUSE, 6, WHEELERS FOLD,  The Coach House, 6 Wheelers Fold,  0.79                         0   
     
                                                                                               EpcInspectionDate_year  \
     Postcode PriceAddress                    EpcAddress                         JaroDistance                           
     WV69QH   142, CODSALL ROAD,              142, Codsall Road,                 1.00                            2013   
     WV108AB  49, CARISBROOKE ROAD,           49, Carisbrooke Road, Bushbury,    0.94                            2016   
     WV46BJ   56, GREENOCK CRESCENT,          56, Greenock Crescent,             1.00                            2012   
     WV112QQ  21, RYAN AVENUE,                21, Ryan Avenue,                   1.00                            2019   
     WV147AP  34, MARBURY DRIVE,              34, Marbury Drive,                 1.00                            2011   
     ...                                                                                                          ...   
     WV22AW   116 - 126, STEELHOUSE LANE,     100, Steelhouse Lane,              0.75                            2020   
     WV108RP  55, PRIMROSE LANE,              55, Primrose Lane,                 1.00                            2019   
     WV106BA  12, THREE TUNS PARADE,          Flat, 12a, Three Tuns Parade,      0.76                            2019   
     WV38NA   201, CASTLECROFT ROAD,          201, Castlecroft Road,             1.00                            2014   
     WV11HN   COACH HOUSE, 6, WHEELERS FOLD,  The Coach House, 6 Wheelers Fold,  0.79                            2020   
     
                                                                                               EpcInspectionDate_month  \
     Postcode PriceAddress                    EpcAddress                         JaroDistance                            
     WV69QH   142, CODSALL ROAD,              142, Codsall Road,                 1.00                             June   
     WV108AB  49, CARISBROOKE ROAD,           49, Carisbrooke Road, Bushbury,    0.94                             June   
     WV46BJ   56, GREENOCK CRESCENT,          56, Greenock Crescent,             1.00                              May   
     WV112QQ  21, RYAN AVENUE,                21, Ryan Avenue,                   1.00                          October   
     WV147AP  34, MARBURY DRIVE,              34, Marbury Drive,                 1.00                        September   
     ...                                                                                                           ...   
     WV22AW   116 - 126, STEELHOUSE LANE,     100, Steelhouse Lane,              0.75                          January   
     WV108RP  55, PRIMROSE LANE,              55, Primrose Lane,                 1.00                        September   
     WV106BA  12, THREE TUNS PARADE,          Flat, 12a, Three Tuns Parade,      0.76                            April   
     WV38NA   201, CASTLECROFT ROAD,          201, Castlecroft Road,             1.00                              May   
     WV11HN   COACH HOUSE, 6, WHEELERS FOLD,  The Coach House, 6 Wheelers Fold,  0.79                          January   
     
                                                                                              EpcInspectionDate_day_of_week  \
     Postcode PriceAddress                    EpcAddress                         JaroDistance                                 
     WV69QH   142, CODSALL ROAD,              142, Codsall Road,                 1.00                                Monday   
     WV108AB  49, CARISBROOKE ROAD,           49, Carisbrooke Road, Bushbury,    0.94                                Friday   
     WV46BJ   56, GREENOCK CRESCENT,          56, Greenock Crescent,             1.00                             Wednesday   
     WV112QQ  21, RYAN AVENUE,                21, Ryan Avenue,                   1.00                             Wednesday   
     WV147AP  34, MARBURY DRIVE,              34, Marbury Drive,                 1.00                              Thursday   
     ...                                                                                                                ...   
     WV22AW   116 - 126, STEELHOUSE LANE,     100, Steelhouse Lane,              0.75                                Friday   
     WV108RP  55, PRIMROSE LANE,              55, Primrose Lane,                 1.00                                Monday   
     WV106BA  12, THREE TUNS PARADE,          Flat, 12a, Three Tuns Parade,      0.76                              Thursday   
     WV38NA   201, CASTLECROFT ROAD,          201, Castlecroft Road,             1.00                              Thursday   
     WV11HN   COACH HOUSE, 6, WHEELERS FOLD,  The Coach House, 6 Wheelers Fold,  0.79                               Tuesday   
     
                                                                                              EpcInspectionDate_hour  
     Postcode PriceAddress                    EpcAddress                         JaroDistance                         
     WV69QH   142, CODSALL ROAD,              142, Codsall Road,                 1.00                              0  
     WV108AB  49, CARISBROOKE ROAD,           49, Carisbrooke Road, Bushbury,    0.94                              0  
     WV46BJ   56, GREENOCK CRESCENT,          56, Greenock Crescent,             1.00                              0  
     WV112QQ  21, RYAN AVENUE,                21, Ryan Avenue,                   1.00                              0  
     WV147AP  34, MARBURY DRIVE,              34, Marbury Drive,                 1.00                              0  
     ...                                                                                                         ...  
     WV22AW   116 - 126, STEELHOUSE LANE,     100, Steelhouse Lane,              0.75                              0  
     WV108RP  55, PRIMROSE LANE,              55, Primrose Lane,                 1.00                              0  
     WV106BA  12, THREE TUNS PARADE,          Flat, 12a, Three Tuns Parade,      0.76                              0  
     WV38NA   201, CASTLECROFT ROAD,          201, Castlecroft Road,             1.00                              0  
     WV11HN   COACH HOUSE, 6, WHEELERS FOLD,  The Coach House, 6 Wheelers Fold,  0.79                              0  
     
     [1121 rows x 21 columns],
     Postcode  PriceAddress                     EpcAddress                          JaroDistance
     WV69QH    142, CODSALL ROAD,               142, Codsall Road,                  1.00            437500
     WV108AB   49, CARISBROOKE ROAD,            49, Carisbrooke Road, Bushbury,     0.94            140000
     WV46BJ    56, GREENOCK CRESCENT,           56, Greenock Crescent,              1.00            102000
     WV112QQ   21, RYAN AVENUE,                 21, Ryan Avenue,                    1.00            141000
     WV147AP   34, MARBURY DRIVE,               34, Marbury Drive,                  1.00            122000
                                                                                                     ...  
     WV22AW    116 - 126, STEELHOUSE LANE,      100, Steelhouse Lane,               0.75            280000
     WV108RP   55, PRIMROSE LANE,               55, Primrose Lane,                  1.00            162000
     WV106BA   12, THREE TUNS PARADE,           Flat, 12a, Three Tuns Parade,       0.76            150000
     WV38NA    201, CASTLECROFT ROAD,           201, Castlecroft Road,              1.00            470000
     WV11HN    COACH HOUSE, 6, WHEELERS FOLD,   The Coach House, 6 Wheelers Fold,   0.79            220000
     Name: Price, Length: 1121, dtype: int64)



# Find the relevent "real estate" related features


```python

X

```




<div>
<style scoped>
    .dataframe tbody tr th:only-of-type {
        vertical-align: middle;
    }

    .dataframe tbody tr th {
        vertical-align: top;
    }

    .dataframe thead th {
        text-align: right;
    }
</style>
<table border="1" class="dataframe">
  <thead>
    <tr style="text-align: right;">
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th>PropertyType</th>
      <th>New</th>
      <th>Duration</th>
      <th>CurrentEnergyRating</th>
      <th>PotentialEnergyRating</th>
      <th>CurrentEnergyEfficiency</th>
      <th>PotentialEnergyEfficiency</th>
      <th>GlazedArea</th>
      <th>HabitableRooms</th>
      <th>HeatedRooms</th>
      <th>...</th>
      <th>TotalFloorArea</th>
      <th>FloorLevel</th>
      <th>PurchaseDate_year</th>
      <th>PurchaseDate_month</th>
      <th>PurchaseDate_day_of_week</th>
      <th>PurchaseDate_hour</th>
      <th>EpcInspectionDate_year</th>
      <th>EpcInspectionDate_month</th>
      <th>EpcInspectionDate_day_of_week</th>
      <th>EpcInspectionDate_hour</th>
    </tr>
    <tr>
      <th>Postcode</th>
      <th>PriceAddress</th>
      <th>EpcAddress</th>
      <th>JaroDistance</th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
    </tr>
  </thead>
  <tbody>
    <tr>
      <th>WV69QH</th>
      <th>142, CODSALL ROAD,</th>
      <th>142, Codsall Road,</th>
      <th>1.00</th>
      <td>D</td>
      <td>N</td>
      <td>F</td>
      <td>D</td>
      <td>C</td>
      <td>55</td>
      <td>75</td>
      <td>Normal</td>
      <td>8.0</td>
      <td>8.0</td>
      <td>...</td>
      <td>348.0</td>
      <td>NODATA!</td>
      <td>2020</td>
      <td>May</td>
      <td>Wednesday</td>
      <td>0</td>
      <td>2013</td>
      <td>June</td>
      <td>Monday</td>
      <td>0</td>
    </tr>
    <tr>
      <th>WV108AB</th>
      <th>49, CARISBROOKE ROAD,</th>
      <th>49, Carisbrooke Road, Bushbury,</th>
      <th>0.94</th>
      <td>S</td>
      <td>N</td>
      <td>F</td>
      <td>B</td>
      <td>B</td>
      <td>81</td>
      <td>85</td>
      <td>Normal</td>
      <td>5.0</td>
      <td>4.0</td>
      <td>...</td>
      <td>84.0</td>
      <td>NODATA!</td>
      <td>2020</td>
      <td>February</td>
      <td>Friday</td>
      <td>0</td>
      <td>2016</td>
      <td>June</td>
      <td>Friday</td>
      <td>0</td>
    </tr>
    <tr>
      <th>WV46BJ</th>
      <th>56, GREENOCK CRESCENT,</th>
      <th>56, Greenock Crescent,</th>
      <th>1.00</th>
      <td>F</td>
      <td>N</td>
      <td>L</td>
      <td>C</td>
      <td>C</td>
      <td>78</td>
      <td>79</td>
      <td>Normal</td>
      <td>3.0</td>
      <td>3.0</td>
      <td>...</td>
      <td>77.0</td>
      <td>2nd</td>
      <td>2020</td>
      <td>March</td>
      <td>Wednesday</td>
      <td>0</td>
      <td>2012</td>
      <td>May</td>
      <td>Wednesday</td>
      <td>0</td>
    </tr>
    <tr>
      <th>WV112QQ</th>
      <th>21, RYAN AVENUE,</th>
      <th>21, Ryan Avenue,</th>
      <th>1.00</th>
      <td>S</td>
      <td>N</td>
      <td>F</td>
      <td>D</td>
      <td>C</td>
      <td>62</td>
      <td>79</td>
      <td>Normal</td>
      <td>5.0</td>
      <td>5.0</td>
      <td>...</td>
      <td>77.0</td>
      <td>NODATA!</td>
      <td>2020</td>
      <td>May</td>
      <td>Monday</td>
      <td>0</td>
      <td>2019</td>
      <td>October</td>
      <td>Wednesday</td>
      <td>0</td>
    </tr>
    <tr>
      <th>WV147AP</th>
      <th>34, MARBURY DRIVE,</th>
      <th>34, Marbury Drive,</th>
      <th>1.00</th>
      <td>S</td>
      <td>N</td>
      <td>L</td>
      <td>C</td>
      <td>C</td>
      <td>75</td>
      <td>75</td>
      <td>Normal</td>
      <td>3.0</td>
      <td>3.0</td>
      <td>...</td>
      <td>56.5</td>
      <td>NODATA!</td>
      <td>2020</td>
      <td>March</td>
      <td>Friday</td>
      <td>0</td>
      <td>2011</td>
      <td>September</td>
      <td>Thursday</td>
      <td>0</td>
    </tr>
    <tr>
      <th>...</th>
      <th>...</th>
      <th>...</th>
      <th>...</th>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
    </tr>
    <tr>
      <th>WV22AW</th>
      <th>116 - 126, STEELHOUSE LANE,</th>
      <th>100, Steelhouse Lane,</th>
      <th>0.75</th>
      <td>O</td>
      <td>N</td>
      <td>F</td>
      <td>A</td>
      <td>A</td>
      <td>96</td>
      <td>97</td>
      <td>NO DATA!</td>
      <td>NaN</td>
      <td>NaN</td>
      <td>...</td>
      <td>107.0</td>
      <td>NO DATA!</td>
      <td>2020</td>
      <td>May</td>
      <td>Tuesday</td>
      <td>0</td>
      <td>2020</td>
      <td>January</td>
      <td>Friday</td>
      <td>0</td>
    </tr>
    <tr>
      <th>WV108RP</th>
      <th>55, PRIMROSE LANE,</th>
      <th>55, Primrose Lane,</th>
      <th>1.00</th>
      <td>S</td>
      <td>N</td>
      <td>F</td>
      <td>C</td>
      <td>B</td>
      <td>72</td>
      <td>85</td>
      <td>Normal</td>
      <td>5.0</td>
      <td>5.0</td>
      <td>...</td>
      <td>102.0</td>
      <td>NODATA!</td>
      <td>2020</td>
      <td>February</td>
      <td>Friday</td>
      <td>0</td>
      <td>2019</td>
      <td>September</td>
      <td>Monday</td>
      <td>0</td>
    </tr>
    <tr>
      <th>WV106BA</th>
      <th>12, THREE TUNS PARADE,</th>
      <th>Flat, 12a, Three Tuns Parade,</th>
      <th>0.76</th>
      <td>O</td>
      <td>N</td>
      <td>F</td>
      <td>D</td>
      <td>C</td>
      <td>57</td>
      <td>72</td>
      <td>Normal</td>
      <td>3.0</td>
      <td>3.0</td>
      <td>...</td>
      <td>67.0</td>
      <td>Ground</td>
      <td>2020</td>
      <td>June</td>
      <td>Wednesday</td>
      <td>0</td>
      <td>2019</td>
      <td>April</td>
      <td>Thursday</td>
      <td>0</td>
    </tr>
    <tr>
      <th>WV38NA</th>
      <th>201, CASTLECROFT ROAD,</th>
      <th>201, Castlecroft Road,</th>
      <th>1.00</th>
      <td>D</td>
      <td>N</td>
      <td>F</td>
      <td>D</td>
      <td>C</td>
      <td>56</td>
      <td>75</td>
      <td>Normal</td>
      <td>7.0</td>
      <td>7.0</td>
      <td>...</td>
      <td>229.0</td>
      <td>NODATA!</td>
      <td>2020</td>
      <td>April</td>
      <td>Monday</td>
      <td>0</td>
      <td>2014</td>
      <td>May</td>
      <td>Thursday</td>
      <td>0</td>
    </tr>
    <tr>
      <th>WV11HN</th>
      <th>COACH HOUSE, 6, WHEELERS FOLD,</th>
      <th>The Coach House, 6 Wheelers Fold,</th>
      <th>0.79</th>
      <td>O</td>
      <td>N</td>
      <td>F</td>
      <td>D</td>
      <td>C</td>
      <td>56</td>
      <td>77</td>
      <td>Normal</td>
      <td>7.0</td>
      <td>7.0</td>
      <td>...</td>
      <td>97.0</td>
      <td>NODATA!</td>
      <td>2020</td>
      <td>June</td>
      <td>Monday</td>
      <td>0</td>
      <td>2020</td>
      <td>January</td>
      <td>Tuesday</td>
      <td>0</td>
    </tr>
  </tbody>
</table>
<p>1121 rows × 21 columns</p>
</div>



# Construct the Price MicroService Pipeline


```python
from evalml.pipelines import RegressionPipeline

class PriceMicroservicePipeline(RegressionPipeline):
    component_graph = ['Imputer', 'DateTime Featurization Component', 'One Hot Encoder', 'XGBoost Regressor']
    custom_name = 'Price Microservice Pipeline'
    parameters = {'Imputer': {'categorical_impute_strategy': 'most_frequent',
                                         'numeric_impute_strategy': 'mean',
                                         'categorical_fill_value': None,
                                         'numeric_fill_value': None
                 },
                 'DateTime Featurization Component':{},
                 'One Hot Encoder': { 'top_n': 10,
                                         'categories': None,
                                         'drop': None,
                                         'handle_unknown': 'ignore',
                                         'handle_missing': 'error'
                 },
                 'XGBoost Regressor':{'eta': 0.1,
                                         'max_depth': 6,
                                         'min_child_weight': 1,
                                         'n_estimators': 100
             }
        }

price_microservice_pipeline = PriceMicroservicePipeline({})
price_microservice_pipeline
```




    PriceMicroservicePipeline(parameters={'Imputer':{'categorical_impute_strategy': 'most_frequent', 'numeric_impute_strategy': 'mean', 'categorical_fill_value': None, 'numeric_fill_value': None}, 'DateTime Featurization Component':{}, 'One Hot Encoder':{'top_n': 10, 'categories': None, 'drop': None, 'handle_unknown': 'ignore', 'handle_missing': 'error'}, 'XGBoost Regressor':{'eta': 0.1, 'max_depth': 6, 'min_child_weight': 1, 'n_estimators': 100},})




```python
X_train, X_holdout, y_train, y_holdout = evalml.preprocessing.split_data(X, y, test_size=0.2, random_state=0, regression=True)
X_train.shape, X_holdout.shape

```




    ((896, 21), (225, 21))




```python
price_microservice_pipeline.fit(X_train, y_train)
```




    PriceMicroservicePipeline(parameters={'Imputer':{'categorical_impute_strategy': 'most_frequent', 'numeric_impute_strategy': 'mean', 'categorical_fill_value': None, 'numeric_fill_value': None}, 'DateTime Featurization Component':{}, 'One Hot Encoder':{'top_n': 10, 'categories': None, 'drop': None, 'handle_unknown': 'ignore', 'handle_missing': 'error'}, 'XGBoost Regressor':{'eta': 0.1, 'max_depth': 6, 'min_child_weight': 1, 'n_estimators': 100},})




```python
price_microservice_pipeline.graph_feature_importance()
```


<div>                            <div id="36de5a03-1b95-40ac-be0e-7f479d74e4db" class="plotly-graph-div" style="height:800px; width:100%;"></div>            <script type="text/javascript">                require(["plotly"], function(Plotly) {                    window.PLOTLYENV=window.PLOTLYENV || {};                                    if (document.getElementById("36de5a03-1b95-40ac-be0e-7f479d74e4db")) {                    Plotly.newPlot(                        "36de5a03-1b95-40ac-be0e-7f479d74e4db",                        [{"orientation": "h", "type": "bar", "x": [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0006102336337789893, 0.0009818108519539237, 0.0011318164179101586, 0.0011416790075600147, 0.0011898907832801342, 0.0015132407424971461, 0.0015840139240026474, 0.0016517146723344922, 0.0017530146287754178, 0.0018164593493565917, 0.0019543820526450872, 0.0023633758537471294, 0.0023924645502120256, 0.002537153661251068, 0.002570198616012931, 0.00262922840192914, 0.0027051030192524195, 0.002760941395536065, 0.002835185034200549, 0.0028888641390949488, 0.0029965585563331842, 0.0030139482114464045, 0.003167911898344755, 0.0033550893422216177, 0.003498314879834652, 0.00355689343996346, 0.003639474045485258, 0.0036741644144058228, 0.0039030902553349733, 0.003904535435140133, 0.003911143634468317, 0.004729517735540867, 0.004948018118739128, 0.004951258189976215, 0.005034882575273514, 0.00518372654914856, 0.00523992208763957, 0.0052411057986319065, 0.005263738799840212, 0.005706917028874159, 0.005736066959798336, 0.005797139368951321, 0.005819236859679222, 0.005952636245638132, 0.0059668985195457935, 0.00632764957845211, 0.006643223110586405, 0.007537259720265865, 0.0077886213548481464, 0.008164912462234497, 0.008406592532992363, 0.009327258914709091, 0.009803230874240398, 0.021110717207193375, 0.022414401173591614, 0.023917246609926224, 0.03085285983979702, 0.030947834253311157, 0.1497509628534317, 0.25091877579689026, 0.2528853714466095], "y": ["PurchaseDate_day_of_week_Sunday", "PurchaseDate_month_August", "FloorLevel_NO DATA!", "FloorLevel_5th", "FloorLevel_3rd", "PotentialEnergyRating_F", "PotentialEnergyRating_E", "PotentialEnergyRating_A", "CurrentEnergyRating_G", "CurrentEnergyRating_B", "CurrentEnergyRating_A", "Duration_L", "New_Y", "New_N", "EpcInspectionDate_hour", "PurchaseDate_hour", "PurchaseDate_year", "FloorLevel_top floor", "PurchaseDate_day_of_week_Saturday", "GlazedArea_Much More Than Typical", "CurrentEnergyRating_D", "FloorLevel_2nd", "EpcInspectionDate_month_September", "PurchaseDate_month_May", "PurchaseDate_day_of_week_Friday", "EpcInspectionDate_month_May", "EpcInspectionDate_month_April", "EpcInspectionDate_month_August", "FloorLevel_NODATA!", "GlazedArea_More Than Typical", "EpcInspectionDate_day_of_week_Monday", "PurchaseDate_day_of_week_Wednesday", "PropertyType_T", "EpcInspectionDate_month_October", "EpcInspectionDate_day_of_week_Wednesday", "PurchaseDate_month_June", "EpcInspectionDate_day_of_week_Sunday", "PurchaseDate_month_February", "PurchaseDate_day_of_week_Monday", "EpcInspectionDate_day_of_week_Tuesday", "PurchaseDate_month_April", "PurchaseDate_month_January", "PotentialEnergyEfficiency", "EpcInspectionDate_day_of_week_Friday", "PotentialEnergyRating_D", "EpcInspectionDate_month_June", "FloorLevel_ground floor", "CurrentEnergyRating_F", "PurchaseDate_month_July", "HabitableRooms", "EpcInspectionDate_day_of_week_Saturday", "EpcInspectionDate_month_January", "GlazedArea_NO DATA!", "EpcInspectionDate_year", "EpcInspectionDate_month_February", "PurchaseDate_day_of_week_Thursday", "PotentialEnergyRating_B", "EpcInspectionDate_day_of_week_Thursday", "EpcInspectionDate_month_November", "Duration_F", "HeatedRooms", "FloorLevel_mid floor", "CurrentEnergyRating_E", "PotentialEnergyRating_C", "PropertyType_F", "PurchaseDate_day_of_week_Tuesday", "GlazedArea_Normal", "CurrentEnergyEfficiency", "EpcInspectionDate_month_March", "FloorLevel_1st", "PropertyType_S", "TotalFloorArea", "PurchaseDate_month_March", "FloorLevel_Ground", "FlatStoreyCount", "PropertyType_O", "PropertyType_D", "CurrentEnergyRating_C"]}],                        {"height": 800, "template": {"data": {"bar": [{"error_x": {"color": "#2a3f5f"}, "error_y": {"color": "#2a3f5f"}, "marker": {"line": {"color": "#E5ECF6", "width": 0.5}}, "type": "bar"}], "barpolar": [{"marker": {"line": {"color": "#E5ECF6", "width": 0.5}}, "type": "barpolar"}], "carpet": [{"aaxis": {"endlinecolor": "#2a3f5f", "gridcolor": "white", "linecolor": "white", "minorgridcolor": "white", "startlinecolor": "#2a3f5f"}, "baxis": {"endlinecolor": "#2a3f5f", "gridcolor": "white", "linecolor": "white", "minorgridcolor": "white", "startlinecolor": "#2a3f5f"}, "type": "carpet"}], "choropleth": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "type": "choropleth"}], "contour": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "colorscale": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]], "type": "contour"}], "contourcarpet": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "type": "contourcarpet"}], "heatmap": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "colorscale": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]], "type": "heatmap"}], "heatmapgl": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "colorscale": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]], "type": "heatmapgl"}], "histogram": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "histogram"}], "histogram2d": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "colorscale": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]], "type": "histogram2d"}], "histogram2dcontour": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "colorscale": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]], "type": "histogram2dcontour"}], "mesh3d": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "type": "mesh3d"}], "parcoords": [{"line": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "parcoords"}], "pie": [{"automargin": true, "type": "pie"}], "scatter": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scatter"}], "scatter3d": [{"line": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scatter3d"}], "scattercarpet": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scattercarpet"}], "scattergeo": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scattergeo"}], "scattergl": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scattergl"}], "scattermapbox": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scattermapbox"}], "scatterpolar": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scatterpolar"}], "scatterpolargl": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scatterpolargl"}], "scatterternary": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scatterternary"}], "surface": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "colorscale": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]], "type": "surface"}], "table": [{"cells": {"fill": {"color": "#EBF0F8"}, "line": {"color": "white"}}, "header": {"fill": {"color": "#C8D4E3"}, "line": {"color": "white"}}, "type": "table"}]}, "layout": {"annotationdefaults": {"arrowcolor": "#2a3f5f", "arrowhead": 0, "arrowwidth": 1}, "coloraxis": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "colorscale": {"diverging": [[0, "#8e0152"], [0.1, "#c51b7d"], [0.2, "#de77ae"], [0.3, "#f1b6da"], [0.4, "#fde0ef"], [0.5, "#f7f7f7"], [0.6, "#e6f5d0"], [0.7, "#b8e186"], [0.8, "#7fbc41"], [0.9, "#4d9221"], [1, "#276419"]], "sequential": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]], "sequentialminus": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]]}, "colorway": ["#636efa", "#EF553B", "#00cc96", "#ab63fa", "#FFA15A", "#19d3f3", "#FF6692", "#B6E880", "#FF97FF", "#FECB52"], "font": {"color": "#2a3f5f"}, "geo": {"bgcolor": "white", "lakecolor": "white", "landcolor": "#E5ECF6", "showlakes": true, "showland": true, "subunitcolor": "white"}, "hoverlabel": {"align": "left"}, "hovermode": "closest", "mapbox": {"style": "light"}, "paper_bgcolor": "white", "plot_bgcolor": "#E5ECF6", "polar": {"angularaxis": {"gridcolor": "white", "linecolor": "white", "ticks": ""}, "bgcolor": "#E5ECF6", "radialaxis": {"gridcolor": "white", "linecolor": "white", "ticks": ""}}, "scene": {"xaxis": {"backgroundcolor": "#E5ECF6", "gridcolor": "white", "gridwidth": 2, "linecolor": "white", "showbackground": true, "ticks": "", "zerolinecolor": "white"}, "yaxis": {"backgroundcolor": "#E5ECF6", "gridcolor": "white", "gridwidth": 2, "linecolor": "white", "showbackground": true, "ticks": "", "zerolinecolor": "white"}, "zaxis": {"backgroundcolor": "#E5ECF6", "gridcolor": "white", "gridwidth": 2, "linecolor": "white", "showbackground": true, "ticks": "", "zerolinecolor": "white"}}, "shapedefaults": {"line": {"color": "#2a3f5f"}}, "ternary": {"aaxis": {"gridcolor": "white", "linecolor": "white", "ticks": ""}, "baxis": {"gridcolor": "white", "linecolor": "white", "ticks": ""}, "bgcolor": "#E5ECF6", "caxis": {"gridcolor": "white", "linecolor": "white", "ticks": ""}}, "title": {"x": 0.05}, "xaxis": {"automargin": true, "gridcolor": "white", "linecolor": "white", "ticks": "", "title": {"standoff": 15}, "zerolinecolor": "white", "zerolinewidth": 2}, "yaxis": {"automargin": true, "gridcolor": "white", "linecolor": "white", "ticks": "", "title": {"standoff": 15}, "zerolinecolor": "white", "zerolinewidth": 2}}}, "title": {"text": "Feature Importance<br><sub>May display fewer features due to feature selection</sub>"}, "xaxis": {"title": {"text": "Feature Importance"}}, "yaxis": {"title": {"text": "Feature"}, "type": "category"}},                        {"responsive": true}                    ).then(function(){

var gd = document.getElementById('36de5a03-1b95-40ac-be0e-7f479d74e4db');
var x = new MutationObserver(function (mutations, observer) {{
        var display = window.getComputedStyle(gd).display;
        if (!display || display === 'none') {{
            console.log([gd, 'removed!']);
            Plotly.purge(gd);
            observer.disconnect();
        }}
}});

// Listen for the removal of the full notebook cells
var notebookContainer = gd.closest('#notebook-container');
if (notebookContainer) {{
    x.observe(notebookContainer, {childList: true});
}}

// Listen for the clearing of the current output cell
var outputEl = gd.closest('.output');
if (outputEl) {{
    x.observe(outputEl, {childList: true});
}}

                        })                };                });            </script>        </div>



```python
from evalml.model_understanding.graphs import graph_prediction_vs_actual

y_pred = price_microservice_pipeline.predict(X_holdout)
graph_prediction_vs_actual(y_holdout, y_pred, outlier_threshold=50)
```


<div>                            <div id="42696be3-db95-47c8-afaa-7fe0ec822980" class="plotly-graph-div" style="height:525px; width:100%;"></div>            <script type="text/javascript">                require(["plotly"], function(Plotly) {                    window.PLOTLYENV=window.PLOTLYENV || {};                                    if (document.getElementById("42696be3-db95-47c8-afaa-7fe0ec822980")) {                    Plotly.newPlot(                        "42696be3-db95-47c8-afaa-7fe0ec822980",                        [{"line": {"color": "grey"}, "name": "y = x line", "type": "scatter", "x": [8275.0, 894331.8625], "y": [8275.0, 894331.8625]}, {"marker": {"color": "#ffff00"}, "mode": "markers", "name": ">= outlier_threshold", "type": "scatter", "x": [385419.71875, 337095.6875, 123666.671875, 121125.328125, 296105.25, 126513.28125, 121328.5, 209080.46875, 291594.15625, 201890.421875, 127222.6796875, 119042.96875, 99921.859375, 81930.546875, 133443.484375, 268091.6875, 152893.390625, 158559.25, 140070.46875, 146105.609375, 143789.140625, 216119.09375, 119623.46875, 161370.28125, 170374.828125, 171383.3125, 151853.40625, 231615.75, 327697.40625, 196263.390625, 149531.0625, 130778.734375, 193997.828125, 144746.96875, 204954.59375, 170362.390625, 118878.421875, 125774.484375, 170109.828125, 165155.046875, 290193.5, 136124.0, 119308.2421875, 184201.875, 107184.421875, 152617.671875, 147998.921875, 212679.0625, 252458.171875, 135006.625, 101543.59375, 173674.65625, 388333.28125, 285616.25, 220063.8125, 124750.953125, 121992.5859375, 155104.203125, 224317.609375, 105484.5078125, 173370.40625, 258644.265625, 75800.9765625, 237761.578125, 123170.0625, 154349.59375, 227405.328125, 103392.2265625, 152814.53125, 138455.8125, 116672.6328125, 171479.9375, 129924.890625, 321584.0625, 119442.28125, 254195.328125, 102538.1640625, 141998.28125, 154699.28125, 153892.578125, 129514.15625, 100992.5, 203365.59375, 119006.65625, 133857.578125, 346109.28125, 122245.5, 211202.75, 127565.15625, 184552.078125, 156404.09375, 161627.953125, 291293.375, 320352.9375, 140838.296875, 145278.484375, 145722.46875, 226549.640625, 129083.5390625, 258737.78125, 205858.875, 218151.53125, 161665.71875, 156863.96875, 218419.6875, 132448.03125, 233508.859375, 171366.765625, 201068.296875, 189981.25, 243972.484375, 119886.90625, 151059.453125, 149187.34375, 311353.4375, 132937.375, 155006.578125, 216948.359375, 161203.015625, 241812.5, 62442.98046875, 116689.984375, 112398.28125, 854341.4375, 109499.046875, 197485.546875, 122926.359375, 148313.390625, 129916.5625, 186999.1875, 276801.75, 149178.03125, 229696.453125, 81612.4765625, 179725.953125, 135615.328125, 87904.3671875, 164487.203125, 126053.9296875, 147118.75, 151162.984375, 101371.8046875, 141781.4375, 360070.46875, 151149.484375, 199141.09375, 141211.46875, 112553.578125, 158046.71875, 99837.953125, 174137.96875, 256859.03125, 184695.265625, 139160.890625, 496080.75, 153855.671875, 145597.625, 252698.21875, 179392.703125, 254769.59375, 114656.921875, 193062.78125, 142999.265625, 121800.8125, 160808.265625, 117882.0078125, 161665.765625, 119725.71875, 106324.1171875, 188780.203125, 72494.578125, 184107.703125, 126737.4140625, 114507.71875, 176504.421875, 127287.1484375, 234637.71875, 144940.453125, 132174.078125, 135354.84375, 339787.25, 146685.625, 193048.4375, 151038.671875, 327694.78125, 104599.0625, 165856.140625, 151901.453125, 140242.09375, 191093.09375, 139292.71875, 121327.6953125, 178827.15625, 183636.484375, 373714.40625, 112650.15625, 54532.921875, 118724.234375, 154482.1875, 117284.375, 137087.328125, 205272.640625, 140929.1875, 270536.15625, 207512.6875, 82150.953125, 105822.421875, 131302.890625, 144370.140625, 151049.03125, 143641.6875, 149950.96875, 157088.015625, 199107.984375, 178770.171875, 159975.1875, 157884.203125, 99043.46875, 160281.703125, 180609.734375, 185413.40625, 248968.28125, 149758.6875, 181342.890625, 171847.53125], "y": [365000, 375000, 97000, 86000, 250000, 158000, 105000, 190000, 281000, 255000, 128000, 195000, 97500, 97000, 128000, 230000, 130000, 237000, 99000, 134500, 140000, 145500, 138000, 120000, 100000, 125000, 165000, 130000, 195000, 130000, 160000, 125000, 210000, 155000, 217500, 160000, 220000, 90000, 140500, 160000, 240000, 101000, 92500, 154000, 115000, 275000, 564000, 190000, 170000, 97000, 71500, 197500, 600000, 325000, 220000, 102000, 177500, 180000, 167000, 135000, 130000, 74000, 68000, 215000, 137500, 148000, 186500, 108000, 142000, 134000, 86500, 151000, 230000, 340000, 133000, 197000, 128000, 140000, 152000, 120000, 145000, 102500, 255000, 116000, 200000, 440000, 77000, 237000, 151000, 144000, 143000, 50000, 319950, 137000, 151000, 140000, 165250, 118000, 70000, 240000, 95000, 115000, 95000, 213000, 160000, 126000, 212000, 230000, 175000, 125000, 172000, 163000, 140000, 99000, 291750, 118000, 203000, 236000, 187500, 189950, 50000, 92500, 92500, 737500, 65000, 170000, 183900, 100000, 138000, 130000, 255000, 152000, 270000, 43000, 127000, 120000, 145000, 162000, 185000, 104000, 179950, 97000, 131000, 435000, 80000, 90000, 125000, 115000, 275000, 159995, 91000, 190000, 147000, 232000, 470000, 108000, 141000, 315000, 200000, 122000, 115000, 160000, 170000, 105000, 162000, 80000, 193600, 103500, 130000, 170000, 70000, 257000, 155000, 98000, 163000, 86250, 212458, 75000, 95000, 86000, 550000, 192500, 85000, 159950, 390000, 95000, 184500, 165000, 110000, 157000, 140000, 130000, 175000, 160000, 500000, 75000, 72500, 105000, 160000, 74000, 140000, 156500, 162000, 189995, 225000, 102000, 65000, 90000, 117100, 210000, 175000, 195000, 165000, 229500, 216500, 152000, 170000, 140000, 150000, 189950, 230000, 366000, 122500, 85000, 245000]}],                        {"template": {"data": {"bar": [{"error_x": {"color": "#2a3f5f"}, "error_y": {"color": "#2a3f5f"}, "marker": {"line": {"color": "#E5ECF6", "width": 0.5}}, "type": "bar"}], "barpolar": [{"marker": {"line": {"color": "#E5ECF6", "width": 0.5}}, "type": "barpolar"}], "carpet": [{"aaxis": {"endlinecolor": "#2a3f5f", "gridcolor": "white", "linecolor": "white", "minorgridcolor": "white", "startlinecolor": "#2a3f5f"}, "baxis": {"endlinecolor": "#2a3f5f", "gridcolor": "white", "linecolor": "white", "minorgridcolor": "white", "startlinecolor": "#2a3f5f"}, "type": "carpet"}], "choropleth": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "type": "choropleth"}], "contour": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "colorscale": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]], "type": "contour"}], "contourcarpet": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "type": "contourcarpet"}], "heatmap": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "colorscale": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]], "type": "heatmap"}], "heatmapgl": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "colorscale": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]], "type": "heatmapgl"}], "histogram": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "histogram"}], "histogram2d": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "colorscale": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]], "type": "histogram2d"}], "histogram2dcontour": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "colorscale": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]], "type": "histogram2dcontour"}], "mesh3d": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "type": "mesh3d"}], "parcoords": [{"line": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "parcoords"}], "pie": [{"automargin": true, "type": "pie"}], "scatter": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scatter"}], "scatter3d": [{"line": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scatter3d"}], "scattercarpet": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scattercarpet"}], "scattergeo": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scattergeo"}], "scattergl": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scattergl"}], "scattermapbox": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scattermapbox"}], "scatterpolar": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scatterpolar"}], "scatterpolargl": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scatterpolargl"}], "scatterternary": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scatterternary"}], "surface": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "colorscale": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]], "type": "surface"}], "table": [{"cells": {"fill": {"color": "#EBF0F8"}, "line": {"color": "white"}}, "header": {"fill": {"color": "#C8D4E3"}, "line": {"color": "white"}}, "type": "table"}]}, "layout": {"annotationdefaults": {"arrowcolor": "#2a3f5f", "arrowhead": 0, "arrowwidth": 1}, "coloraxis": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "colorscale": {"diverging": [[0, "#8e0152"], [0.1, "#c51b7d"], [0.2, "#de77ae"], [0.3, "#f1b6da"], [0.4, "#fde0ef"], [0.5, "#f7f7f7"], [0.6, "#e6f5d0"], [0.7, "#b8e186"], [0.8, "#7fbc41"], [0.9, "#4d9221"], [1, "#276419"]], "sequential": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]], "sequentialminus": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]]}, "colorway": ["#636efa", "#EF553B", "#00cc96", "#ab63fa", "#FFA15A", "#19d3f3", "#FF6692", "#B6E880", "#FF97FF", "#FECB52"], "font": {"color": "#2a3f5f"}, "geo": {"bgcolor": "white", "lakecolor": "white", "landcolor": "#E5ECF6", "showlakes": true, "showland": true, "subunitcolor": "white"}, "hoverlabel": {"align": "left"}, "hovermode": "closest", "mapbox": {"style": "light"}, "paper_bgcolor": "white", "plot_bgcolor": "#E5ECF6", "polar": {"angularaxis": {"gridcolor": "white", "linecolor": "white", "ticks": ""}, "bgcolor": "#E5ECF6", "radialaxis": {"gridcolor": "white", "linecolor": "white", "ticks": ""}}, "scene": {"xaxis": {"backgroundcolor": "#E5ECF6", "gridcolor": "white", "gridwidth": 2, "linecolor": "white", "showbackground": true, "ticks": "", "zerolinecolor": "white"}, "yaxis": {"backgroundcolor": "#E5ECF6", "gridcolor": "white", "gridwidth": 2, "linecolor": "white", "showbackground": true, "ticks": "", "zerolinecolor": "white"}, "zaxis": {"backgroundcolor": "#E5ECF6", "gridcolor": "white", "gridwidth": 2, "linecolor": "white", "showbackground": true, "ticks": "", "zerolinecolor": "white"}}, "shapedefaults": {"line": {"color": "#2a3f5f"}}, "ternary": {"aaxis": {"gridcolor": "white", "linecolor": "white", "ticks": ""}, "baxis": {"gridcolor": "white", "linecolor": "white", "ticks": ""}, "bgcolor": "#E5ECF6", "caxis": {"gridcolor": "white", "linecolor": "white", "ticks": ""}}, "title": {"x": 0.05}, "xaxis": {"automargin": true, "gridcolor": "white", "linecolor": "white", "ticks": "", "title": {"standoff": 15}, "zerolinecolor": "white", "zerolinewidth": 2}, "yaxis": {"automargin": true, "gridcolor": "white", "linecolor": "white", "ticks": "", "title": {"standoff": 15}, "zerolinecolor": "white", "zerolinewidth": 2}}}, "title": {"text": "Predicted vs Actual Values Scatter Plot"}, "xaxis": {"range": [8275.0, 894331.8625], "title": {"text": "Prediction"}}, "yaxis": {"range": [8275.0, 894331.8625], "title": {"text": "Actual"}}},                        {"responsive": true}                    ).then(function(){

var gd = document.getElementById('42696be3-db95-47c8-afaa-7fe0ec822980');
var x = new MutationObserver(function (mutations, observer) {{
        var display = window.getComputedStyle(gd).display;
        if (!display || display === 'none') {{
            console.log([gd, 'removed!']);
            Plotly.purge(gd);
            observer.disconnect();
        }}
}});

// Listen for the removal of the full notebook cells
var notebookContainer = gd.closest('#notebook-container');
if (notebookContainer) {{
    x.observe(notebookContainer, {childList: true});
}}

// Listen for the clearing of the current output cell
var outputEl = gd.closest('.output');
if (outputEl) {{
    x.observe(outputEl, {childList: true});
}}

                        })                };                });            </script>        </div>



```python
price_microservice_pipeline.describe()
```

    *******************************
    * Price Microservice Pipeline *
    *******************************
    
    Problem Type: regression
    Model Family: XGBoost
    Number of features: 78
    
    Pipeline Steps
    ==============
    1. Imputer
    	 * categorical_impute_strategy : most_frequent
    	 * numeric_impute_strategy : mean
    	 * categorical_fill_value : None
    	 * numeric_fill_value : None
    2. DateTime Featurization Component
    	 * features_to_extract : ['year', 'month', 'day_of_week', 'hour']
    3. One Hot Encoder
    	 * top_n : 10
    	 * features_to_encode : None
    	 * categories : None
    	 * drop : None
    	 * handle_unknown : ignore
    	 * handle_missing : error
    4. XGBoost Regressor
    	 * eta : 0.1
    	 * max_depth : 6
    	 * min_child_weight : 1
    	 * n_estimators : 100



```python
print(price_microservice_pipeline.score(X_holdout, y_holdout, objectives=['MAE']))
```

    OrderedDict([('MAE', 41639.26171875)])



```python
save_path = '../../models/price_microservice_pipeline.pkl'
price_microservice_pipeline.save(save_path)
```


```python
!evalml info
```

    numpy.ufunc size changed, may indicate binary incompatibility. Expected 192 from C header, got 216 from PyObject
    numpy.ufunc size changed, may indicate binary incompatibility. Expected 192 from C header, got 216 from PyObject
    numpy.ufunc size changed, may indicate binary incompatibility. Expected 192 from C header, got 216 from PyObject
    numpy.ufunc size changed, may indicate binary incompatibility. Expected 192 from C header, got 216 from PyObject
    numpy.ufunc size changed, may indicate binary incompatibility. Expected 192 from C header, got 216 from PyObject
    EvalML version: 0.15.0
    EvalML installation directory: /usr/local/anaconda3/envs/microservice_price_env/lib/python3.7/site-packages/evalml
    
    SYSTEM INFO
    -----------
    python: 3.7.9.final.0
    python-bits: 64
    OS: Darwin
    OS-release: 19.6.0
    machine: x86_64
    processor: i386
    byteorder: little
    LC_ALL: None
    LANG: en_US.UTF-8
    LOCALE: en_US.UTF-8
    # of CPUS: 8
    Available memory: 5.3G
    unclosed file <_io.TextIOWrapper name='/usr/local/anaconda3/envs/microservice_price_env/lib/python3.7/site-packages/evalml/../core-requirements.txt' mode='r' encoding='UTF-8'>
    
    INSTALLED VERSIONS
    ------------------
    numpy: 1.19.2
    pandas: 1.1.3
    scipy: 1.5.2
    scikit-learn: 0.23.2
    scikit-optimize: 0.8.1
    colorama: 0.4.3
    cloudpickle: 1.6.0
    click: 7.1.2
    psutil: 5.7.2
    requirements-parser: 0.2.0
    shap: 0.36.0
    texttable: 1.6.3
    woodwork: 0.0.5
    featuretools: 0.21.0
    nlp-primitives: 1.1.0
    [0m


```python
from evalml.pipelines import RegressionPipeline

price_microservice_pipeline = RegressionPipeline.load(save_path)
price_microservice_pipeline
```




    PriceMicroservicePipeline(parameters={'Imputer':{'categorical_impute_strategy': 'most_frequent', 'numeric_impute_strategy': 'mean', 'categorical_fill_value': None, 'numeric_fill_value': None}, 'DateTime Featurization Component':{}, 'One Hot Encoder':{'top_n': 10, 'categories': None, 'drop': None, 'handle_unknown': 'ignore', 'handle_missing': 'error'}, 'XGBoost Regressor':{'eta': 0.1, 'max_depth': 6, 'min_child_weight': 1, 'n_estimators': 100},})




```python
price_microservice_pipeline.predict(X)
```




    0       426248.250000
    1       151059.453125
    2        82150.953125
    3       142689.812500
    4       127317.421875
                ...      
    1116    279921.843750
    1117    169645.578125
    1118    153830.046875
    1119    496080.750000
    1120    214928.062500
    Length: 1121, dtype: float32




```python
from evalml.model_understanding.graphs import graph_partial_dependence
graph_partial_dependence(price_microservice_pipeline, X_holdout, feature='TotalFloorArea')
```


<div>                            <div id="834cd22b-18d2-4dc1-ab36-5c7f1c8f8d47" class="plotly-graph-div" style="height:525px; width:100%;"></div>            <script type="text/javascript">                require(["plotly"], function(Plotly) {                    window.PLOTLYENV=window.PLOTLYENV || {};                                    if (document.getElementById("834cd22b-18d2-4dc1-ab36-5c7f1c8f8d47")) {                    Plotly.newPlot(                        "834cd22b-18d2-4dc1-ab36-5c7f1c8f8d47",                        [{"line": {"width": 3}, "name": "Partial Dependence", "type": "scatter", "x": [55.5124, 56.6858101010101, 57.859220202020204, 59.0326303030303, 60.2060404040404, 61.37945050505051, 62.552860606060605, 63.72627070707071, 64.8996808080808, 66.07309090909091, 67.24650101010101, 68.4199111111111, 69.59332121212121, 70.76673131313132, 71.94014141414141, 73.11355151515151, 74.28696161616162, 75.46037171717171, 76.63378181818182, 77.80719191919192, 78.98060202020201, 80.15401212121212, 81.32742222222222, 82.50083232323232, 83.67424242424242, 84.84765252525253, 86.02106262626262, 87.19447272727272, 88.36788282828283, 89.54129292929292, 90.71470303030303, 91.88811313131313, 93.06152323232322, 94.23493333333333, 95.40834343434344, 96.58175353535353, 97.75516363636363, 98.92857373737374, 100.10198383838383, 101.27539393939394, 102.44880404040404, 103.62221414141413, 104.79562424242424, 105.96903434343434, 107.14244444444444, 108.31585454545454, 109.48926464646465, 110.66267474747474, 111.83608484848484, 113.00949494949495, 114.18290505050504, 115.35631515151516, 116.52972525252525, 117.70313535353534, 118.87654545454546, 120.04995555555556, 121.22336565656566, 122.39677575757575, 123.57018585858586, 124.74359595959596, 125.91700606060606, 127.09041616161616, 128.26382626262625, 129.43723636363637, 130.61064646464646, 131.78405656565656, 132.95746666666668, 134.13087676767677, 135.30428686868686, 136.47769696969698, 137.65110707070707, 138.82451717171716, 139.99792727272728, 141.17133737373737, 142.34474747474746, 143.51815757575758, 144.69156767676768, 145.86497777777777, 147.0383878787879, 148.21179797979798, 149.38520808080807, 150.5586181818182, 151.73202828282828, 152.90543838383837, 154.0788484848485, 155.25225858585858, 156.42566868686868, 157.5990787878788, 158.7724888888889, 159.94589898989898, 161.1193090909091, 162.2927191919192, 163.46612929292928, 164.6395393939394, 165.8129494949495, 166.98635959595958, 168.1597696969697, 169.3331797979798, 170.5065898989899, 171.68], "y": [138399.3125, 138597.203125, 138581.46875, 138499.015625, 138525.765625, 133345.25, 134604.546875, 136473.25, 141164.34375, 144128.671875, 143990.53125, 147300.890625, 147079.796875, 147209.296875, 149460.6875, 149860.484375, 147246.046875, 147832.53125, 147714.703125, 146515.328125, 146176.296875, 146347.078125, 147206.875, 152471.921875, 153971.515625, 156552.28125, 161022.359375, 160941.421875, 159542.1875, 159630.46875, 159211.453125, 159119.640625, 160594.078125, 173299.015625, 181243.09375, 180992.671875, 180992.671875, 178943.640625, 176667.703125, 187609.8125, 189956.8125, 190609.171875, 190609.171875, 192466.625, 192092.296875, 188267.5625, 188267.5625, 188861.671875, 188861.671875, 189287.359375, 189287.359375, 190393.640625, 190787.265625, 190787.265625, 190995.390625, 190995.390625, 190995.390625, 191982.703125, 232062.421875, 232062.421875, 232062.421875, 233285.625, 233285.625, 233285.625, 232241.171875, 230957.125, 230957.125, 230957.125, 230957.125, 228961.0625, 228961.0625, 220900.59375, 239301.515625, 236502.359375, 236502.359375, 236502.359375, 236502.359375, 236502.359375, 236502.359375, 236502.359375, 236502.359375, 236640.9375, 236640.9375, 236640.9375, 236218.71875, 236883.78125, 236883.78125, 238338.140625, 238338.140625, 238364.625, 238364.625, 238364.625, 236558.75, 238770.546875, 238770.546875, 237627.640625, 237627.640625, 237627.640625, 237627.640625, 237627.640625]}],                        {"template": {"data": {"bar": [{"error_x": {"color": "#2a3f5f"}, "error_y": {"color": "#2a3f5f"}, "marker": {"line": {"color": "#E5ECF6", "width": 0.5}}, "type": "bar"}], "barpolar": [{"marker": {"line": {"color": "#E5ECF6", "width": 0.5}}, "type": "barpolar"}], "carpet": [{"aaxis": {"endlinecolor": "#2a3f5f", "gridcolor": "white", "linecolor": "white", "minorgridcolor": "white", "startlinecolor": "#2a3f5f"}, "baxis": {"endlinecolor": "#2a3f5f", "gridcolor": "white", "linecolor": "white", "minorgridcolor": "white", "startlinecolor": "#2a3f5f"}, "type": "carpet"}], "choropleth": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "type": "choropleth"}], "contour": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "colorscale": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]], "type": "contour"}], "contourcarpet": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "type": "contourcarpet"}], "heatmap": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "colorscale": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]], "type": "heatmap"}], "heatmapgl": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "colorscale": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]], "type": "heatmapgl"}], "histogram": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "histogram"}], "histogram2d": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "colorscale": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]], "type": "histogram2d"}], "histogram2dcontour": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "colorscale": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]], "type": "histogram2dcontour"}], "mesh3d": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "type": "mesh3d"}], "parcoords": [{"line": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "parcoords"}], "pie": [{"automargin": true, "type": "pie"}], "scatter": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scatter"}], "scatter3d": [{"line": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scatter3d"}], "scattercarpet": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scattercarpet"}], "scattergeo": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scattergeo"}], "scattergl": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scattergl"}], "scattermapbox": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scattermapbox"}], "scatterpolar": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scatterpolar"}], "scatterpolargl": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scatterpolargl"}], "scatterternary": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scatterternary"}], "surface": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "colorscale": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]], "type": "surface"}], "table": [{"cells": {"fill": {"color": "#EBF0F8"}, "line": {"color": "white"}}, "header": {"fill": {"color": "#C8D4E3"}, "line": {"color": "white"}}, "type": "table"}]}, "layout": {"annotationdefaults": {"arrowcolor": "#2a3f5f", "arrowhead": 0, "arrowwidth": 1}, "coloraxis": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "colorscale": {"diverging": [[0, "#8e0152"], [0.1, "#c51b7d"], [0.2, "#de77ae"], [0.3, "#f1b6da"], [0.4, "#fde0ef"], [0.5, "#f7f7f7"], [0.6, "#e6f5d0"], [0.7, "#b8e186"], [0.8, "#7fbc41"], [0.9, "#4d9221"], [1, "#276419"]], "sequential": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]], "sequentialminus": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]]}, "colorway": ["#636efa", "#EF553B", "#00cc96", "#ab63fa", "#FFA15A", "#19d3f3", "#FF6692", "#B6E880", "#FF97FF", "#FECB52"], "font": {"color": "#2a3f5f"}, "geo": {"bgcolor": "white", "lakecolor": "white", "landcolor": "#E5ECF6", "showlakes": true, "showland": true, "subunitcolor": "white"}, "hoverlabel": {"align": "left"}, "hovermode": "closest", "mapbox": {"style": "light"}, "paper_bgcolor": "white", "plot_bgcolor": "#E5ECF6", "polar": {"angularaxis": {"gridcolor": "white", "linecolor": "white", "ticks": ""}, "bgcolor": "#E5ECF6", "radialaxis": {"gridcolor": "white", "linecolor": "white", "ticks": ""}}, "scene": {"xaxis": {"backgroundcolor": "#E5ECF6", "gridcolor": "white", "gridwidth": 2, "linecolor": "white", "showbackground": true, "ticks": "", "zerolinecolor": "white"}, "yaxis": {"backgroundcolor": "#E5ECF6", "gridcolor": "white", "gridwidth": 2, "linecolor": "white", "showbackground": true, "ticks": "", "zerolinecolor": "white"}, "zaxis": {"backgroundcolor": "#E5ECF6", "gridcolor": "white", "gridwidth": 2, "linecolor": "white", "showbackground": true, "ticks": "", "zerolinecolor": "white"}}, "shapedefaults": {"line": {"color": "#2a3f5f"}}, "ternary": {"aaxis": {"gridcolor": "white", "linecolor": "white", "ticks": ""}, "baxis": {"gridcolor": "white", "linecolor": "white", "ticks": ""}, "bgcolor": "#E5ECF6", "caxis": {"gridcolor": "white", "linecolor": "white", "ticks": ""}}, "title": {"x": 0.05}, "xaxis": {"automargin": true, "gridcolor": "white", "linecolor": "white", "ticks": "", "title": {"standoff": 15}, "zerolinecolor": "white", "zerolinewidth": 2}, "yaxis": {"automargin": true, "gridcolor": "white", "linecolor": "white", "ticks": "", "title": {"standoff": 15}, "zerolinecolor": "white", "zerolinewidth": 2}}}, "title": {"text": "Partial Dependence of 'TotalFloorArea'"}, "xaxis": {"range": [49.70402, 177.48838], "title": {"text": "TotalFloorArea"}}, "yaxis": {"range": [128047.43671875, 244599.32890625], "title": {"text": "Partial Dependence"}}},                        {"responsive": true}                    ).then(function(){

var gd = document.getElementById('834cd22b-18d2-4dc1-ab36-5c7f1c8f8d47');
var x = new MutationObserver(function (mutations, observer) {{
        var display = window.getComputedStyle(gd).display;
        if (!display || display === 'none') {{
            console.log([gd, 'removed!']);
            Plotly.purge(gd);
            observer.disconnect();
        }}
}});

// Listen for the removal of the full notebook cells
var notebookContainer = gd.closest('#notebook-container');
if (notebookContainer) {{
    x.observe(notebookContainer, {childList: true});
}}

// Listen for the clearing of the current output cell
var outputEl = gd.closest('.output');
if (outputEl) {{
    x.observe(outputEl, {childList: true});
}}

                        })                };                });            </script>        </div>



```python
graph_partial_dependence(price_microservice_pipeline, X_holdout, feature='HabitableRooms')
```

    /usr/local/anaconda3/envs/microservice_price_env/lib/python3.7/site-packages/evalml/model_understanding/graphs.py:441: NullsInColumnWarning:
    
    There are null values in the features, which will cause NaN values in the partial dependence output. Fill in these values to remove the NaN values.
    



<div>                            <div id="147a4a73-0cbe-40f1-b524-fee64c778a0e" class="plotly-graph-div" style="height:525px; width:100%;"></div>            <script type="text/javascript">                require(["plotly"], function(Plotly) {                    window.PLOTLYENV=window.PLOTLYENV || {};                                    if (document.getElementById("147a4a73-0cbe-40f1-b524-fee64c778a0e")) {                    Plotly.newPlot(                        "147a4a73-0cbe-40f1-b524-fee64c778a0e",                        [{"line": {"width": 3}, "name": "Partial Dependence", "type": "scatter", "x": [1.0, 2.0, 3.0, 4.0, 5.0, 6.0, 7.0, 8.0, 11.0, null, null, null, null, null, null, null, null, null, null, null, null, null], "y": [172547.796875, 172547.796875, 173272.53125, 175981.796875, 176637.6875, 178386.03125, 179231.640625, 172837.234375, 177014.109375, 177224.171875, 177224.171875, 177224.171875, 177224.171875, 177224.171875, 177224.171875, 177224.171875, 177224.171875, 177224.171875, 177224.171875, 177224.171875, 177224.171875, 177224.171875]}],                        {"template": {"data": {"bar": [{"error_x": {"color": "#2a3f5f"}, "error_y": {"color": "#2a3f5f"}, "marker": {"line": {"color": "#E5ECF6", "width": 0.5}}, "type": "bar"}], "barpolar": [{"marker": {"line": {"color": "#E5ECF6", "width": 0.5}}, "type": "barpolar"}], "carpet": [{"aaxis": {"endlinecolor": "#2a3f5f", "gridcolor": "white", "linecolor": "white", "minorgridcolor": "white", "startlinecolor": "#2a3f5f"}, "baxis": {"endlinecolor": "#2a3f5f", "gridcolor": "white", "linecolor": "white", "minorgridcolor": "white", "startlinecolor": "#2a3f5f"}, "type": "carpet"}], "choropleth": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "type": "choropleth"}], "contour": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "colorscale": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]], "type": "contour"}], "contourcarpet": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "type": "contourcarpet"}], "heatmap": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "colorscale": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]], "type": "heatmap"}], "heatmapgl": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "colorscale": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]], "type": "heatmapgl"}], "histogram": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "histogram"}], "histogram2d": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "colorscale": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]], "type": "histogram2d"}], "histogram2dcontour": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "colorscale": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]], "type": "histogram2dcontour"}], "mesh3d": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "type": "mesh3d"}], "parcoords": [{"line": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "parcoords"}], "pie": [{"automargin": true, "type": "pie"}], "scatter": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scatter"}], "scatter3d": [{"line": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scatter3d"}], "scattercarpet": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scattercarpet"}], "scattergeo": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scattergeo"}], "scattergl": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scattergl"}], "scattermapbox": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scattermapbox"}], "scatterpolar": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scatterpolar"}], "scatterpolargl": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scatterpolargl"}], "scatterternary": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scatterternary"}], "surface": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "colorscale": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]], "type": "surface"}], "table": [{"cells": {"fill": {"color": "#EBF0F8"}, "line": {"color": "white"}}, "header": {"fill": {"color": "#C8D4E3"}, "line": {"color": "white"}}, "type": "table"}]}, "layout": {"annotationdefaults": {"arrowcolor": "#2a3f5f", "arrowhead": 0, "arrowwidth": 1}, "coloraxis": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "colorscale": {"diverging": [[0, "#8e0152"], [0.1, "#c51b7d"], [0.2, "#de77ae"], [0.3, "#f1b6da"], [0.4, "#fde0ef"], [0.5, "#f7f7f7"], [0.6, "#e6f5d0"], [0.7, "#b8e186"], [0.8, "#7fbc41"], [0.9, "#4d9221"], [1, "#276419"]], "sequential": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]], "sequentialminus": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]]}, "colorway": ["#636efa", "#EF553B", "#00cc96", "#ab63fa", "#FFA15A", "#19d3f3", "#FF6692", "#B6E880", "#FF97FF", "#FECB52"], "font": {"color": "#2a3f5f"}, "geo": {"bgcolor": "white", "lakecolor": "white", "landcolor": "#E5ECF6", "showlakes": true, "showland": true, "subunitcolor": "white"}, "hoverlabel": {"align": "left"}, "hovermode": "closest", "mapbox": {"style": "light"}, "paper_bgcolor": "white", "plot_bgcolor": "#E5ECF6", "polar": {"angularaxis": {"gridcolor": "white", "linecolor": "white", "ticks": ""}, "bgcolor": "#E5ECF6", "radialaxis": {"gridcolor": "white", "linecolor": "white", "ticks": ""}}, "scene": {"xaxis": {"backgroundcolor": "#E5ECF6", "gridcolor": "white", "gridwidth": 2, "linecolor": "white", "showbackground": true, "ticks": "", "zerolinecolor": "white"}, "yaxis": {"backgroundcolor": "#E5ECF6", "gridcolor": "white", "gridwidth": 2, "linecolor": "white", "showbackground": true, "ticks": "", "zerolinecolor": "white"}, "zaxis": {"backgroundcolor": "#E5ECF6", "gridcolor": "white", "gridwidth": 2, "linecolor": "white", "showbackground": true, "ticks": "", "zerolinecolor": "white"}}, "shapedefaults": {"line": {"color": "#2a3f5f"}}, "ternary": {"aaxis": {"gridcolor": "white", "linecolor": "white", "ticks": ""}, "baxis": {"gridcolor": "white", "linecolor": "white", "ticks": ""}, "bgcolor": "#E5ECF6", "caxis": {"gridcolor": "white", "linecolor": "white", "ticks": ""}}, "title": {"x": 0.05}, "xaxis": {"automargin": true, "gridcolor": "white", "linecolor": "white", "ticks": "", "title": {"standoff": 15}, "zerolinecolor": "white", "zerolinewidth": 2}, "yaxis": {"automargin": true, "gridcolor": "white", "linecolor": "white", "ticks": "", "title": {"standoff": 15}, "zerolinecolor": "white", "zerolinewidth": 2}}}, "title": {"text": "Partial Dependence of 'HabitableRooms'"}, "xaxis": {"range": [0.5, 11.5], "title": {"text": "HabitableRooms"}}, "yaxis": {"range": [172213.6046875, 179565.8328125], "title": {"text": "Partial Dependence"}}},                        {"responsive": true}                    ).then(function(){

var gd = document.getElementById('147a4a73-0cbe-40f1-b524-fee64c778a0e');
var x = new MutationObserver(function (mutations, observer) {{
        var display = window.getComputedStyle(gd).display;
        if (!display || display === 'none') {{
            console.log([gd, 'removed!']);
            Plotly.purge(gd);
            observer.disconnect();
        }}
}});

// Listen for the removal of the full notebook cells
var notebookContainer = gd.closest('#notebook-container');
if (notebookContainer) {{
    x.observe(notebookContainer, {childList: true});
}}

// Listen for the clearing of the current output cell
var outputEl = gd.closest('.output');
if (outputEl) {{
    x.observe(outputEl, {childList: true});
}}

                        })                };                });            </script>        </div>



```python
graph_partial_dependence(price_microservice_pipeline, X_holdout, feature='PotentialEnergyEfficiency')
```


<div>                            <div id="1cc91b5f-1282-489b-a65e-be3dca91b4db" class="plotly-graph-div" style="height:525px; width:100%;"></div>            <script type="text/javascript">                require(["plotly"], function(Plotly) {                    window.PLOTLYENV=window.PLOTLYENV || {};                                    if (document.getElementById("1cc91b5f-1282-489b-a65e-be3dca91b4db")) {                    Plotly.newPlot(                        "1cc91b5f-1282-489b-a65e-be3dca91b4db",                        [{"line": {"width": 3}, "name": "Partial Dependence", "type": "scatter", "x": [54, 62, 63, 66, 68, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 93, 94, 95, 97, 99], "y": [184508.625, 182567.0, 182663.65625, 176601.3125, 176826.625, 179341.65625, 179280.53125, 179217.03125, 179526.234375, 174557.453125, 174448.46875, 174421.265625, 174658.703125, 174629.015625, 174471.3125, 176189.65625, 183074.671875, 181566.8125, 178701.796875, 178555.84375, 175985.71875, 175582.921875, 174536.953125, 171433.265625, 171294.1875, 172642.609375, 172642.609375, 172642.609375, 172642.609375, 171950.40625, 172495.0, 172495.0]}],                        {"template": {"data": {"bar": [{"error_x": {"color": "#2a3f5f"}, "error_y": {"color": "#2a3f5f"}, "marker": {"line": {"color": "#E5ECF6", "width": 0.5}}, "type": "bar"}], "barpolar": [{"marker": {"line": {"color": "#E5ECF6", "width": 0.5}}, "type": "barpolar"}], "carpet": [{"aaxis": {"endlinecolor": "#2a3f5f", "gridcolor": "white", "linecolor": "white", "minorgridcolor": "white", "startlinecolor": "#2a3f5f"}, "baxis": {"endlinecolor": "#2a3f5f", "gridcolor": "white", "linecolor": "white", "minorgridcolor": "white", "startlinecolor": "#2a3f5f"}, "type": "carpet"}], "choropleth": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "type": "choropleth"}], "contour": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "colorscale": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]], "type": "contour"}], "contourcarpet": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "type": "contourcarpet"}], "heatmap": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "colorscale": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]], "type": "heatmap"}], "heatmapgl": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "colorscale": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]], "type": "heatmapgl"}], "histogram": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "histogram"}], "histogram2d": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "colorscale": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]], "type": "histogram2d"}], "histogram2dcontour": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "colorscale": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]], "type": "histogram2dcontour"}], "mesh3d": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "type": "mesh3d"}], "parcoords": [{"line": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "parcoords"}], "pie": [{"automargin": true, "type": "pie"}], "scatter": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scatter"}], "scatter3d": [{"line": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scatter3d"}], "scattercarpet": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scattercarpet"}], "scattergeo": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scattergeo"}], "scattergl": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scattergl"}], "scattermapbox": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scattermapbox"}], "scatterpolar": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scatterpolar"}], "scatterpolargl": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scatterpolargl"}], "scatterternary": [{"marker": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "type": "scatterternary"}], "surface": [{"colorbar": {"outlinewidth": 0, "ticks": ""}, "colorscale": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]], "type": "surface"}], "table": [{"cells": {"fill": {"color": "#EBF0F8"}, "line": {"color": "white"}}, "header": {"fill": {"color": "#C8D4E3"}, "line": {"color": "white"}}, "type": "table"}]}, "layout": {"annotationdefaults": {"arrowcolor": "#2a3f5f", "arrowhead": 0, "arrowwidth": 1}, "coloraxis": {"colorbar": {"outlinewidth": 0, "ticks": ""}}, "colorscale": {"diverging": [[0, "#8e0152"], [0.1, "#c51b7d"], [0.2, "#de77ae"], [0.3, "#f1b6da"], [0.4, "#fde0ef"], [0.5, "#f7f7f7"], [0.6, "#e6f5d0"], [0.7, "#b8e186"], [0.8, "#7fbc41"], [0.9, "#4d9221"], [1, "#276419"]], "sequential": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]], "sequentialminus": [[0.0, "#0d0887"], [0.1111111111111111, "#46039f"], [0.2222222222222222, "#7201a8"], [0.3333333333333333, "#9c179e"], [0.4444444444444444, "#bd3786"], [0.5555555555555556, "#d8576b"], [0.6666666666666666, "#ed7953"], [0.7777777777777778, "#fb9f3a"], [0.8888888888888888, "#fdca26"], [1.0, "#f0f921"]]}, "colorway": ["#636efa", "#EF553B", "#00cc96", "#ab63fa", "#FFA15A", "#19d3f3", "#FF6692", "#B6E880", "#FF97FF", "#FECB52"], "font": {"color": "#2a3f5f"}, "geo": {"bgcolor": "white", "lakecolor": "white", "landcolor": "#E5ECF6", "showlakes": true, "showland": true, "subunitcolor": "white"}, "hoverlabel": {"align": "left"}, "hovermode": "closest", "mapbox": {"style": "light"}, "paper_bgcolor": "white", "plot_bgcolor": "#E5ECF6", "polar": {"angularaxis": {"gridcolor": "white", "linecolor": "white", "ticks": ""}, "bgcolor": "#E5ECF6", "radialaxis": {"gridcolor": "white", "linecolor": "white", "ticks": ""}}, "scene": {"xaxis": {"backgroundcolor": "#E5ECF6", "gridcolor": "white", "gridwidth": 2, "linecolor": "white", "showbackground": true, "ticks": "", "zerolinecolor": "white"}, "yaxis": {"backgroundcolor": "#E5ECF6", "gridcolor": "white", "gridwidth": 2, "linecolor": "white", "showbackground": true, "ticks": "", "zerolinecolor": "white"}, "zaxis": {"backgroundcolor": "#E5ECF6", "gridcolor": "white", "gridwidth": 2, "linecolor": "white", "showbackground": true, "ticks": "", "zerolinecolor": "white"}}, "shapedefaults": {"line": {"color": "#2a3f5f"}}, "ternary": {"aaxis": {"gridcolor": "white", "linecolor": "white", "ticks": ""}, "baxis": {"gridcolor": "white", "linecolor": "white", "ticks": ""}, "bgcolor": "#E5ECF6", "caxis": {"gridcolor": "white", "linecolor": "white", "ticks": ""}}, "title": {"x": 0.05}, "xaxis": {"automargin": true, "gridcolor": "white", "linecolor": "white", "ticks": "", "title": {"standoff": 15}, "zerolinecolor": "white", "zerolinewidth": 2}, "yaxis": {"automargin": true, "gridcolor": "white", "linecolor": "white", "ticks": "", "title": {"standoff": 15}, "zerolinecolor": "white", "zerolinewidth": 2}}}, "title": {"text": "Partial Dependence of 'PotentialEnergyEfficiency'"}, "xaxis": {"range": [51.75, 101.25], "title": {"text": "PotentialEnergyEfficiency"}}, "yaxis": {"range": [170633.465625, 185169.346875], "title": {"text": "Partial Dependence"}}},                        {"responsive": true}                    ).then(function(){

var gd = document.getElementById('1cc91b5f-1282-489b-a65e-be3dca91b4db');
var x = new MutationObserver(function (mutations, observer) {{
        var display = window.getComputedStyle(gd).display;
        if (!display || display === 'none') {{
            console.log([gd, 'removed!']);
            Plotly.purge(gd);
            observer.disconnect();
        }}
}});

// Listen for the removal of the full notebook cells
var notebookContainer = gd.closest('#notebook-container');
if (notebookContainer) {{
    x.observe(notebookContainer, {childList: true});
}}

// Listen for the clearing of the current output cell
var outputEl = gd.closest('.output');
if (outputEl) {{
    x.observe(outputEl, {childList: true});
}}

                        })                };                });            </script>        </div>



```python
price_microservice_pipeline.input_feature_names
```




    {'Imputer': ['PropertyType',
      'New',
      'Duration',
      'CurrentEnergyRating',
      'PotentialEnergyRating',
      'CurrentEnergyEfficiency',
      'PotentialEnergyEfficiency',
      'GlazedArea',
      'HabitableRooms',
      'HeatedRooms',
      'FlatStoreyCount',
      'TotalFloorArea',
      'FloorLevel',
      'PurchaseDate_year',
      'PurchaseDate_month',
      'PurchaseDate_day_of_week',
      'PurchaseDate_hour',
      'EpcInspectionDate_year',
      'EpcInspectionDate_month',
      'EpcInspectionDate_day_of_week',
      'EpcInspectionDate_hour'],
     'DateTime Featurization Component': ['PropertyType',
      'New',
      'Duration',
      'CurrentEnergyRating',
      'PotentialEnergyRating',
      'CurrentEnergyEfficiency',
      'PotentialEnergyEfficiency',
      'GlazedArea',
      'HabitableRooms',
      'HeatedRooms',
      'FlatStoreyCount',
      'TotalFloorArea',
      'FloorLevel',
      'PurchaseDate_year',
      'PurchaseDate_month',
      'PurchaseDate_day_of_week',
      'PurchaseDate_hour',
      'EpcInspectionDate_year',
      'EpcInspectionDate_month',
      'EpcInspectionDate_day_of_week',
      'EpcInspectionDate_hour'],
     'One Hot Encoder': ['PropertyType',
      'New',
      'Duration',
      'CurrentEnergyRating',
      'PotentialEnergyRating',
      'CurrentEnergyEfficiency',
      'PotentialEnergyEfficiency',
      'GlazedArea',
      'HabitableRooms',
      'HeatedRooms',
      'FlatStoreyCount',
      'TotalFloorArea',
      'FloorLevel',
      'PurchaseDate_year',
      'PurchaseDate_month',
      'PurchaseDate_day_of_week',
      'PurchaseDate_hour',
      'EpcInspectionDate_year',
      'EpcInspectionDate_month',
      'EpcInspectionDate_day_of_week',
      'EpcInspectionDate_hour'],
     'XGBoost Regressor': ['CurrentEnergyEfficiency',
      'PotentialEnergyEfficiency',
      'HabitableRooms',
      'HeatedRooms',
      'FlatStoreyCount',
      'TotalFloorArea',
      'PurchaseDate_year',
      'PurchaseDate_hour',
      'EpcInspectionDate_year',
      'EpcInspectionDate_hour',
      'PropertyType_D',
      'PropertyType_F',
      'PropertyType_O',
      'PropertyType_S',
      'PropertyType_T',
      'New_N',
      'New_Y',
      'Duration_F',
      'Duration_L',
      'CurrentEnergyRating_A',
      'CurrentEnergyRating_B',
      'CurrentEnergyRating_C',
      'CurrentEnergyRating_D',
      'CurrentEnergyRating_E',
      'CurrentEnergyRating_F',
      'CurrentEnergyRating_G',
      'PotentialEnergyRating_A',
      'PotentialEnergyRating_B',
      'PotentialEnergyRating_C',
      'PotentialEnergyRating_D',
      'PotentialEnergyRating_E',
      'PotentialEnergyRating_F',
      'GlazedArea_More Than Typical',
      'GlazedArea_Much More Than Typical',
      'GlazedArea_NO DATA!',
      'GlazedArea_Normal',
      'FloorLevel_1st',
      'FloorLevel_2nd',
      'FloorLevel_3rd',
      'FloorLevel_5th',
      'FloorLevel_Ground',
      'FloorLevel_NO DATA!',
      'FloorLevel_NODATA!',
      'FloorLevel_ground floor',
      'FloorLevel_mid floor',
      'FloorLevel_top floor',
      'PurchaseDate_month_April',
      'PurchaseDate_month_August',
      'PurchaseDate_month_February',
      'PurchaseDate_month_January',
      'PurchaseDate_month_July',
      'PurchaseDate_month_June',
      'PurchaseDate_month_March',
      'PurchaseDate_month_May',
      'PurchaseDate_day_of_week_Friday',
      'PurchaseDate_day_of_week_Monday',
      'PurchaseDate_day_of_week_Saturday',
      'PurchaseDate_day_of_week_Sunday',
      'PurchaseDate_day_of_week_Thursday',
      'PurchaseDate_day_of_week_Tuesday',
      'PurchaseDate_day_of_week_Wednesday',
      'EpcInspectionDate_month_April',
      'EpcInspectionDate_month_August',
      'EpcInspectionDate_month_February',
      'EpcInspectionDate_month_January',
      'EpcInspectionDate_month_June',
      'EpcInspectionDate_month_March',
      'EpcInspectionDate_month_May',
      'EpcInspectionDate_month_November',
      'EpcInspectionDate_month_October',
      'EpcInspectionDate_month_September',
      'EpcInspectionDate_day_of_week_Friday',
      'EpcInspectionDate_day_of_week_Monday',
      'EpcInspectionDate_day_of_week_Saturday',
      'EpcInspectionDate_day_of_week_Sunday',
      'EpcInspectionDate_day_of_week_Thursday',
      'EpcInspectionDate_day_of_week_Tuesday',
      'EpcInspectionDate_day_of_week_Wednesday']}




```python

```
