"use strict";
const assert_1 = require("@aws-cdk/assert");
const autoscaling = require("@aws-cdk/aws-autoscaling");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const ec2 = require("@aws-cdk/aws-ec2");
const lbv2 = require("@aws-cdk/aws-elasticloadbalancingv2");
const cdk = require("@aws-cdk/core");
const codedeploy = require("../../lib");
module.exports = {
    'CodeDeploy Server Deployment Group': {
        'can be created by explicitly passing an Application'(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.ServerApplication(stack, 'MyApp');
            new codedeploy.ServerDeploymentGroup(stack, 'MyDG', {
                application,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                'ApplicationName': {
                    'Ref': 'MyApp3CE31C26',
                },
            }));
            test.done();
        },
        'can be imported'(test) {
            const stack = new cdk.Stack();
            const application = codedeploy.ServerApplication.fromServerApplicationName(stack, 'MyApp', 'MyApp');
            const deploymentGroup = codedeploy.ServerDeploymentGroup.fromServerDeploymentGroupAttributes(stack, 'MyDG', {
                application,
                deploymentGroupName: 'MyDG',
            });
            test.notEqual(deploymentGroup, undefined);
            test.done();
        },
        'created with ASGs contains the ASG names'(test) {
            const stack = new cdk.Stack();
            const asg = new autoscaling.AutoScalingGroup(stack, 'ASG', {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.STANDARD3, ec2.InstanceSize.SMALL),
                machineImage: new ec2.AmazonLinuxImage(),
                vpc: new ec2.Vpc(stack, 'VPC'),
            });
            new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                autoScalingGroups: [asg],
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                'AutoScalingGroups': [
                    {
                        'Ref': 'ASG46ED3070',
                    },
                ],
            }));
            test.done();
        },
        'created without ASGs but adding them later contains the ASG names'(test) {
            const stack = new cdk.Stack();
            const asg = new autoscaling.AutoScalingGroup(stack, 'ASG', {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.STANDARD3, ec2.InstanceSize.SMALL),
                machineImage: new ec2.AmazonLinuxImage(),
                vpc: new ec2.Vpc(stack, 'VPC'),
            });
            const deploymentGroup = new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup');
            deploymentGroup.addAutoScalingGroup(asg);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                'AutoScalingGroups': [
                    {
                        'Ref': 'ASG46ED3070',
                    },
                ],
            }));
            test.done();
        },
        'can be created with an ALB Target Group as the load balancer'(test) {
            const stack = new cdk.Stack();
            const alb = new lbv2.ApplicationLoadBalancer(stack, 'ALB', {
                vpc: new ec2.Vpc(stack, 'VPC'),
            });
            const listener = alb.addListener('Listener', { protocol: lbv2.ApplicationProtocol.HTTP });
            const targetGroup = listener.addTargets('Fleet', { protocol: lbv2.ApplicationProtocol.HTTP });
            new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                loadBalancer: codedeploy.LoadBalancer.application(targetGroup),
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                'LoadBalancerInfo': {
                    'TargetGroupInfoList': [
                        {
                            'Name': {
                                'Fn::GetAtt': [
                                    'ALBListenerFleetGroup008CEEE4',
                                    'TargetGroupName',
                                ],
                            },
                        },
                    ],
                },
                'DeploymentStyle': {
                    'DeploymentOption': 'WITH_TRAFFIC_CONTROL',
                },
            }));
            test.done();
        },
        'can be created with an NLB Target Group as the load balancer'(test) {
            const stack = new cdk.Stack();
            const nlb = new lbv2.NetworkLoadBalancer(stack, 'NLB', {
                vpc: new ec2.Vpc(stack, 'VPC'),
            });
            const listener = nlb.addListener('Listener', { port: 80 });
            const targetGroup = listener.addTargets('Fleet', { port: 80 });
            new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                loadBalancer: codedeploy.LoadBalancer.network(targetGroup),
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                'LoadBalancerInfo': {
                    'TargetGroupInfoList': [
                        {
                            'Name': {
                                'Fn::GetAtt': [
                                    'NLBListenerFleetGroupB882EC86',
                                    'TargetGroupName',
                                ],
                            },
                        },
                    ],
                },
                'DeploymentStyle': {
                    'DeploymentOption': 'WITH_TRAFFIC_CONTROL',
                },
            }));
            test.done();
        },
        'can be created with a single EC2 instance tag set with a single or no value'(test) {
            const stack = new cdk.Stack();
            new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                ec2InstanceTags: new codedeploy.InstanceTagSet({
                    'some-key': ['some-value'],
                    'other-key': [],
                }),
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                'Ec2TagSet': {
                    'Ec2TagSetList': [
                        {
                            'Ec2TagGroup': [
                                {
                                    'Key': 'some-key',
                                    'Value': 'some-value',
                                    'Type': 'KEY_AND_VALUE',
                                },
                                {
                                    'Key': 'other-key',
                                    'Type': 'KEY_ONLY',
                                },
                            ],
                        },
                    ],
                },
            }));
            test.done();
        },
        'can be created with two on-premise instance tag sets with multiple values or without a key'(test) {
            const stack = new cdk.Stack();
            new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                onPremiseInstanceTags: new codedeploy.InstanceTagSet({
                    'some-key': ['some-value', 'another-value'],
                }, {
                    '': ['keyless-value'],
                }),
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                'OnPremisesTagSet': {
                    'OnPremisesTagSetList': [
                        {
                            'OnPremisesTagGroup': [
                                {
                                    'Key': 'some-key',
                                    'Value': 'some-value',
                                    'Type': 'KEY_AND_VALUE',
                                },
                                {
                                    'Key': 'some-key',
                                    'Value': 'another-value',
                                    'Type': 'KEY_AND_VALUE',
                                },
                            ],
                        },
                        {
                            'OnPremisesTagGroup': [
                                {
                                    'Value': 'keyless-value',
                                    'Type': 'VALUE_ONLY',
                                },
                            ],
                        },
                    ],
                },
            }));
            test.done();
        },
        'cannot be created with an instance tag set containing a keyless, valueless filter'(test) {
            const stack = new cdk.Stack();
            test.throws(() => {
                new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                    onPremiseInstanceTags: new codedeploy.InstanceTagSet({
                        '': [],
                    }),
                });
            });
            test.done();
        },
        'cannot be created with an instance tag set containing 4 instance tag groups'(test) {
            const stack = new cdk.Stack();
            test.throws(() => {
                new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                    onPremiseInstanceTags: new codedeploy.InstanceTagSet({}, {}, {}, {}),
                });
            }, /3/);
            test.done();
        },
        'can have alarms added to it after being created'(test) {
            const stack = new cdk.Stack();
            const alarm = new cloudwatch.Alarm(stack, 'Alarm1', {
                metric: new cloudwatch.Metric({
                    metricName: 'Errors',
                    namespace: 'my.namespace',
                }),
                threshold: 1,
                evaluationPeriods: 1,
            });
            const deploymentGroup = new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup');
            deploymentGroup.addAlarm(alarm);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                'AlarmConfiguration': {
                    'Alarms': [
                        {
                            'Name': {
                                'Ref': 'Alarm1F9009D71',
                            },
                        },
                    ],
                    'Enabled': true,
                },
            }));
            test.done();
        },
        'only automatically rolls back failed deployments by default'(test) {
            const stack = new cdk.Stack();
            new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup');
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                'AutoRollbackConfiguration': {
                    'Enabled': true,
                    'Events': [
                        'DEPLOYMENT_FAILURE',
                    ],
                },
            }));
            test.done();
        },
        'rolls back alarmed deployments if at least one alarm has been added'(test) {
            const stack = new cdk.Stack();
            const alarm = new cloudwatch.Alarm(stack, 'Alarm1', {
                metric: new cloudwatch.Metric({
                    metricName: 'Errors',
                    namespace: 'my.namespace',
                }),
                threshold: 1,
                evaluationPeriods: 1,
            });
            const deploymentGroup = new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                autoRollback: {
                    failedDeployment: false,
                },
            });
            deploymentGroup.addAlarm(alarm);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                'AutoRollbackConfiguration': {
                    'Enabled': true,
                    'Events': [
                        'DEPLOYMENT_STOP_ON_ALARM',
                    ],
                },
            }));
            test.done();
        },
        'setting to roll back on alarms without providing any results in an exception'(test) {
            const stack = new cdk.Stack();
            new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                autoRollback: {
                    deploymentInAlarm: true,
                },
            });
            test.throws(() => assert_1.SynthUtils.toCloudFormation(stack), /deploymentInAlarm/);
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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