# -*- encoding: utf-8 -*-
import os
import re
import subprocess
from subprocess import CalledProcessError
import sys
import semver

from halo import Halo

from suite_py.lib import logger


class GitHandler:
    def __init__(self, repo, config):
        self._repo = repo
        self._projects_home = config.user["projects_home"]
        self._path = os.path.join(self._projects_home, repo)

    def get_repo(self):
        return self._repo

    def get_path(self):
        return self._path

    def clone(self):
        subprocess.run(
            ["git", "clone", f"git@github.com:primait/{self._repo}.git"],
            cwd=self._projects_home,
            check=True,
        )

    def checkout(self, branch, new=False):
        try:
            if new:
                subprocess.run(
                    ["git", "checkout", "-b", branch],
                    cwd=self._path,
                    check=True,
                    stdout=subprocess.DEVNULL,
                    stderr=subprocess.DEVNULL,
                )
            else:
                subprocess.run(
                    ["git", "checkout", branch],
                    cwd=self._path,
                    check=True,
                    stdout=subprocess.DEVNULL,
                    stderr=subprocess.DEVNULL,
                )
                self.pull(rebase=True)
        except (CalledProcessError) as e:
            if e.returncode == 128:
                self.checkout(branch)  # caso checkout -b e branch gia' esistente
            elif e.returncode == 1:
                # caso checkout normale e branch non esistente
                self.checkout(branch, new=True)
            else:
                logger.error(f"Errore eseguendo il comando: {e}")
                sys.exit(-1)

    def commit(self, commit_message="", dummy=False):
        try:
            if dummy:
                subprocess.run(
                    ["git", "commit", "--allow-empty", "-m", commit_message],
                    cwd=self._path,
                    check=True,
                    stdout=subprocess.DEVNULL,
                    stderr=subprocess.DEVNULL,
                )
            else:
                subprocess.run(
                    ["git", "commit", "-am", commit_message],
                    cwd=self._path,
                    check=True,
                    stdout=subprocess.DEVNULL,
                    stderr=subprocess.DEVNULL,
                )
        except (CalledProcessError) as e:
            logger.error(f"Errore eseguendo il comando: {e}")
            sys.exit(-1)

    def add(self):
        try:
            subprocess.run(["git", "add", "."], cwd=self._path, check=True)
        except (CalledProcessError) as e:
            logger.error(f"Errore eseguendo il comando: {e}")
            sys.exit(-1)

    def push(self, branch, remote="origin"):
        with Halo(text=f"Pushing {self._repo}...", spinner="dots", color="magenta"):
            try:
                subprocess.run(
                    ["git", "push", remote, branch],
                    cwd=self._path,
                    check=True,
                    stdout=subprocess.DEVNULL,
                    stderr=subprocess.DEVNULL,
                )
            except (CalledProcessError) as e:
                logger.error(f"Errore eseguendo il comando: {e}")
                sys.exit(-1)

    def pull(self, rebase=False):
        with Halo(text=f"Pulling {self._repo}...", spinner="dots", color="magenta"):
            try:
                if rebase:
                    subprocess.run(
                        ["git", "pull", "--rebase"],
                        cwd=self._path,
                        check=True,
                        stdout=subprocess.DEVNULL,
                        stderr=subprocess.DEVNULL,
                    )
                else:
                    subprocess.run(
                        ["git", "pull"],
                        cwd=self._path,
                        check=True,
                        stdout=subprocess.DEVNULL,
                        stderr=subprocess.DEVNULL,
                    )
            except (CalledProcessError) as e:
                logger.error(f"Errore eseguendo il comando: {e}")
                sys.exit(-1)

    def fetch(self, remote="origin"):
        with Halo(text=f"Fetching {self._repo}...", spinner="dots", color="magenta"):
            try:
                subprocess.run(
                    ["git", "fetch", "--quiet"],
                    cwd=self._path,
                    check=True,
                    stdout=subprocess.DEVNULL,
                    stderr=subprocess.DEVNULL,
                )
                subprocess.run(
                    ["git", "fetch", "-p", remote, "--quiet"],
                    cwd=self._path,
                    check=True,
                    stdout=subprocess.DEVNULL,
                    stderr=subprocess.DEVNULL,
                )
            except (CalledProcessError) as e:
                logger.error(f"Errore eseguendo il comando: {e}")
                sys.exit(-1)

    def sync(self):
        self.fetch()
        self.checkout("master")
        self.pull()

    def check_repo_cloned(self):
        if self._repo not in os.listdir(self._projects_home):
            logger.warning(
                "Il progetto non e' presente nella tua home dei progetti, lo clono adesso.."
            )
            self.clone()

    def delete_remote_branch(self, branch):
        try:
            self.push(f":{branch}")
            return True
        except BaseException:
            return False

    def get_last_tag(self):
        self.sync()
        return (
            subprocess.check_output(["git", "describe", "--abbrev=0", "--tags"])
            .decode("utf-8")
            .strip()
        )

    def search_remote_branch(self, regex):
        try:
            output = subprocess.run(
                ["git", "branch", "-r", "--list", regex],
                cwd=self._path,
                check=True,
                stdout=subprocess.PIPE,
            )

            return output.stdout.decode("utf-8").strip("\n").strip().strip("origin/")
        except CalledProcessError:
            return ""

    def local_branch_exists(self, branch):
        try:
            subprocess.run(
                ["git", "show-ref", "--quiet", f"refs/heads/{branch}"],
                cwd=self._path,
                check=True,
            )
            return True
        except CalledProcessError:
            return False

    def remote_branch_exists(self, branch):
        try:
            self.fetch()
            subprocess.run(
                ["git", "show-ref", "--quiet", f"refs/remotes/origin/{branch}"],
                cwd=self._path,
                check=True,
            )
            return True
        except CalledProcessError:
            return False

    def current_branch_name(self):
        try:
            output = subprocess.run(
                ["git", "rev-parse", "--abbrev-ref", "HEAD"],
                cwd=self._path,
                stdout=subprocess.PIPE,
                check=True,
            ).stdout.decode("utf-8")
            return re.sub("\n", "", output)
        except (CalledProcessError) as e:
            logger.error(f"Errore eseguendo il comando: {e}")
            sys.exit(-1)

    def is_dirty(self):
        try:
            output = subprocess.run(
                ["git", "status", "--porcelain"],
                cwd=self._path,
                check=True,
                stdout=subprocess.PIPE,
            ).stdout.decode("utf-8")

            return len(output) != 0
        except (CalledProcessError) as e:
            logger.error(f"Errore eseguendo il comando: {e}")
            sys.exit(-1)

    def reset(self):
        try:
            subprocess.run(
                ["git", "reset", "HEAD", "--hard", "--quiet"],
                cwd=self._path,
                check=True,
            )
        except CalledProcessError as e:
            logger.error(f"Errore eseguendo git reset su {self._repo}: {e}")
            sys.exit(-1)


def is_repo(directory):
    try:
        subprocess.run(
            ["git", "status"],
            cwd=directory,
            check=True,
            stdout=subprocess.DEVNULL,
            stderr=subprocess.DEVNULL,
        )
        return True
    except CalledProcessError:
        return False


def get_username():
    try:
        output = subprocess.run(
            ["git", "config", "user.name"], check=True, stdout=subprocess.PIPE
        ).stdout.decode("utf-8")
        output = re.sub("[^A-Za-z]", "", output)
        return re.sub("\n", "", output)
    except CalledProcessError:
        return None


def files_changed_between_commits(from_commit, to_commit):
    p = subprocess.Popen(
        ["git", "diff", from_commit, to_commit, "--name-only"], stdout=subprocess.PIPE
    )
    result = p.communicate()[0]
    return result.decode("utf-8").splitlines()


def migrations_found(files_changed):
    for file in files_changed:
        if "migration" in file.lower():
            return True
    return False


def get_last_tag_number(tags):
    for tag in tags:
        if is_semver(tag.name):
            return tag.name
    return None


def is_semver(tag):
    try:
        return semver.parse(tag)
    except Exception:
        return None
