"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StateMachineFragment = void 0;
const chain_1 = require("./chain");
const parallel_1 = require("./states/parallel");
const state_1 = require("./states/state");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("@aws-cdk/core");
/**
 * Base class for reusable state machine fragments.
 *
 * @stability stable
 */
class StateMachineFragment extends core_1.Construct {
    /**
     * Descriptive identifier for this chainable.
     *
     * @stability stable
     */
    get id() {
        return this.node.id;
    }
    /**
     * Prefix the IDs of all states in this state machine fragment.
     *
     * Use this to avoid multiple copies of the state machine all having the
     * same state IDs.
     *
     * @param prefix The prefix to add.
     * @stability stable
     */
    prefixStates(prefix) {
        state_1.State.prefixStates(this, prefix || `${this.id}: `);
        return this;
    }
    /**
     * Wrap all states in this state machine fragment up into a single state.
     *
     * This can be used to add retry or error handling onto this state
     * machine fragment.
     *
     * Be aware that this changes the result of the inner state machine
     * to be an array with the result of the state machine in it. Adjust
     * your paths accordingly. For example, change 'outputPath' to
     * '$[0]'.
     *
     * @stability stable
     */
    toSingleState(options = {}) {
        const stateId = options.stateId || this.id;
        this.prefixStates(options.prefixStates || `${stateId}: `);
        return new parallel_1.Parallel(this, stateId, options).branch(this);
    }
    /**
     * Continue normal execution with the given state.
     *
     * @stability stable
     */
    next(next) {
        return chain_1.Chain.start(this).next(next);
    }
}
exports.StateMachineFragment = StateMachineFragment;
//# sourceMappingURL=data:application/json;base64,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