"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const s3deploy = require("@aws-cdk/aws-s3-deployment");
const s3 = require("@aws-cdk/aws-s3");
const aws_cloudfront_1 = require("@aws-cdk/aws-cloudfront");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const aws_route53_patterns_1 = require("@aws-cdk/aws-route53-patterns");
const aws_route53_targets_1 = require("@aws-cdk/aws-route53-targets");
class SPADeploy extends cdk.Construct {
    constructor(scope, id, config) {
        super(scope, id);
        if (typeof config != 'undefined') {
            this.globalConfig = config;
        }
        else {
            this.globalConfig = {
                encryptBucket: false,
                ipFilter: false
            };
        }
    }
    /**
     * Helper method to provide a configured s3 bucket
     */
    getS3Bucket(config) {
        let bucketConfig = {
            websiteIndexDocument: config.indexDoc,
            websiteErrorDocument: config.errorDoc,
            publicReadAccess: true
        };
        if (this.globalConfig.encryptBucket === true) {
            bucketConfig.encryption = s3.BucketEncryption.S3_MANAGED;
        }
        if (this.globalConfig.ipFilter === true) {
            bucketConfig.publicReadAccess = false;
        }
        let bucket = new s3.Bucket(this, 'WebsiteBucket', bucketConfig);
        if (this.globalConfig.ipFilter === true) {
            if (typeof this.globalConfig.ipList == 'undefined') {
                this.node.addError('When IP Filter is true then the IP List is required');
            }
            const bucketPolicy = new aws_iam_1.PolicyStatement();
            bucketPolicy.addAnyPrincipal();
            bucketPolicy.addActions('s3:GetObject');
            bucketPolicy.addResources(bucket.bucketArn + '/*');
            bucketPolicy.addCondition('IpAddress', {
                'aws:SourceIp': this.globalConfig.ipList
            });
            bucket.addToResourcePolicy(bucketPolicy);
        }
        return bucket;
    }
    /**
     * Helper method to provide configuration for cloudfront
     */
    getCFConfig(websiteBucket, config, cert) {
        let cfConfig = {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: websiteBucket
                    },
                    behaviors: [{ isDefaultBehavior: true }]
                }
            ],
            //We need to redirect all unknown routes back to index.html for angular routing to work
            errorConfigurations: [{
                    errorCode: 403,
                    responsePagePath: (config.errorDoc ? '/' + config.errorDoc : '/' + config.indexDoc),
                    responseCode: 200
                },
                {
                    errorCode: 404,
                    responsePagePath: (config.errorDoc ? '/' + config.errorDoc : '/' + config.indexDoc),
                    responseCode: 200
                }]
        };
        if (typeof config.certificateARN != 'undefined' && typeof config.cfAliases != 'undefined') {
            cfConfig.aliasConfiguration = {
                acmCertRef: config.certificateARN,
                names: config.cfAliases
            };
        }
        if (typeof config.zoneName != 'undefined' && typeof cert != 'undefined') {
            cfConfig.viewerCertificate = aws_cloudfront_1.ViewerCertificate.fromAcmCertificate(cert, {
                aliases: [config.zoneName],
            });
        }
        return cfConfig;
    }
    /**
     * Basic setup needed for a non-ssl, non vanity url, non cached s3 website
     */
    createBasicSite(config) {
        const websiteBucket = this.getS3Bucket(config);
        new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            sources: [s3deploy.Source.asset(config.websiteFolder)],
            destinationBucket: websiteBucket,
        });
        let cfnOutputConfig = {
            description: 'The url of the website',
            value: websiteBucket.bucketWebsiteUrl
        };
        if (config.exportWebsiteUrlOutput === true) {
            if (typeof config.exportWebsiteUrlName == 'undefined' || config.exportWebsiteUrlName === '') {
                this.node.addError('When Output URL as AWS Export property is true then the output name is required');
            }
            cfnOutputConfig.exportName = config.exportWebsiteUrlName;
        }
        new cdk.CfnOutput(this, 'URL', cfnOutputConfig);
    }
    /**
     * This will create an s3 deployment fronted by a cloudfront distribution
     * It will also setup error forwarding and unauth forwarding back to indexDoc
     */
    createSiteWithCloudfront(config) {
        const websiteBucket = this.getS3Bucket(config);
        const distribution = new aws_cloudfront_1.CloudFrontWebDistribution(this, 'cloudfrontDistribution', this.getCFConfig(websiteBucket, config));
        new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            sources: [s3deploy.Source.asset(config.websiteFolder)],
            destinationBucket: websiteBucket,
            //Invalidate the cache for / and index.html when we deploy so that cloudfront serves latest site
            distribution: distribution,
            distributionPaths: ['/', '/' + config.indexDoc]
        });
        new cdk.CfnOutput(this, 'cloudfront domain', {
            description: 'The domain of the website',
            value: distribution.domainName
        });
    }
    /**
     * S3 Deployment, cloudfront distribution, ssl cert and error forwarding auto
     * configured by using the details in the hosted zone provided
     */
    createSiteFromHostedZone(config) {
        const websiteBucket = this.getS3Bucket(config);
        let zone = aws_route53_1.HostedZone.fromLookup(this, 'HostedZone', { domainName: config.zoneName });
        let cert = new aws_certificatemanager_1.DnsValidatedCertificate(this, 'Certificate', {
            hostedZone: zone,
            domainName: config.zoneName,
            region: 'us-east-1',
        });
        const distribution = new aws_cloudfront_1.CloudFrontWebDistribution(this, 'cloudfrontDistribution', this.getCFConfig(websiteBucket, config, cert));
        new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            sources: [s3deploy.Source.asset(config.websiteFolder)],
            destinationBucket: websiteBucket,
            //Invalidate the cache for / and index.html when we deploy so that cloudfront serves latest site
            distribution: distribution,
            distributionPaths: ['/', '/' + config.indexDoc]
        });
        new aws_route53_1.ARecord(this, 'Alias', {
            zone,
            recordName: config.zoneName,
            target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.CloudFrontTarget(distribution))
        });
        new aws_route53_patterns_1.HttpsRedirect(this, 'Redirect', {
            zone,
            recordNames: ['www.' + config.zoneName],
            targetDomain: config.zoneName,
        });
    }
}
exports.SPADeploy = SPADeploy;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3BhLWRlcGxveS1jb25zdHJ1Y3QuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJzcGEtZGVwbG95LWNvbnN0cnVjdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBLHFDQUFzQztBQUN0Qyx1REFBdUQ7QUFDdkQsc0NBQXVDO0FBQ3ZDLDREQUFzRjtBQUN0Riw4Q0FBbUQ7QUFDbkQsc0RBQXlFO0FBQ3pFLDRFQUEwRTtBQUMxRSx3RUFBOEQ7QUFDOUQsc0VBQWdFO0FBeUJoRSxNQUFhLFNBQVUsU0FBUSxHQUFHLENBQUMsU0FBUztJQUd4QyxZQUFZLEtBQW9CLEVBQUUsRUFBUyxFQUFFLE1BQXVCO1FBQ2hFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsSUFBRyxPQUFPLE1BQU0sSUFBSSxXQUFXLEVBQUM7WUFDOUIsSUFBSSxDQUFDLFlBQVksR0FBRyxNQUFNLENBQUM7U0FDNUI7YUFBTTtZQUNMLElBQUksQ0FBQyxZQUFZLEdBQUc7Z0JBQ2xCLGFBQWEsRUFBQyxLQUFLO2dCQUNuQixRQUFRLEVBQUUsS0FBSzthQUNoQixDQUFBO1NBQ0Y7SUFDTCxDQUFDO0lBRUQ7O09BRUc7SUFDSyxXQUFXLENBQUMsTUFBc0I7UUFFeEMsSUFBSSxZQUFZLEdBQU87WUFDbkIsb0JBQW9CLEVBQUUsTUFBTSxDQUFDLFFBQVE7WUFDckMsb0JBQW9CLEVBQUUsTUFBTSxDQUFDLFFBQVE7WUFDckMsZ0JBQWdCLEVBQUUsSUFBSTtTQUN2QixDQUFDO1FBRUosSUFBRyxJQUFJLENBQUMsWUFBWSxDQUFDLGFBQWEsS0FBSyxJQUFJLEVBQUM7WUFDMUMsWUFBWSxDQUFDLFVBQVUsR0FBRyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsVUFBVSxDQUFBO1NBQ3pEO1FBRUQsSUFBRyxJQUFJLENBQUMsWUFBWSxDQUFDLFFBQVEsS0FBSyxJQUFJLEVBQUM7WUFDckMsWUFBWSxDQUFDLGdCQUFnQixHQUFHLEtBQUssQ0FBQztTQUN2QztRQUVELElBQUksTUFBTSxHQUFHLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQyxJQUFJLEVBQUUsZUFBZSxFQUFFLFlBQVksQ0FBQyxDQUFDO1FBRWhFLElBQUcsSUFBSSxDQUFDLFlBQVksQ0FBQyxRQUFRLEtBQUssSUFBSSxFQUFDO1lBQ3JDLElBQUcsT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDLE1BQU0sSUFBSSxXQUFXLEVBQUU7Z0JBQ2pELElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLHFEQUFxRCxDQUFDLENBQUM7YUFDM0U7WUFFRCxNQUFNLFlBQVksR0FBRyxJQUFJLHlCQUFlLEVBQUUsQ0FBQztZQUMzQyxZQUFZLENBQUMsZUFBZSxFQUFFLENBQUM7WUFDL0IsWUFBWSxDQUFDLFVBQVUsQ0FBQyxjQUFjLENBQUMsQ0FBQztZQUN4QyxZQUFZLENBQUMsWUFBWSxDQUFDLE1BQU0sQ0FBQyxTQUFTLEdBQUcsSUFBSSxDQUFDLENBQUM7WUFDbkQsWUFBWSxDQUFDLFlBQVksQ0FBQyxXQUFXLEVBQUU7Z0JBQ3JDLGNBQWMsRUFBRSxJQUFJLENBQUMsWUFBWSxDQUFDLE1BQU07YUFDekMsQ0FBQyxDQUFDO1lBRUgsTUFBTSxDQUFDLG1CQUFtQixDQUFDLFlBQVksQ0FBQyxDQUFDO1NBQzFDO1FBRUQsT0FBTyxNQUFNLENBQUM7SUFDaEIsQ0FBQztJQUVEOztPQUVHO0lBQ0ssV0FBVyxDQUFDLGFBQXVCLEVBQUUsTUFBVSxFQUFFLElBQTZCO1FBQ2pGLElBQUksUUFBUSxHQUFPO1lBQ2xCLGFBQWEsRUFBRTtnQkFDYjtvQkFDRSxjQUFjLEVBQUU7d0JBQ2QsY0FBYyxFQUFFLGFBQWE7cUJBQzlCO29CQUNELFNBQVMsRUFBRyxDQUFFLEVBQUMsaUJBQWlCLEVBQUUsSUFBSSxFQUFDLENBQUM7aUJBQ3pDO2FBQ0Y7WUFDRCx1RkFBdUY7WUFDdkYsbUJBQW1CLEVBQUUsQ0FBQztvQkFDcEIsU0FBUyxFQUFFLEdBQUc7b0JBQ2QsZ0JBQWdCLEVBQUUsQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxHQUFHLEdBQUMsTUFBTSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsR0FBRyxHQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUM7b0JBQy9FLFlBQVksRUFBRSxHQUFHO2lCQUNsQjtnQkFDRDtvQkFDRSxTQUFTLEVBQUUsR0FBRztvQkFDZCxnQkFBZ0IsRUFBRSxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLEdBQUcsR0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxHQUFHLEdBQUMsTUFBTSxDQUFDLFFBQVEsQ0FBQztvQkFDL0UsWUFBWSxFQUFFLEdBQUc7aUJBQ2xCLENBQUM7U0FDSCxDQUFBO1FBRUQsSUFBRyxPQUFPLE1BQU0sQ0FBQyxjQUFjLElBQUksV0FBVyxJQUFJLE9BQU8sTUFBTSxDQUFDLFNBQVMsSUFBSSxXQUFXLEVBQUU7WUFDdEYsUUFBUSxDQUFDLGtCQUFrQixHQUFHO2dCQUMxQixVQUFVLEVBQUUsTUFBTSxDQUFDLGNBQWM7Z0JBQ2pDLEtBQUssRUFBRSxNQUFNLENBQUMsU0FBUzthQUMxQixDQUFBO1NBQ0o7UUFFRCxJQUFHLE9BQU8sTUFBTSxDQUFDLFFBQVEsSUFBRyxXQUFXLElBQUksT0FBTyxJQUFJLElBQUksV0FBVyxFQUFDO1lBQ3BFLFFBQVEsQ0FBQyxpQkFBaUIsR0FBRyxrQ0FBaUIsQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLEVBQUU7Z0JBQ3RFLE9BQU8sRUFBRSxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUM7YUFDM0IsQ0FBQyxDQUFBO1NBQ0g7UUFFRCxPQUFPLFFBQVEsQ0FBQztJQUNwQixDQUFDO0lBRUQ7O09BRUc7SUFDSSxlQUFlLENBQUMsTUFBc0I7UUFDekMsTUFBTSxhQUFhLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUUvQyxJQUFJLFFBQVEsQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLEVBQUUsa0JBQWtCLEVBQUU7WUFDdEQsT0FBTyxFQUFFLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLGFBQWEsQ0FBQyxDQUFDO1lBQ3RELGlCQUFpQixFQUFFLGFBQWE7U0FDakMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxlQUFlLEdBQU87WUFDeEIsV0FBVyxFQUFFLHdCQUF3QjtZQUNyQyxLQUFLLEVBQUUsYUFBYSxDQUFDLGdCQUFnQjtTQUN0QyxDQUFDO1FBRUYsSUFBSSxNQUFNLENBQUMsc0JBQXNCLEtBQUssSUFBSSxFQUFFO1lBQzFDLElBQUcsT0FBTyxNQUFNLENBQUMsb0JBQW9CLElBQUksV0FBVyxJQUFJLE1BQU0sQ0FBQyxvQkFBb0IsS0FBSyxFQUFFLEVBQUU7Z0JBQzFGLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLGlGQUFpRixDQUFDLENBQUM7YUFDdkc7WUFDRCxlQUFlLENBQUMsVUFBVSxHQUFHLE1BQU0sQ0FBQyxvQkFBb0IsQ0FBQztTQUMxRDtRQUVELElBQUksR0FBRyxDQUFDLFNBQVMsQ0FBQyxJQUFJLEVBQUUsS0FBSyxFQUFFLGVBQWUsQ0FBQyxDQUFDO0lBQ3BELENBQUM7SUFFRDs7O09BR0c7SUFDSSx3QkFBd0IsQ0FBQyxNQUFzQjtRQUNsRCxNQUFNLGFBQWEsR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQy9DLE1BQU0sWUFBWSxHQUFHLElBQUksMENBQXlCLENBQUMsSUFBSSxFQUFFLHdCQUF3QixFQUFFLElBQUksQ0FBQyxXQUFXLENBQUMsYUFBYSxFQUFFLE1BQU0sQ0FBQyxDQUFDLENBQUM7UUFFNUgsSUFBSSxRQUFRLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxFQUFFLGtCQUFrQixFQUFFO1lBQ3RELE9BQU8sRUFBRSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxhQUFhLENBQUMsQ0FBQztZQUN0RCxpQkFBaUIsRUFBRSxhQUFhO1lBQ2hDLGdHQUFnRztZQUNoRyxZQUFZLEVBQUUsWUFBWTtZQUMxQixpQkFBaUIsRUFBRSxDQUFDLEdBQUcsRUFBRSxHQUFHLEdBQUMsTUFBTSxDQUFDLFFBQVEsQ0FBQztTQUM5QyxDQUFDLENBQUM7UUFFSCxJQUFJLEdBQUcsQ0FBQyxTQUFTLENBQUMsSUFBSSxFQUFFLG1CQUFtQixFQUFFO1lBQzNDLFdBQVcsRUFBRSwyQkFBMkI7WUFDeEMsS0FBSyxFQUFFLFlBQVksQ0FBQyxVQUFVO1NBQy9CLENBQUMsQ0FBQTtJQUNOLENBQUM7SUFFRDs7O09BR0c7SUFDSSx3QkFBd0IsQ0FBQyxNQUF1QjtRQUNwRCxNQUFNLGFBQWEsR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQy9DLElBQUksSUFBSSxHQUFHLHdCQUFVLENBQUMsVUFBVSxDQUFDLElBQUksRUFBRSxZQUFZLEVBQUUsRUFBRSxVQUFVLEVBQUUsTUFBTSxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUM7UUFDdEYsSUFBSSxJQUFJLEdBQUcsSUFBSSxnREFBdUIsQ0FBQyxJQUFJLEVBQUUsYUFBYSxFQUFFO1lBQ25ELFVBQVUsRUFBRSxJQUFJO1lBQ2hCLFVBQVUsRUFBRSxNQUFNLENBQUMsUUFBUTtZQUMzQixNQUFNLEVBQUUsV0FBVztTQUN0QixDQUFDLENBQUM7UUFFUCxNQUFNLFlBQVksR0FBRyxJQUFJLDBDQUF5QixDQUFDLElBQUksRUFBRSx3QkFBd0IsRUFBRSxJQUFJLENBQUMsV0FBVyxDQUFDLGFBQWEsRUFBRSxNQUFNLEVBQUUsSUFBSSxDQUFDLENBQUMsQ0FBQztRQUVsSSxJQUFJLFFBQVEsQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLEVBQUUsa0JBQWtCLEVBQUU7WUFDdEQsT0FBTyxFQUFFLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLGFBQWEsQ0FBQyxDQUFDO1lBQ3RELGlCQUFpQixFQUFFLGFBQWE7WUFDaEMsZ0dBQWdHO1lBQ2hHLFlBQVksRUFBRSxZQUFZO1lBQzFCLGlCQUFpQixFQUFFLENBQUMsR0FBRyxFQUFFLEdBQUcsR0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDO1NBQzlDLENBQUMsQ0FBQztRQUVILElBQUkscUJBQU8sQ0FBQyxJQUFJLEVBQUUsT0FBTyxFQUFFO1lBQ3pCLElBQUk7WUFDSixVQUFVLEVBQUUsTUFBTSxDQUFDLFFBQVE7WUFDM0IsTUFBTSxFQUFFLDBCQUFZLENBQUMsU0FBUyxDQUFDLElBQUksc0NBQWdCLENBQUMsWUFBWSxDQUFDLENBQUM7U0FDbkUsQ0FBQyxDQUFBO1FBRUYsSUFBSSxvQ0FBYSxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDaEMsSUFBSTtZQUNKLFdBQVcsRUFBRSxDQUFDLE1BQU0sR0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDO1lBQ3JDLFlBQVksRUFBRSxNQUFNLENBQUMsUUFBUTtTQUNoQyxDQUFDLENBQUM7SUFDUCxDQUFDO0NBRUo7QUF0TEQsOEJBc0xDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IGNkayA9IHJlcXVpcmUoJ0Bhd3MtY2RrL2NvcmUnKTtcbmltcG9ydCBzM2RlcGxveT0gcmVxdWlyZSgnQGF3cy1jZGsvYXdzLXMzLWRlcGxveW1lbnQnKTtcbmltcG9ydCBzMyA9IHJlcXVpcmUoJ0Bhd3MtY2RrL2F3cy1zMycpO1xuaW1wb3J0IHsgQ2xvdWRGcm9udFdlYkRpc3RyaWJ1dGlvbiwgVmlld2VyQ2VydGlmaWNhdGUgfSBmcm9tICdAYXdzLWNkay9hd3MtY2xvdWRmcm9udCdcbmltcG9ydCB7IFBvbGljeVN0YXRlbWVudCB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0IHsgSG9zdGVkWm9uZSwgQVJlY29yZCwgUmVjb3JkVGFyZ2V0IH0gZnJvbSAnQGF3cy1jZGsvYXdzLXJvdXRlNTMnO1xuaW1wb3J0IHsgRG5zVmFsaWRhdGVkQ2VydGlmaWNhdGUgfSBmcm9tICdAYXdzLWNkay9hd3MtY2VydGlmaWNhdGVtYW5hZ2VyJztcbmltcG9ydCB7IEh0dHBzUmVkaXJlY3QgfSBmcm9tICdAYXdzLWNkay9hd3Mtcm91dGU1My1wYXR0ZXJucyc7XG5pbXBvcnQgeyBDbG91ZEZyb250VGFyZ2V0IH0gZnJvbSAnQGF3cy1jZGsvYXdzLXJvdXRlNTMtdGFyZ2V0cyc7XG5cbmV4cG9ydCBpbnRlcmZhY2UgU1BBRGVwbG95Q29uZmlnIHtcbiAgcmVhZG9ubHkgaW5kZXhEb2M6c3RyaW5nLFxuICByZWFkb25seSBlcnJvckRvYz86c3RyaW5nLFxuICByZWFkb25seSB3ZWJzaXRlRm9sZGVyOiBzdHJpbmcsXG4gIHJlYWRvbmx5IGNlcnRpZmljYXRlQVJOPzogc3RyaW5nLFxuICByZWFkb25seSBjZkFsaWFzZXM/OiBzdHJpbmdbXSxcbiAgcmVhZG9ubHkgZXhwb3J0V2Vic2l0ZVVybE91dHB1dD86Ym9vbGVhbixcbiAgcmVhZG9ubHkgZXhwb3J0V2Vic2l0ZVVybE5hbWU/OiBzdHJpbmdcbn1cblxuZXhwb3J0IGludGVyZmFjZSBIb3N0ZWRab25lQ29uZmlnIHtcbiAgcmVhZG9ubHkgaW5kZXhEb2M6c3RyaW5nLFxuICByZWFkb25seSBlcnJvckRvYz86c3RyaW5nLFxuICByZWFkb25seSB3ZWJzaXRlRm9sZGVyOiBzdHJpbmcsXG4gIHJlYWRvbmx5IHpvbmVOYW1lOiBzdHJpbmdcbn1cblxuZXhwb3J0IGludGVyZmFjZSBTUEFHbG9iYWxDb25maWcge1xuICByZWFkb25seSBlbmNyeXB0QnVja2V0Pzpib29sZWFuLFxuICByZWFkb25seSBpcEZpbHRlcj86Ym9vbGVhbixcbiAgcmVhZG9ubHkgaXBMaXN0PzpzdHJpbmdbXVxufVxuXG5leHBvcnQgY2xhc3MgU1BBRGVwbG95IGV4dGVuZHMgY2RrLkNvbnN0cnVjdCB7XG4gICAgZ2xvYmFsQ29uZmlnOiBTUEFHbG9iYWxDb25maWc7XG4gICAgXG4gICAgY29uc3RydWN0b3Ioc2NvcGU6IGNkay5Db25zdHJ1Y3QsIGlkOnN0cmluZywgY29uZmlnPzpTUEFHbG9iYWxDb25maWcpe1xuICAgICAgICBzdXBlcihzY29wZSwgaWQpO1xuICAgICAgICBcbiAgICAgICAgaWYodHlwZW9mIGNvbmZpZyAhPSAndW5kZWZpbmVkJyl7XG4gICAgICAgICAgdGhpcy5nbG9iYWxDb25maWcgPSBjb25maWc7XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgdGhpcy5nbG9iYWxDb25maWcgPSB7XG4gICAgICAgICAgICBlbmNyeXB0QnVja2V0OmZhbHNlLFxuICAgICAgICAgICAgaXBGaWx0ZXI6IGZhbHNlXG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgfVxuICAgIFxuICAgIC8qKlxuICAgICAqIEhlbHBlciBtZXRob2QgdG8gcHJvdmlkZSBhIGNvbmZpZ3VyZWQgczMgYnVja2V0XG4gICAgICovXG4gICAgcHJpdmF0ZSBnZXRTM0J1Y2tldChjb25maWc6U1BBRGVwbG95Q29uZmlnKSB7XG4gICAgICBcbiAgICAgIGxldCBidWNrZXRDb25maWc6YW55ID0ge1xuICAgICAgICAgIHdlYnNpdGVJbmRleERvY3VtZW50OiBjb25maWcuaW5kZXhEb2MsXG4gICAgICAgICAgd2Vic2l0ZUVycm9yRG9jdW1lbnQ6IGNvbmZpZy5lcnJvckRvYyxcbiAgICAgICAgICBwdWJsaWNSZWFkQWNjZXNzOiB0cnVlXG4gICAgICAgIH07XG4gICAgICAgIFxuICAgICAgaWYodGhpcy5nbG9iYWxDb25maWcuZW5jcnlwdEJ1Y2tldCA9PT0gdHJ1ZSl7XG4gICAgICAgIGJ1Y2tldENvbmZpZy5lbmNyeXB0aW9uID0gczMuQnVja2V0RW5jcnlwdGlvbi5TM19NQU5BR0VEXG4gICAgICB9XG4gICAgICBcbiAgICAgIGlmKHRoaXMuZ2xvYmFsQ29uZmlnLmlwRmlsdGVyID09PSB0cnVlKXtcbiAgICAgICAgYnVja2V0Q29uZmlnLnB1YmxpY1JlYWRBY2Nlc3MgPSBmYWxzZTtcbiAgICAgIH1cbiAgICAgICAgXG4gICAgICBsZXQgYnVja2V0ID0gbmV3IHMzLkJ1Y2tldCh0aGlzLCAnV2Vic2l0ZUJ1Y2tldCcsIGJ1Y2tldENvbmZpZyk7XG4gICAgICBcbiAgICAgIGlmKHRoaXMuZ2xvYmFsQ29uZmlnLmlwRmlsdGVyID09PSB0cnVlKXtcbiAgICAgICAgaWYodHlwZW9mIHRoaXMuZ2xvYmFsQ29uZmlnLmlwTGlzdCA9PSAndW5kZWZpbmVkJykge1xuICAgICAgICAgIHRoaXMubm9kZS5hZGRFcnJvcignV2hlbiBJUCBGaWx0ZXIgaXMgdHJ1ZSB0aGVuIHRoZSBJUCBMaXN0IGlzIHJlcXVpcmVkJyk7XG4gICAgICAgIH1cbiAgICAgICAgXG4gICAgICAgIGNvbnN0IGJ1Y2tldFBvbGljeSA9IG5ldyBQb2xpY3lTdGF0ZW1lbnQoKTtcbiAgICAgICAgYnVja2V0UG9saWN5LmFkZEFueVByaW5jaXBhbCgpO1xuICAgICAgICBidWNrZXRQb2xpY3kuYWRkQWN0aW9ucygnczM6R2V0T2JqZWN0Jyk7XG4gICAgICAgIGJ1Y2tldFBvbGljeS5hZGRSZXNvdXJjZXMoYnVja2V0LmJ1Y2tldEFybiArICcvKicpO1xuICAgICAgICBidWNrZXRQb2xpY3kuYWRkQ29uZGl0aW9uKCdJcEFkZHJlc3MnLCB7XG4gICAgICAgICAgJ2F3czpTb3VyY2VJcCc6IHRoaXMuZ2xvYmFsQ29uZmlnLmlwTGlzdFxuICAgICAgICB9KTtcbiAgICAgICAgXG4gICAgICAgIGJ1Y2tldC5hZGRUb1Jlc291cmNlUG9saWN5KGJ1Y2tldFBvbGljeSk7XG4gICAgICB9XG4gICAgICBcbiAgICAgIHJldHVybiBidWNrZXQ7XG4gICAgfVxuICAgIFxuICAgIC8qKlxuICAgICAqIEhlbHBlciBtZXRob2QgdG8gcHJvdmlkZSBjb25maWd1cmF0aW9uIGZvciBjbG91ZGZyb250IFxuICAgICAqL1xuICAgIHByaXZhdGUgZ2V0Q0ZDb25maWcod2Vic2l0ZUJ1Y2tldDpzMy5CdWNrZXQsIGNvbmZpZzphbnksIGNlcnQ/OkRuc1ZhbGlkYXRlZENlcnRpZmljYXRlKSB7XG4gICAgICAgICBsZXQgY2ZDb25maWc6YW55ID0ge1xuICAgICAgICAgIG9yaWdpbkNvbmZpZ3M6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgczNPcmlnaW5Tb3VyY2U6IHtcbiAgICAgICAgICAgICAgICBzM0J1Y2tldFNvdXJjZTogd2Vic2l0ZUJ1Y2tldFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICBiZWhhdmlvcnMgOiBbIHtpc0RlZmF1bHRCZWhhdmlvcjogdHJ1ZX1dXG4gICAgICAgICAgICB9XG4gICAgICAgICAgXSxcbiAgICAgICAgICAvL1dlIG5lZWQgdG8gcmVkaXJlY3QgYWxsIHVua25vd24gcm91dGVzIGJhY2sgdG8gaW5kZXguaHRtbCBmb3IgYW5ndWxhciByb3V0aW5nIHRvIHdvcmtcbiAgICAgICAgICBlcnJvckNvbmZpZ3VyYXRpb25zOiBbe1xuICAgICAgICAgICAgZXJyb3JDb2RlOiA0MDMsXG4gICAgICAgICAgICByZXNwb25zZVBhZ2VQYXRoOiAoY29uZmlnLmVycm9yRG9jID8gJy8nK2NvbmZpZy5lcnJvckRvYyA6ICcvJytjb25maWcuaW5kZXhEb2MpLFxuICAgICAgICAgICAgcmVzcG9uc2VDb2RlOiAyMDBcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIGVycm9yQ29kZTogNDA0LFxuICAgICAgICAgICAgcmVzcG9uc2VQYWdlUGF0aDogKGNvbmZpZy5lcnJvckRvYyA/ICcvJytjb25maWcuZXJyb3JEb2MgOiAnLycrY29uZmlnLmluZGV4RG9jKSxcbiAgICAgICAgICAgIHJlc3BvbnNlQ29kZTogMjAwXG4gICAgICAgICAgfV1cbiAgICAgICAgfVxuICAgICAgICBcbiAgICAgICAgaWYodHlwZW9mIGNvbmZpZy5jZXJ0aWZpY2F0ZUFSTiAhPSAndW5kZWZpbmVkJyAmJiB0eXBlb2YgY29uZmlnLmNmQWxpYXNlcyAhPSAndW5kZWZpbmVkJykge1xuICAgICAgICAgICAgY2ZDb25maWcuYWxpYXNDb25maWd1cmF0aW9uID0ge1xuICAgICAgICAgICAgICAgIGFjbUNlcnRSZWY6IGNvbmZpZy5jZXJ0aWZpY2F0ZUFSTixcbiAgICAgICAgICAgICAgICBuYW1lczogY29uZmlnLmNmQWxpYXNlc1xuICAgICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIFxuICAgICAgICBpZih0eXBlb2YgY29uZmlnLnpvbmVOYW1lICE9J3VuZGVmaW5lZCcgJiYgdHlwZW9mIGNlcnQgIT0gJ3VuZGVmaW5lZCcpe1xuICAgICAgICAgIGNmQ29uZmlnLnZpZXdlckNlcnRpZmljYXRlID0gVmlld2VyQ2VydGlmaWNhdGUuZnJvbUFjbUNlcnRpZmljYXRlKGNlcnQsIHtcbiAgICAgICAgICAgIGFsaWFzZXM6IFtjb25maWcuem9uZU5hbWVdLFxuICAgICAgICAgIH0pXG4gICAgICAgIH1cbiAgICAgICAgXG4gICAgICAgIHJldHVybiBjZkNvbmZpZztcbiAgICB9XG4gICAgXG4gICAgLyoqXG4gICAgICogQmFzaWMgc2V0dXAgbmVlZGVkIGZvciBhIG5vbi1zc2wsIG5vbiB2YW5pdHkgdXJsLCBub24gY2FjaGVkIHMzIHdlYnNpdGVcbiAgICAgKi9cbiAgICBwdWJsaWMgY3JlYXRlQmFzaWNTaXRlKGNvbmZpZzpTUEFEZXBsb3lDb25maWcpIHtcbiAgICAgICAgY29uc3Qgd2Vic2l0ZUJ1Y2tldCA9IHRoaXMuZ2V0UzNCdWNrZXQoY29uZmlnKTtcbiAgICAgICAgXG4gICAgICAgIG5ldyBzM2RlcGxveS5CdWNrZXREZXBsb3ltZW50KHRoaXMsICdCdWNrZXREZXBsb3ltZW50Jywge1xuICAgICAgICAgIHNvdXJjZXM6IFtzM2RlcGxveS5Tb3VyY2UuYXNzZXQoY29uZmlnLndlYnNpdGVGb2xkZXIpXSwgXG4gICAgICAgICAgZGVzdGluYXRpb25CdWNrZXQ6IHdlYnNpdGVCdWNrZXQsXG4gICAgICAgIH0pO1xuICAgICAgICBcbiAgICAgICAgbGV0IGNmbk91dHB1dENvbmZpZzphbnkgPSB7XG4gICAgICAgICAgZGVzY3JpcHRpb246ICdUaGUgdXJsIG9mIHRoZSB3ZWJzaXRlJyxcbiAgICAgICAgICB2YWx1ZTogd2Vic2l0ZUJ1Y2tldC5idWNrZXRXZWJzaXRlVXJsXG4gICAgICAgIH07XG5cbiAgICAgICAgaWYgKGNvbmZpZy5leHBvcnRXZWJzaXRlVXJsT3V0cHV0ID09PSB0cnVlKSB7XG4gICAgICAgICAgaWYodHlwZW9mIGNvbmZpZy5leHBvcnRXZWJzaXRlVXJsTmFtZSA9PSAndW5kZWZpbmVkJyB8fCBjb25maWcuZXhwb3J0V2Vic2l0ZVVybE5hbWUgPT09ICcnKSB7XG4gICAgICAgICAgICB0aGlzLm5vZGUuYWRkRXJyb3IoJ1doZW4gT3V0cHV0IFVSTCBhcyBBV1MgRXhwb3J0IHByb3BlcnR5IGlzIHRydWUgdGhlbiB0aGUgb3V0cHV0IG5hbWUgaXMgcmVxdWlyZWQnKTtcbiAgICAgICAgICB9XG4gICAgICAgICAgY2ZuT3V0cHV0Q29uZmlnLmV4cG9ydE5hbWUgPSBjb25maWcuZXhwb3J0V2Vic2l0ZVVybE5hbWU7XG4gICAgICAgIH1cblxuICAgICAgICBuZXcgY2RrLkNmbk91dHB1dCh0aGlzLCAnVVJMJywgY2ZuT3V0cHV0Q29uZmlnKTtcbiAgICB9XG4gICAgXG4gICAgLyoqXG4gICAgICogVGhpcyB3aWxsIGNyZWF0ZSBhbiBzMyBkZXBsb3ltZW50IGZyb250ZWQgYnkgYSBjbG91ZGZyb250IGRpc3RyaWJ1dGlvblxuICAgICAqIEl0IHdpbGwgYWxzbyBzZXR1cCBlcnJvciBmb3J3YXJkaW5nIGFuZCB1bmF1dGggZm9yd2FyZGluZyBiYWNrIHRvIGluZGV4RG9jXG4gICAgICovXG4gICAgcHVibGljIGNyZWF0ZVNpdGVXaXRoQ2xvdWRmcm9udChjb25maWc6U1BBRGVwbG95Q29uZmlnKSB7XG4gICAgICAgIGNvbnN0IHdlYnNpdGVCdWNrZXQgPSB0aGlzLmdldFMzQnVja2V0KGNvbmZpZyk7XG4gICAgICAgIGNvbnN0IGRpc3RyaWJ1dGlvbiA9IG5ldyBDbG91ZEZyb250V2ViRGlzdHJpYnV0aW9uKHRoaXMsICdjbG91ZGZyb250RGlzdHJpYnV0aW9uJywgdGhpcy5nZXRDRkNvbmZpZyh3ZWJzaXRlQnVja2V0LCBjb25maWcpKTtcbiAgICAgICAgXG4gICAgICAgIG5ldyBzM2RlcGxveS5CdWNrZXREZXBsb3ltZW50KHRoaXMsICdCdWNrZXREZXBsb3ltZW50Jywge1xuICAgICAgICAgIHNvdXJjZXM6IFtzM2RlcGxveS5Tb3VyY2UuYXNzZXQoY29uZmlnLndlYnNpdGVGb2xkZXIpXSwgXG4gICAgICAgICAgZGVzdGluYXRpb25CdWNrZXQ6IHdlYnNpdGVCdWNrZXQsXG4gICAgICAgICAgLy9JbnZhbGlkYXRlIHRoZSBjYWNoZSBmb3IgLyBhbmQgaW5kZXguaHRtbCB3aGVuIHdlIGRlcGxveSBzbyB0aGF0IGNsb3VkZnJvbnQgc2VydmVzIGxhdGVzdCBzaXRlXG4gICAgICAgICAgZGlzdHJpYnV0aW9uOiBkaXN0cmlidXRpb24sXG4gICAgICAgICAgZGlzdHJpYnV0aW9uUGF0aHM6IFsnLycsICcvJytjb25maWcuaW5kZXhEb2NdXG4gICAgICAgIH0pO1xuICAgICAgICBcbiAgICAgICAgbmV3IGNkay5DZm5PdXRwdXQodGhpcywgJ2Nsb3VkZnJvbnQgZG9tYWluJywge1xuICAgICAgICAgIGRlc2NyaXB0aW9uOiAnVGhlIGRvbWFpbiBvZiB0aGUgd2Vic2l0ZScsXG4gICAgICAgICAgdmFsdWU6IGRpc3RyaWJ1dGlvbi5kb21haW5OYW1lXG4gICAgICAgIH0pXG4gICAgfVxuICAgIFxuICAgIC8qKlxuICAgICAqIFMzIERlcGxveW1lbnQsIGNsb3VkZnJvbnQgZGlzdHJpYnV0aW9uLCBzc2wgY2VydCBhbmQgZXJyb3IgZm9yd2FyZGluZyBhdXRvXG4gICAgICogY29uZmlndXJlZCBieSB1c2luZyB0aGUgZGV0YWlscyBpbiB0aGUgaG9zdGVkIHpvbmUgcHJvdmlkZWRcbiAgICAgKi9cbiAgICBwdWJsaWMgY3JlYXRlU2l0ZUZyb21Ib3N0ZWRab25lKGNvbmZpZzpIb3N0ZWRab25lQ29uZmlnKSB7XG4gICAgICAgY29uc3Qgd2Vic2l0ZUJ1Y2tldCA9IHRoaXMuZ2V0UzNCdWNrZXQoY29uZmlnKTtcbiAgICAgICBsZXQgem9uZSA9IEhvc3RlZFpvbmUuZnJvbUxvb2t1cCh0aGlzLCAnSG9zdGVkWm9uZScsIHsgZG9tYWluTmFtZTogY29uZmlnLnpvbmVOYW1lIH0pO1xuICAgICAgIGxldCBjZXJ0ID0gbmV3IERuc1ZhbGlkYXRlZENlcnRpZmljYXRlKHRoaXMsICdDZXJ0aWZpY2F0ZScsIHtcbiAgICAgICAgICAgICAgICBob3N0ZWRab25lOiB6b25lLFxuICAgICAgICAgICAgICAgIGRvbWFpbk5hbWU6IGNvbmZpZy56b25lTmFtZSxcbiAgICAgICAgICAgICAgICByZWdpb246ICd1cy1lYXN0LTEnLFxuICAgICAgICAgICAgfSk7XG4gICAgICAgICAgICBcbiAgICAgICAgY29uc3QgZGlzdHJpYnV0aW9uID0gbmV3IENsb3VkRnJvbnRXZWJEaXN0cmlidXRpb24odGhpcywgJ2Nsb3VkZnJvbnREaXN0cmlidXRpb24nLCB0aGlzLmdldENGQ29uZmlnKHdlYnNpdGVCdWNrZXQsIGNvbmZpZywgY2VydCkpO1xuICAgICAgICBcbiAgICAgICAgbmV3IHMzZGVwbG95LkJ1Y2tldERlcGxveW1lbnQodGhpcywgJ0J1Y2tldERlcGxveW1lbnQnLCB7XG4gICAgICAgICAgc291cmNlczogW3MzZGVwbG95LlNvdXJjZS5hc3NldChjb25maWcud2Vic2l0ZUZvbGRlcildLCBcbiAgICAgICAgICBkZXN0aW5hdGlvbkJ1Y2tldDogd2Vic2l0ZUJ1Y2tldCxcbiAgICAgICAgICAvL0ludmFsaWRhdGUgdGhlIGNhY2hlIGZvciAvIGFuZCBpbmRleC5odG1sIHdoZW4gd2UgZGVwbG95IHNvIHRoYXQgY2xvdWRmcm9udCBzZXJ2ZXMgbGF0ZXN0IHNpdGVcbiAgICAgICAgICBkaXN0cmlidXRpb246IGRpc3RyaWJ1dGlvbixcbiAgICAgICAgICBkaXN0cmlidXRpb25QYXRoczogWycvJywgJy8nK2NvbmZpZy5pbmRleERvY11cbiAgICAgICAgfSk7XG4gICAgICAgIFxuICAgICAgICBuZXcgQVJlY29yZCh0aGlzLCAnQWxpYXMnLCB7XG4gICAgICAgICAgem9uZSxcbiAgICAgICAgICByZWNvcmROYW1lOiBjb25maWcuem9uZU5hbWUsXG4gICAgICAgICAgdGFyZ2V0OiBSZWNvcmRUYXJnZXQuZnJvbUFsaWFzKG5ldyBDbG91ZEZyb250VGFyZ2V0KGRpc3RyaWJ1dGlvbikpXG4gICAgICAgIH0pXG5cbiAgICAgICAgbmV3IEh0dHBzUmVkaXJlY3QodGhpcywgJ1JlZGlyZWN0Jywge1xuICAgICAgICAgICAgem9uZSxcbiAgICAgICAgICAgIHJlY29yZE5hbWVzOiBbJ3d3dy4nK2NvbmZpZy56b25lTmFtZV0sXG4gICAgICAgICAgICB0YXJnZXREb21haW46IGNvbmZpZy56b25lTmFtZSxcbiAgICAgICAgfSk7XG4gICAgfVxuICAgIFxufSJdfQ==