"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const yaml = require("yaml");
const __1 = require("..");
const deps_1 = require("../deps");
const logging = require("../logging");
const node_package_1 = require("../node-package");
const node_project_1 = require("../node-project");
const util_1 = require("./util");
logging.disable();
test('license file is added by default', () => {
    // WHEN
    const project = new TestNodeProject();
    // THEN
    expect(util_1.synthSnapshot(project).LICENSE).toContain('Apache License');
});
test('license file is not added if licensed is false', () => {
    // WHEN
    const project = new TestNodeProject({
        licensed: false,
    });
    // THEN
    const snapshot = util_1.synthSnapshot(project);
    expect(snapshot.LICENSE).toBeUndefined();
    expect(snapshot['.gitignore']).not.toContain('LICENSE');
    expect(snapshot['package.json'].license).toEqual('UNLICENSED');
});
describe('deps', () => {
    test('runtime deps', () => {
        // GIVEN
        const project = new TestNodeProject({
            deps: [
                'aaa@^1.2.3',
                'bbb@~4.5.6',
            ],
        });
        // WHEN
        project.addDeps('ccc');
        project.deps.addDependency('ddd', deps_1.DependencyType.RUNTIME);
        // THEN
        const pkgjson = packageJson(project);
        expect(pkgjson.dependencies).toStrictEqual({
            aaa: '^1.2.3',
            bbb: '~4.5.6',
            ccc: '*',
            ddd: '*',
        });
        expect(pkgjson.peerDependencies).toStrictEqual({});
    });
    test('dev dependencies', () => {
        // GIVEN
        const project = new TestNodeProject({
            devDeps: [
                'aaa@^1.2.3',
                'bbb@~4.5.6',
            ],
        });
        // WHEN
        project.addDevDeps('ccc');
        project.deps.addDependency('ddd', deps_1.DependencyType.TEST);
        project.deps.addDependency('eee@^1', deps_1.DependencyType.DEVENV);
        project.deps.addDependency('fff@^2', deps_1.DependencyType.BUILD);
        // THEN
        const pkgjson = packageJson(project);
        expect(pkgjson.devDependencies.aaa).toStrictEqual('^1.2.3');
        expect(pkgjson.devDependencies.bbb).toStrictEqual('~4.5.6');
        expect(pkgjson.devDependencies.ccc).toStrictEqual('*');
        expect(pkgjson.devDependencies.ddd).toStrictEqual('*');
        expect(pkgjson.devDependencies.eee).toStrictEqual('^1');
        expect(pkgjson.devDependencies.fff).toStrictEqual('^2');
        expect(pkgjson.peerDependencies).toStrictEqual({});
        expect(pkgjson.dependencieds).toBeUndefined();
    });
    test('peerDependencies', () => {
        // GIVEN
        const project = new TestNodeProject({
            peerDeps: [
                'aaa@^1.2.3',
                'bbb@~4.5.6',
            ],
        });
        // WHEN
        project.addPeerDeps('ccc');
        project.deps.addDependency('ddd', deps_1.DependencyType.PEER);
        // THEN
        const pkgjson = packageJson(project);
        expect(pkgjson.peerDependencies).toStrictEqual({
            aaa: '^1.2.3',
            bbb: '~4.5.6',
            ccc: '*',
            ddd: '*',
        });
        // devDependencies are added with pinned versions
        expect(pkgjson.devDependencies.aaa).toStrictEqual('1.2.3');
        expect(pkgjson.devDependencies.bbb).toStrictEqual('4.5.6');
        expect(pkgjson.devDependencies.ccc).toStrictEqual('*');
        expect(pkgjson.devDependencies.ddd).toStrictEqual('*');
        expect(pkgjson.dependencieds).toBeUndefined();
    });
    test('peerDependencies without pinnedDevDep', () => {
        // GIVEN
        const project = new TestNodeProject({
            peerDependencyOptions: {
                pinnedDevDependency: false,
            },
            peerDeps: [
                'aaa@^1.2.3',
                'bbb@~4.5.6',
            ],
        });
        // WHEN
        project.addPeerDeps('ccc');
        project.deps.addDependency('ddd', deps_1.DependencyType.PEER);
        // THEN
        const pkgjson = packageJson(project);
        expect(pkgjson.peerDependencies).toStrictEqual({
            aaa: '^1.2.3',
            bbb: '~4.5.6',
            ccc: '*',
            ddd: '*',
        });
        // sanitize
        ['npm-check-updates', 'jest', 'jest-junit', 'projen', 'standard-version'].forEach(d => delete pkgjson.devDependencies[d]);
        expect(pkgjson.devDependencies).toStrictEqual({});
        expect(pkgjson.dependencieds).toBeUndefined();
    });
    test('bundled deps are automatically added as normal deps', () => {
        // GIVEN
        const project = new TestNodeProject({
            bundledDeps: ['hey@2.1.1'],
        });
        // WHEN
        project.addBundledDeps('foo@^1.2.3');
        project.deps.addDependency('bar@~1.0.0', deps_1.DependencyType.BUNDLED);
        // THEN
        const pkgjson = packageJson(project);
        expect(pkgjson.dependencies).toStrictEqual({
            hey: '2.1.1',
            foo: '^1.2.3',
            bar: '~1.0.0',
        });
        expect(pkgjson.bundledDependencies).toStrictEqual([
            'bar',
            'foo',
            'hey',
        ]);
    });
});
describe('deps upgrade', () => {
    test('default - with projen secret', () => {
        const project = new TestNodeProject({ projenUpgradeSecret: 'PROJEN_GITHUB_TOKEN' });
        const snapshot = util_1.synthSnapshot(project);
        expect(snapshot['.github/workflows/upgrade-dependencies.yml']).toBeDefined();
        expect(snapshot['.github/workflows/upgrade-projen.yml']).toBeDefined();
    });
    test('default - no projen secret', () => {
        const project = new TestNodeProject();
        const snapshot = util_1.synthSnapshot(project);
        expect(snapshot['.github/workflows/upgrade-dependencies.yml']).toBeDefined();
        expect(snapshot['.github/workflows/upgrade-projen.yml']).toBeUndefined();
    });
    test('dependabot - with projen secret', () => {
        const project = new TestNodeProject({
            depsUpgrade: node_project_1.DependenciesUpgradeMechanism.dependabot(),
            projenUpgradeSecret: 'PROJEN_GITHUB_TOKEN',
        });
        const snapshot = util_1.synthSnapshot(project);
        expect(snapshot['.github/dependabot.yml']).toBeDefined();
        expect(snapshot['.github/workflows/upgrade-projen.yml']).toBeDefined();
    });
    test('dependabot - no projen secret', () => {
        const project = new TestNodeProject({
            depsUpgrade: node_project_1.DependenciesUpgradeMechanism.dependabot(),
        });
        const snapshot = util_1.synthSnapshot(project);
        expect(snapshot['.github/dependabot.yml']).toBeDefined();
        expect(snapshot['.github/workflows/upgrade-projen.yml']).toBeUndefined();
    });
    test('github actions - with projen secret', () => {
        const project = new TestNodeProject({
            depsUpgrade: node_project_1.DependenciesUpgradeMechanism.githubWorkflow(),
            projenUpgradeSecret: 'PROJEN_GITHUB_TOKEN',
        });
        const snapshot = util_1.synthSnapshot(project);
        expect(snapshot['.github/workflows/upgrade-dependencies.yml']).toBeDefined();
        expect(snapshot['.github/workflows/upgrade-projen.yml']).toBeDefined();
    });
    test('github actions - no projen secret', () => {
        const project = new TestNodeProject({
            depsUpgrade: node_project_1.DependenciesUpgradeMechanism.githubWorkflow(),
        });
        const snapshot = util_1.synthSnapshot(project);
        expect(snapshot['.github/workflows/upgrade-dependencies.yml']).toBeDefined();
        expect(snapshot['.github/workflows/upgrade-projen.yml']).toBeUndefined();
    });
    test('throws when depracated dependabot is configued with dependenciesUpgrade', () => {
        expect(() => {
            new TestNodeProject({ dependabot: true, depsUpgrade: node_project_1.DependenciesUpgradeMechanism.githubWorkflow() });
        }).toThrow("'dependabot' cannot be configured together with 'depsUpgrade'");
    });
});
describe('npm publishing options', () => {
    test('defaults', () => {
        // GIVEN
        const project = new util_1.TestProject();
        // WHEN
        const npm = new node_package_1.NodePackage(project, {
            packageName: 'my-package',
        });
        // THEN
        expect(npm.npmAccess).toStrictEqual(node_package_1.NpmAccess.PUBLIC);
        expect(npm.npmDistTag).toStrictEqual('latest');
        expect(npm.npmRegistry).toStrictEqual('registry.npmjs.org');
        expect(npm.npmRegistryUrl).toStrictEqual('https://registry.npmjs.org/');
        expect(npm.npmTokenSecret).toStrictEqual('NPM_TOKEN');
        // since these are all defaults, publishConfig is not defined.
        expect(util_1.synthSnapshot(project)['package.json'].publishConfig).toBeUndefined();
    });
    test('scoped packages default to RESTRICTED access', () => {
        // GIVEN
        const project = new util_1.TestProject();
        // WHEN
        const npm = new node_package_1.NodePackage(project, {
            packageName: 'scoped@my-package',
        });
        // THEN
        expect(npm.npmAccess).toStrictEqual(node_package_1.NpmAccess.RESTRICTED);
        // since these are all defaults, publishConfig is not defined.
        expect(packageJson(project).publishConfig).toBeUndefined();
    });
    test('non-scoped package cannot be RESTRICTED', () => {
        // GIVEN
        const project = new util_1.TestProject();
        // THEN
        expect(() => new node_package_1.NodePackage(project, {
            packageName: 'my-package',
            npmAccess: node_package_1.NpmAccess.RESTRICTED,
        })).toThrow(/"npmAccess" cannot be RESTRICTED for non-scoped npm package/);
    });
    test('custom settings', () => {
        // GIVEN
        const project = new util_1.TestProject();
        // WHEN
        const npm = new node_package_1.NodePackage(project, {
            packageName: 'scoped@my-package',
            npmDistTag: 'next',
            npmRegistryUrl: 'https://foo.bar',
            npmAccess: node_package_1.NpmAccess.PUBLIC,
            npmTokenSecret: 'GITHUB_TOKEN',
        });
        // THEN
        expect(npm.npmDistTag).toStrictEqual('next');
        expect(npm.npmRegistry).toStrictEqual('foo.bar');
        expect(npm.npmRegistryUrl).toStrictEqual('https://foo.bar/');
        expect(npm.npmAccess).toStrictEqual(node_package_1.NpmAccess.PUBLIC);
        expect(npm.npmTokenSecret).toStrictEqual('GITHUB_TOKEN');
        expect(packageJson(project).publishConfig).toStrictEqual({
            access: 'public',
            registry: 'https://foo.bar/',
            tag: 'next',
        });
    });
    test('deprecated npmRegistry can be used instead of npmRegistryUrl and then https:// is assumed', () => {
        // GIVEN
        const project = new util_1.TestProject();
        // WHEN
        const npm = new node_package_1.NodePackage(project, {
            packageName: 'scoped@my-package',
            npmRegistry: 'foo.bar.com',
        });
        // THEN
        expect(npm.npmRegistry).toStrictEqual('foo.bar.com');
        expect(npm.npmRegistryUrl).toStrictEqual('https://foo.bar.com/');
        expect(packageJson(project).publishConfig).toStrictEqual({
            registry: 'https://foo.bar.com/',
        });
    });
});
test('extend github release workflow', () => {
    var _a;
    const project = new TestNodeProject();
    (_a = project.releaseWorkflow) === null || _a === void 0 ? void 0 : _a.addJobs({
        publish_docker_hub: {
            'runs-on': 'ubuntu-latest',
            'env': {
                CI: 'true',
            },
            'steps': [
                {
                    name: 'Check out the repo',
                    uses: 'actions/checkout@v2',
                },
                {
                    name: 'Push to Docker Hub',
                    uses: 'docker/build-push-action@v1',
                    with: {
                        username: '${{ secrets.DOCKER_USERNAME }}',
                        password: '${{ secrets.DOCKER_PASSWORD }}',
                        repository: 'projen/projen-docker',
                        tag_with_ref: true,
                    },
                },
            ],
        },
    });
    const workflow = util_1.synthSnapshot(project)['.github/workflows/release.yml'];
    expect(workflow).toContain('publish_docker_hub:\n    runs-on: ubuntu-latest\n');
    expect(workflow).toContain('username: ${{ secrets.DOCKER_USERNAME }}\n          password: ${{ secrets.DOCKER_PASSWORD }}');
});
describe('scripts', () => {
    test('removeScript will remove tasks and scripts', () => {
        const p = new TestNodeProject();
        p.addTask('chortle', { exec: 'echo "frabjous day!"' });
        p.setScript('slithy-toves', 'gyre && gimble');
        expect(packageJson(p).scripts).toHaveProperty('chortle');
        expect(packageJson(p).scripts).toHaveProperty('slithy-toves');
        p.removeScript('chortle');
        p.removeScript('slithy-toves');
        expect(packageJson(p).scripts).not.toHaveProperty('chortle');
        expect(packageJson(p).scripts).not.toHaveProperty('slithy-toves');
    });
});
test('buildWorkflowMutable will push changes to PR branches', () => {
    // WHEN
    const project = new TestNodeProject({
        mutableBuild: true,
    });
    // THEN
    const workflowYaml = util_1.synthSnapshot(project)['.github/workflows/build.yml'];
    const workflow = yaml.parse(workflowYaml);
    expect(workflow.jobs.build.steps).toMatchSnapshot();
});
test('projenDuringBuild can be used to disable "projen synth" during build', () => {
    const enabled = new TestNodeProject({
        projenDuringBuild: true,
    });
    const disabled = new TestNodeProject({
        projenDuringBuild: false,
    });
    const buildTaskEnabled = util_1.synthSnapshot(enabled)['.projen/tasks.json'].tasks.build;
    const buildTaskDisabled = util_1.synthSnapshot(disabled)['.projen/tasks.json'].tasks.build;
    expect(buildTaskEnabled.steps[0].exec).toEqual('npx projen');
    expect(buildTaskDisabled.steps).toBeUndefined();
});
test('projen synth is only executed for subprojects', () => {
    // GIVEN
    const root = new TestNodeProject();
    // WHEN
    new TestNodeProject({ parent: root, outdir: 'child' });
    // THEN
    const snapshot = util_1.synthSnapshot(root);
    const rootBuildTask = snapshot['.projen/tasks.json'].tasks.build;
    const childBuildTask = snapshot['child/.projen/tasks.json'].tasks.build;
    expect(rootBuildTask).toStrictEqual({
        category: '00.build',
        description: 'Full release build (test+compile)',
        name: 'build',
        steps: [{ exec: 'npx projen' }],
    });
    expect(childBuildTask).toStrictEqual({
        category: '00.build',
        description: 'Full release build (test+compile)',
        name: 'build',
    });
});
test('enabling dependabot does not overturn mergify: false', () => {
    // WHEN
    const project = new TestNodeProject({
        dependabot: true,
        mergify: false,
    });
    // THEN
    const snapshot = util_1.synthSnapshot(project);
    // Note: brackets important, they prevent "." in filenames to be interpreted
    //       as JSON object path delimiters.
    expect(snapshot).not.toHaveProperty(['.mergify.yml']);
    expect(snapshot).toHaveProperty(['.github/dependabot.yml']);
});
function packageJson(project) {
    return util_1.synthSnapshot(project)['package.json'];
}
class TestNodeProject extends __1.NodeProject {
    constructor(options = {}) {
        super({
            outdir: util_1.mkdtemp(),
            name: 'test-node-project',
            logging: {
                level: __1.LogLevel.OFF,
            },
            defaultReleaseBranch: 'master',
            ...options,
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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