import * as cloudfront from '@aws-cdk/aws-cloudfront';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as iam from '@aws-cdk/aws-iam';
import * as s3 from '@aws-cdk/aws-s3';
import * as cdk from '@aws-cdk/core';
import { Construct } from 'constructs';
import { ISource } from './source';
import { Construct as CoreConstruct } from '@aws-cdk/core';
/**
 * Properties for `BucketDeployment`.
 *
 * @stability stable
 */
export interface BucketDeploymentProps {
    /**
     * The sources from which to deploy the contents of this bucket.
     *
     * @stability stable
     */
    readonly sources: ISource[];
    /**
     * The S3 bucket to sync the contents of the zip file to.
     *
     * @stability stable
     */
    readonly destinationBucket: s3.IBucket;
    /**
     * Key prefix in the destination bucket.
     *
     * @default "/" (unzip to root of the destination bucket)
     * @stability stable
     */
    readonly destinationKeyPrefix?: string;
    /**
     * If this is set to false, files in the destination bucket that do not exist in the asset, will NOT be deleted during deployment (create/update).
     *
     * @default true
     * @see https://docs.aws.amazon.com/cli/latest/reference/s3/sync.html
     * @stability stable
     */
    readonly prune?: boolean;
    /**
     * If this is set to "false", the destination files will be deleted when the resource is deleted or the destination is updated.
     *
     * NOTICE: Configuring this to "false" might have operational implications. Please
     * visit to the package documentation referred below to make sure you fully understand those implications.
     *
     * @default true - when resource is deleted/updated, files are retained
     * @see https://github.com/aws/aws-cdk/tree/master/packages/%40aws-cdk/aws-s3-deployment#retain-on-delete
     * @stability stable
     */
    readonly retainOnDelete?: boolean;
    /**
     * The CloudFront distribution using the destination bucket as an origin.
     *
     * Files in the distribution's edge caches will be invalidated after
     * files are uploaded to the destination bucket.
     *
     * @default - No invalidation occurs
     * @stability stable
     */
    readonly distribution?: cloudfront.IDistribution;
    /**
     * The file paths to invalidate in the CloudFront distribution.
     *
     * @default - All files under the destination bucket key prefix will be invalidated.
     * @stability stable
     */
    readonly distributionPaths?: string[];
    /**
     * The amount of memory (in MiB) to allocate to the AWS Lambda function which replicates the files from the CDK bucket to the destination bucket.
     *
     * If you are deploying large files, you will need to increase this number
     * accordingly.
     *
     * @default 128
     * @stability stable
     */
    readonly memoryLimit?: number;
    /**
     * Execution role associated with this function.
     *
     * @default - A role is automatically created
     * @stability stable
     */
    readonly role?: iam.IRole;
    /**
     * User-defined object metadata to be set on all objects in the deployment.
     *
     * @default - No user metadata is set
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#UserMetadata
     * @stability stable
     */
    readonly metadata?: UserDefinedObjectMetadata;
    /**
     * System-defined cache-control metadata to be set on all objects in the deployment.
     *
     * @default - Not set.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
     * @stability stable
     */
    readonly cacheControl?: CacheControl[];
    /**
     * System-defined cache-disposition metadata to be set on all objects in the deployment.
     *
     * @default - Not set.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
     * @stability stable
     */
    readonly contentDisposition?: string;
    /**
     * System-defined content-encoding metadata to be set on all objects in the deployment.
     *
     * @default - Not set.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
     * @stability stable
     */
    readonly contentEncoding?: string;
    /**
     * System-defined content-language metadata to be set on all objects in the deployment.
     *
     * @default - Not set.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
     * @stability stable
     */
    readonly contentLanguage?: string;
    /**
     * System-defined content-type metadata to be set on all objects in the deployment.
     *
     * @default - Not set.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
     * @stability stable
     */
    readonly contentType?: string;
    /**
     * System-defined expires metadata to be set on all objects in the deployment.
     *
     * @default - The objects in the distribution will not expire.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
     * @stability stable
     */
    readonly expires?: cdk.Expiration;
    /**
     * System-defined x-amz-server-side-encryption metadata to be set on all objects in the deployment.
     *
     * @default - Server side encryption is not used.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
     * @stability stable
     */
    readonly serverSideEncryption?: ServerSideEncryption;
    /**
     * System-defined x-amz-storage-class metadata to be set on all objects in the deployment.
     *
     * @default - Default storage-class for the bucket is used.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
     * @stability stable
     */
    readonly storageClass?: StorageClass;
    /**
     * System-defined x-amz-website-redirect-location metadata to be set on all objects in the deployment.
     *
     * @default - No website redirection.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
     * @stability stable
     */
    readonly websiteRedirectLocation?: string;
    /**
     * System-defined x-amz-server-side-encryption-aws-kms-key-id metadata to be set on all objects in the deployment.
     *
     * @default - Not set.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
     * @stability stable
     */
    readonly serverSideEncryptionAwsKmsKeyId?: string;
    /**
     * System-defined x-amz-server-side-encryption-customer-algorithm metadata to be set on all objects in the deployment.
     *
     * Warning: This is not a useful parameter until this bug is fixed: https://github.com/aws/aws-cdk/issues/6080
     *
     * @default - Not set.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/ServerSideEncryptionCustomerKeys.html#sse-c-how-to-programmatically-intro
     * @stability stable
     */
    readonly serverSideEncryptionCustomerAlgorithm?: string;
    /**
     * System-defined x-amz-acl metadata to be set on all objects in the deployment.
     *
     * @default - Not set.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/userguide/acl-overview.html#canned-acl
     * @stability stable
     */
    readonly accessControl?: s3.BucketAccessControl;
    /**
     * The VPC network to place the deployment lambda handler in.
     *
     * @default None
     * @stability stable
     */
    readonly vpc?: ec2.IVpc;
    /**
     * Where in the VPC to place the deployment lambda handler.
     *
     * Only used if 'vpc' is supplied.
     *
     * @default - the Vpc default strategy if not specified
     * @stability stable
     */
    readonly vpcSubnets?: ec2.SubnetSelection;
}
/**
 * `BucketDeployment` populates an S3 bucket with the contents of .zip files from other S3 buckets or from local disk.
 *
 * @stability stable
 */
export declare class BucketDeployment extends CoreConstruct {
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: BucketDeploymentProps);
    private renderSingletonUuid;
}
/**
 * Used for HTTP cache-control header, which influences downstream caches.
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 * @stability stable
 */
export declare class CacheControl {
    /**
     * The raw cache control setting.
     */
    readonly value: any;
    /**
     * Sets 'must-revalidate'.
     *
     * @stability stable
     */
    static mustRevalidate(): CacheControl;
    /**
     * Sets 'no-cache'.
     *
     * @stability stable
     */
    static noCache(): CacheControl;
    /**
     * Sets 'no-transform'.
     *
     * @stability stable
     */
    static noTransform(): CacheControl;
    /**
     * Sets 'public'.
     *
     * @stability stable
     */
    static setPublic(): CacheControl;
    /**
     * Sets 'private'.
     *
     * @stability stable
     */
    static setPrivate(): CacheControl;
    /**
     * Sets 'proxy-revalidate'.
     *
     * @stability stable
     */
    static proxyRevalidate(): CacheControl;
    /**
     * Sets 'max-age=<duration-in-seconds>'.
     *
     * @stability stable
     */
    static maxAge(t: cdk.Duration): CacheControl;
    /**
     * Sets 's-maxage=<duration-in-seconds>'.
     *
     * @stability stable
     */
    static sMaxAge(t: cdk.Duration): CacheControl;
    /**
     * Constructs a custom cache control key from the literal value.
     *
     * @stability stable
     */
    static fromString(s: string): CacheControl;
    private constructor();
}
/**
 * Indicates whether server-side encryption is enabled for the object, and whether that encryption is from the AWS Key Management Service (AWS KMS) or from Amazon S3 managed encryption (SSE-S3).
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 * @stability stable
 */
export declare enum ServerSideEncryption {
    /**
     * 'AES256'.
     *
     * @stability stable
     */
    AES_256 = "AES256",
    /**
     * 'aws:kms'.
     *
     * @stability stable
     */
    AWS_KMS = "aws:kms"
}
/**
 * Storage class used for storing the object.
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 * @stability stable
 */
export declare enum StorageClass {
    /**
     * 'STANDARD'.
     *
     * @stability stable
     */
    STANDARD = "STANDARD",
    /**
     * 'REDUCED_REDUNDANCY'.
     *
     * @stability stable
     */
    REDUCED_REDUNDANCY = "REDUCED_REDUNDANCY",
    /**
     * 'STANDARD_IA'.
     *
     * @stability stable
     */
    STANDARD_IA = "STANDARD_IA",
    /**
     * 'ONEZONE_IA'.
     *
     * @stability stable
     */
    ONEZONE_IA = "ONEZONE_IA",
    /**
     * 'INTELLIGENT_TIERING'.
     *
     * @stability stable
     */
    INTELLIGENT_TIERING = "INTELLIGENT_TIERING",
    /**
     * 'GLACIER'.
     *
     * @stability stable
     */
    GLACIER = "GLACIER",
    /**
     * 'DEEP_ARCHIVE'.
     *
     * @stability stable
     */
    DEEP_ARCHIVE = "DEEP_ARCHIVE"
}
/**
 * (deprecated) Used for HTTP expires header, which influences downstream caches.
 *
 * Does NOT influence deletion of the object.
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 * @deprecated use core.Expiration
 */
export declare class Expires {
    /**
     * The raw expiration date expression.
     */
    readonly value: any;
    /**
     * (deprecated) Expire at the specified date.
     *
     * @param d date to expire at.
     * @deprecated
     */
    static atDate(d: Date): Expires;
    /**
     * (deprecated) Expire at the specified timestamp.
     *
     * @param t timestamp in unix milliseconds.
     * @deprecated
     */
    static atTimestamp(t: number): Expires;
    /**
     * (deprecated) Expire once the specified duration has passed since deployment time.
     *
     * @param t the duration to wait before expiring.
     * @deprecated
     */
    static after(t: cdk.Duration): Expires;
    /**
     * (deprecated) Create an expiration date from a raw date string.
     *
     * @deprecated
     */
    static fromString(s: string): Expires;
    private constructor();
}
/**
 * Custom user defined metadata.
 *
 * @stability stable
 */
export interface UserDefinedObjectMetadata {
    /**
     * Arbitrary metadata key-values
     * The `x-amz-meta-` prefix will automatically be added to keys.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#UserMetadata
     */
    readonly [key: string]: string;
}
