# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['dowhy',
 'dowhy.api',
 'dowhy.causal_estimators',
 'dowhy.causal_identifier',
 'dowhy.causal_refuters',
 'dowhy.data_transformers',
 'dowhy.do_samplers',
 'dowhy.gcm',
 'dowhy.gcm.independence_test',
 'dowhy.gcm.ml',
 'dowhy.gcm.util',
 'dowhy.graph_learners',
 'dowhy.interpreters',
 'dowhy.utils',
 'dowhy.utils.dgps']

package_data = \
{'': ['*']}

install_requires = \
['causal-learn>=0.1.3.0,<0.2.0.0',
 'joblib>=1.1.0,<2.0.0',
 'networkx>=2.8.5,<3.0.0',
 'numpy>=1.23.1,<2.0.0',
 'pandas>=1.4.3,<2.0.0',
 'scikit-learn==1.0.2',
 'scipy>=1.8.1,<2.0.0',
 'sphinx_design>=0.3.0,<0.4.0',
 'statsmodels>=0.13.2,<0.14.0',
 'sympy>=1.10.1,<2.0.0',
 'tqdm>=4.64.0,<5.0.0']

extras_require = \
{':extra == "causalml"': ['cython>=0.29.32,<0.30.0'],
 'causalml': ['llvmlite>=0.36.0,<0.37.0'],
 'econml': ['econml'],
 'plotting': ['matplotlib>=3.5.3,<4.0.0'],
 'pydot': ['pydot>=1.4.2,<2.0.0'],
 'pygraphviz': ['pygraphviz>=1.9,<2.0']}

setup_kwargs = {
    'name': 'dowhy',
    'version': '0.9',
    'description': 'DoWhy is a Python library for causal inference that supports explicit modeling and testing of causal assumptions',
    'long_description': '|BuildStatus|_ |PyPiVersion|_ |PythonSupport|_ |Downloads|_\n\n.. |PyPiVersion| image:: https://img.shields.io/pypi/v/dowhy.svg\n.. _PyPiVersion: https://pypi.org/project/dowhy/\n\n.. |PythonSupport| image:: https://img.shields.io/pypi/pyversions/dowhy.svg\n.. _PythonSupport: https://pypi.org/project/dowhy/\n\n.. |BuildStatus| image:: https://github.com/microsoft/dowhy/workflows/Python%20package/badge.svg\n.. _BuildStatus: https://github.com/microsoft/dowhy/actions\n\n.. |Downloads| image:: https://pepy.tech/badge/dowhy\n.. _Downloads: https://pepy.tech/project/dowhy\n\n\n.. image:: dowhy-logo-large.png\n  :width: 50%\n  :align: center\n\n\\\n===============================\n\n  Introducing DoWhy and the 4 steps of causal inference | `Microsoft Research Blog <https://www.microsoft.com/en-us/research/blog/dowhy-a-library-for-causal-inference/>`_ | `Video Tutorial <https://note.microsoft.com/MSR-Webinar-DoWhy-Library-Registration-On-Demand.html>`_ | `Arxiv Paper <https://arxiv.org/abs/2011.04216>`_ | `Arxiv Paper (GCM-extension) <https://arxiv.org/abs/2206.06821>`_ | `Slides <https://www2.slideshare.net/AmitSharma315/dowhy-an-endtoend-library-for-causal-inference>`_\n\n  Read the `docs <https://py-why.github.io/dowhy/>`_ | Try it online! |Binder|_\n\n.. |Binder| image:: https://mybinder.org/badge_logo.svg\n.. _Binder: https://mybinder.org/v2/gh/microsoft/dowhy/main?filepath=docs%2Fsource%2F\n\n**Case Studies using DoWhy**: `Hotel booking cancellations <https://towardsdatascience.com/beyond-predictive-models-the-causal-story-behind-hotel-booking-cancellations-d29e8558cbaf>`_ | `Effect of customer loyalty programs <https://github.com/microsoft/dowhy/blob/main/docs/source/example_notebooks/dowhy_example_effect_of_memberrewards_program.ipynb>`_ | `Optimizing article headlines <https://medium.com/@akelleh/introducing-the-do-sampler-for-causal-inference-a3296ea9e78d>`_ | `Effect of home visits on infant health (IHDP) <https://towardsdatascience.com/implementing-causal-inference-a-key-step-towards-agi-de2cde8ea599>`_ | `Causes of customer churn/attrition <https://medium.com/geekculture/a-quickstart-for-causal-analysis-decision-making-with-dowhy-2ce2d4d1efa9>`_\n\n.. image:: https://raw.githubusercontent.com/microsoft/dowhy/main/docs/images/dowhy-schematic.png\n\nAs computing systems are more frequently and more actively intervening in societally critical domains such as healthcare, education, and governance, it is critical to correctly predict and understand the causal effects of these interventions. Without an A/B test, conventional machine learning methods, built on pattern recognition and correlational analyses, are insufficient for decision-making.\n\nMuch like machine learning libraries have done for prediction, **"DoWhy" is a Python library that aims to spark causal thinking and analysis**. DoWhy provides a principled four-step interface for causal inference that focuses on explicitly modeling causal assumptions and validating them as much as possible. The key feature of DoWhy is its state-of-the-art refutation API that can automatically test causal assumptions for any estimation method, thus making inference more robust and accessible to non-experts. DoWhy supports estimation of the average causal effect for backdoor, frontdoor, instrumental variable and other identification methods, and estimation of the conditional effect (CATE) through an integration with the EconML library.\n\nFor a quick introduction to causal inference, check out `amit-sharma/causal-inference-tutorial <https://github.com/amit-sharma/causal-inference-tutorial/>`_. We also gave a more comprehensive tutorial at the ACM Knowledge Discovery and Data Mining (`KDD 2018 <http://www.kdd.org/kdd2018/>`_) conference: `causalinference.gitlab.io/kdd-tutorial <http://causalinference.gitlab.io/kdd-tutorial/>`_. For an introduction to the four steps of causal inference and its implications for machine learning, you can access this video tutorial from Microsoft Research: `DoWhy Webinar <https://note.microsoft.com/MSR-Webinar-DoWhy-Library-Registration-On-Demand.html>`_.\n\nDocumentation for DoWhy is available at `py-why.github.io/dowhy <https://py-why.github.io/dowhy/>`_.\n\n.. i here comment toctree::\n.. i here comment   :maxdepth: 4\n.. i here comment   :caption: Contents:\n.. contents:: **Contents**\n\nNews\n-----\n**2022.05.27**:\n\n* **DoWhy now part of PyWhy**\n\n  We have moved DoWhy from microsoft/dowhy to py-why/dowhy. While GitHub will automatically\n  redirect your git command for cloning, pulling, etc., we recommend updating git remotes and bookmarks. Please note\n  that the **documentation** has now moved to https://py-why.github.io/dowhy with **no** redirect from the old URL.\n\n* **Experimental support for GCM-based inference**\n\n  We have started adding support for graphical causal model-based inference (or in short GCM-based). At the moment,\n  this includes support for interventions, counterfactuals, and attributing distribution changes. As part of this,\n  we also added features for Shapley value estimation and independence tests. We\'re still in the process of fleshing\n  everything out, including `documentation <https://py-why.github.io/dowhy/main/user_guide/gcm_based_inference/index.html>`_. Some of it is already on `main\n  <https://py-why.github.io/dowhy/main/user_guide/gcm_based_inference/index.html>`_, other parts are on feature branches (prefixed with ``gcm-``) with open\n  pull-requests, other parts will appear as new pull-requests in the next couple of weeks. Be sure to watch this space\n  here as we quickly expand functionality and documentation.\n\nThe need for causal inference\n----------------------------------\n\nPredictive models uncover patterns that connect the inputs and outcome in observed data. To intervene, however, we need to estimate the effect of changing an input from its current value, for which no data exists. Such questions, involving estimating a *counterfactual*, are common in decision-making scenarios.\n\n* Will it work?\n    * Does a proposed change to a system improve people\'s outcomes?\n* Why did it work?\n    * What led to a change in a system\'s outcome?\n* What should we do?\n    * What changes to a system are likely to improve outcomes for people?\n* What are the overall effects?\n    * How does the system interact with human behavior?\n    * What is the effect of a system\'s recommendations on people\'s activity?\n\nAnswering these questions requires causal reasoning. While many methods exist\nfor causal inference, it is hard to compare their assumptions and robustness of results. DoWhy makes three contributions,\n\n1. Provides a principled way of modeling a given problem as a causal graph so\n   that all assumptions are explicit.\n2. Provides a unified interface for many popular causal inference methods, combining the two major frameworks of graphical models and potential outcomes.\n3. Automatically tests for the validity of assumptions if possible and assesses\n   the robustness of the estimate to violations.\n\nTo see DoWhy in action, check out how it can be applied to estimate the effect\nof a subscription or rewards program for customers [`Rewards notebook\n<https://github.com/microsoft/dowhy/blob/main/docs/source/example_notebooks/dowhy_example_effect_of_memberrewards_program.ipynb>`_] and for implementing and evaluating causal inference methods on benchmark datasets like the `Infant Health and Development Program (IHDP) <https://github.com/microsoft/dowhy/blob/main/docs/source/example_notebooks/dowhy_ihdp_data_example.ipynb>`_ dataset, `Infant Mortality (Twins) <https://github.com/microsoft/dowhy/blob/main/docs/source/example_notebooks/dowhy_twins_example.ipynb>`_ dataset, and the `Lalonde Jobs <https://github.com/microsoft/dowhy/blob/main/docs/source/example_notebooks/dowhy_lalonde_example.ipynb>`_ dataset.\n\n\nInstallation\n-------------\n\nDoWhy support Python 3.8+. To install, you can use pip, poetry, or conda.\n\n**Latest Release**\n\nInstall the latest `release <https://pypi.org/project/dowhy/>`__ using pip.\n\n.. code:: shell\n\n   pip install dowhy\n\nInstall the latest `release <https://pypi.org/project/dowhy/>`__ using poetry.\n\n.. code:: shell\n\n   poetry add dowhy\n\nInstall the latest `release <https://anaconda.org/conda-forge/dowhy>`__ using conda.\n\n.. code:: shell\n\n   conda install -c conda-forge dowhy\n\nIf you face "Solving environment" problems with conda, then try :code:`conda update --all` and then install dowhy. If that does not work, then use :code:`conda config --set channel_priority false` and try to install again. If the problem persists, please `add your issue here <https://github.com/microsoft/dowhy/issues/197>`_.\n\n**Development Version**\n\nIf you prefer to use the latest dev version, your dependency management tool will need to point at our GitHub repository.\n\n.. code:: shell\n\n    pip install git+https://github.com/py-why/dowhy@main\n\n**Requirements**\n\nDoWhy requires a few dependencies. \nDetails on specific versions can be found in `pyproject.toml <./pyproject.toml>`_, under the `tool.poetry.dependencies` section.\n\nIf you face any problems, try installing dependencies manually.\n\n.. code:: shell\n\n    pip install \'<dependency-name>==<version>\'\n\nOptionally, if you wish to input graphs in the dot format, then install pydot (or pygraphviz).\n\n\nFor better-looking graphs, you can optionally install pygraphviz. To proceed,\nfirst install graphviz and then pygraphviz (on Ubuntu and Ubuntu WSL).\n\n.. code:: shell\n\n    sudo apt install graphviz libgraphviz-dev graphviz-dev pkg-config\n    ## from https://github.com/pygraphviz/pygraphviz/issues/71\n    pip install pygraphviz --install-option="--include-path=/usr/include/graphviz" \\\n    --install-option="--library-path=/usr/lib/graphviz/"\n\nSample causal inference analysis in DoWhy\n-------------------------------------------\nMost DoWhy\nanalyses for causal inference take 4 lines to write, assuming a\npandas dataframe df that contains the data:\n\n.. code:: python\n\n    from dowhy import CausalModel\n    import dowhy.datasets\n\n    # Load some sample data\n    data = dowhy.datasets.linear_dataset(\n        beta=10,\n        num_common_causes=5,\n        num_instruments=2,\n        num_samples=10000,\n        treatment_is_binary=True)\n\nDoWhy supports two formats for providing the causal graph: `gml <https://github.com/GunterMueller/UNI_PASSAU_FMI_Graph_Drawing>`_ (preferred) and `dot <http://www.graphviz.org/documentation/>`_. After loading in the data, we use the four main operations in DoWhy: *model*,\n*estimate*, *identify* and *refute*:\n\n.. code:: python\n\n    # I. Create a causal model from the data and given graph.\n    model = CausalModel(\n        data=data["df"],\n        treatment=data["treatment_name"],\n        outcome=data["outcome_name"],\n        graph=data["gml_graph"])\n\n    # II. Identify causal effect and return target estimands\n    identified_estimand = model.identify_effect()\n\n    # III. Estimate the target estimand using a statistical method.\n    estimate = model.estimate_effect(identified_estimand,\n                                     method_name="backdoor.propensity_score_matching")\n\n    # IV. Refute the obtained estimate using multiple robustness checks.\n    refute_results = model.refute_estimate(identified_estimand, estimate,\n                                           method_name="random_common_cause")\n\nDoWhy stresses on the interpretability of its output. At any point in the analysis,\nyou can inspect the untested assumptions, identified estimands (if any) and the\nestimate (if any). Here\'s a sample output of the linear regression estimator.\n\n.. image:: https://raw.githubusercontent.com/microsoft/dowhy/main/docs/images/regression_output.png\n\nFor a full code example, check out the `Getting Started with DoWhy <https://github.com/microsoft/dowhy/blob/main/docs/source/example_notebooks/dowhy_simple_example.ipynb>`_ notebook. You can also use Conditional Average Treatment Effect (CATE) estimation methods from other libraries such as EconML and CausalML, as shown in the `Conditional Treatment Effects <https://github.com/microsoft/dowhy/blob/main/docs/source/example_notebooks/dowhy-conditional-treatment-effects.ipynb>`_ notebook. For more examples of using DoWhy, check out the Jupyter notebooks in `docs/source/example_notebooks <https://github.com/microsoft/dowhy/tree/main/docs/source/example_notebooks/>`_ or try them online at `Binder <https://mybinder.org/v2/gh/microsoft/dowhy/main?filepath=docs%2Fsource%2F>`_.\n\n\nGCM-based inference (experimental)\n----------------------------------\n\nGraphical causal model-based inference, or GCM-based inference for short, is an experimental addition to DoWhy. For\ndetails, check out the `documentation for the gcm sub-package <https://py-why.github.io/dowhy/main/user_guide/gcm_based_inference/index.html>`_. The basic\nrecipe for this API works as follows:\n\n.. code:: python\n\n    # 1. Modeling cause-effect relationships as a structural causal model\n    #    (causal graph + functional causal models):\n    scm = gcm.StructuralCausalModel(nx.DiGraph([(\'X\', \'Y\'), (\'Y\', \'Z\')])) # X -> Y -> Z\n    scm.set_causal_mechanism(\'X\', gcm.EmpiricalDistribution())\n    scm.set_causal_mechanism(\'Y\', gcm.AdditiveNoiseModel(gcm.ml.create_linear_regressor()))\n    scm.set_causal_mechanism(\'Z\', gcm.AdditiveNoiseModel(gcm.ml.create_linear_regressor()))\n\n    # 2. Fitting the SCM to the data:\n    gcm.fit(scm, data)\n\n    # 3. Answering a causal query based on the SCM:\n    results = gcm.<causal_query>(scm, ...)\n\nWhere <causal_query> can be one of multiple functions explained in `Answering Causal Questions <https://py-why.github.io/dowhy/main/user_guide/gcm_based_inference/answering_causal_questions/index.html>`_.\n\n\nA high-level Pandas API\n-----------------------\n\nWe\'ve made an even simpler API for dowhy which is a light layer on top of the standard one. The goal is to make causal analysis much more like regular exploratory analysis. To use this API, simply\nimport :code:`dowhy.api`. This will magically add the :code:`causal` namespace to your\n:code:`pandas.DataFrame` s. Then,\nyou can use the namespace as follows.\n\n.. code:: python\n\n    import dowhy.api\n    import dowhy.datasets\n\n    data = dowhy.datasets.linear_dataset(beta=5,\n        num_common_causes=1,\n        num_instruments = 0,\n        num_samples=1000,\n        treatment_is_binary=True)\n\n    # data[\'df\'] is just a regular pandas.DataFrame\n    data[\'df\'].causal.do(x=\'v0\', # name of treatment variable\n                         variable_types={\'v0\': \'b\', \'y\': \'c\', \'W0\': \'c\'},\n                         outcome=\'y\',\n                         common_causes=[\'W0\']).groupby(\'v0\').mean().plot(y=\'y\', kind=\'bar\')\n\n.. image:: https://raw.githubusercontent.com/microsoft/dowhy/main/docs/images/do_barplot.png\n\nFor some methods, the :code:`variable_types` field must be specified. It should be a :code:`dict`, where the keys are\nvariable names, and values are \'o\' for ordered discrete, \'u\' for un-ordered discrete, \'d\' for discrete, or \'c\'\nfor continuous.\n\n**Note:If the** :code:`variable_types` **is not specified we make use of the following implicit conversions:**\n::\n\n   int -> \'c\'\n   float -> \'c\'\n   binary -> \'b\'\n   category -> \'d\'\n\n**Currently we have not added support for timestamps.**\n\nThe :code:`do` method in the causal namespace generates a random sample from $P(outcome|do(X=x))$ of the\nsame length as your data set, and returns this outcome as a new :code:`DataFrame`. You can continue to perform\nthe usual :code:`DataFrame` operations with this sample, and so you can compute statistics and create plots\nfor causal outcomes!\n\nThe :code:`do` method is built on top of the lower-level :code:`dowhy` objects, so can still take a graph and perform\nidentification automatically when you provide a graph instead of :code:`common_causes`.\n\nFor more details, check out the `Pandas API\n<https://github.com/microsoft/dowhy/blob/main/docs/source/example_notebooks/dowhy_causal_api.ipynb>`_ notebook or the `Do Sampler <https://github.com/microsoft/dowhy/blob/main/docs/source/example_notebooks/do_sampler_demo.ipynb>`_\nnotebook.\n\nGraphical Models and Potential Outcomes: Best of both worlds\n============================================================\nDoWhy builds on two of the most powerful frameworks for causal inference:\ngraphical models and potential outcomes. It uses graph-based criteria and\ndo-calculus for modeling assumptions and identifying a non-parametric causal effect.\nFor estimation, it switches to methods based primarily on potential outcomes.\n\nA unifying language for causal inference\n----------------------------------------\n\nDoWhy is based on a simple unifying language for causal inference. Causal\ninference may seem tricky, but almost all methods follow four key steps:\n\n1. Model a causal inference problem using assumptions.\n2. Identify an expression for the causal effect under these assumptions ("causal estimand").\n3. Estimate the expression using statistical methods such as matching or instrumental variables.\n4. Finally, verify the validity of the estimate using a variety of robustness checks.\n\nThis workflow can be captured by four key verbs in DoWhy:\n\n- model\n- identify\n- estimate\n- refute\n\nUsing these verbs, DoWhy implements a causal inference engine that can support\na variety of methods. *model* encodes prior knowledge as a formal causal graph, *identify* uses\ngraph-based methods to identify the causal effect, *estimate* uses\nstatistical methods for estimating the identified estimand, and finally *refute*\ntries to refute the obtained estimate by testing robustness to assumptions.\n\nKey differences compared to available causal inference software\n----------------------------------------------------------------\nDoWhy brings three key differences compared to available software for causal inference:\n\n**Explicit identifying assumptions**\n    Assumptions are first-class citizens in DoWhy.\n\n    Each analysis starts with a\n    building a causal model. The assumptions can be viewed graphically or in terms\n    of conditional independence statements. Wherever possible, DoWhy can also\n    automatically test for stated assumptions using observed data.\n\n**Separation between identification and estimation**\n    Identification is the causal problem. Estimation is simply a statistical problem.\n\n    DoWhy\n    respects this boundary and treats them separately. This focuses the causal\n    inference effort on identification, and frees up estimation using any\n    available statistical estimator for a target estimand. In addition, multiple\n    estimation methods can be used for a single identified_estimand and\n    vice-versa.\n\n**Automated robustness checks**\n    What happens when key identifying assumptions may not be satisfied?\n\n    The most critical, and often skipped, part of causal analysis is checking the\n    robustness of an estimate to unverified assumptions. DoWhy makes it easy to\n    automatically run sensitivity and robustness checks on the obtained estimate.\n\nFinally, DoWhy is easily extensible, allowing other implementations of the\nfour verbs to co-exist (e.g., we support implementations of the *estimation* verb from\nEconML and CausalML libraries). The four verbs are mutually independent, so their\nimplementations can be combined in any way.\n\n\n\nBelow are more details about the current implementation of each of these verbs.\n\nFour steps of causal inference\n===============================\n\nI. Model a causal problem\n-----------------------------\n\nDoWhy creates an underlying causal graphical model for each problem. This\nserves to make each causal assumption explicit. This graph need not be\ncomplete---you can provide a partial graph, representing prior\nknowledge about some of the variables. DoWhy automatically considers the rest\nof the variables as potential confounders.\n\nCurrently, DoWhy supports two formats for graph input: `gml <https://github.com/GunterMueller/UNI_PASSAU_FMI_Graph_Drawing>`_ (preferred) and\n`dot <http://www.graphviz.org/documentation/>`_. We strongly suggest to use gml as the input format, as it works well with networkx. You can provide the graph either as a .gml file or as a string. If you prefer to use dot format, you will need to install additional packages (pydot or pygraphviz, see the installation section above). Both .dot files and string format are supported.\n\nWhile not recommended, you can also specify common causes and/or instruments directly\ninstead of providing a graph.\n\nSupported formats for specifying causal assumptions\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\n* **Graph**: Provide a causal graph in either gml or dot format. Can be a text file\n  or a string.\n* **Named variable sets**: Instead of the graph, provide variable names that\n  correspond to relevant categories, such as common causes, instrumental variables, effect\n  modifiers, frontdoor variables, etc.\n\nExamples of how to instantiate a causal model are in the `Getting Started\n<https://github.com/microsoft/dowhy/blob/main/docs/source/example_notebooks/dowhy_simple_example.ipynb>`_\nnotebook.\n\n.. i comment image:: causal_model.png\n\nII. Identify a target estimand under the model\n----------------------------------------------\n\nBased on the causal graph, DoWhy finds all possible ways of identifying a desired causal effect based on\nthe graphical model. It uses graph-based criteria and do-calculus to find\npotential ways find expressions that can identify the causal effect.\n\nSupported identification criteria\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\n* Back-door criterion\n* Front-door criterion\n* Instrumental Variables\n* Mediation (Direct and indirect effect identification)\n\nDifferent notebooks illustrate how to use these identification criteria. Check\nout the `Simple Backdoor <https://github.com/microsoft/dowhy/blob/main/docs/source/example_notebooks/dowhy_confounder_example.ipynb>`_ notebook for the back-door criterion, and the `Simple IV <https://github.com/microsoft/dowhy/blob/main/docs/source/example_notebooks/dowhy-simple-iv-example.ipynb>`_ notebook for the instrumental variable criterion.\n\nIII. Estimate causal effect based on the identified estimand\n------------------------------------------------------------\n\nDoWhy supports methods based on both back-door criterion and instrumental\nvariables. It also provides a non-parametric confidence intervals and a permutation test for testing\nthe statistical significance of obtained estimate.\n\nSupported estimation methods\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\n* Methods based on estimating the treatment assignment\n    * Propensity-based Stratification\n    * Propensity Score Matching\n    * Inverse Propensity Weighting\n\n* Methods based on estimating the outcome model\n    * Linear Regression\n    * Generalized Linear Models\n\n* Methods based on the instrumental variable equation\n    * Binary Instrument/Wald Estimator\n    * Two-stage least squares\n    * Regression discontinuity\n\n* Methods for front-door criterion and general mediation\n    * Two-stage linear regression\n\nExamples of using these methods are in the `Estimation methods\n<https://github.com/microsoft/dowhy/blob/main/docs/source/example_notebooks/dowhy_estimation_methods.ipynb>`_\nnotebook.\n\nUsing EconML and CausalML estimation methods in DoWhy\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\nIt is easy to call external estimation methods using DoWhy. Currently we\nsupport integrations with the `EconML <https://github.com/microsoft/econml>`_ and `CausalML <https://github.com/uber/causalml>`_ packages. Here\'s an example\nof estimating conditional treatment effects using EconML\'s double machine\nlearning estimator.\n\n.. code:: python\n\n\tfrom sklearn.preprocessing import PolynomialFeatures\n\tfrom sklearn.linear_model import LassoCV\n\tfrom sklearn.ensemble import GradientBoostingRegressor\n\tdml_estimate = model.estimate_effect(identified_estimand, method_name="backdoor.econml.dml.DML",\n                        control_value = 0,\n                        treatment_value = 1,\n                        target_units = lambda df: df["X0"]>1,\n                        confidence_intervals=False,\n                        method_params={\n                            "init_params":{\'model_y\':GradientBoostingRegressor(),\n                                           \'model_t\': GradientBoostingRegressor(),\n                                           \'model_final\':LassoCV(),\n                                           \'featurizer\':PolynomialFeatures(degree=1, include_bias=True)},\n                            "fit_params":{}}\n\t\t\t\t\t\t)\n\n\nMore examples are in the `Conditional Treatment Effects with DoWhy\n<https://github.com/microsoft/dowhy/blob/main/docs/source/example_notebooks/dowhy-conditional-treatment-effects.ipynb>`_ notebook.\n\nIV. Refute the obtained estimate\n-------------------------------------\nHaving access to multiple refutation methods to validate an effect estimate from a\ncausal estimator is\na key benefit of using DoWhy.\n\nSupported refutation methods\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\n* **Add Random Common Cause**: Does the estimation method change its estimate after\n  we add an independent random variable as a common cause to the dataset?\n  (*Hint: It should not*)\n* **Placebo Treatment**: What happens to the estimated causal effect when we\n  replace the true treatment variable with an independent random variable?\n  (*Hint: the effect should go to zero*)\n* **Dummy Outcome**: What happens to the estimated causal effect when we replace\n  the true outcome variable with an independent random variable? (*Hint: The\n  effect should go to zero*)\n* **Simulated Outcome**: What happens to the estimated causal effect when we\n  replace the dataset with a simulated dataset based on a known data-generating\n  process closest to the given dataset? (*Hint: It should match the effect parameter\n  from the data-generating process*)\n* **Add Unobserved Common Causes**: How sensitive is the effect estimate when we\n  add an additional common cause (confounder) to the dataset that is correlated\n  with the treatment and the outcome? (*Hint: It should not be too sensitive*)\n* **Data Subsets Validation**: Does the estimated effect change significantly when\n  we replace the given dataset with a randomly selected subset? (*Hint: It\n  should not*)\n* **Bootstrap Validation**: Does the estimated effect change significantly when we\n  replace the given dataset with bootstrapped samples from the same dataset? (*Hint: It should not*)\n\nExamples of using refutation methods are in the `Refutations <https://github.com/microsoft/dowhy/blob/main/docs/source/example_notebooks/dowhy_refuter_notebook.ipynb>`_ notebook. For an advanced refutation that uses a simulated dataset based on user-provided or learnt data-generating processes, check out the `Dummy Outcome Refuter <https://github.com/microsoft/dowhy/blob/main/docs/source/example_notebooks/dowhy_demo_dummy_outcome_refuter.ipynb>`_ notebook.\nAs a practical example, `this notebook <https://github.com/microsoft/dowhy/blob/main/docs/source/example_notebooks/dowhy_refutation_testing.ipynb>`_ shows an application of refutation methods on evaluating effect estimators for the Infant Health and Development Program (IHDP) and Lalonde datasets.\n\nCiting this package\n====================\nIf you find DoWhy useful for your research work, please cite us as follows:\n\nAmit Sharma, Emre Kiciman, et al. DoWhy: A Python package for causal inference. 2019. https://github.com/microsoft/dowhy\n\nBibtex::\n\n  @misc{dowhy,\n  author={Sharma, Amit and Kiciman, Emre and others},\n  title={Do{W}hy: {A Python package for causal inference}},\n  howpublished={https://github.com/microsoft/dowhy},\n  year={2019}\n  }\n\nAlternatively, you can cite our Arxiv paper on DoWhy.\n\nAmit Sharma, Emre Kiciman. DoWhy: An End-to-End Library for Causal Inference. 2020. https://arxiv.org/abs/2011.04216\n\nBibtex::\n\n  @article{dowhypaper,\n  title={DoWhy: An End-to-End Library for Causal Inference},\n  author={Sharma, Amit and Kiciman, Emre},\n  journal={arXiv preprint arXiv:2011.04216},\n  year={2020}\n  }\n\nAnd if you find the gcm package useful for your work, please also cite us as:\n\nPatrick Blöbaum, Peter Götz, Kailash Budhathoki, Atalanti A. Mastakouri, Dominik Janzing. DoWhy-GCM: An extension of DoWhy for causal inference in graphical causal models. 2022. https://arxiv.org/abs/2206.06821\n\nBibtex::\n\n    @article{dowhy_gcm,\n      author = {Bl{\\"o}baum, Patrick and G{\\"o}tz, Peter and Budhathoki, Kailash and Mastakouri, Atalanti A. and Janzing, Dominik},\n      title = {DoWhy-GCM: An extension of DoWhy for causal inference in graphical causal models},\n      journal={arXiv preprint arXiv:2206.06821},\n      year={2022}\n    }\n\nRoadmap\n=======\nThe `projects <https://github.com/microsoft/dowhy/projects>`_ page lists the next steps for DoWhy. If you would like to contribute, have a look at the current projects. If you have a specific request for DoWhy, please `raise an issue <https://github.com/microsoft/dowhy/issues>`_.\n\nContributing\n============\n\nThis project welcomes contributions and suggestions. For a guide to contributing and a list of all contributors, check out `CONTRIBUTING.md <https://github.com/microsoft/dowhy/blob/main/CONTRIBUTING.md>`_ and our `docs for contributing code <https://github.com/py-why/dowhy/blob/main/docs/source/contributing/contributing-code.rst>`_. Our `contributor code of conduct is available here <https://github.com/py-why/governance/blob/main/CODE-OF-CONDUCT.md>`_. You can also join the DoWhy development channel on Discord: |discord|_\n\n.. |discord| image:: https://img.shields.io/discord/818456847551168542\n.. _discord: https://discord.gg/cSBGb3vsZb\n\n',
    'author': 'PyWhy Community',
    'author_email': 'amshar@microsoft.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/pywhy/dowhy',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.8,<3.10',
}


setup(**setup_kwargs)
