"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PrivateCA = void 0;
const acmpca = require("@aws-cdk/aws-acmpca");
const acm = require("@aws-cdk/aws-certificatemanager");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const route53 = require("@aws-cdk/aws-route53");
const core_1 = require("@aws-cdk/core");
class PrivateCA extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.route53_zone = new route53.PrivateHostedZone(this, 'CpInternalHostedZone', {
            vpc: props.cluster.vpc,
            zoneName: props.hostedZoneName,
        });
        //TODO : Customize this
        const cfnCertificateAuthority = new acmpca.CfnCertificateAuthority(this, 'CA', {
            type: 'ROOT',
            keyAlgorithm: 'RSA_2048',
            signingAlgorithm: 'SHA256WITHRSA',
            subject: {
                country: 'US',
                organization: 'Amazon',
                organizationalUnit: 'AWS',
                state: 'WA',
                commonName: props.hostedZoneName,
                locality: 'Seattle',
            },
        });
        this.private_ca_arn = cfnCertificateAuthority.attrArn;
        const internalRootCaCert = new acmpca.CfnCertificate(this, 'PrivCaCert', {
            certificateAuthorityArn: this.private_ca_arn,
            certificateSigningRequest: cfnCertificateAuthority.attrCertificateSigningRequest,
            templateArn: 'arn:aws:acm-pca:::template/RootCACertificate/V1',
            signingAlgorithm: 'SHA256WITHRSA',
            validity: {
                value: 10,
                type: 'YEARS',
            },
        });
        const internalRootCaActivation = new acmpca.CfnCertificateAuthorityActivation(this, 'PrivCaActivation', {
            certificateAuthorityArn: this.private_ca_arn,
            certificate: internalRootCaCert.attrCertificate,
        });
        const wild_card_cert = new acm.CfnCertificate(this, 'CpCert', {
            domainName: `*.${props.hostedZoneName}`,
            certificateAuthorityArn: this.private_ca_arn,
        });
        wild_card_cert.node.addDependency(internalRootCaActivation);
        const cluster_issuer_namespace_manifest = {
            apiVersion: 'v1',
            kind: 'Namespace',
            metadata: { name: 'aws-pca-issuer' },
        };
        const cert_issuer_namespace = props.cluster.addManifest('ClusterIssuerNamespaceManifest', cluster_issuer_namespace_manifest);
        const service_account = props.cluster.addServiceAccount('PrivateCASA');
        service_account.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            resources: [this.private_ca_arn],
            actions: [
                'acm-pca:Get*',
                'acm-pca:Issue*',
                'acm-pca:Describe*',
            ],
        }));
        const service_account_external_dns = props.cluster.addServiceAccount('ExternalDns');
        service_account_external_dns.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            resources: ['*'],
            actions: [
                'route53:*',
            ],
        }));
        // props.nodegroup.role.addToPrincipalPolicy(new PolicyStatement({
        //   resources: ['*'],
        //   actions: [
        //     'route53:*',
        //   ],
        // }));
        props.cluster.addHelmChart('ExternalDnsHelm', {
            repository: 'https://kubernetes-sigs.github.io/external-dns/',
            chart: 'external-dns',
            // namespace: 'external-dns',
            release: 'external-dns',
            wait: true,
            values: {
                provider: 'aws',
                policy: 'sync',
                aws: {
                    region: core_1.Stack.of(this).region,
                    zoneType: 'private',
                },
                // domainFilters: [
                //   hosted_zone.zoneName,
                // ],
                serviceAccount: {
                    create: false,
                    name: service_account_external_dns.serviceAccountName,
                },
            },
        });
        // TODO : Dont know why, but pod SA still wants node to lookup Describe* calls on PCA
        props.nodegroup.role.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            resources: [this.private_ca_arn],
            actions: [
                'acm-pca:Get*',
                'acm-pca:Issue*',
                'acm-pca:Describe*',
            ],
        }));
        service_account.node.addDependency(cert_issuer_namespace);
        const deploy_cert_manager = props.cluster.addHelmChart('CertManager', {
            repository: 'https://charts.jetstack.io',
            chart: 'cert-manager',
            namespace: 'cert-manager',
            release: 'cert-manager',
            wait: true,
            version: 'v1.5.0',
            values: {
                installCRDs: true,
                webhook: {
                    timeoutSeconds: 30,
                },
            },
        });
        const priv_ca_issuer_helm = props.cluster.addHelmChart('PrivateCaIssuer', {
            repository: 'https://cert-manager.github.io/aws-privateca-issuer',
            chart: 'aws-pca-issuer',
            namespace: 'aws-pca-issuer',
            release: 'aws-pca-issuer',
            wait: true,
            values: {
                serviceAccount: {
                    create: 'false',
                    name: service_account.serviceAccountName,
                },
            },
        });
        priv_ca_issuer_helm.node.addDependency(internalRootCaActivation, deploy_cert_manager);
        const deploy_waiter = props.cluster.addHelmChart('DummyWaiterPostCertManager', {
            repository: 'https://anshrma.github.io/helm-charts',
            chart: 'helm-waiter',
            // namespace: 'default',
            release: 'helm-waiter',
            wait: true,
        });
        deploy_waiter.node.addDependency(priv_ca_issuer_helm);
        const cluster_issuer_manifest = {
            apiVersion: 'awspca.cert-manager.io/v1beta1',
            kind: 'AWSPCAClusterIssuer',
            metadata: { name: props.clusterIssuerName },
            spec: {
                arn: this.private_ca_arn,
                // arn: cfnCertificateAuthority.attrArn,
                region: core_1.Stack.of(this).region,
            },
        };
        const cluster_issuer = props.cluster.addManifest('ClusterIssuerManifest', cluster_issuer_manifest);
        cluster_issuer.node.addDependency(deploy_waiter);
        const namespace_manifest = {
            apiVersion: 'v1',
            kind: 'Namespace',
            metadata: { name: props.namespace },
        };
        const ns = props.cluster.addManifest('NSManifests', namespace_manifest);
        const cert_manifest = {
            apiVersion: 'cert-manager.io/v1',
            kind: 'Certificate',
            metadata: {
                name: props.cacertname,
                namespace: props.namespace,
            },
            spec: {
                commonName: props.hostedZoneName,
                dnsNames: props.dnsNames,
                duration: '2160h0m0s',
                issuerRef: {
                    group: 'awspca.cert-manager.io',
                    kind: 'AWSPCAClusterIssuer',
                    name: props.clusterIssuerName,
                },
                renewBefore: '360h0m0s',
                secretName: `${props.clusterIssuerName}-secret`,
                usages: [
                    'server auth',
                    'client auth',
                ],
                privateKey: {
                    algorithm: 'RSA',
                    size: 2048,
                },
            },
        };
        const cert = props.cluster.addManifest('CertManifests', cert_manifest);
        cert.node.addDependency(ns, cluster_issuer);
        // new CfnOutput(this, 'ExtDnsRoleArn', { value: service_account_external_dns.role.roleArn });
        // new CfnOutput(this, 'CertIssuerRoleArnOutput', { value: service_account.serviceAccountName });
    }
}
exports.PrivateCA = PrivateCA;
//# sourceMappingURL=data:application/json;base64,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