"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KongControlPlane = void 0;
const aws_iam_1 = require("@aws-cdk/aws-iam");
// import * as sm from '@aws-cdk/aws-secretsmanager';
const core_1 = require("@aws-cdk/core");
class KongControlPlane extends core_1.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        // TODO : Dont know why, but pod SA still wants node to lookup Describe* calls on PCA
        props.nodegroup.role.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            resources: ['*'],
            actions: [
                'secretsmanager:GetResourcePolicy',
                'secretsmanager:GetSecretValue',
                'secretsmanager:DescribeSecret',
                'secretsmanager:ListSecretVersionIds',
            ],
        }));
        const external_secrets_helm = props.cluster.addHelmChart('ExternalSecretsHandler', {
            repository: 'https://external-secrets.github.io/kubernetes-external-secrets/',
            chart: 'kubernetes-external-secrets',
            release: 'kubernetes-external-secrets',
            namespace: props.namespace,
            wait: true,
            values: {
                securityContext: {
                    fsGroup: 65534,
                },
                env: {
                    AWS_REGION: core_1.Stack.of(this).region,
                },
            },
        });
        const secretNameParts = core_1.Fn.split('-', (_a = props.rds.secret) === null || _a === void 0 ? void 0 : _a.secretName);
        const secretNameWithoutSuffix = core_1.Fn.join('-', [core_1.Fn.select(0, secretNameParts), core_1.Fn.select(1, secretNameParts)]);
        const secret_provider_class = {
            apiVersion: 'kubernetes-client.io/v1',
            kind: 'ExternalSecret',
            metadata: {
                name: 'rds-secrets',
                namespace: props.namespace,
            },
            spec: {
                backendType: 'secretsManager',
                data: [
                    {
                        key: secretNameWithoutSuffix,
                        name: 'password',
                        property: 'password',
                    },
                    {
                        key: secretNameWithoutSuffix,
                        name: 'host',
                        property: 'host',
                    },
                    {
                        key: secretNameWithoutSuffix,
                        name: 'username',
                        property: 'username',
                    },
                    {
                        key: secretNameWithoutSuffix,
                        name: 'dbname',
                        property: 'dbname',
                    },
                ],
            },
        };
        const kong_rds_secrets = props.cluster.addManifest('KongRdsSecrets', secret_provider_class);
        kong_rds_secrets.node.addDependency(external_secrets_helm);
        const kong_helm = props.cluster.addHelmChart('KongCpHelm', {
            repository: 'https://charts.konghq.com',
            chart: 'kong',
            release: 'kong',
            namespace: props.namespace,
            timeout: core_1.Duration.minutes(15),
            wait: true,
            createNamespace: true,
            values: {
                ingressController: {
                    enabled: true,
                    installCRDs: false,
                    image: {
                        repository: 'kong/kubernetes-ingress-controller',
                        tag: '1.3.1-alpine',
                    },
                },
                image: {
                    repository: 'kong/kong-gateway',
                    tag: '2.5.0.0-alpine',
                },
                env: {
                    database: 'postgres',
                    role: 'control_plane',
                    cluster_mtls: 'pki',
                    cluster_cert: '/etc/secrets/kong-cp-cluster-issuer-secret/tls.crt',
                    cluster_cert_key: '/etc/secrets/kong-cp-cluster-issuer-secret/tls.key',
                    cluster_ca_cert: '/etc/secrets/kong-cp-cluster-issuer-secret/ca.crt',
                    pg_user: {
                        valueFrom: {
                            secretKeyRef: {
                                name: 'rds-secrets',
                                key: 'username',
                            },
                        },
                    },
                    pg_password: {
                        valueFrom: {
                            secretKeyRef: {
                                name: 'rds-secrets',
                                key: 'password',
                            },
                        },
                    },
                    pg_database: {
                        valueFrom: {
                            secretKeyRef: {
                                name: 'rds-secrets',
                                key: 'dbname',
                            },
                        },
                    },
                    pg_host: {
                        valueFrom: {
                            secretKeyRef: {
                                name: 'rds-secrets',
                                key: 'host',
                            },
                        },
                    },
                },
                cluster: {
                    enabled: 'true',
                    type: 'LoadBalancer',
                    tls: {
                        enabled: 'true',
                        servicePort: 8005,
                        containerPort: 8005,
                    },
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'external-dns.alpha.kubernetes.io/hostname': 'cluster.kong-cp.internal',
                    },
                },
                clustertelemetry: {
                    enabled: true,
                    type: 'LoadBalancer',
                    tls: {
                        enabled: 'true',
                    },
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'external-dns.alpha.kubernetes.io/hostname': 'telemetry.kong-cp.internal',
                    },
                },
                proxy: {
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'service.beta.kubernetes.io/aws-load-balancer-additional-resource-tags': 'Type=proxy',
                        'external-dns.alpha.kubernetes.io/hostname': 'proxy.kong-cp.internal',
                    },
                },
                admin: {
                    enabled: true,
                    type: 'LoadBalancer',
                    labels: {
                        'enable-metrics': true,
                    },
                    http: {
                        enabled: true,
                    },
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'service.beta.kubernetes.io/aws-load-balancer-additional-resource-tags': 'Type=admin',
                        'external-dns.alpha.kubernetes.io/hostname': 'admin.kong-cp.internal',
                    },
                },
                enterprise: {
                    enabled: true,
                },
                serviceMonitor: {
                    enabled: true,
                },
                manager: {
                    enabled: true,
                    type: 'LoadBalancer',
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'external-dns.alpha.kubernetes.io/hostname': 'manager.kong-cp.internal',
                    },
                },
                postgresql: {
                    enabled: false,
                },
                secretVolumes: [
                    'kong-cp-cluster-issuer-secret',
                ],
            },
        });
        kong_helm.node.addDependency(kong_rds_secrets);
        //TODO : https://github.com/aws/aws-cdk/issues/14933
        // const kong_admin = new KubernetesObjectValue(this, 'KongAdminAttribute', {
        //   cluster: props.cluster,
        //   objectType: 'service',
        //   objectName: 'kong-kong-admin',
        //   jsonPath: '.status.loadBalancer.ingress[0].hostname',
        //   timeout: Duration.minutes(30),
        //   objectNamespace: props.namespace,
        // });
        // const kong_telemetry = new KubernetesObjectValue(this, 'KongTelemetryAttribute', {
        //   cluster: props.cluster,
        //   objectType: 'service',
        //   objectName: 'kong-kong-clustertelemetry',
        //   jsonPath: '.status.loadBalancer.ingress[0].hostname',
        //   timeout: Duration.minutes(30),
        //   objectNamespace: props.namespace,
        // });
        // const kong_cluster = new KubernetesObjectValue(this, 'KongClusterAttribute', {
        //   cluster: props.cluster,
        //   objectType: 'service',
        //   objectName: 'kong-kong-cluster',
        //   jsonPath: '.status.loadBalancer.ingress[0].hostname',
        //   timeout: Duration.minutes(30),
        //   objectNamespace: props.namespace,
        // });
        // // const kong_cluster= props.cluster.getServiceLoadBalancerAddress('kong-kong-cluster', { namespace: props.namespace }) ;
        // // const kong_telemetry = props.cluster.getServiceLoadBalancerAddress('kong-kong-clustertelemetry', { namespace: props.namespace }) ;
        // new CfnOutput(this, 'KongAdminOutput', { value: kong_admin.value, description: 'Kong Admin Endpoint' });
        // new CfnOutput(this, 'KongTelemetryOutput', { value: kong_telemetry.value, description: 'Kong Telemetry Endpoint' });
        // new CfnOutput(this, 'KongClusterOutput', { value: kong_cluster.value, description: 'Kong Cluster Endpoint' });
    }
}
exports.KongControlPlane = KongControlPlane;
//# sourceMappingURL=data:application/json;base64,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