# Terraform Kubernetes Engine ASM Submodule

This module installs [Anthos Service Mesh](https://cloud.google.com/service-mesh/docs) (ASM) in a Kubernetes Engine (GKE) cluster.

## Usage

 The ASM module requires Terraform 0.14+. There are a few prerequisites to using this module that can be done either through Terraform and/or manually:

1. Enable the `mesh.googleapis.com` service
1. Enable the `servicemesh` feature on the cluster hub
1. Register target cluster to the servicemesh-enabled hub
1. Configure the [Kubernetes Provider](https://registry.terraform.io/providers/hashicorp/kubernetes/latest/docs) for the target cluster, for example:

```tf
provider "kubernetes" {
  host                   = "https://${module.gke.endpoint}"
  token                  = data.google_client_config.default.access_token
  cluster_ca_certificate = base64decode(module.gke.ca_certificate)
}

data "google_client_config" "default" {}
```

There is a full example provided [here](../../examples/simple_zonal_with_asm). Detailed usage example is as follows:

```tf
module "asm" {
  source            = "terraform-google-modules/kubernetes-engine/google//modules/asm"
  project_id        = var.project_id
  cluster_name      = module.gke.name
  cluster_location  = module.gke.location
  enable_cni        = true
}
```

Note that the [`mesh_id` label on the cluster](https://cloud.google.com/service-mesh/docs/managed/auto-control-plane-with-fleet#apply_the_mesh_id_label) is required for metrics to get displayed on the Anthos Service Mesh pages in the Cloud console (Topology, etc.). Illustrated with the full example mentioned above, here is an example of what your cluster should have:
```tf
module "gke" {
...
  cluster_resource_labels = { "mesh_id" : "proj-${data.google_project.project.number}" }
...
}
```

To deploy this config:

1. Run `terraform apply`

 <!-- BEGINNING OF PRE-COMMIT-TERRAFORM DOCS HOOK -->
## Inputs

| Name | Description | Type | Default | Required |
|------|-------------|------|---------|:--------:|
| channel | The channel to use for this ASM installation. | `string` | `""` | no |
| cluster\_location | The cluster location for this ASM installation. | `string` | n/a | yes |
| cluster\_name | The unique name to identify the cluster in ASM. | `string` | n/a | yes |
| enable\_cni | Determines whether to enable CNI for this ASM installation. Required to use Managed Data Plane (MDP). | `bool` | `false` | no |
| enable\_fleet\_registration | Determines whether the module registers the cluster to the fleet. | `bool` | `false` | no |
| enable\_mesh\_feature | Determines whether the module enables the mesh feature on the fleet. | `bool` | `false` | no |
| enable\_vpc\_sc | Determines whether to enable VPC-SC for this ASM installation. For more information read https://cloud.google.com/service-mesh/docs/managed/vpc-sc | `bool` | `false` | no |
| fleet\_id | The fleet to use for this ASM installation. | `string` | `""` | no |
| internal\_ip | Use internal ip for the cluster endpoint when running kubectl commands. | `bool` | `false` | no |
| multicluster\_mode | [Preview] Determines whether remote secrets should be autogenerated across fleet cluster. | `string` | `"manual"` | no |
| project\_id | The project in which the resource belongs. | `string` | n/a | yes |

## Outputs

| Name | Description |
|------|-------------|
| revision\_name | The name of the installed managed ASM revision. |
| wait | An output to use when depending on the ASM installation finishing. |

<!-- END OF PRE-COMMIT-TERRAFORM DOCS HOOK -->
