import os
import shutil
import tempfile
import unittest
from pathlib import Path

from findfile import (
    find_file,
    find_files,
    find_dir,
    find_dirs,
    find_cwd_files,
    rm_file,
    rm_files,
    rm_dir,
    rm_dirs,
    FileManager,
)


class FindFileTestCase(unittest.TestCase):
    def setUp(self):
        self._orig_cwd = os.getcwd()
        self.tmp = tempfile.TemporaryDirectory()
        self.root = Path(self.tmp.name)
        # layout
        # root/
        #   a.txt
        #   b.md
        #   logs/run.log
        #   src/a.py
        #   src/data/a.txt
        #   nested/sub/file.tmp
        (self.root / "logs").mkdir(parents=True, exist_ok=True)
        (self.root / "src" / "data").mkdir(parents=True, exist_ok=True)
        (self.root / "nested" / "sub").mkdir(parents=True, exist_ok=True)

        (self.root / "a.txt").write_text("root a\n")
        (self.root / "b.md").write_text("x\n")
        (self.root / "logs" / "run.log").write_text("log\n")
        (self.root / "src" / "a.py").write_text("print('x')\n")
        (self.root / "src" / "data" / "a.txt").write_text("deep a\n")
        (self.root / "nested" / "sub" / "file.tmp").write_text("t\n")
        # ignored markers shouldn't affect unrelated results
        (self.root / ".FFIGNORE").write_text("ignored\n")

    def tearDown(self):
        os.chdir(self._orig_cwd)
        self.tmp.cleanup()

    def test_find_files_basic_and_exclude(self):
        # find all txt files
        res = find_files(self.root, and_key=[".txt"], recursive=5, return_relative_path=False)
        self.assertCountEqual(res, [str(self.root / "a.txt"), str(self.root / "src" / "data" / "a.txt")])

        # exclude by OR logic (default): exclude anything under 'src/data' only
        # use a more specific pattern to avoid matching the OS 'AppData' in temp path on Windows
        res2 = find_files(
            self.root,
            and_key=[".txt"],
            exclude_key=[str(Path("src") / "data")],
            recursive=5,
            return_relative_path=False,
        )
        self.assertEqual(res2, [str(self.root / "a.txt")])

        # nothing should match .log when excluding 'logs'
        res3 = find_files(self.root, and_key=[".log"], exclude_key=["logs"], recursive=5, return_relative_path=False)
        self.assertEqual(res3, [])

    def test_find_file_deepest_and_shortest(self):
        deepest = find_file(self.root, and_key=["a", ".txt"], recursive=5, return_relative_path=False, return_deepest_path=True)
        self.assertEqual(deepest, str(self.root / "src" / "data" / "a.txt"))

        shortest = find_file(self.root, and_key=["a", ".txt"], recursive=5, return_relative_path=False, return_deepest_path=False)
        self.assertEqual(shortest, str(self.root / "a.txt"))

    def test_find_dirs_leaf_only(self):
        # match any path containing 'src' should return only leaf dirs by default
        dirs = find_dirs(self.root, and_key=["src"], recursive=5, return_relative_path=False)
        # leaf-only should yield src/data (not src)
        self.assertEqual(dirs, [str(self.root / "src" / "data")])

    def test_rm_files_and_dirs(self):
        # remove a single tmp file by key
        rm_files(self.root, and_key=["file.tmp"], recursive=5, return_relative_path=False)
        self.assertFalse((self.root / "nested" / "sub" / "file.tmp").exists())

        # remove the logs directory
        rm_dirs(self.root, and_key=["logs"], recursive=2, return_relative_path=False)
        self.assertFalse((self.root / "logs").exists())

    def test_filemanager_read_and_lines(self):
        # operate in temp cwd to exercise find_cwd_* helpers on fm
        os.chdir(self.root)
        fm = FileManager(str(self.root), recursive=5)

        # list txt files in cwd
        txts = fm.find_cwd_files(key=".txt", recursive=5, return_relative_path=False)
        self.assertCountEqual(txts, [str(self.root / "a.txt"), str(self.root / "src" / "data" / "a.txt")])

        # read content using fm helpers
        lines = fm.readlines(file_type="txt")
        self.assertIn("root a\n", lines)
        self.assertIn("deep a\n", lines)

        # read() should behave same as readlines()
        lines2 = fm.read(file_type=["txt"])  # accept list too
        self.assertEqual(sorted(lines), sorted(lines2))


if __name__ == "__main__":
    unittest.main()
