"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Expires = exports.StorageClass = exports.ServerSideEncryption = exports.CacheControl = exports.BucketDeployment = void 0;
const path = require("path");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const lambda_layer_awscli_1 = require("@aws-cdk/lambda-layer-awscli");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("@aws-cdk/core");
/**
 * @experimental
 */
class BucketDeployment extends core_1.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        if (props.distributionPaths && !props.distribution) {
            throw new Error('Distribution must be specified if distribution paths are specified');
        }
        const handler = new lambda.SingletonFunction(this, 'CustomResourceHandler', {
            uuid: this.renderSingletonUuid(props.memoryLimit),
            code: lambda.Code.fromAsset(path.join(__dirname, 'lambda')),
            layers: [new lambda_layer_awscli_1.AwsCliLayer(this, 'AwsCliLayer')],
            runtime: lambda.Runtime.PYTHON_3_6,
            handler: 'index.handler',
            lambdaPurpose: 'Custom::CDKBucketDeployment',
            timeout: cdk.Duration.minutes(15),
            role: props.role,
            memorySize: props.memoryLimit,
            vpc: props.vpc,
            vpcSubnets: props.vpcSubnets,
        });
        const handlerRole = handler.role;
        if (!handlerRole) {
            throw new Error('lambda.SingletonFunction should have created a Role');
        }
        const sources = props.sources.map((source) => source.bind(this, { handlerRole }));
        props.destinationBucket.grantReadWrite(handler);
        if (props.distribution) {
            handler.addToRolePolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['cloudfront:GetInvalidation', 'cloudfront:CreateInvalidation'],
                resources: ['*'],
            }));
        }
        new cdk.CustomResource(this, 'CustomResource', {
            serviceToken: handler.functionArn,
            resourceType: 'Custom::CDKBucketDeployment',
            properties: {
                SourceBucketNames: sources.map(source => source.bucket.bucketName),
                SourceObjectKeys: sources.map(source => source.zipObjectKey),
                DestinationBucketName: props.destinationBucket.bucketName,
                DestinationBucketKeyPrefix: props.destinationKeyPrefix,
                RetainOnDelete: props.retainOnDelete,
                Prune: (_a = props.prune) !== null && _a !== void 0 ? _a : true,
                UserMetadata: props.metadata ? mapUserMetadata(props.metadata) : undefined,
                SystemMetadata: mapSystemMetadata(props),
                DistributionId: props.distribution ? props.distribution.distributionId : undefined,
                DistributionPaths: props.distributionPaths,
            },
        });
    }
    renderSingletonUuid(memoryLimit) {
        let uuid = '8693BB64-9689-44B6-9AAF-B0CC9EB8756C';
        // if user specify a custom memory limit, define another singleton handler
        // with this configuration. otherwise, it won't be possible to use multiple
        // configurations since we have a singleton.
        if (memoryLimit) {
            if (cdk.Token.isUnresolved(memoryLimit)) {
                throw new Error('Can\'t use tokens when specifying "memoryLimit" since we use it to identify the singleton custom resource handler');
            }
            uuid += `-${memoryLimit.toString()}MiB`;
        }
        return uuid;
    }
}
exports.BucketDeployment = BucketDeployment;
/**
 * Metadata
 */
function mapUserMetadata(metadata) {
    const mapKey = (key) => key.toLowerCase();
    return Object.keys(metadata).reduce((o, key) => ({ ...o, [mapKey(key)]: metadata[key] }), {});
}
function mapSystemMetadata(metadata) {
    const res = {};
    if (metadata.cacheControl) {
        res['cache-control'] = metadata.cacheControl.map(c => c.value).join(', ');
    }
    if (metadata.expires) {
        res.expires = metadata.expires.date.toUTCString();
    }
    if (metadata.contentDisposition) {
        res['content-disposition'] = metadata.contentDisposition;
    }
    if (metadata.contentEncoding) {
        res['content-encoding'] = metadata.contentEncoding;
    }
    if (metadata.contentLanguage) {
        res['content-language'] = metadata.contentLanguage;
    }
    if (metadata.contentType) {
        res['content-type'] = metadata.contentType;
    }
    if (metadata.serverSideEncryption) {
        res.sse = metadata.serverSideEncryption;
    }
    if (metadata.storageClass) {
        res['storage-class'] = metadata.storageClass;
    }
    if (metadata.websiteRedirectLocation) {
        res['website-redirect'] = metadata.websiteRedirectLocation;
    }
    if (metadata.serverSideEncryptionAwsKmsKeyId) {
        res['sse-kms-key-id'] = metadata.serverSideEncryptionAwsKmsKeyId;
    }
    if (metadata.serverSideEncryptionCustomerAlgorithm) {
        res['sse-c-copy-source'] = metadata.serverSideEncryptionCustomerAlgorithm;
    }
    return Object.keys(res).length === 0 ? undefined : res;
}
/**
 * (experimental) Used for HTTP cache-control header, which influences downstream caches.
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 * @experimental
 */
class CacheControl {
    constructor(value) {
        this.value = value;
    }
    /**
     * @experimental
     */
    static mustRevalidate() { return new CacheControl('must-revalidate'); }
    /**
     * @experimental
     */
    static noCache() { return new CacheControl('no-cache'); }
    /**
     * @experimental
     */
    static noTransform() { return new CacheControl('no-transform'); }
    /**
     * @experimental
     */
    static setPublic() { return new CacheControl('public'); }
    /**
     * @experimental
     */
    static setPrivate() { return new CacheControl('private'); }
    /**
     * @experimental
     */
    static proxyRevalidate() { return new CacheControl('proxy-revalidate'); }
    /**
     * @experimental
     */
    static maxAge(t) { return new CacheControl(`max-age=${t.toSeconds()}`); }
    /**
     * @experimental
     */
    static sMaxAge(t) { return new CacheControl(`s-maxage=${t.toSeconds()}`); }
    /**
     * @experimental
     */
    static fromString(s) { return new CacheControl(s); }
}
exports.CacheControl = CacheControl;
/**
 * (experimental) Indicates whether server-side encryption is enabled for the object, and whether that encryption is from the AWS Key Management Service (AWS KMS) or from Amazon S3 managed encryption (SSE-S3).
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 * @experimental
 */
var ServerSideEncryption;
(function (ServerSideEncryption) {
    ServerSideEncryption["AES_256"] = "AES256";
    ServerSideEncryption["AWS_KMS"] = "aws:kms";
})(ServerSideEncryption = exports.ServerSideEncryption || (exports.ServerSideEncryption = {}));
/**
 * (experimental) Storage class used for storing the object.
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 * @experimental
 */
var StorageClass;
(function (StorageClass) {
    StorageClass["STANDARD"] = "STANDARD";
    StorageClass["REDUCED_REDUNDANCY"] = "REDUCED_REDUNDANCY";
    StorageClass["STANDARD_IA"] = "STANDARD_IA";
    StorageClass["ONEZONE_IA"] = "ONEZONE_IA";
    StorageClass["INTELLIGENT_TIERING"] = "INTELLIGENT_TIERING";
    StorageClass["GLACIER"] = "GLACIER";
    StorageClass["DEEP_ARCHIVE"] = "DEEP_ARCHIVE";
})(StorageClass = exports.StorageClass || (exports.StorageClass = {}));
/**
 * (deprecated) Used for HTTP expires header, which influences downstream caches.
 *
 * Does NOT influence deletion of the object.
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 * @deprecated use core.Expiration
 */
class Expires {
    constructor(value) {
        this.value = value;
    }
    /**
     * (deprecated) Expire at the specified date.
     *
     * @param d date to expire at.
     * @deprecated
     */
    static atDate(d) { return new Expires(d.toUTCString()); }
    /**
     * (deprecated) Expire at the specified timestamp.
     *
     * @param t timestamp in unix milliseconds.
     * @deprecated
     */
    static atTimestamp(t) { return Expires.atDate(new Date(t)); }
    /**
     * (deprecated) Expire once the specified duration has passed since deployment time.
     *
     * @param t the duration to wait before expiring.
     * @deprecated
     */
    static after(t) { return Expires.atDate(new Date(Date.now() + t.toMilliseconds())); }
    /**
     * @deprecated
     */
    static fromString(s) { return new Expires(s); }
}
exports.Expires = Expires;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYnVja2V0LWRlcGxveW1lbnQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJidWNrZXQtZGVwbG95bWVudC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSw2QkFBNkI7QUFHN0Isd0NBQXdDO0FBQ3hDLDhDQUE4QztBQUU5QyxxQ0FBcUM7QUFDckMsc0VBQTJEO0FBSTNELGlHQUFpRztBQUNqRyw4REFBOEQ7QUFDOUQsd0NBQTJEOzs7O0FBd0szRCxNQUFhLGdCQUFpQixTQUFRLGdCQUFhOzs7O0lBQ2pELFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBNEI7O1FBQ3BFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsSUFBSSxLQUFLLENBQUMsaUJBQWlCLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWSxFQUFFO1lBQ2xELE1BQU0sSUFBSSxLQUFLLENBQUMsb0VBQW9FLENBQUMsQ0FBQztTQUN2RjtRQUVELE1BQU0sT0FBTyxHQUFHLElBQUksTUFBTSxDQUFDLGlCQUFpQixDQUFDLElBQUksRUFBRSx1QkFBdUIsRUFBRTtZQUMxRSxJQUFJLEVBQUUsSUFBSSxDQUFDLG1CQUFtQixDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUM7WUFDakQsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1lBQzNELE1BQU0sRUFBRSxDQUFDLElBQUksaUNBQVcsQ0FBQyxJQUFJLEVBQUUsYUFBYSxDQUFDLENBQUM7WUFDOUMsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsVUFBVTtZQUNsQyxPQUFPLEVBQUUsZUFBZTtZQUN4QixhQUFhLEVBQUUsNkJBQTZCO1lBQzVDLE9BQU8sRUFBRSxHQUFHLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7WUFDakMsSUFBSSxFQUFFLEtBQUssQ0FBQyxJQUFJO1lBQ2hCLFVBQVUsRUFBRSxLQUFLLENBQUMsV0FBVztZQUM3QixHQUFHLEVBQUUsS0FBSyxDQUFDLEdBQUc7WUFDZCxVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVU7U0FDN0IsQ0FBQyxDQUFDO1FBRUgsTUFBTSxXQUFXLEdBQUcsT0FBTyxDQUFDLElBQUksQ0FBQztRQUNqQyxJQUFJLENBQUMsV0FBVyxFQUFFO1lBQUUsTUFBTSxJQUFJLEtBQUssQ0FBQyxxREFBcUQsQ0FBQyxDQUFDO1NBQUU7UUFFN0YsTUFBTSxPQUFPLEdBQW1CLEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLENBQUMsTUFBZSxFQUFFLEVBQUUsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxFQUFFLFdBQVcsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUUzRyxLQUFLLENBQUMsaUJBQWlCLENBQUMsY0FBYyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQ2hELElBQUksS0FBSyxDQUFDLFlBQVksRUFBRTtZQUN0QixPQUFPLENBQUMsZUFBZSxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztnQkFDOUMsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSztnQkFDeEIsT0FBTyxFQUFFLENBQUMsNEJBQTRCLEVBQUUsK0JBQStCLENBQUM7Z0JBQ3hFLFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQzthQUNqQixDQUFDLENBQUMsQ0FBQztTQUNMO1FBRUQsSUFBSSxHQUFHLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSxnQkFBZ0IsRUFBRTtZQUM3QyxZQUFZLEVBQUUsT0FBTyxDQUFDLFdBQVc7WUFDakMsWUFBWSxFQUFFLDZCQUE2QjtZQUMzQyxVQUFVLEVBQUU7Z0JBQ1YsaUJBQWlCLEVBQUUsT0FBTyxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsVUFBVSxDQUFDO2dCQUNsRSxnQkFBZ0IsRUFBRSxPQUFPLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLFlBQVksQ0FBQztnQkFDNUQscUJBQXFCLEVBQUUsS0FBSyxDQUFDLGlCQUFpQixDQUFDLFVBQVU7Z0JBQ3pELDBCQUEwQixFQUFFLEtBQUssQ0FBQyxvQkFBb0I7Z0JBQ3RELGNBQWMsRUFBRSxLQUFLLENBQUMsY0FBYztnQkFDcEMsS0FBSyxRQUFFLEtBQUssQ0FBQyxLQUFLLG1DQUFJLElBQUk7Z0JBQzFCLFlBQVksRUFBRSxLQUFLLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxlQUFlLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTO2dCQUMxRSxjQUFjLEVBQUUsaUJBQWlCLENBQUMsS0FBSyxDQUFDO2dCQUN4QyxjQUFjLEVBQUUsS0FBSyxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDLFNBQVM7Z0JBQ2xGLGlCQUFpQixFQUFFLEtBQUssQ0FBQyxpQkFBaUI7YUFDM0M7U0FDRixDQUFDLENBQUM7SUFFTCxDQUFDO0lBRU8sbUJBQW1CLENBQUMsV0FBb0I7UUFDOUMsSUFBSSxJQUFJLEdBQUcsc0NBQXNDLENBQUM7UUFFbEQsMEVBQTBFO1FBQzFFLDJFQUEyRTtRQUMzRSw0Q0FBNEM7UUFDNUMsSUFBSSxXQUFXLEVBQUU7WUFDZixJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLFdBQVcsQ0FBQyxFQUFFO2dCQUN2QyxNQUFNLElBQUksS0FBSyxDQUFDLG1IQUFtSCxDQUFDLENBQUM7YUFDdEk7WUFFRCxJQUFJLElBQUksSUFBSSxXQUFXLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQztTQUN6QztRQUVELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztDQUNGO0FBdkVELDRDQXVFQztBQUVEOztHQUVHO0FBRUgsU0FBUyxlQUFlLENBQUMsUUFBbUM7SUFDMUQsTUFBTSxNQUFNLEdBQUcsQ0FBQyxHQUFXLEVBQUUsRUFBRSxDQUFDLEdBQUcsQ0FBQyxXQUFXLEVBQUUsQ0FBQztJQUVsRCxPQUFPLE1BQU0sQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLEdBQUcsRUFBRSxFQUFFLENBQUMsQ0FBQyxFQUFFLEdBQUcsQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLEVBQUUsUUFBUSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQztBQUNoRyxDQUFDO0FBRUQsU0FBUyxpQkFBaUIsQ0FBQyxRQUErQjtJQUN4RCxNQUFNLEdBQUcsR0FBOEIsRUFBRSxDQUFDO0lBRTFDLElBQUksUUFBUSxDQUFDLFlBQVksRUFBRTtRQUFFLEdBQUcsQ0FBQyxlQUFlLENBQUMsR0FBRyxRQUFRLENBQUMsWUFBWSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7S0FBRTtJQUN6RyxJQUFJLFFBQVEsQ0FBQyxPQUFPLEVBQUU7UUFBRSxHQUFHLENBQUMsT0FBTyxHQUFHLFFBQVEsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxDQUFDO0tBQUU7SUFDNUUsSUFBSSxRQUFRLENBQUMsa0JBQWtCLEVBQUU7UUFBRSxHQUFHLENBQUMscUJBQXFCLENBQUMsR0FBRyxRQUFRLENBQUMsa0JBQWtCLENBQUM7S0FBRTtJQUM5RixJQUFJLFFBQVEsQ0FBQyxlQUFlLEVBQUU7UUFBRSxHQUFHLENBQUMsa0JBQWtCLENBQUMsR0FBRyxRQUFRLENBQUMsZUFBZSxDQUFDO0tBQUU7SUFDckYsSUFBSSxRQUFRLENBQUMsZUFBZSxFQUFFO1FBQUUsR0FBRyxDQUFDLGtCQUFrQixDQUFDLEdBQUcsUUFBUSxDQUFDLGVBQWUsQ0FBQztLQUFFO0lBQ3JGLElBQUksUUFBUSxDQUFDLFdBQVcsRUFBRTtRQUFFLEdBQUcsQ0FBQyxjQUFjLENBQUMsR0FBRyxRQUFRLENBQUMsV0FBVyxDQUFDO0tBQUU7SUFDekUsSUFBSSxRQUFRLENBQUMsb0JBQW9CLEVBQUU7UUFBRSxHQUFHLENBQUMsR0FBRyxHQUFHLFFBQVEsQ0FBQyxvQkFBb0IsQ0FBQztLQUFFO0lBQy9FLElBQUksUUFBUSxDQUFDLFlBQVksRUFBRTtRQUFFLEdBQUcsQ0FBQyxlQUFlLENBQUMsR0FBRyxRQUFRLENBQUMsWUFBWSxDQUFDO0tBQUU7SUFDNUUsSUFBSSxRQUFRLENBQUMsdUJBQXVCLEVBQUU7UUFBRSxHQUFHLENBQUMsa0JBQWtCLENBQUMsR0FBRyxRQUFRLENBQUMsdUJBQXVCLENBQUM7S0FBRTtJQUNyRyxJQUFJLFFBQVEsQ0FBQywrQkFBK0IsRUFBRTtRQUFFLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLFFBQVEsQ0FBQywrQkFBK0IsQ0FBQztLQUFFO0lBQ25ILElBQUksUUFBUSxDQUFDLHFDQUFxQyxFQUFFO1FBQUUsR0FBRyxDQUFDLG1CQUFtQixDQUFDLEdBQUcsUUFBUSxDQUFDLHFDQUFxQyxDQUFDO0tBQUU7SUFFbEksT0FBTyxNQUFNLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLE1BQU0sS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDO0FBQ3pELENBQUM7Ozs7Ozs7QUFNRCxNQUFhLFlBQVk7SUFXdkIsWUFBb0MsS0FBVTtRQUFWLFVBQUssR0FBTCxLQUFLLENBQUs7SUFBRyxDQUFDOzs7O0lBVjNDLE1BQU0sQ0FBQyxjQUFjLEtBQUssT0FBTyxJQUFJLFlBQVksQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLENBQUMsQ0FBQzs7OztJQUN2RSxNQUFNLENBQUMsT0FBTyxLQUFLLE9BQU8sSUFBSSxZQUFZLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxDQUFDOzs7O0lBQ3pELE1BQU0sQ0FBQyxXQUFXLEtBQUssT0FBTyxJQUFJLFlBQVksQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDLENBQUM7Ozs7SUFDakUsTUFBTSxDQUFDLFNBQVMsS0FBSyxPQUFPLElBQUksWUFBWSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQzs7OztJQUN6RCxNQUFNLENBQUMsVUFBVSxLQUFLLE9BQU8sSUFBSSxZQUFZLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDOzs7O0lBQzNELE1BQU0sQ0FBQyxlQUFlLEtBQUssT0FBTyxJQUFJLFlBQVksQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUMsQ0FBQzs7OztJQUN6RSxNQUFNLENBQUMsTUFBTSxDQUFDLENBQWUsSUFBSSxPQUFPLElBQUksWUFBWSxDQUFDLFdBQVcsQ0FBQyxDQUFDLFNBQVMsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7Ozs7SUFDdkYsTUFBTSxDQUFDLE9BQU8sQ0FBQyxDQUFlLElBQUksT0FBTyxJQUFJLFlBQVksQ0FBQyxZQUFZLENBQUMsQ0FBQyxTQUFTLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDOzs7O0lBQ3pGLE1BQU0sQ0FBQyxVQUFVLENBQUMsQ0FBUyxJQUFJLE9BQU8sSUFBSSxZQUFZLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO0NBR3BFO0FBWkQsb0NBWUM7Ozs7Ozs7QUFPRCxJQUFZLG9CQUdYO0FBSEQsV0FBWSxvQkFBb0I7SUFDOUIsMENBQWtCLENBQUE7SUFDbEIsMkNBQW1CLENBQUE7QUFDckIsQ0FBQyxFQUhXLG9CQUFvQixHQUFwQiw0QkFBb0IsS0FBcEIsNEJBQW9CLFFBRy9COzs7Ozs7O0FBTUQsSUFBWSxZQVFYO0FBUkQsV0FBWSxZQUFZO0lBQ3RCLHFDQUFxQixDQUFBO0lBQ3JCLHlEQUF5QyxDQUFBO0lBQ3pDLDJDQUEyQixDQUFBO0lBQzNCLHlDQUF5QixDQUFBO0lBQ3pCLDJEQUEyQyxDQUFBO0lBQzNDLG1DQUFtQixDQUFBO0lBQ25CLDZDQUE2QixDQUFBO0FBQy9CLENBQUMsRUFSVyxZQUFZLEdBQVosb0JBQVksS0FBWixvQkFBWSxRQVF2Qjs7Ozs7Ozs7O0FBUUQsTUFBYSxPQUFPO0lBcUJsQixZQUFvQyxLQUFVO1FBQVYsVUFBSyxHQUFMLEtBQUssQ0FBSztJQUFHLENBQUM7Ozs7Ozs7SUFoQjNDLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBTyxJQUFJLE9BQU8sSUFBSSxPQUFPLENBQUMsQ0FBQyxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDOzs7Ozs7O0lBTS9ELE1BQU0sQ0FBQyxXQUFXLENBQUMsQ0FBUyxJQUFJLE9BQU8sT0FBTyxDQUFDLE1BQU0sQ0FBQyxJQUFJLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQzs7Ozs7OztJQU1yRSxNQUFNLENBQUMsS0FBSyxDQUFDLENBQWUsSUFBSSxPQUFPLE9BQU8sQ0FBQyxNQUFNLENBQUMsSUFBSSxJQUFJLENBQUMsSUFBSSxDQUFDLEdBQUcsRUFBRSxHQUFHLENBQUMsQ0FBQyxjQUFjLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDOzs7O0lBRW5HLE1BQU0sQ0FBQyxVQUFVLENBQUMsQ0FBUyxJQUFJLE9BQU8sSUFBSSxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO0NBRy9EO0FBdEJELDBCQXNCQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgKiBhcyBjbG91ZGZyb250IGZyb20gJ0Bhd3MtY2RrL2F3cy1jbG91ZGZyb250JztcbmltcG9ydCAqIGFzIGVjMiBmcm9tICdAYXdzLWNkay9hd3MtZWMyJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tICdAYXdzLWNkay9hd3MtbGFtYmRhJztcbmltcG9ydCAqIGFzIHMzIGZyb20gJ0Bhd3MtY2RrL2F3cy1zMyc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBBd3NDbGlMYXllciB9IGZyb20gJ0Bhd3MtY2RrL2xhbWJkYS1sYXllci1hd3NjbGknO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBJU291cmNlLCBTb3VyY2VDb25maWcgfSBmcm9tICcuL3NvdXJjZSc7XG5cbi8vIGtlZXAgdGhpcyBpbXBvcnQgc2VwYXJhdGUgZnJvbSBvdGhlciBpbXBvcnRzIHRvIHJlZHVjZSBjaGFuY2UgZm9yIG1lcmdlIGNvbmZsaWN0cyB3aXRoIHYyLW1haW5cbi8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBuby1kdXBsaWNhdGUtaW1wb3J0cywgaW1wb3J0L29yZGVyXG5pbXBvcnQgeyBDb25zdHJ1Y3QgYXMgQ29yZUNvbnN0cnVjdCB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuXG5leHBvcnQgaW50ZXJmYWNlIEJ1Y2tldERlcGxveW1lbnRQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgc291cmNlczogSVNvdXJjZVtdO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBkZXN0aW5hdGlvbkJ1Y2tldDogczMuSUJ1Y2tldDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZGVzdGluYXRpb25LZXlQcmVmaXg/OiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHBydW5lPzogYm9vbGVhblxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHJldGFpbk9uRGVsZXRlPzogYm9vbGVhbjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBkaXN0cmlidXRpb24/OiBjbG91ZGZyb250LklEaXN0cmlidXRpb247XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZGlzdHJpYnV0aW9uUGF0aHM/OiBzdHJpbmdbXTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbWVtb3J5TGltaXQ/OiBudW1iZXI7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSByb2xlPzogaWFtLklSb2xlO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBtZXRhZGF0YT86IFVzZXJEZWZpbmVkT2JqZWN0TWV0YWRhdGE7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGNhY2hlQ29udHJvbD86IENhY2hlQ29udHJvbFtdO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGNvbnRlbnREaXNwb3NpdGlvbj86IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGNvbnRlbnRFbmNvZGluZz86IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGNvbnRlbnRMYW5ndWFnZT86IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgY29udGVudFR5cGU/OiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZXhwaXJlcz86IGNkay5FeHBpcmF0aW9uO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgc2VydmVyU2lkZUVuY3J5cHRpb24/OiBTZXJ2ZXJTaWRlRW5jcnlwdGlvbjtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBzdG9yYWdlQ2xhc3M/OiBTdG9yYWdlQ2xhc3M7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB3ZWJzaXRlUmVkaXJlY3RMb2NhdGlvbj86IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHNlcnZlclNpZGVFbmNyeXB0aW9uQXdzS21zS2V5SWQ/OiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgc2VydmVyU2lkZUVuY3J5cHRpb25DdXN0b21lckFsZ29yaXRobT86IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHZwYz86IGVjMi5JVnBjO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB2cGNTdWJuZXRzPzogZWMyLlN1Ym5ldFNlbGVjdGlvbjtcbn1cblxuZXhwb3J0IGNsYXNzIEJ1Y2tldERlcGxveW1lbnQgZXh0ZW5kcyBDb3JlQ29uc3RydWN0IHtcbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEJ1Y2tldERlcGxveW1lbnRQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICBpZiAocHJvcHMuZGlzdHJpYnV0aW9uUGF0aHMgJiYgIXByb3BzLmRpc3RyaWJ1dGlvbikge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdEaXN0cmlidXRpb24gbXVzdCBiZSBzcGVjaWZpZWQgaWYgZGlzdHJpYnV0aW9uIHBhdGhzIGFyZSBzcGVjaWZpZWQnKTtcbiAgICB9XG5cbiAgICBjb25zdCBoYW5kbGVyID0gbmV3IGxhbWJkYS5TaW5nbGV0b25GdW5jdGlvbih0aGlzLCAnQ3VzdG9tUmVzb3VyY2VIYW5kbGVyJywge1xuICAgICAgdXVpZDogdGhpcy5yZW5kZXJTaW5nbGV0b25VdWlkKHByb3BzLm1lbW9yeUxpbWl0KSxcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChwYXRoLmpvaW4oX19kaXJuYW1lLCAnbGFtYmRhJykpLFxuICAgICAgbGF5ZXJzOiBbbmV3IEF3c0NsaUxheWVyKHRoaXMsICdBd3NDbGlMYXllcicpXSxcbiAgICAgIHJ1bnRpbWU6IGxhbWJkYS5SdW50aW1lLlBZVEhPTl8zXzYsXG4gICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgICBsYW1iZGFQdXJwb3NlOiAnQ3VzdG9tOjpDREtCdWNrZXREZXBsb3ltZW50JyxcbiAgICAgIHRpbWVvdXQ6IGNkay5EdXJhdGlvbi5taW51dGVzKDE1KSxcbiAgICAgIHJvbGU6IHByb3BzLnJvbGUsXG4gICAgICBtZW1vcnlTaXplOiBwcm9wcy5tZW1vcnlMaW1pdCxcbiAgICAgIHZwYzogcHJvcHMudnBjLFxuICAgICAgdnBjU3VibmV0czogcHJvcHMudnBjU3VibmV0cyxcbiAgICB9KTtcblxuICAgIGNvbnN0IGhhbmRsZXJSb2xlID0gaGFuZGxlci5yb2xlO1xuICAgIGlmICghaGFuZGxlclJvbGUpIHsgdGhyb3cgbmV3IEVycm9yKCdsYW1iZGEuU2luZ2xldG9uRnVuY3Rpb24gc2hvdWxkIGhhdmUgY3JlYXRlZCBhIFJvbGUnKTsgfVxuXG4gICAgY29uc3Qgc291cmNlczogU291cmNlQ29uZmlnW10gPSBwcm9wcy5zb3VyY2VzLm1hcCgoc291cmNlOiBJU291cmNlKSA9PiBzb3VyY2UuYmluZCh0aGlzLCB7IGhhbmRsZXJSb2xlIH0pKTtcblxuICAgIHByb3BzLmRlc3RpbmF0aW9uQnVja2V0LmdyYW50UmVhZFdyaXRlKGhhbmRsZXIpO1xuICAgIGlmIChwcm9wcy5kaXN0cmlidXRpb24pIHtcbiAgICAgIGhhbmRsZXIuYWRkVG9Sb2xlUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgICBhY3Rpb25zOiBbJ2Nsb3VkZnJvbnQ6R2V0SW52YWxpZGF0aW9uJywgJ2Nsb3VkZnJvbnQ6Q3JlYXRlSW52YWxpZGF0aW9uJ10sXG4gICAgICAgIHJlc291cmNlczogWycqJ10sXG4gICAgICB9KSk7XG4gICAgfVxuXG4gICAgbmV3IGNkay5DdXN0b21SZXNvdXJjZSh0aGlzLCAnQ3VzdG9tUmVzb3VyY2UnLCB7XG4gICAgICBzZXJ2aWNlVG9rZW46IGhhbmRsZXIuZnVuY3Rpb25Bcm4sXG4gICAgICByZXNvdXJjZVR5cGU6ICdDdXN0b206OkNES0J1Y2tldERlcGxveW1lbnQnLFxuICAgICAgcHJvcGVydGllczoge1xuICAgICAgICBTb3VyY2VCdWNrZXROYW1lczogc291cmNlcy5tYXAoc291cmNlID0+IHNvdXJjZS5idWNrZXQuYnVja2V0TmFtZSksXG4gICAgICAgIFNvdXJjZU9iamVjdEtleXM6IHNvdXJjZXMubWFwKHNvdXJjZSA9PiBzb3VyY2UuemlwT2JqZWN0S2V5KSxcbiAgICAgICAgRGVzdGluYXRpb25CdWNrZXROYW1lOiBwcm9wcy5kZXN0aW5hdGlvbkJ1Y2tldC5idWNrZXROYW1lLFxuICAgICAgICBEZXN0aW5hdGlvbkJ1Y2tldEtleVByZWZpeDogcHJvcHMuZGVzdGluYXRpb25LZXlQcmVmaXgsXG4gICAgICAgIFJldGFpbk9uRGVsZXRlOiBwcm9wcy5yZXRhaW5PbkRlbGV0ZSxcbiAgICAgICAgUHJ1bmU6IHByb3BzLnBydW5lID8/IHRydWUsXG4gICAgICAgIFVzZXJNZXRhZGF0YTogcHJvcHMubWV0YWRhdGEgPyBtYXBVc2VyTWV0YWRhdGEocHJvcHMubWV0YWRhdGEpIDogdW5kZWZpbmVkLFxuICAgICAgICBTeXN0ZW1NZXRhZGF0YTogbWFwU3lzdGVtTWV0YWRhdGEocHJvcHMpLFxuICAgICAgICBEaXN0cmlidXRpb25JZDogcHJvcHMuZGlzdHJpYnV0aW9uID8gcHJvcHMuZGlzdHJpYnV0aW9uLmRpc3RyaWJ1dGlvbklkIDogdW5kZWZpbmVkLFxuICAgICAgICBEaXN0cmlidXRpb25QYXRoczogcHJvcHMuZGlzdHJpYnV0aW9uUGF0aHMsXG4gICAgICB9LFxuICAgIH0pO1xuXG4gIH1cblxuICBwcml2YXRlIHJlbmRlclNpbmdsZXRvblV1aWQobWVtb3J5TGltaXQ/OiBudW1iZXIpIHtcbiAgICBsZXQgdXVpZCA9ICc4NjkzQkI2NC05Njg5LTQ0QjYtOUFBRi1CMENDOUVCODc1NkMnO1xuXG4gICAgLy8gaWYgdXNlciBzcGVjaWZ5IGEgY3VzdG9tIG1lbW9yeSBsaW1pdCwgZGVmaW5lIGFub3RoZXIgc2luZ2xldG9uIGhhbmRsZXJcbiAgICAvLyB3aXRoIHRoaXMgY29uZmlndXJhdGlvbi4gb3RoZXJ3aXNlLCBpdCB3b24ndCBiZSBwb3NzaWJsZSB0byB1c2UgbXVsdGlwbGVcbiAgICAvLyBjb25maWd1cmF0aW9ucyBzaW5jZSB3ZSBoYXZlIGEgc2luZ2xldG9uLlxuICAgIGlmIChtZW1vcnlMaW1pdCkge1xuICAgICAgaWYgKGNkay5Ub2tlbi5pc1VucmVzb2x2ZWQobWVtb3J5TGltaXQpKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignQ2FuXFwndCB1c2UgdG9rZW5zIHdoZW4gc3BlY2lmeWluZyBcIm1lbW9yeUxpbWl0XCIgc2luY2Ugd2UgdXNlIGl0IHRvIGlkZW50aWZ5IHRoZSBzaW5nbGV0b24gY3VzdG9tIHJlc291cmNlIGhhbmRsZXInKTtcbiAgICAgIH1cblxuICAgICAgdXVpZCArPSBgLSR7bWVtb3J5TGltaXQudG9TdHJpbmcoKX1NaUJgO1xuICAgIH1cblxuICAgIHJldHVybiB1dWlkO1xuICB9XG59XG5cbi8qKlxuICogTWV0YWRhdGFcbiAqL1xuXG5mdW5jdGlvbiBtYXBVc2VyTWV0YWRhdGEobWV0YWRhdGE6IFVzZXJEZWZpbmVkT2JqZWN0TWV0YWRhdGEpIHtcbiAgY29uc3QgbWFwS2V5ID0gKGtleTogc3RyaW5nKSA9PiBrZXkudG9Mb3dlckNhc2UoKTtcblxuICByZXR1cm4gT2JqZWN0LmtleXMobWV0YWRhdGEpLnJlZHVjZSgobywga2V5KSA9PiAoeyAuLi5vLCBbbWFwS2V5KGtleSldOiBtZXRhZGF0YVtrZXldIH0pLCB7fSk7XG59XG5cbmZ1bmN0aW9uIG1hcFN5c3RlbU1ldGFkYXRhKG1ldGFkYXRhOiBCdWNrZXREZXBsb3ltZW50UHJvcHMpIHtcbiAgY29uc3QgcmVzOiB7IFtrZXk6IHN0cmluZ106IHN0cmluZyB9ID0ge307XG5cbiAgaWYgKG1ldGFkYXRhLmNhY2hlQ29udHJvbCkgeyByZXNbJ2NhY2hlLWNvbnRyb2wnXSA9IG1ldGFkYXRhLmNhY2hlQ29udHJvbC5tYXAoYyA9PiBjLnZhbHVlKS5qb2luKCcsICcpOyB9XG4gIGlmIChtZXRhZGF0YS5leHBpcmVzKSB7IHJlcy5leHBpcmVzID0gbWV0YWRhdGEuZXhwaXJlcy5kYXRlLnRvVVRDU3RyaW5nKCk7IH1cbiAgaWYgKG1ldGFkYXRhLmNvbnRlbnREaXNwb3NpdGlvbikgeyByZXNbJ2NvbnRlbnQtZGlzcG9zaXRpb24nXSA9IG1ldGFkYXRhLmNvbnRlbnREaXNwb3NpdGlvbjsgfVxuICBpZiAobWV0YWRhdGEuY29udGVudEVuY29kaW5nKSB7IHJlc1snY29udGVudC1lbmNvZGluZyddID0gbWV0YWRhdGEuY29udGVudEVuY29kaW5nOyB9XG4gIGlmIChtZXRhZGF0YS5jb250ZW50TGFuZ3VhZ2UpIHsgcmVzWydjb250ZW50LWxhbmd1YWdlJ10gPSBtZXRhZGF0YS5jb250ZW50TGFuZ3VhZ2U7IH1cbiAgaWYgKG1ldGFkYXRhLmNvbnRlbnRUeXBlKSB7IHJlc1snY29udGVudC10eXBlJ10gPSBtZXRhZGF0YS5jb250ZW50VHlwZTsgfVxuICBpZiAobWV0YWRhdGEuc2VydmVyU2lkZUVuY3J5cHRpb24pIHsgcmVzLnNzZSA9IG1ldGFkYXRhLnNlcnZlclNpZGVFbmNyeXB0aW9uOyB9XG4gIGlmIChtZXRhZGF0YS5zdG9yYWdlQ2xhc3MpIHsgcmVzWydzdG9yYWdlLWNsYXNzJ10gPSBtZXRhZGF0YS5zdG9yYWdlQ2xhc3M7IH1cbiAgaWYgKG1ldGFkYXRhLndlYnNpdGVSZWRpcmVjdExvY2F0aW9uKSB7IHJlc1snd2Vic2l0ZS1yZWRpcmVjdCddID0gbWV0YWRhdGEud2Vic2l0ZVJlZGlyZWN0TG9jYXRpb247IH1cbiAgaWYgKG1ldGFkYXRhLnNlcnZlclNpZGVFbmNyeXB0aW9uQXdzS21zS2V5SWQpIHsgcmVzWydzc2Uta21zLWtleS1pZCddID0gbWV0YWRhdGEuc2VydmVyU2lkZUVuY3J5cHRpb25Bd3NLbXNLZXlJZDsgfVxuICBpZiAobWV0YWRhdGEuc2VydmVyU2lkZUVuY3J5cHRpb25DdXN0b21lckFsZ29yaXRobSkgeyByZXNbJ3NzZS1jLWNvcHktc291cmNlJ10gPSBtZXRhZGF0YS5zZXJ2ZXJTaWRlRW5jcnlwdGlvbkN1c3RvbWVyQWxnb3JpdGhtOyB9XG5cbiAgcmV0dXJuIE9iamVjdC5rZXlzKHJlcykubGVuZ3RoID09PSAwID8gdW5kZWZpbmVkIDogcmVzO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgQ2FjaGVDb250cm9sIHtcbiAgcHVibGljIHN0YXRpYyBtdXN0UmV2YWxpZGF0ZSgpIHsgcmV0dXJuIG5ldyBDYWNoZUNvbnRyb2woJ211c3QtcmV2YWxpZGF0ZScpOyB9XG4gIHB1YmxpYyBzdGF0aWMgbm9DYWNoZSgpIHsgcmV0dXJuIG5ldyBDYWNoZUNvbnRyb2woJ25vLWNhY2hlJyk7IH1cbiAgcHVibGljIHN0YXRpYyBub1RyYW5zZm9ybSgpIHsgcmV0dXJuIG5ldyBDYWNoZUNvbnRyb2woJ25vLXRyYW5zZm9ybScpOyB9XG4gIHB1YmxpYyBzdGF0aWMgc2V0UHVibGljKCkgeyByZXR1cm4gbmV3IENhY2hlQ29udHJvbCgncHVibGljJyk7IH1cbiAgcHVibGljIHN0YXRpYyBzZXRQcml2YXRlKCkgeyByZXR1cm4gbmV3IENhY2hlQ29udHJvbCgncHJpdmF0ZScpOyB9XG4gIHB1YmxpYyBzdGF0aWMgcHJveHlSZXZhbGlkYXRlKCkgeyByZXR1cm4gbmV3IENhY2hlQ29udHJvbCgncHJveHktcmV2YWxpZGF0ZScpOyB9XG4gIHB1YmxpYyBzdGF0aWMgbWF4QWdlKHQ6IGNkay5EdXJhdGlvbikgeyByZXR1cm4gbmV3IENhY2hlQ29udHJvbChgbWF4LWFnZT0ke3QudG9TZWNvbmRzKCl9YCk7IH1cbiAgcHVibGljIHN0YXRpYyBzTWF4QWdlKHQ6IGNkay5EdXJhdGlvbikgeyByZXR1cm4gbmV3IENhY2hlQ29udHJvbChgcy1tYXhhZ2U9JHt0LnRvU2Vjb25kcygpfWApOyB9XG4gIHB1YmxpYyBzdGF0aWMgZnJvbVN0cmluZyhzOiBzdHJpbmcpIHsgcmV0dXJuIG5ldyBDYWNoZUNvbnRyb2wocyk7IH1cblxuICBwcml2YXRlIGNvbnN0cnVjdG9yKHB1YmxpYyByZWFkb25seSB2YWx1ZTogYW55KSB7fVxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgZW51bSBTZXJ2ZXJTaWRlRW5jcnlwdGlvbiB7XG4gIEFFU18yNTYgPSAnQUVTMjU2JyxcbiAgQVdTX0tNUyA9ICdhd3M6a21zJ1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGVudW0gU3RvcmFnZUNsYXNzIHtcbiAgU1RBTkRBUkQgPSAnU1RBTkRBUkQnLFxuICBSRURVQ0VEX1JFRFVOREFOQ1kgPSAnUkVEVUNFRF9SRURVTkRBTkNZJyxcbiAgU1RBTkRBUkRfSUEgPSAnU1RBTkRBUkRfSUEnLFxuICBPTkVaT05FX0lBID0gJ09ORVpPTkVfSUEnLFxuICBJTlRFTExJR0VOVF9USUVSSU5HID0gJ0lOVEVMTElHRU5UX1RJRVJJTkcnLFxuICBHTEFDSUVSID0gJ0dMQUNJRVInLFxuICBERUVQX0FSQ0hJVkUgPSAnREVFUF9BUkNISVZFJ1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgRXhwaXJlcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGF0RGF0ZShkOiBEYXRlKSB7IHJldHVybiBuZXcgRXhwaXJlcyhkLnRvVVRDU3RyaW5nKCkpOyB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgYXRUaW1lc3RhbXAodDogbnVtYmVyKSB7IHJldHVybiBFeHBpcmVzLmF0RGF0ZShuZXcgRGF0ZSh0KSk7IH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgYWZ0ZXIodDogY2RrLkR1cmF0aW9uKSB7IHJldHVybiBFeHBpcmVzLmF0RGF0ZShuZXcgRGF0ZShEYXRlLm5vdygpICsgdC50b01pbGxpc2Vjb25kcygpKSk7IH1cblxuICBwdWJsaWMgc3RhdGljIGZyb21TdHJpbmcoczogc3RyaW5nKSB7IHJldHVybiBuZXcgRXhwaXJlcyhzKTsgfVxuXG4gIHByaXZhdGUgY29uc3RydWN0b3IocHVibGljIHJlYWRvbmx5IHZhbHVlOiBhbnkpIHt9XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgVXNlckRlZmluZWRPYmplY3RNZXRhZGF0YSB7XG4gIC8qKlxuICAgKiBBcmJpdHJhcnkgbWV0YWRhdGEga2V5LXZhbHVlc1xuICAgKiBUaGUgYHgtYW16LW1ldGEtYCBwcmVmaXggd2lsbCBhdXRvbWF0aWNhbGx5IGJlIGFkZGVkIHRvIGtleXMuXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9kZXYvVXNpbmdNZXRhZGF0YS5odG1sI1VzZXJNZXRhZGF0YVxuICAgKi9cbiAgcmVhZG9ubHkgW2tleTogc3RyaW5nXTogc3RyaW5nO1xufVxuIl19