"""
This module provides functions to create molecules from core and R-groups.

Attributes:
    most_common (List[str]): a list of the most common R-groups.
    most_common_in_NP (List[str]): a list of the most common R-groups in natural products.
"""

from collections import defaultdict
from itertools import product
from concurrent.futures import ProcessPoolExecutor

from tqdm import tqdm

from rdkit import Chem
from rdkit.Chem import BRICS, AllChem
from rdkit.Chem.rdchem import EditableMol

from rdworks.mol import Mol
from rdworks.mollibr import MolLibr

import rdworks


def fragmented(rdmol:Chem.Mol, min_atoms:int=5, max_atoms:int=17) -> list[Chem.Mol]:
    """BRICS decompose and returns fragments"""
    dummy = Chem.MolFromSmiles('*')
    hydro = Chem.MolFromSmiles('[H]')
    frag_set = BRICS.BRICSDecompose(rdmol)
    lfrag_rdmol = []
    for frag_smi in frag_set:
        # if '.' in frag_smi :
        (_, frag_rdmol) = rdworks.desalt_smiles(frag_smi)
        # replace dummy atom(s) with [H]
        frag_rdmol_H = AllChem.ReplaceSubstructs(frag_rdmol, dummy, hydro, True)[0]
        frag_rdmol = Chem.RemoveHs(frag_rdmol_H)
        frag_smi = Chem.MolToSmiles(frag_rdmol)
        # filter out molecules which are too small or too big
        na = frag_rdmol.GetNumAtoms()
        if na < min_atoms or na > max_atoms : 
            continue
        lfrag_rdmol.append(frag_rdmol)
    return lfrag_rdmol


def connect_core_and_chains(combi_smiles:str) -> Mol | None:
    """Connects scaffold core and chains(R-groups).

    `combi_smiles` has [`expr`:`n`] format where `expr` is an expression of atom mapping
    and n is the atom map number. 
    It is modified from the Patrick Walters' work, 
    [Re: [Rdkit-discuss] reassembling a molecule from R-groups](https://sourceforge.net/p/rdkit/mailman/message/36294482/).
    Examples:
        >>> combi_smiles = "CN(C)CC(Br)c1cc([*:2])c([*:1])cn1.[H]C([H])([H])[*:1].[H][*:2]"
        >>> m = connect_core_and_chains(combi_smiles)

    Args:
        combi_smiles (str): concatenated SMILES containing core followed by chains.

    Returns:
        Union[Mol, None]: connected rdworks.Mol object.
    """
    connect = defaultdict(list)
    core_and_chains = Chem.MolFromSmiles(combi_smiles, sanitize=False)
    # default sanitize=True causes WARNING: not removing hydrogen atom with dummy atom neighbors
    for atom in core_and_chains.GetAtoms():
        atom_map_num = atom.GetAtomMapNum()
        if atom_map_num > 0:
            connect[atom_map_num].append(atom)
    # Second pass, transfer the atom maps to the neighbor atoms
    for atom_map_num, atom_list in connect.items():
        if len(atom_list) == 2:
            atm_1, atm_2 = atom_list
            nbr_1 = [x.GetOtherAtom(atm_1) for x in atm_1.GetBonds()][0]
            nbr_1.SetAtomMapNum(atom_map_num)
            nbr_2 = [x.GetOtherAtom(atm_2) for x in atm_2.GetBonds()][0]
            nbr_2.SetAtomMapNum(atom_map_num)
    # Nuke all of the dummy atoms
    new_mol = Chem.DeleteSubstructs(core_and_chains, Chem.MolFromSmarts('[#0]'))
    # Third pass - arrange the atoms with AtomMapNum, these will be connected
    connect_bond = defaultdict(list)
    for atom in new_mol.GetAtoms():
        atom_map_num = atom.GetAtomMapNum()
        if atom_map_num > 0:
            connect_bond[atom_map_num].append(atom.GetIdx())
    # Make an editable molecule and add bonds between atoms with correspoing AtomMapNum
    em = EditableMol(new_mol)
    for atom_map_num, atom_list in connect_bond.items():
        if len(atom_list) == 2:
            start_atm, end_atm = atom_list
            em.AddBond(start_atm, end_atm, order=Chem.rdchem.BondType.SINGLE)
    bonded_mol = em.GetMol()
    # remove the AtomMapNum values
    for atom in bonded_mol.GetAtoms():
        atom.SetAtomMapNum(0) 
    try:
        return Mol(Chem.RemoveHs(bonded_mol)) # returns a new molecule
    except:
        return None
        

def expand_rgroup(core:str, r:dict[int, list[str]], prefix:str='RGX', start:int=1, **kwargs) -> MolLibr:
    """Expands or enumerates the core with R groups.

    `core` SMILES has `[#0:n]` where n is the attachment index. All attachments are enumerated.

    Examples:
        >>> core = "[#0:1]C1C([#0:4])C([#0:3])CN2[C@@H]([C@H](C3=CC=C(C#CC4=CC=CC=C4)C=C3)[C@]2([H])CN1C(NC5=CC=C(C=C5)OC)=O)[#0:2]"
        >>> V = "[*]C#N.[*]C(O)=O.[*]CO.[*]C(NC)=O.[*]CNC(C)=O.[*]CC=C.[*][H]".split(".")
        >>> W = "[*][H].[*]COC".split(".")
        >>> X = "[*][H].[*]CC(F)(F)F.[*]OCCOC".split(".")
        >>> Y = "[*][H].[*]O.[*]OC".split(".")
        >>> output = expand_rgroup(core=core, r={1:V, 2:W, 3:X, 4:Y})
    
    Args:
        core (str): SMILES for the core.
        r (Dict[int, List[str]]): dictionary of {atom mapping number: list of SMILES for R groups}.
        prefix (str, optional): name prefix of generated molecules. Defaults to 'RGX'.
        start (int, optional): start number for serial numbers. Defaults to 1.

    Returns:
        MolLibr: a library of generated molecules.
    """
    obj = MolLibr().compute(**kwargs)
    r_groups = defaultdict(list)
    for atom_map_num, smiles_list in r.items():
        for smiles in smiles_list:
            smiles = smiles.replace("[H]","[#1]")
            smiles = smiles.replace("[*]", f"[*:{atom_map_num}]")
            r_groups[atom_map_num].append(smiles)
    combi_smiles = [ core + "." + ".".join(c) for c in product(*(r_groups[i] for i in r_groups)) ]
    with ProcessPoolExecutor(max_workers=obj.max_workers) as executor:
        if obj.progress:
            obj.libr = list(tqdm(executor.map(
                connect_core_and_chains, combi_smiles, chunksize=obj.chunksize),
                desc="Expand R-Group", 
                total=len(combi_smiles)))
        else:
            obj.libr = list(executor.map(
                connect_core_and_chains, combi_smiles, chunksize=obj.chunksize))
    return obj.rename(prefix=prefix, start=start)


most_common = [
    "[*]O",
    "[*]C",
    "[*]OC",
    "[*]OC(=O)C",
    "[*]CO",
    "[*]C(=O)O",
    "[*]C(C)C",
    "[*]C(=O)OC",
    "[*]N",
    "[*]CC",
    "[*]C(=O)",
    "[*]c1ccccc1",
    "[*]CC=C(C)(C)",
    "[*]C(=O)C",
    "[*]Cl",
    "[*]COC(=O)C",
]

# the most commmon substituents present in natural products
# Peter Ertl, Substituents of life: The most common substituent patterns present in natural products
# Bioorganic & Medicinal Chemistry, 54,116562 (2022)
# https://doi.org/10.1016/j.bmc.2021.116562
# this data are released under the Creative Commons Attribution 4.0 International License

most_common_in_NP = [
    "[*]O",
    "[*]C",
    "[*]OC",
    "[*]OC(C)=O",
    "[*]CO",
    "[*]C(O)=O",
    "[*]C(C)C",
    "[*]C(=O)OC",
    "[*]N",
    "[*]CC",
    "[*]C=O",
    "[*]c1ccccc1",
    "[*]CC=C(C)C",
    "[*]C(C)=O",
    "[*]Cl",
    "[*]COC(C)=O",
    "[*]C(C)=C",
    "[*]c1ccc(O)cc1",
    "[*]Br",
    "[*]Cc1ccccc1",
    "[*]OC(=O)C(C)=CC",
    "[*]C=C",
    "[*]C(C)(C)O",
    "[*]c1ccoc1",
    "[*]CCO",
    "[*]OC(=O)c1ccccc1",
    "[*]CC(C)C",
    "[*]CC(O)=O",
    "[*]CCCO",
    "[*]CCC",
    "[*]c1ccc(O)c(O)c1",
    "[*]CCCCC",
    "[*]CNC",
    "[*]c1ccc(OC)cc1",
    "[*]NC",
    "[*]C(C)CC",
    "[*]CC(=O)OC",
    "[*]C(C)O",
    "[*]c1ccc(O)c(OC)c1",
    "[*]COC",
    "[*]OC(=O)C(C)C",
    "[*]OS(O)(=O)=O",
    "[*]C(=O)OCC",
    "[*]C1CCCCC1",
    "[*]OC(=O)C(C)CC",
    "[*]CCC(O)=O",
    "[*]Cc1ccc(O)cc1",
    "[*]C1CCCC1",
    "[*]OCC",
    "[*]CC=C",
    "[*]CCCC",
    "[*]c1ccc(OC)c(OC)c1",
    "[*]CCc1ccccc1",
    "[*]OC1CCCC1",
    "[*]CCC=C(C)C",
    "[*]c2ccc1OCOc1c2",
    "[*]NCC(C)O",
    "[*]OC(=O)CC(C)C",
    "[*]NC(C)=O",
    "[*]C(C)(C)C=C",
    "[*]c1cccc(O)c1",
    "[*]CC=C(C)CCC=C(C)C",
    "[*]OC(=O)C=Cc1ccccc1",
    "[*]C(C)CCCC(C)C",
    "[*]OCC=C(C)C",
    "[*]C=CC",
    "[*]C(C)CCC=C(C)C",
    "[*]c1ccc(O)cc1O",
    "[*]OC(=O)CC",
    "[*]CCNC",
    "[*]N(C)C",
    "[*]OC(=O)CCC",
    "[*]CC(C)=O",
    "[*]C(N)=O",
    "[*]CCC(C)C",
    "[*]C=C(C)C",
    "[*]C1=CC(=O)OC1",
    "[*]F",
    "[*]CCC1=CC(=O)OC1",
    "[*]OO",
    "[*]C(O)CO",
    "[*]OC(=O)C=C(C)C",
    "[*]CCCN",
    "[*]N(=O)=O",
    "[*]c1ccc(OC)c(O)c1",
    "[*]c1cc(OC)c(O)c(OC)c1",
    "[*]C=Cc1ccccc1",
    "[*]CCOC",
    "[*]C#N",
    "[*]CC=O",
    "[*]n1ccnc1",
    "[*]OC(=O)C(C)=C",
    "[*]CCc1ccoc1",
    "[*]NC(=N)NC",
    "[*]c1cc(O)cc(O)c1",
    "[*]OCCO",
    "[*]SC",
    "[*]Cc1ccc(O)c(OC)c1",
    "[*]CCC(C)=O",
    "[*]C(=O)c1ccccc1",
    "[*]N=C(N)N",
    "[*]C(C)CCC(=C)C(C)C",
    "[*]C=CC(O)=O",
    "[*]CCCCCC",
    "[*]OC(=O)c1cc(O)c(O)c(O)c1",
    "[*]Cc1ccnc(N)c1",
    "[*]Cc1c[nH]c2ccccc12",
    "[*]CCC(=O)OC",
    "[*]CCN",
    "[*]C(C)C=CC(C)C(C)C",
    "[*]C(=O)C(C)C",
    "[*]OCO",
    "[*]NCC",
    "[*]C(C)C(C)C",
    "[*]COC(=O)c1ccccc1",
    "[*]C(N)N",
    "[*]CC(C)O",
    "[*]c1cc(OC)c(OC)c(OC)c1",
    "[*]OC(=O)c1cccnc1",
    "[*]CCCOC",
    "[*]C(=C)CO",
    "[*]c1c[nH]c2ccccc12",
    "[*]C(=O)CCC",
    "[*]C(C)CO",
    "[*]C(=O)CO",
    "[*]c1ccc(=O)oc1",
    "[*]C(C)CCC(CC)C(C)C",
    "[*]C(C)N(C)C",
    "[*]Cc1ccc(OC)cc1",
    "[*]CCl",
    "[*]OC=O",
    "[*]Cc1ccc[nH]1",
    "[*]c1cc[nH]c1",
    "[*]CCCCCCC",
    "[*]C=CC(O)CCCCC",
    "[*]C(=C)C(O)=O",
    "[*]C(=O)C=Cc1ccc(O)cc1",
    "[*]OCc1ccccc1",
    "[*]C(C)CCC(C)C(C)C",
    "[*]CCC(=O)OCC",
    "[*]c1ccccc1OC",
    "[*]C(=O)CC",
    "[*]c1cc(O)cc(CNC)c1",
    "[*]c1ccccc1O",
    "[*]CCc1ccc(O)cc1",
    "[*]C(=O)C=Cc1ccccc1",
    "[*]CC(O)CO",
    "[*]C(C)(C)C",
    "[*]C=Cc1ccc(O)c(O)c1",
    "[*]C(=O)C(C)CC",
    "[*]OCCCO",
    "[*]C(=O)CC(C)C",
    "[*]NC=O",
    "[*]CC(N)=O",
    "[*]OC(=O)C=Cc1ccc(O)cc1",
    "[*]n1cnc2c(N)ncnc12",
    "[*]CC(O)C(C)=C",
    "[*]C=CCO",
    "[*]C(=C)C(=O)OC",
    "[*]COC(=O)C(C)=CC",
    "[*]OC(=O)C1(C)OC1C",
    "[*]C#CC",
    "[*]CCCCCCCC",
    "[*]c1cc(O)c(O)c(O)c1",
    "[*]Cc1cccc(O)c1",
    "[*]C(C)(O)CO",
    "[*]C(C)C(O)=O",
    "[*]CCC(C)(C)O",
    "[*]CC(O)C1OC1(C)C",
    "[*]C#C",
    "[*]C(=O)CCc1ccccc1",
    "[*]OP(O)(O)=O",
    "[*]C(O)O",
    "[*]C(C)CCC=C(C)C(O)=O",
    "[*]c2ccc1[nH]ccc1c2",
    "[*]Cc1cc[nH]c1",
    "[*]OC(C)C",
    "[*]CCC(C)(O)C=C",
    "[*]CNC(=N)NC",
    "[*]OC1CCCCC1",
    "[*]CCN=C(N)N",
    "[*]C(C)=CC",
    "[*]Cc1ccc(N)nc1",
    "[*]OCC(C)=C",
    "[*]c1cccc(N)c1",
    "[*]C1CCCC1c2cccc(N)c2",
    "[*]c1ccnc(N)c1",
    "[*]S(O)(=O)=O",
    "[*]c2ccc1OCCOc1c2",
    "[*]C(C)OC(C)=O",
    "[*]Cc1cnc[nH]1",
    "[*]CCN(C)C",
    "[*]C(C)(C)OC(C)=O",
    "[*]C(C)C1CCCC1",
    "[*]C(CCO)COC",
    "[*]C(C)(O)CCC=C(C)C",
    "[*]CC(N)C(O)=O",
    "[*]n1ccc(=O)[nH]c1=O",
    "[*]OCC(O)=O",
    "[*]OCNC",
    "[*]C1CNCN1",
    "[*]CC=C(C)C=C",
    "[*]c1ccc(C#CC)s1",
    "[*]OC(=O)c1ccc(O)cc1",
    "[*]C=CC(C)=O",
    "[*]CNCC",
    "[*]C(=N)NC",
    "[*]CCC(C)=CCO",
    "[*]I",
    "[*]C(=C)C=O",
    "[*]C=CC(=O)OC",
    "[*]CCCNC(N)=N",
    "[*]c1cc(O)c(O)c(OC)c1",
    "[*]OCC(C)C",
    "[*]CCC(N)=O",
    "[*]OCC=C(C)CCC=C(C)C",
    "[*]COC(=O)CC(C)C",
    "[*]c2cc(OC)c1OCOc1c2",
    "[*]CCC(C)CC(O)=O",
    "[*]Cc1cccc(CO)c1",
    "[*]C(=O)COC(C)=O",
    "[*][N+]#[C-]",
    "[*]CCC(C)=CC(O)=O",
    "[*]CNCC(C)O",
    "[*]CC=C(C)CO",
    "[*]C=CC=O",
    "[*]OC(=O)CC(O)=O",
    "[*]c1ccc(N)nc1",
    "[*]N1CNC2C1NC(N)NC2=O",
    "[*]OC(=O)C(=C)CO",
    "[*]C(O)C(=O)OC",
    "[*]Cc2ccc1OCOc1c2",
    "[*]CCCCCCCCCCC",
    "[*]C(C)CCC(O)=O",
    "[*]Cc1ccncc1",
    "[*]CC(O)C(C)(C)O",
    "[*]c1ccc(OC)cc1OC",
    "[*]OCCc1ccccc1",
    "[*]Oc1ccccc1",
    "[*]OC(N)=O",
    "[*]C=CC(C)(C)O",
    "[*]OC(=O)c1c(C)cc(O)cc1O",
    "[*]CCC(C)CO",
    "[*]C(O)CC",
    "[*]C=Cc1ccc(O)cc1",
    "[*]CNCCO",
    "[*]C1COC(=O)C1c2cc[nH]c2",
    "[*]c1cc(OC)c(OC)cc1OC",
    "[*]C(C)CCC(O)C(C)(C)O",
    "[*]OCCCOC",
    "[*]OC(=O)c1ccc(OC)cc1",
    "[*]C(C)C=CC(CC)C(C)C",
    "[*]CCCCCCCCC",
    "[*]COC(=O)C(C)C",
    "[*]OCC(=O)OC",
    "[*]c1c[nH]cn1",
    "[*]CCC(=C)C=C",
    "[*]CCC=C(C)CCC=C(C)C",
    "[*]OC(=O)c1ccoc1",
    "[*]C(C)(O)C1CC(C)=C(C)C(=O)O1",
    "[*]C(C)CC(O)C1OC1(C)C",
    "[*]CN",
    "[*]c1cccnc1",
    "[*]CCCC(C)C",
    "[*]c1ccc(Cl)cc1",
    "[*]NCC(C)=O",
    "[*]C(C)C(=O)OC",
    "[*]OCN",
    "[*]C1OC1C(C)C(C)C",
    "[*]CC=CCC",
    "[*]OC(=O)Cc1ccccc1",
    "[*]CCC1=CCOC1=O",
    "[*]C(C)CC(=O)C=C(C)C",
    "[*]c1ccc(O)cc1OC",
    "[*]C(C)C=CCC(C)C",
    "[*]OCCCC",
    "[*]OCc1cccc(O)c1",
    "[*]CC(=O)OCC",
    "[*]c1ccco1",
    "[*]CC1CCCCC1",
    "[*]c2ccc1=N[CH+]C=c1c2",
    "[*]CN=C(N)NC(C)O",
    "[*]C1=CC(=O)OC1O",
    "[*]c1cccc(OC)c1",
    "[*]OC(=O)C(CO)=CC",
    "[*]c1ccc[nH]1",
    "[*]C(C)C1CC(C)=C(C)C(=O)O1",
    "[*]CCc1ccc(N)nc1",
    "[*]C(C)C1CC=C(C)C(=O)O1",
    "[*]c2coc1cc(O)ccc1c2=O",
    "[*]COC(=O)C(C)CC",
    "[*]C(=O)c1ccc(O)c(O)c1",
    "[*]C1OC1(C)C",
    "[*]OC(=O)CCCCC",
    "[*]CCOC(C)=O",
    "[*]CC=CCCCC(O)=O",
    "[*]CN=C(N)N",
    "[*]CCC(C)O",
    "[*]C=CC(O)C(O)CCCC",
    "[*]c1ccc(OC)cc1O",
    "[*]C(CCO)COCO",
    "[*]OCCc1cccnc1",
    "[*]OCNCC",
    "[*]C(C)CCC(=O)OC",
    "[*]COCC",
    "[*]C(OC(C)=O)C(=O)OC",
    "[*]OC1CC(CO)C(O)C(O)C1O",
    "[*]c1ccc(OC(C)=O)cc1",
    "[*]C1=CC(O)OC1=O",
    "[*]c1ccc(C)cc1",
    "[*]OC(=O)CCCCCCC",
    "[*]C=CC(C)C",
    "[*]OCCNC",
    "[*]C(C)C1CC(C)=C(CO)C(=O)O1",
    "[*]CCC(=O)CC(O)CCCCC",
    "[*]Cc1ccccc1O",
    "[*]C=CC(C)O",
    "[*]CCCc1ccccc1",
    "[*]CC(CC=C(C)C)C(C)=C",
    "[*]C1CC(C)C(=O)O1",
    "[*]Cc2ccc1[nH]ccc1c2",
    "[*]Cc1ccc(O)c(O)c1",
    "[*]OCC(O)CC=O",
    "[*]Cc1ccc(CCOC)cc1",
    "[*]C1(C)CO1",
    "[*]c1cc(C)cc(C)c1",
    "[*]CCC=O",
    "[*]CCC1(O)CCCC(O)C1",
    "[*]C(C)CCC(=CC)C(C)C",
    "[*]CCC(C)CCO",
    "[*]Cc1ccc(OC)c(OC)c1",
    "[*]COC(=O)CC(O)=O",
    "[*]C(=O)C=C(C)C",
    "[*]c2cc1ccc(O)cc1o2",
    "[*]NC(=O)c1ccccc1",
    "[*]CC1OC1(C)C",
    "[*]Oc1cc(N)cc(Cl)c1",
    "[*]OC(=O)C=C(C)CC",
    "[*]c1cc(O)c(O)c(CCO)c1",
    "[*]C(=O)CCO",
    "[*]c1cc(O)c(OC)c(OC)c1",
    "[*]OC(=O)c1ccc(O)c(OC)c1",
    "[*]C(CCC(N)O)c1c[nH]cn1",
    "[*]CC=C(C)C(O)=O",
    "[*]C(=O)c1ccc(O)cc1O",
    "[*]CCCCCCC(O)=O",
    "[*]C(CCC(=C)C(C)C)C(O)=O",
    "[*]c2cc1ccccc1o2",
    "[*]N=C=S",
    "[*]Cc1cc(OC)c(O)c(OC)c1",
    "[*]OCCC",
    "[*]CBr",
    "[*]CCC(C)=CC(=O)OC",
    "[*]C(O)c1ccccc1",
    "[*]c1ccc(C(O)=O)cc1",
    "[*]n2cc1cc[nH]c1c2",
    "[*]C(C)CC=CC(C)(C)O",
    "[*]CCCCO",
    "[*]C(C)(O)C1OC1C(C)C(C)C",
    "[*]C=CCC",
    "[*]C(O)C(C)O",
    "[*]OC(=O)C(O)C(C)CC",
    "[*]c1ccc(OC)c(OC)c1OC",
    "[*]Cn1c(=O)ccc1=O",
    "[*]c1ccc(N)nc1CCCO",
    "[*]OCCC(C)C",
    "[*]C2=Cc1ccccc1CC2",
    "[*]NCNC",
    "[*]OC(=O)CCC(O)=O",
    "[*]OC(=O)C(C)=CCO",
    "[*]c1cc(OC)ccc1OC",
    "[*]c1cc(O)c(OC)c(O)c1",
    "[*]CCCCCCCCCC",
    "[*]C(C)=CCC=C(C)C",
    "[*]C(O)CNC",
    "[*]Cc1cccc(C)c1",
    "[*]OC1CCNCC1",
    "[*]Cn2cc1cc[nH]c1c2",
    "[*]CCc1cccc(O)c1",
    "[*]CCCC(O)=O",
    "[*]CCC(C)CC(=O)OC",
    "[*]C1CO1",
    "[*]CC(O)CC",
    "[*]c1ccc(O)c(CC=C(C)C)c1",
    "[*]C(=C)CCC=C(C)C",
    "[*]C(C)C(O)CC",
    "[*]Oc1c(O)cc(O)cc1O",
    "[*]C(C)C1CC1(C)C(C)C(C)C",
    "[*]c1cc(O)ccc1O",
    "[*]C(C)COC(C)=O",
    "[*]C(F)(F)F",
    "[*]OCCN",
    "[*]C(C)CCC=C(C)CO",
    "[*]CCC(C)=CCOC(C)=O",
    "[*]C(CCCN)c1c[nH]cn1",
    "[*]C(C)CCCC(C)CO",
    "[*]CCC(CO)=CCO",
    "[*]C=Cc1cc(O)cc(O)c1",
    "[*]OC1CCC(=O)C(C)O1",
    "[*]Cc2cc1ccccc1[nH]2",
    "[*]OC(=O)C(C)(C)C",
    "[*]c1cccc(CNC)c1",
    "[*]CC(O)c1ccoc1",
    "[*]c1cc(OC)c(O)cc1CO",
    "[*]C(C)NC",
    "[*]C(C)CC(=O)CC(C)C",
    "[*]Cc1cc(C)cc(C)c1",
    "[*]CNCCNC",
    "[*]OCNC(=N)NC",
    "[*]C1CNC(=O)C1",
    "[*]CC(=O)c1ccoc1",
    "[*]CCC(=O)C=CCCC",
    "[*]OCC=C",
    "[*]Oc1ccc(CC=C)cc1",
    "[*]C(C)CC(=O)CC(C)C(O)=O",
    "[*]C(C)OC",
    "[*]C(N)C=O",
    "[*]C(=O)c1ccc(O)cc1",
    "[*]OCCc2ccc1OCCc1c2",
    "[*][n+]1cnc2c(=O)nc(N)nc12",
    "[*]C=CC(C)=C",
    "[*]OCCc1ccc(OC)cc1",
    "[*]C=C(C)C(O)=O",
    "[*]CCCCCCCCCCCC",
    "[*]C(=C)Cc2cc1ccccc1[nH]2",
    "[*]CC=CC#C",
    "[*]OCC(O)C(C)(C)O",
    "[*]c1c[nH]c2cc(Br)ccc12",
    "[*]OC(=O)C=CC",
    "[*]C=C=CBr",
    "[*]C1(O)CCCC1",
    "[*]Oc1ccc(OC)cc1",
    "[*]C=CC=CC",
    "[*]C1COC(=O)C1",
    "[*]C1OC(=O)C(C)=C1",
    "[*]C(CCC=C(C)C)C(O)=O",
    "[*]CCCCN",
    "[*]C2(c1ccccc1)CCCCC2",
    "[*]CC=CCCCCC",
    "[*]CCc1ccc(OC)cc1",
    "[*]OC(=O)CCCCCCCCC",
    "[*]n2cc1[nH]ccc1c2CC",
    "[*]C=CC(C)=CC(O)=O",
    "[*]Cc1ccc(OC)c(O)c1",
    "[*]OC(=O)C1CCCCC1",
    "[*]C2COc1cc(O)ccc1C2",
    "[*]C(=C)COC(C)=O",
    "[*]OC(=O)C(COC(C)=O)=CC",
    "[*]C(C)CCC(O)C(C)C",
    "[*]C(O)C(O)=O",
    "[*]CCCN=C(N)N",
    "[*]C=CC(C)=CC(C)CC",
    "[*]NO",
    "[*]Oc1cc(C)cc(O)c1",
    "[*]C(OC(C)=O)c1ccoc1",
    "[*]C(=O)CCCCC",
    "[*]C1(CO)CCCC1",
    "[*]CC1=CN=C[CH-]1",
    "[*]C(=O)CCc1ccc(O)cc1",
    "[*]C(C)C(C)O",
    "[*]COC(=O)CCC(O)=O",
    "[*]OCNCC(C)O",
    "[*]C(C)CCC(CC)C(C)=C",
    "[*]CNC1CCCCC1",
    "[*]C(=O)C=Cc1ccc(O)c(O)c1",
    "[*]OC(=O)c1ccc[nH]1",
    "[*]OC(=O)C(CCO)=CC",
    "[*]C(=O)OC(C)(C)C",
    "[*]C(C)CC(O)C=C(C)C(O)=O",
    "[*]c1cc(O)cc(OC)c1",
    "[*]c2cc1OCOc1cc2OC",
    "[*]COC(=O)C=C(C)C",
    "[*]C(O)C(O)CO",
    "[*]c2cc1cccc(OC)c1o2",
    "[*]S",
    "[*]c1cccc(CCCCC)c1",
    "[*]C(O)(O)CC=O",
    "[*]C1(C)CCCC1(C)C",
    "[*]OC(=O)C(O)C(C)C",
    "[*]OC(C)C(=O)OC",
    "[*]C1(C)CCCC(C)(C)C1",
    "[*]C(=O)C=CC",
    "[*]C#CC(C)=C",
    "[*]C(=O)C=Cc1ccc(OC)cc1",
    "[*]C=CC=CC(O)=O",
    "[*]Oc1cccc(O)c1",
    "[*]OCC(=O)OC(C)C",
    "[*]C(C)(C)OC",
    "[*]CCc1ccnc(N)c1",
    "[*]CCc1ccc(O)c(OC)c1",
    "[*]OC(CO)C(O)c1ccc(N)[nH]1",
    "[*]CC(C)(C)O",
    "[*]C(C)c1ccc(C=O)cc1",
    "[*]c2cc1ccccc1oc2=O",
    "[*]C(OC(=O)CC(O)=O)C(N)N",
    "[*]c1ccc(OC(C)=O)c(OC)c1",
    "[*]OCC=Cc1ccccc1",
    "[*]c2ccc1OCCCOc1c2",
    "[*]C(=O)OCCCC",
    "[*]C(C)(O)CC=CC(C)(C)O",
    "[*]N=C",
    "[*]c2coc1ccccc1c2=O",
    "[*]COC(=O)C=Cc1ccccc1",
    "[*]C=CC(=O)c1ccc(O)cc1O",
    "[*]C(O)c1ccc(O)cc1",
    "[*]OCC(N)=O",
    "[*]C(O)CCCO",
    "[*]OC(CO)CO",
    "[*]CN(C)C",
    "[*]c2cc1OCOc1cc2O",
    "[*]C=CC=C(C)C",
    "[*]c1ccc(Br)cc1",
    "[*]CCC1=CC(=O)OC1O",
    "[*]C(=O)C=CC=CC",
    "[*]Cc1ccoc1",
    "[*]CC(O)OC",
    "[*]c1ccccn1",
    "[*][N+]1=CC=N[CH+]1",
    "[*]C(C)C(O)CCC(C)C",
    "[*]c1cc(CC=C)ccc1O",
    "[*]C(C)(C)C(O)=O",
    "[*]C1(C)CCC(=O)O1",
    "[*]CC=CC(C)(C)O",
    "[*]CCCCCC(O)=O",
    "[*]CNCCCOC",
    "[*]C1COC(C)(C)O1",
    "[*]c1ccc(C(=O)OC)cc1",
    "[*]c2ccc1ccccc1c2",
    "[*]OCC(C)=O",
    "[*]CCCCc1cccc(O)c1",
    "[*]CC#N",
    "[*]OC(C)C(O)=O",
    "[*]C2CC(=O)c1ccc(O)cc1O2",
    "[*]CC=Cc1ccccc1",
    "[*]CCC1(O)CCCCC1",
    "[*]Cn2cc1cc[nH]c1c2CO",
    "[*]C=C(C)CCC=C(C)C",
    "[*]C(O)=N",
    "[*]NCO",
    "[*]CCC(=O)C=CCCCCC",
    "[*]OC(=O)c1ccco1",
    "[*]C=CCOC(C)=O",
    "[*]C(OC)OC",
    "[*]c1c(O)cc(O)cc1O",
    "[*]OC(=O)CC(C)O",
    "[*]C(O)C(C)C",
    "[*]C(C)CCC=C(C)C=O",
    "[*]C=CC(=O)c1ccccc1",
    "[*]n1ccc(N)nc1=O",
    "[*]OCC(=O)OCC",
    "[*]C(C)CC(O)C(O)C(C)(C)O",
    "[*]CCCCCCCN",
    "[*]OC(=O)C(CO)=CCO",
    "[*]OC(=O)CC(C)CC",
    "[*]C(C)(O)CCCC(C)C",
    "[*]CCCC#C",
    "[*]C(=COC)C(=O)OC",
    "[*]OC(=O)CCCC",
    "[*]C=CC1=CC(=O)OC1",
    "[*]C(C)CNCC(C)O",
    "[*]C(C)C=CC(C)C",
    "[*]Cc1ccc(C(O)=O)cc1",
    "[*]CCSC",
    "[*]c1cccc(O)c1O",
    "[*]OC(=O)CO",
    "[*]C1CCNC(NC)C1",
    "[*]CC(C)=C",
    "[*]C1=CCOC1=O",
    "[*]OC(=O)c2ccc1OCOc1c2",
    "[*]C(O)C(C)(C)O",
    "[*]OC(C)C(=O)OCC",
    "[*]C(=O)OCc1ccccc1",
    "[*]C(C)CCCC(C)(C)O",
    "[*]OCc1ccc(C=C)cc1",
    "[*]CCC(=O)C=C(CO)CCC",
    "[*]CCC(=O)C(C)C",
    "[*]Cc1cccc(C(N)N)c1",
    "[*]C(C)(O)C(O)CCC(C)(C)O",
    "[*]c1ccc(OC)c(OC)c1O",
    "[*]C2CCC1(CCCC1)C2",
    "[*]NCCCc1ccnc(N)c1",
    "[*]C(=O)CCC(O)=O",
    "[*]C=COC(=O)C=C(C)C",
    "[*]CC(=O)CCCCC",
    "[*]c1cccc(OC)c1OC",
    "[*]C(C)CC(O)C=C(C)C",
    "[*]C1CCC(O)CC1",
    "[*]OCCc1cccc(CC)c1",
    "[*]COC(N)=O",
    "[*]C1CC(C)=C(C)C(=O)O1",
    "[*]Cc1cccc2ccccc12",
    "[*]OC(=O)CCc1ccccc1",
    "[*]OC(=O)C=CC=CCCCCC",
    "[*]CCC1(O)CCCC1",
    "[*]CN1CCCCC1",
    "[*]C(C)CCC(O)C(C)=C",
    "[*]Oc1ccccc1OC",
    "[*]CC(C)CC",
    "[*]c1c(O)cccc1O",
    "[*]C(C)CCC(C)C(C)=C",
    "[*]c1ccccc1C(O)=O",
    "[*]Cc2ccc1=N[CH+]C=c1c2",
    "[*]C(=O)NC",
    "[*]C1CCC(O)C(O)C1",
    "[*]CNCCC",
    "[*]C2COc1cc(O)ccc1C2=O",
    "[*]c1cc(OC)cc(OC)c1",
    "[*]OC(=O)CC(C)(C)O",
    "[*]c1ccc(O)c(CCCCC)c1",
    "[*]C(C)C=CC=C(C)C",
    "[*]NC(=O)Cc1ccccc1",
    "[*]CCCCC(O)=O",
    "[*]OC1OC(C)C(=O)C=C1",
    "[*]C=CCCC",
    "[*]C(=O)CCCCCCC",
    "[*]C(C)C2CC1(C)OC1(C)C(O)O2",
    "[*]CCCC(C)(C)O",
    "[*]n1c(=O)ccc1=O",
    "[*]COS(O)(=O)=O",
    "[*]C(=O)C(C)=CC",
    "[*]Oc1cc(O)cc(O)c1",
    "[*]CCCCCCCC(O)=O",
    "[*]CNC1CCCC1",
    "[*]Oc1ccc(C(=O)OC)cc1",
    "[*]C=Cc1ccc(O)c(OC)c1",
    "[*]OCC1OC1(C)C",
    "[*]C1OCc2cnc(C)c(O)c12",
    "[*]C(=O)CCc1ccc(OC)cc1",
    "[*]CC(C(O)=O)[N+](C)(C)C",
    "[*]C=NO",
    "[*]COC(=O)c1cccnc1",
    "[*]C(Br)CC",
    "[*]C1CCC(=O)O1",
    "[*]OCCCN(C)C",
    "[*]C(CNC)NCC(C)O",
    "[*]c1ccc(OCC=C(C)C)cc1",
    "[*]C(=O)N(C)C",
    "[*]OCNC1CCCCC1",
    "[*]CNc1ccc[nH]1",
    "[*]C(C)C1CCCCC1",
    "[*]COC(=O)CC",
    "[*]C2CC1C=COC1O2",
    "[*]c2cc(OC)c1OCCOc1c2",
    "[*]NC(=N)NCCCO",
    "[*]CNC(=O)c1cc(Br)c(Br)[nH]1",
    "[*]C(C(C)O)C(O)=O",
    "[*]C(C)(O)CCC(O)C(C)=C",
    "[*]N=O",
    "[*]CCC(O)C(C)=C",
    "[*]C(C)=Cc1csc(C)n1",
    "[*]C=CCCCCC",
    "[*][n+]2cc1ccnc1c2",
    "[*]C(C)(O)Cc1ccc(O)cc1",
    "[*]OC(=O)C(C)=CCOC(C)=O",
    "[*]OCC(O)C(C)=C",
    "[*]CC1CNCN1",
    "[*]OCCCCCCCC",
    "[*]OC(=O)C(C)C(C)O",
    "[*]C(=O)CCC=CC",
    "[*]NC(=N)NC1CCCCC1",
    "[*]C(C)CCC(=O)C(C)C",
    "[*]CCCC(C)O",
    "[*]NC(=O)C=C(C)C",
    "[*]C1(C)CC(C)CC=C1",
    "[*]C(O)C(C)C(C)C",
    "[*]COC(=O)CCC(=O)OC",
    "[*]COC(=O)c1ccc(O)cc1",
    "[*]CC=CCC=CCC",
    "[*]Oc1ccc(O)cc1",
    "[*]C(=O)OCCCCNC(N)=N",
    "[*]C1(CC)CCCC1",
    "[*]C(O)CCCCC",
    "[*]Oc1cccc(OC)c1",
    "[*]CC=CC",
    "[*]COC(=O)CC(C)(O)CC(O)=O",
    "[*]C(=O)CCCC",
    "[*]NC(CO)C(C)=O",
    "[*]OCCc1ccc(N)[nH]1",
    "[*]CC1=NC=N[CH+]1",
    "[*]C=CC(C)CC",
    "[*]C(=O)C(C)O",
    "[*]Oc1cc(O)c(O)c(O)c1",
    "[*]c1oc(OC)c(C)c(=O)c1C",
    "[*]CCc1ccc(CCCC)o1",
    "[*]C(C)N",
    "[*]C2=Cc1ccccc1OC2",
    "[*]CC1=C[CH+]N=C1",
    "[*]N=C(C)O",
    "[*]CCc1cccc(OC)c1",
    "[*]C(=O)C(O)=O",
    "[*]S(C)=O",
    "[*]C(=O)N1CCCCC1",
    "[*]C(C)Cc1ccccc1",
    "[*]C(N)C(O)=O",
    "[*]N1C(=O)CC(C)C1=O",
    "[*]OCC=C(C)CO",
    "[*]C(C)C=C(C)C(C)C(C)C",
    "[*]CC=CCCCC(=O)OC",
    "[*]C(C)(O)C1CCNCC1",
    "[*]CCCCc1ccccc1",
    "[*]CCC(C)CCOC(C)=O",
    "[*]C(O)CN",
    "[*]COP(O)(O)=O",
    "[*]C(C)CC(=O)CC(C)C(=O)OC",
    "[*]CC(=O)C=C(C)C",
    "[*]Cc1c(O)c(C)c(CC)oc1=O",
    "[*]OCC(=O)OC(C)(C)C",
    "[*]c1ccc(F)cc1",
    "[*]c1cccc(C)c1",
    "[*]C(=O)NC(CCC(O)=O)C(O)=O",
    "[*]C2C1CC3CC(C1)CC2C3",
    "[*]c1ccc(NO)cc1",
    "[*]CC(O)CCC",
    "[*]OCCCC(C)C",
    "[*]C=CC(=O)c1ccc(O)cc1",
    "[*]CC=C(C)C=O",
    "[*]C1OC1C",
    "[*]Cc1ccc(O)cc1O",
    "[*]C(O)C1CC(C)CC=C1",
    "[*]CC1=N[CH+]N=C1",
    "[*]C(O)CCC",
    "[*]C(C)(O)CCCCCC",
    "[*]CCS(C)=O",
    "[*]C(=O)OCCC",
    "[*]C1=C[CH+]N=C1",
    "[*]C(C)C=O",
    "[*]C(C)(O)C(=O)C=CC(C)(C)O",
    "[*]C1CCC(O)C(C)N1",
    "[*]CC(O)c1ccccc1",
    "[*]OC(O)=N",
    "[*]C1(O)CCCCC1",
    "[*]CCC1CCCCC1",
    "[*]C(C)CCC(=C)C(C)C(O)=O",
    "[*]CCCCCCCC(=O)OC",
    "[*]CCC(=O)c1ccc(O)cc1O",
    "[*]CCC(=C)CO",
    "[*]C(C)(C)OC(=O)C(C)=CC",
    "[*]c1cc(O)c(O)cc1O",
    "[*]C(CO)C(=O)OC",
    "[*]C(C)(O)CCCCO",
    "[*]CCC1CCNC(N)C1",
    "[*]C1OC(=O)CC1C",
    "[*]c2cc(=O)c1ccccc1o2",
    "[*]CCOCO",
    "[*]C(C)(C)OO",
    "[*]OS(C)(=O)=O",
    "[*]C=Cc1ccc(O)cc1O",
    "[*]CCCC=C",
    "[*]CCC1=CC(O)OC1=O",
    "[*]CC(=O)NC(C(C)CC)C(O)=O",
    "[*]OS(=O)(=O)c1ccc(C)cc1",
    "[*]C(CCC(O)=O)C(O)=O",
    "[*]NCOC",
    "[*]OCCc1ccnc(N)c1",
    "[*]CCC1CC(O)CC(=O)O1",
    "[*]OC(=O)Cc1ccc(O)cc1",
    "[*]OC(=O)C1(C)CO1",
    "[*]C(CCCO)NCC(C)O",
    "[*]CCC(C)=C",
    "[*]NCC(O)=O",
    "[*]CCC(N)C(O)=O",
    "[*]CCCOC(C)=O",
    "[*]C(C)(O)C1CCCC1",
    "[*]NCC(O)C1CCCCC1",
    "[*]NCC1CCCCC1",
    "[*]C(C)(O)C(O)CCC(C)C",
    "[*]CCC1CNC(=O)C1",
    "[*]n1ccc(O)nc1=O",
    "[*]OC(C)=C",
    "[*]Cc1cc(O)c(O)c(Br)c1Br",
    "[*]OCNC1CCCC1",
    "[*]C(CO)CO",
    "[*]N(C)C=O",
    "[*]SCC(NC(C)=O)C(O)=O",
    "[*]c1cc(OC)cc(=O)o1",
    "[*]OCNC(=N)NCCCCO",
    "[*]NCCS(O)(=O)=O",
    "[*]C1(C)OC1CC=C(C)C",
    "[*]C(=O)c1ccoc1",
    "[*]CCOCC",
    "[*]CNCNCC(C)(O)CC(C)C",
    "[*]COC(=O)CCC",
    "[*]C(=O)C=Cc1ccc(O)cc1O",
    "[*]CCc1c[nH]cc1CC",
    "[*]C(C)=NO",
    "[*]C(O)C(N)=O",
    "[*]c1cc(N)cc(CC(C)C)c1",
    "[*]C(=O)NC(CC(C)C)C(N)=O",
    "[*]CCc1ccc(O)c(O)c1",
    "[*]CCc1cc(O)ccc1C",
    "[*]CC2NC(=O)C1CCCN1C2=O",
    "[*]C(=O)c1c[nH]c2ccccc12",
    "[*]NCC(C)(O)C(O)COC",
    "[*]CCCCNC(N)=N",
    "[*]CC(CCC(C)=C)C(C)=C",
    "[*]C(C)CCCO",
    "[*]C(C)CCCC",
    "[*]Cc1c[nH]cn1",
    "[*]OC(=O)c1cccc(OC)c1",
    "[*]NC(=O)C(C)C",
    "[*]c2cc1ccccc1[nH]2",
    "[*]CCC1=C[CH-]C(CCCC)=[O+]1",
    "[*]C(=O)c1cccc(O)c1",
    "[*]c1ccc(O)c(Cl)c1",
    "[*]C[n+]2cc1ccnc1c2",
    "[*]C(=O)C(=O)OC",
    "[*]CCCl",
    "[*]c1ccncc1",
    "[*]C(C)(O)CCC(=C)C(C)C",
    "[*]C=CC=C",
    "[*]OC(=O)C(=C)C(O)CO",
    "[*]c1cc(OC)c(O)cc1O",
    "[*]c1cc(OC)c(O)c(OC)c1O",
    "[*]C2CC(C1OC1(C)C)OC2O",
    "[*]NC1CCCCC1",
    "[*]COC(=O)C(O)(C(C)C)C(C)O",
    "[*]C(=O)OC(C)C",
    "[*]C=CC=CCC",
    "[*]C(C)C(O)CC=C(C)C",
    "[*]CCNC(C)=O",
    "[*]CC1=C[CH-]C=N1",
    "[*]c1ccc(C(C)C)cc1",
    "[*]C(C)C(O)C(C)CC",
    "[*]OC(=O)C=CSC",
    "[*]CCC=C(C)CO",
    "[*]C2(c1ccccc1)CCCC2",
    "[*]c2ccc1OC(C)(C)C=Cc1c2",
    "[*]Oc1ccc(Br)cc1Br",
    "[*]CC(=O)C(C)C",
    "[*]NC(=O)C(C)=CC",
    "[*]C=CCOC",
    "[*]C(C)(O)CCC=C(C)CO",
    "[*]C(C)CCC",
    "[*]Cc1ccc(O)c(Cl)c1",
    "[*]C(C)CC=CC(C)(C)OC",
    "[*]Cc1cc(O)cc(=O)o1",
    "[*]C1(C)OC(=O)C=C1",
    "[*]COC(=O)SC",
    "[*]NC(=O)CC(C)(C)C",
    "[*]COC=O",
    "[*]OCOC",
    "[*]C2(c1ccnc(N)c1)CCCC2",
    "[*]OC(=O)C(CO)c1ccccc1",
    "[*]CCC=C",
    "[*]Oc1ccccc1OCC",
    "[*]Cn1ccnc1",
    "[*]c1ccc(OC)c(O)c1OC",
    "[*]C1CCCCN1",
    "[*]C(C)CCCCC",
    "[*]OCc1ccc(CNC)cc1",
    "[*]C#Cc1ccc(O)cc1",
    "[*]C(N)=N",
    "[*]C(C)CCC(O)C(C)O",
    "[*]OC(=O)C(C)(O)C(C)O",
    "[*]c1cc(O)cc(CO)c1",
    "[*]C=CC(=O)OCC",
    "[*]C(=O)CCCCCCCCC",
    "[*]C=CC(=O)c1ccccc1O",
    "[*]CNC(C)=O",
    "[*]C(O)CC=C(C)C",
    "[*]C=CC=CC=CC",
    "[*]C=C(C)CO",
    "[*]C2CC1CCOC1O2",
    "[*]C(=O)NC(C)C(O)=O",
    "[*]N1CCCCC1",
    "[*]NC(N)=N",
    "[*]c1c[nH]c2ccc(O)cc12",
    "[*]C(CNCC(C)O)C1CCCC1",
    "[*]CC(O)C(O)=O",
    "[*]C(C)C(C)=O",
    "[*]C=CC(=O)CC(=O)CCCCC",
    "[*]OCC(=O)OCC=C",
    "[*]CNC(=O)c1cc(Br)c[nH]1",
    "[*]NCC(C)(O)CCOC",
    "[*]c1ccc(OC(C)C)cc1",
    "[*]C(O)CC(O)=O",
    "[*]Oc2ccc1ccc(=O)oc1c2",
    "[*]CCCCCCCCC(C)C",
    "[*]N=CO",
    "[*]OC(=O)C(C)(O)C(C)OC(C)=O",
    "[*]c1cc(O)cc(=O)o1",
    "[*]NC(=O)C(C)O",
    "[*]CCCC(C)=C",
    "[*]OC(=O)CC(C)(O)CC(O)=O",
    "[*]C(=O)c1ccc(OC)cc1",
    "[*]C(C)CC(C)=O",
    "[*]N1CCNCC1",
    "[*]Cc1cc(O)ccc1O",
    "[*]COC1CCCCC1",
    "[*]C2CNC1CC(=O)CCC1C2",
    "[*]Cc2cc(OC)c1OCOc1c2",
    "[*]C(O)CCCCCC",
    "[*]CCNC(=N)NC",
    "[*]OCCCN",
    "[*]C(CC)OC(C)=O",
    "[*]CC(O)C(C)O",
    "[*]CC(=O)c1ccccc1",
    "[*]C(C)C(O)CC(CC)C(C)C",
    "[*]CC(O)CCc1cccc(O)c1",
    "[*]C(C)OC(=O)C(C)=CC",
    "[*]CCC(O)C(C)(C)O",
    "[*]C(O)C=C(C)C",
    "[*]C=CC=CC=CC(O)=O",
    "[*]CC(O)OCCCO",
    "[*]C(Cl)Cl",
    "[*]CCC(C)=CC=O",
    "[*]CC(=O)CC",
    "[*]OC(C)C(C)=O",
    "[*]c1ccc(C)cc1O",
    "[*]CCCC(C)=O",
    "[*]C(C(O)CC(N)O)c1c[nH]cn1",
    "[*]Cc1cccc(CCCCC)c1",
    "[*]NCCO",
    "[*]NCNCC(C)(O)CC(C)C",
    "[*]Nc2ccc1cc(O)ccc1c2",
    "[*]C(CC(O)=O)c1ccccc1",
    "[*]C(C)C1CC1C(C)C(C)C",
    "[*]C(=O)NC(CC(C)C)C(O)=O",
    "[*]c1cc(O)c(OC)cc1O",
    "[*]C(=O)NCC(O)=O",
    "[*]C=CC=C(C)C(O)=O",
    "[*]C(=O)CCC(C)C",
    "[*]NC(=O)COC",
    "[*]C(C)(O)CC",
    "[*]C(C)CCC(CCO)C(C)C",
    "[*]c1nccs1",
    "[*]CC(=O)N1CCOCC1",
    "[*]n1cc(C)c(=O)[nH]c1=O",
    "[*]C(=O)NC(C(C)O)C(O)=O",
    "[*]C(=O)NC(C(C)CC)C(O)=O",
    "[*]C#CC#CC",
    "[*]NC(=O)c1ccc(OC)cc1",
    "[*]CCNC(=O)c1ccccc1",
    "[*]C(C)C=CC(=C)C(C)C",
    "[*]N=C(O)CCCC(N)C(O)=O",
    "[*]NC(=O)NC(C(C)CC)C(O)=O",
    "[*]CF",
    "[*]C=CCl",
    "[*]CCc1ccc(OC)c(O)c1",
    "[*]OC(=O)CCCCCC",
    "[*]C#Cc1ccccc1",
    "[*]OC(=O)CC(=O)OC",
    "[*]C(O)c1ccoc1",
    "[*]CCCCCCC1CCC=C1",
    "[*]C(C)C=c1cc(C)c(=O)o1",
    "[*]C=C1NC(=O)C(C)NC1=O",
    "[*]C(C)=CC(=O)CC(C)C(O)=O",
    "[*]C=Cc1ccc(OC)cc1",
    "[*]CCC1=CC(OC)OC1=O",
    "[*]CCc1ccccc1NCC",
    "[*]OC(=O)C(C)C(C)C",
    "[*]C(=N)NCCNC",
    "[*]C(=O)OC1CC2CCC(C1)N2C",
    "[*]c1ccc(OC)c(O)c1O",
    "[*]C=CC=CCCC",
    "[*]c2cc1ccccc1[nH]c2=O",
    "[*]N1CCCC1",
    "[*]CC(=O)NCCOC",
    "[*]N(C)C(C)=O",
    "[*]C(C)CCC1OC1(C)C",
    "[*]C(C)(C)C(=O)OC",
    "[*]c1cc(Br)c(O)c(Br)c1",
    "[*]c1cccc(CO)c1",
    "[*]CC(=O)C(O)=O",
    "[*]c1cc(OC)c(O)cc1OC",
    "[*]CCCC(=O)OC",
    "[*]CCc1ccccc1O",
    "[*]CC(=O)c1ccc(O)cc1O",
    "[*]c1cccs1",
    "[*]Cc1cnco1",
    "[*]OCC(O)c1ccc(N)[nH]1",
    "[*]CC(O)C(O)C(O)CO",
    "[*]c1cccc(C(N)N)c1",
    "[*]CC(S)=NOS(O)(=O)=O",
    "[*]CCCCCCC1CNCN1",
    "[*]Cc1ccc(C(C)C)cc1",
    "[*]C(O)CC(C)C",
    "[*]C(O)CCO",
    "[*]OCCCCC",
    "[*]c2cc1ccc(O)cc1oc2=O",
    "[*]c2ccc(c1ccccc1)cc2",
    "[*]C(=O)OCCCCCC",
    "[*]CC(O)C1=CC(=O)OC1",
    "[*]Cc1c(O)c(C)c(C)oc1=O",
    "[*]C(=O)c1ccc(O)c(OC)c1",
    "[*]C(O)C(O)C(O)CO",
    "[*]C=CC=CCCCCC",
    "[*]NCC(C)C",
    "[*]C(Cc1ccccc1CO)NC",
    "[*]c1cc(OC)c(OC)cc1O",
    "[*]C(C)C1CCC(C)CN1",
    "[*]C(=O)NC(C)C(N)=O",
    "[*]NC(=O)OC",
    "[*]CC(=O)NC(CC(C)C)C(O)=O",
    "[*]C(=O)OCC(O)CO",
    "[*]CCC1C(=C)CCCC1(C)C",
    "[*]CCCCCCCO",
    "[*]C(C)CCCC(C)C(O)=O",
    "[*]OC(=O)C(CCNC)=CC",
    "[*]C(=O)Oc1ccccc1",
    "[*]OC(=O)C(CNCC)=CC",
    "[*]C(=O)Nc1ccc(OC)cc1",
    "[*]CS(O)(=O)=O",
    "[*]CCC=C(C)C(O)=O",
    "[*]C1COC2(C)CC1OC(=O)C2=C",
    "[*]OC(=O)C=CC=CC=CC",
    "[*]C2=c1ccccc1=N[CH+]2",
    "[*]C(=O)NC(C(C)C)C(O)=O",
    "[*]Nc1nc(=O)n(C)c1=O",
    "[*]C(=O)CC(C)O",
    "[*]C1(C)CCCC1",
    "[*]C(=O)Nc1ccccc1",
    "[*]c1ccc(O)c(Br)c1",
    "[*]c1ccc(N(C)C)cc1",
    "[*]C=CC(=O)c1ccc(OC)cc1",
    "[*]c1cc(O)cc(CC)c1",
    "[*]C(O)CCC(C)C",
    "[*]OC(=O)c1ccccc1O",
    "[*]Oc1ccc(C(=O)OCC)cc1",
    "[*]C(C)=CC(O)=O",
    "[*]Cc1ccc(OC)c(Cl)c1",
    "[*]OC1(O)CCCCC1",
    "[*]CC(O)=N",
    "[*]Cc1[nH]cnc1CC",
    "[*]CCC(=C)C(O)CO",
    "[*]C(=O)NCC(N)=O",
    "[*]OC(=O)C=C(C)C(C)C",
    "[*]OC(=O)C(C)CO",
    "[*]c1cccc(Cl)c1",
    "[*]OC(=O)C=C(C)CO",
    "[*]OC(=O)c1ccccc1OC",
    "[*]OC(=O)c1ccccc1NC",
    "[*]OC(=O)c1ccc(O)c(O)c1",
    "[*]C=CCCC(O)=O",
    "[*]CCC(C)CNC(C)=O",
    "[*]C(C)CCCCCC",
    "[*]c1ccc(O)c(OCCO)c1",
    "[*]CCC=CC",
    "[*]CC=C(C)CCC(O)C(C)=C",
    "[*]CCCc1ccc(O)cc1",
    "[*]CCCC=CCCCCCC",
    "[*]C=c2oc1cc(O)ccc1c2=O",
    "[*]Oc1ccccc1Br",
    "[*]C2CCCC1(CCCC1)C2",
    "[*]OC(=O)C=CC=CCCC",
    "[*]C(C)C=CC(C)CC",
    "[*]C(C)(O)C(=O)CCC(C)(C)O",
    "[*]C1CC(CC)C(CCO)C=C1",
    "[*]Cc1ccc(O)c(Br)c1",
    "[*]NC(=O)NC1CCCCC1",
    "[*]CC(C)(O)C(O)=O",
    "[*]NC(NC)=NCCCO",
    "[*]C(CCC)CC(O)=O",
    "[*]NC(=O)C(C)CC",
    "[*]C(=O)C(N)C(C)C",
    "[*]C(Cl)(Cl)Cl",
    "[*]C(O)CC(C)=O",
    "[*]C1OC(=O)C=C1C",
    "[*]OC(=O)C(C)(O)CC",
    "[*]C(C)C=CC",
    "[*]C(C)=CC(C)CC",
    "[*]c1ccc(OC)c2ccccc12",
    "[*]c1cccc(O)c1OC",
    "[*]n2cnc1ccccc1c2=O",
    "[*]OC(=O)C=CC=CC=CCCC",
    "[*]OC(=O)c1c[nH]c2ccccc12",
    "[*]C(C)CCC=C(C)C(=O)OC",
    "[*]c1c(OC)cccc1OC",
    "[*]CC=C1C(O)COC1=O",
    "[*]C(C)CC(O)CC(C)C",
    "[*]COc1ccccc1",
    "[*]CCc1cc(O)cc(O)c1",
    "[*]C(C)(C)CCn1ccnc1",
    "[*]CN1CCCC1C(O)=O",
    "[*]c1ccc(O)c(OS(O)(=O)=O)c1",
    "[*]CCC(=O)CC",
    "[*]N(C)C(=O)c1ccccc1",
    "[*]C(O)CCCC",
    "[*]CC(NC(C)=O)C(O)=O",
    "[*]C(C)CC(=O)CC(C)(C)O",
    "[*]CC(=O)CC(C)C",
    "[*]c1c(OC)cc(OC)cc1OC",
    "[*]C(C)CC(O)=O",
    "[*]C1CCCCC1CNC",
    "[*]C(C=C)c1ccccc1",
    "[*]C(C)OC(=O)c1ccccc1",
    "[*]CC(O)OC1CCCCC1",
    "[*]C(=O)c1c(O)cc(O)cc1O",
    "[*]CCC(C)CC",
    "[*]CCC(=O)N1CCCC1C(O)=O",
    "[*]CC(CO)CCCO",
    "[*]c1cccc2ccccc12",
    "[*]NC(=O)CCCC(N)C(O)=O",
    "[*]C(=O)c1ccc(Cl)cc1",
    "[*]C(CC(=O)OC)c1ccccc1",
    "[*]C(C)C=CC(CC)C(C)=C",
    "[*]CC(C)CO",
    "[*]C(=O)NCCc1ccc(O)cc1",
    "[*]C(C)C(O)C(O)C(C)C(C)C",
    "[*]CC(=O)N1CCCC1",
    "[*]CCC1CCCC1",
    "[*]CC1NC(=O)C(C)NC1=O",
    "[*]COC(=O)c1ccc[nH]1",
    "[*]C(C)(O)CCCC(C)(C)O",
    "[*]C(CO)CCC=C(C)C",
    "[*]C(=O)c1c(O)cc(C)cc1O",
    "[*]c1c(C)cccc1O",
    "[*]NCCCC(C)C",
    "[*]OC1CC(C)C(O)C(O)C1O",
    "[*]CC=C(C)CCCC(C)(C)O",
    "[*]CC(=O)CCC",
    "[*]C(C)CCC(C)=C(C)C",
    "[*]CC1CCC(=O)O1",
    "[*]C1=C[CH-]C=N1",
    "[*]C(C)CC(=O)C=C(C)C(O)=O",
    "[*]C1(C)OC1C",
    "[*]CCCCC(C)C",
    "[*]OC(C(O)=O)c1ccccc1",
    "[*]OC(O)c1cccc(O)c1",
    "[*]C(=O)NCCCCNC(N)=N",
    "[*]C(N)CC(O)=O",
    "[*]C1CC(C)CC=C1",
    "[*]C(C)CC=CC(C)(C)OO",
    "[*]C=Cc2ccc1OCOc1c2",
    "[*]NC(=O)CO",
    "[*]OC1CCCCC1=O",
    "[*]C#CC=C",
    "[*]C(C)(O)CCC(O)C(C)(C)O",
    "[*]C(C)(O)C(C)C",
    "[*]c2ccc1OCOc1c2OC",
    "[*]C1NCCc2cc(O)ccc12",
    "[*]CC(C)OC(C)=O",
    "[*]C(C)OS(O)(=O)=O",
    "[*]CCC(=O)C=CC(CC)CCO",
    "[*]CC(CO)Cc1cc[nH]c1",
    "[*]c1ccc(O)c(C)c1",
    "[*]C(CC1=CC(=O)OC1)OC(C)=O",
    "[*]CCOS(O)(=O)=O",
    "[*]C(C)CCC(=O)C(C)=C",
    "[*]C(C)CC1OC(=O)C(C)=C1",
    "[*]C(C)=CCC",
    "[*]C(Br)CO",
    "[*]C=c1[nH]c(=O)c(C=C)c1C",
    "[*]CCc1ccc(OC)c(OC)c1",
    "[*]c1ccccc1CCC(C)C",
    "[*]C2(C)CCC1CC12C",
    "[*]C(=O)OCc1ccccc1O",
    "[*]CSC",
    "[*]Cc1c(Br)[nH]c2ccccc12",
    "[*]CC[N+](C)(C)C",
    "[*]C(=O)CC(C)=O",
    "[*]C(=O)C=Cc1ccccc1O",
    "[*]CC(CO)CC1=C[CH+]N=C1",
    "[*]Cc1ccc(O)c2OCCCc12",
    "[*]OC(=O)COC(C)=O",
    "[*]CCC(=O)c1ccc(O)cc1",
    "[*]C(Cl)CC=CC#C",
    "[*]CC(=O)NC(C(C)C)C(O)=O",
    "[*]C(O)C1CCCC=C1",
    "[*]c1ccc(O)c(OC)c1OC",
    "[*]CC(=O)NC(C)C",
    "[*]C(C)(O)CC(=O)CC(C)C(O)=O",
    "[*]C(Cl)=O",
    "[*]CC(=O)c1cc(O)ccc1O",
    "[*]C(C)C1=NCC(C)CC1",
    "[*]OC(=O)CCC(C)C",
    "[*]C(=O)N1CC1",
    "[*]NC(Cc1ccccc1)C(C)O",
    "[*]C1C=C(C)CCC1C(C)C",
    "[*]C(=O)NCCc1ccccc1",
    "[*]NC(=O)CCOC",
    "[*]OCc1ccc(C(=O)OC)cc1",
    "[*]NCCc1ccccc1",
    "[*]CC1CCCC1",
    "[*]C2(O)CCCC1CC(=O)CCC12",
    "[*]C=CC1=CCOC1=O",
    "[*]COC(=O)c1ccccc1N",
    "[*]CC(O)OC1CCCC1",
    "[*]OCC1CC(C)CC=C1",
    "[*]C12CCC(C(N)C1)C(C)(C)O2",
    "[*]NC(=O)OC(C)(C)C",
    "[*]C=Cc1ccoc1",
    "[*]C(O)(CO)CO",
    "[*]NC(=O)C(C(C)CC)N(C)C",
    "[*]OC(C)C(=O)c1ccccc1",
    "[*]C(=O)OCC(C)C",
    "[*]C(C)=CCO",
    "[*]OC(=O)C(=CC)C1CCCC1",
    "[*]CCCCCCCCC(O)=O",
    "[*]OC(=O)C=Cc1cn(C)cn1",
    "[*]CCCc1ccc(O)c(OC)c1",
    "[*]COCc1c[nH]c2ccccc12",
    "[*]C(C)C(O)CC(=C)C(C)C",
    "[*]OCc1cc(O)cc(CC)c1",
    "[*]c1c(C)cc(O)cc1O",
    "[*]CCc1cc(O)cc(OC)c1",
    "[*]C(=O)OC1CCCCC1",
    "[*]C(C)CCC(C)C(C)(C)O",
    "[*]OC(=O)OCC",
    "[*]OC(=O)CC(C)OC(C)=O",
    "[*]CCCCCCCCCC(C)C",
    "[*]SCCO",
    "[*]C(=O)C(C)=O",
    "[*]C=Cc1cc(OC)cc(OC)c1",
    "[*]C(C)CCCOC(C)=O",
    "[*]CCCCCC(CCCO)C=O",
    "[*]C(=O)C=Cc1ccccc1OC",
    "[*]C(O)CC(=O)OC",
    "[*]C2=Cc1ccc(O)cc1OC2",
    "[*]C=CO",
    "[*]C(C)c1ccccc1",
    "[*]C(CC(=O)OC)C(C)C",
    "[*]C(=O)Cc1ccccc1",
    "[*]C(C)CCO",
    "[*]C(O)c1cc(O)cc(O)c1",
    "[*]C(C)CCC(O)C(C)(O)CO",
    "[*]CSCC(NC(C)=O)C(O)=O",
    "[*]C=c1[nH]c(=O)c(CC)c1C",
    "[*]N1CCCC1=O",
    "[*]C(=CC)C=O",
    "[*]C2CCC1(CCCC1)CC2",
    "[*]C(=O)CC(N)C(O)=O",
    "[*]C(=O)CC=C(C)C",
    "[*]C(O)C(C)=O",
    "[*]c1ccsc1",
    "[*]CC(C)C(O)=O",
    "[*]C(=O)C(C)CCCCCC",
    "[*]CCCC(C)(Cl)Cl",
    "[*]c2cc(=O)c1ccc(O)cc1o2",
    "[*]N=C(O)C(C)=CC",
    "[*]C(CC(C)C)OC(C)=O",
    "[*]OC1C(O)C(O)C(O)C(O)C1O",
    "[*]C(OC(C)=O)C(C)(C)O",
    "[*]OCCc1ccc(O)cc1",
    "[*]C(C)=CC1CCC(O)C(OC)C1",
    "[*]NC(C)CO",
    "[*]OCC(O)CO",
    "[*]CC(O)C(C)(C)OC",
    "[*]C(C)CCC(C)C1CC1C",
    "[*]OC(F)(F)F",
    "[*]CNCCCc1ccccc1",
    "[*]C(O)COC(C)=O",
    "[*]CCN=C(N)NCC(C)C",
    "[*]c1ccnc(NCNC)c1",
    "[*]Cc2ccc1ccccc1c2",
    "[*]CN1CCOCC1",
    "[*]CC(=O)c1ccc(OC)cc1",
    "[*]CN1CCCC1",
    "[*]C2CC1CC(O)OC1O2",
    "[*]C1(C)CCC(=C)C1C",
    "[*]NC(=O)C1CCCCC1",
    "[*]C(C)=CCCC(C)C",
    "[*]COC(=O)CC(=O)OC",
    "[*]c1cc(O)c(OC)cc1OC",
    "[*]CCC(COC(C)=O)=CC(O)=O",
    "[*]C(=O)c1ccccc1O",
    "[*]C(C)(C)C(O)C=CC",
    "[*]CC(=O)NC(CCC)C(O)=O",
    "[*]C(O)Cc1ccccc1",
    "[*]C(=O)Oc1ccc(N(=O)=O)cc1",
    "[*]Cc1cc(C(O)=O)ccc1O",
    "[*]C(C)(O)CC(=O)C=C(C)C",
    "[*]Cc1ccc(O)c(I)c1",
    "[*]C1(COC(C)=O)CO1",
    "[*]C(C)CCC(OO)C(C)=C",
    "[*]c2cc(=O)c1cc(O)ccc1o2",
    "[*]CC(O)C(=O)OC",
    "[*]CCC(C)C(O)=O",
    "[*]C(O)CC(N)=O",
    "[*]CC(CCC(C)(C)O)C(C)=C",
    "[*]CC(=O)NCc1ccccc1",
    "[*]C(C)(O)CCCO",
    "[*]Cc1ccc(NO)cc1",
    "[*]C=CCCCC",
    "[*]CCCN(O)C(C)=O",
    "[*]CC(=O)NC(C)C(O)=O",
    "[*]C=CC(C)=CCO",
    "[*]C=C(C)C(=O)OC",
    "[*]C1CSC(=O)N1",
    "[*]C=CC(O)C(C)O",
    "[*]OC(=O)c1ccc(OC)c(O)c1",
    "[*]c2ccc1ncccc1c2",
    "[*]CC=C1CCOC1=O",
    "[*]C(C)CCC(C)(C)C(C)=C",
    "[*]C2CCc1ccc(O)cc1O2",
    "[*]C(=O)COC(=O)CC",
    "[*]C=Cc1ccc(OC)c(OC)c1",
    "[*]CCCC(N)CCCC",
    "[*]C(C)C(=O)CC",
    "[*]OC(=O)C(C)(C)O",
    "[*]OC(=O)c1cccn1C",
    "[*]CC(CO)CCO",
    "[*]OCC(=O)NC(C(C)C)C(O)=O",
    "[*]OCc1cccc(OC)c1",
    "[*]C1CC2(C)CC1OC(=O)C2(C)O",
    "[*]Oc1cc(C)cc(O)c1O",
    "[*]N=C(O)C=C(C)C",
    "[*]Oc1cc(C(O)=O)cc(O)c1O",
    "[*]CCCCCC(C)O",
    "[*]C(C)C(O)CCC(C)(C)O",
    "[*]CN1CC2(C)CC1CC(C)(C)C2",
    "[*]C(C)(CCC=C(C)C)C=C",
    "[*]C(Br)Br",
    "[*]CCC(O)=N",
    "[*]C2(c1ccccc1)CCCC2N",
    "[*]C1COC(C)(C)C(O)C(O)C1",
    "[*]C(=O)COC",
    "[*]c2cc(=O)c1c(O)cccc1o2",
    "[*]OC(=O)OC",
    "[*]CCCC(N)CCC(C)C=O",
    "[*]C(C)(O)C1CC(C)C(C)C(=O)O1",
    "[*]C1CCNCC1",
    "[*]C(=O)NC(CC(O)=O)C(O)=O",
    "[*]C=C(C)C=C",
    "[*]CC1CC1",
    "[*]C(C)(O)CC=CC(C)(C)OO",
    "[*]C(C)C1CCC(C)C(=O)O1",
    "[*]CC(=O)NC(CCCC)C(O)=O",
    "[*]C(C)CC(=O)C(=C)C(C)C",
    "[*]CC2(c1ccccc1)CCCC2",
    "[*]C(C)CCC(=CCO)C(C)C",
    "[*]C(C)C(OC(C)=O)C(C)CC",
    "[*]NC(=O)C(CC(C)C)NC",
    "[*]OC(=O)NCC=C",
    "[*]C(CO)CCC(O)C(C)(C)O",
    "[*]OC1(NCC)CCCCC1",
    "[*]c2cc1cc(C=CC)ccc1o2",
    "[*]OC(=O)CC(O)CC",
    "[*]C1CC=CC=C1",
    "[*]CCC(=O)CCCCC",
    "[*]c1ccccc1OCC(O)=O",
    "[*]C(=O)OCCc1ccc(O)cc1",
    "[*]c2oc1ccccc1c(=O)c2O",
    "[*]C(C)CC=C(C)C(C)C",
    "[*]C(C)CCC=C(CO)CO",
    "[*]OCCC(C)CO",
    "[*]CC(O)C1CC(=O)NC1=O",
    "[*]C(N)NCCc1ccccc1",
    "[*]NC(=O)C1CCC(=O)N1",
    "[*]CC(OC(C)=O)c1ccoc1",
    "[*]C(C)C=CC(O)C(C)C",
    "[*]c1ccc(O)c(CC=C)c1",
    "[*]c1cccc(OC)c1CO",
    "[*]CCC(CC)CCCO",
    "[*]CCC(O)CC(O)CC(O)=O",
    "[*]CN1CCCC(C)C1",
    "[*]N=C=O",
    "[*]CC=CC=C",
    "[*]CC=CCCC",
    "[*]C(C)OC(C)(C)C=C",
    "[*]C1CCC(C)=CC1",
    "[*]COC(O)=N",
    "[*]C(=O)NCCC(O)=O",
    "[*]C(C)CC=CC(C)=C",
    "[*]OCC(C)CC",
    "[*]CCC=C(C)C=O",
    "[*]C2COC1(CCCCC1)O2",
    "[*]Cc1ccc(NCC)nc1",
    "[*]C1C=C(C)CCC1C(C)=C",
    "[*]OC(=O)c1ccc(=O)n(C)c1",
    "[*]OCc1ccc(C(C)C)cc1",
    "[*]CC1NC(=O)C(C=C)=C1C",
    "[*]OCC[C+]1C=CC(N)=N1",
    "[*]OC(O)=O",
    "[*]OC(=O)c1ccc(Br)cc1",
    "[*]C(O)C=CC",
    "[*]C(=O)CN",
    "[*]c1cc(=O)ccc1=O",
    "[*]c1ccc(C)c(O)c1",
    "[*]C(C)=CC=CC(C)(C)O",
    "[*]CCC(=O)CC(O)CCC",
    "[*]CCCCCC(=O)C1CO1",
    "[*]N=c1cccc(C#N)c1=O",
    "[*]CCCCCCC(=O)OC",
    "[*]C(CC(C)C)CC(=O)OC",
    "[*]CCCCCCCCC(C)CC",
    "[*]Cc1cccc(CC)c1",
    "[*]CC(=O)CC(C)O",
    "[*]CNC(=N)NC1CCCCC1",
    "[*]C(O)CCCC(C)C",
    "[*]C1CC(O)C(O)C(O)C1",
    "[*]OC(=O)COC(=O)CC(C)C",
    "[*]C(O)C=C(C)C(O)=O",
    "[*]OC(=O)C=CC(C)CC(C)CC",
    "[*]OCC=C(C)CC=CC(C)(C)O",
    "[*]C=Cc1cc(O)cc(OC)c1",
    "[*]NCc1ccccc1",
    "[*]C(CO)CCC",
    "[*]CCNC(N)=N",
    "[*]C=CC(C)C(C)O",
    "[*]C(C)(O)COC(C)=O",
    "[*]C2CCC1CCC(O)CC1(O)C2",
    "[*]Nc1ccccc1",
    "[*]COC(=O)C(C)=C",
    "[*]C2CC(=O)c1ccccc1O2",
    "[*]C(COC(C)=O)OC(C)=O",
    "[*]S(C)(=O)=O",
    "[*]C=C(C)C=O",
    "[*]CC(OC)OC",
    "[*]C(C)C(C)=C",
    "[*]c1ccc(C)c(C)c1",
    "[*]CC=CC(C)(C)OO",
    "[*]C(=O)CC(=O)c1ccccc1",
    "[*]C(C)CCCC(C)=C",
    "[*]c1ccccc1C",
    "[*]Cc1cc(O)cc(CC)c1",
    "[*]C(O)=C",
    "[*]CC1CNC(=O)C1",
    "[*]CC(C)C(Cl)(Cl)Cl",
    "[*]C1CCCN1C",
    "[*]CN1CCCC1=O",
    "[*]C(=O)c1c(O)cccc1O",
    "[*]C(=O)c1c(O)cc(O)cc1OC",
    "[*]c1ccc(C)o1",
    "[*]COC(=O)Cc1ccccc1",
    "[*]c1cc(=O)c(OC)cc1=O",
    "[*]CC(C)CCCC",
    "[*]C(=O)OCc1cc(O)ccc1O",
    "[*]C(C)CCCC(C)COC(C)=O",
    "[*]C(CCO)NC(=N)NC",
    "[*]c1cc(OC)ccc1O",
    "[*]CS",
    "[*]c1ccc(OC)c(OC(C)=O)c1",
    "[*]OC(=O)C=CCCC",
    "[*]C(=O)C=CC(C)(C)O",
    "[*]Cc1c(O)c(=O)cc(OC)c1=O",
    "[*]CCC(=O)c1ccccc1",
    "[*]C(C)C(CCC(C)C)OC(C)=O",
    "[*]c1ccoc1CCCCC",
    "[*]C(C)C(=O)NC1CCCCC1",
    "[*]C(CO)OC",
    "[*]C(C)C=CC(C)C(C)=C",
    "[*]C(C)CCC(CC)=C(C)C",
    "[*]C(O)C(O)c1ccc(O)cc1",
    "[*]C(O)c2ccc1OCOc1c2",
    "[*]CC1=NC=C[CH-]1",
    "[*]C(=O)CCc1ccc(O)c(O)c1",
    "[*]CNC(=O)OC(C)(C)C",
    "[*]CN1CCC(C)CC1",
    "[*]C(=O)C(C)C(C)O",
    "[*]CCCCCC=C",
    "[*]C(O)C1CC=CC(=O)O1",
    "[*]CC=C(C)C(O)CC=C(C)C",
    "[*]OC(=O)C=CC=CC(O)C(C)O",
    "[*]OC(C)(C)C=C",
    "[*]OCC=C=C",
    "[*]CCOCNCC(C)O",
    "[*]C(C)NC(C)=O",
    "[*]C(CC(=O)OC)c1cccnc1",
    "[*]C(C)CCC(=C)C(C)CO",
    "[*]C2Oc1cc(O)ccc1CC2O",
    "[*]NCCC(C)C",
    "[*]C1(NC)CCCC1",
    "[*]NC(=O)c1ncccc1O",
    "[*]CCS(O)(=O)=O",
    "[*]CCCCCCCCO",
    "[*]C=CC#C",
    "[*]OC(=O)C=CC=CC(C)CC",
    "[*]CCC(=O)C(C)=C",
    "[*]OC(=O)C(=C)COC(C)=O",
    "[*]COC(=O)c1ccc(O)c(O)c1",
    "[*]C=CC(O)CC=CCC",
    "[*]CC=C(C)COC(C)=O",
    "[*]Cc1cccc(C(N)N)c1C=O",
    "[*]c1ccnc2NC(NC)CCc12",
    "[*]CCc2ccc1OCOc1c2",
    "[*]C(=O)CCl",
    "[*]CCNC(=O)C=CS(C)(=O)=O",
    "[*]C(C)CCC(=O)OCC",
    "[*]C(C)(O)C1CCCCC1",
    "[*]C(=O)c1ccco1",
    "[*]CC(=O)C(C)=C",
    "[*]C(C)C(=O)C(C)CC",
    "[*]CCC(CO)=CC(O)=O",
    "[*]C(C)C=CC(C)C(C)(C)O",
    "[*]C(CC(O)=O)C(O)=O",
    "[*]C(CO)CCCO",
    "[*]C(O)CCc1ccccc1",
    "[*]c2ccc1cc(O)ccc1c2",
    "[*]OC1C(N)CC(N)C(O)C1O",
    "[*]Cc1c[nH]c2cc(Br)ccc12",
    "[*]C(O)C=C",
    "[*]OCc1ccc(OC)cc1",
    "[*]n1cnc2c(=O)[nH]c(N)nc12",
    "[*][n+]1cc[nH]c1",
    "[*]C(O)CCl",
    "[*]CCCCC(N)C(O)=O",
    "[*]C(CO)=CC",
    "[*]CCN(C)(C)=O",
    "[*]C(C)C(=O)CCC(C)CO",
    "[*]C1CCC(Br)C(C)(C)C1",
    "[*]C(O)c1ccc(O)c(OC)c1",
    "[*]C(C)CC1(O)OC(=O)C(C)=C1",
    "[*]C(C)CC(=O)C1OC1(C)C",
    "[*]CNCCCCOC",
    "[*]C(=O)C=C(O)c1ccccc1",
    "[*]CC(C)(C)Cc1ccc(N)nc1",
    "[*]C(C)C=Cc1ccccc1",
    "[*]C(=O)C=Cc1cccc(OC)c1",
    "[*]C(=C)C1CCCCC1",
    "[*]C(=O)OCCO",
    "[*]C=CC(O)CCCC(C)O",
    "[*]CC(=O)N1CCCC1CO",
    "[*]Nc1cccc(C(C)=O)c1",
    "[*]C(=O)N1C(C)C=CC1=O",
    "[*]C(OC)c1ccccc1",
    "[*]C(O)C(O)c1ccccc1",
    "[*]c1cc(O)c(O)cc1OC",
    "[*]CCC(CC)CC=O",
    "[*]C=CC=CC=CCCC",
    "[*]OC(=O)CCN1CCOCC1",
    "[*]C1OCC2C(C)C=C(C)C1C2C",
    "[*]C(C)C=CC(C)C(C)CO",
    "[*]C(=O)C=Cc1ccc(C)cc1",
    "[*]C(=O)C(O)CO",
    "[*]C=Cc1ccc(OC)c(O)c1",
    "[*]C(=O)c2ccc1OCOc1c2",
    "[*]NC(=O)CC",
    "[*]SC(C)=O",
    "[*]OC(C)CC",
    "[*]OCNCCC(C)C",
    "[*]N=C(O)c1ccccc1",
    "[*]C(=C)CCC(O)C(C)=C",
    "[*]C(C)C(O)CC(C)C(C)C",
    "[*]COC(=O)Nc1ccccc1",
    "[*]CC(=O)OCCCC",
    "[*]Oc1cc(Br)cc(Br)c1O",
    "[*]CC(=O)c1ccc(O)cc1",
    "[*]CCCCCCC(C)C",
    "[*]C(=O)SC",
    "[*]c1c(O)cc(C)oc1=O",
    "[*]C(C)C1OC(=O)C(C)=CC1O",
    "[*]C(C)CCC(O)(CO)C(C)C",
    "[*]CC=CCCC1CCCCC1",
    "[*]CC(=O)N1CCCCC1",
    "[*]CNCCCOCC",
    "[*]c1c[nH]c2ccc(CNC)cc12",
    "[*]c1cccc(C(O)=O)c1",
    "[*]NC(N)=O",
    "[*]COCC2=C1C=CC=C[C+]1N=C2",
    "[*]C(=O)CCCOC",
    "[*]c1cccc2cocc12",
    "[*]CCC(C)=CCCC(C)=O",
    "[*]C=CC(C)(C)OC",
    "[*]CC(O)OCCCCCC",
    "[*]CCCCC=C",
    "[*]NC(=O)C=Cc1ccccc1",
    "[*]CCC(C)(OC(C)=O)C=C",
    "[*]OC(=O)c1ccc(N(=O)=O)cc1",
    "[*]N(CC)CC",
    "[*]c2cc1ccccc1n(C)c2=O",
    "[*]CCC1CCC(O)C(C)N1",
    "[*]CCCCN=C(N)N",
    "[*]C(C)(O)C=C",
    "[*]NC(=O)C=CC=CCC(C)C",
    "[*]COC(=O)CCCC",
    "[*]CNC1CCN(C)CC1",
    "[*]COO",
    "[*]c1cccc(OC)c1O",
    "[*]CNC(=O)C(C)=O",
    "[*]C=CC1=CC(=O)OC1O",
    "[*]C(C)(O)CCC(OO)C(C)=C",
    "[*]OCc1ccc(C(O)=O)cc1",
    "[*]c1ccc(OC(C)C)c(OC)c1",
    "[*]CC=C(C)CC(=O)C=C(C)C",
    "[*]c1cnc(N)c(CC)c1",
    "[*]OC(=O)NC(C)=O",
    "[*]COC(=O)C(C)C(C)O",
    "[*]c1cc(=O)cco1",
    "[*]CCCCCNC(N)=N",
    "[*]NC(=O)CC(C)C",
    "[*]NCCc1ccc(O)cc1",
    "[*]CC1C(C(C)=C)CCC1(C)O",
    "[*]CCC1OC1(C)C",
    "[*]C(C)CC(C)C",
    "[*]C(O)NC",
    "[*]C(C)C(OC(C)=O)C(C)=CC",
    "[*]C=Cc1ccc[nH]1",
    "[*]CC1C(C)(C)C(Cl)CCC1(C)O",
    "[*]C(C)(O)CC(=O)CC(C)C",
    "[*]C(C)C1CCC(C)CN1C",
    "[*]C(O)=C(C)C(O)=O",
    "[*]c1ccccc1OCC(C)=C",
    "[*]C(=O)C=COC",
    "[*]C(C)Cl",
    "[*]c2cc(=O)c1cc(C)ccc1o2",
    "[*]CCCCCCCCC=C",
    "[*]CC(C)OC(=O)CC(C)O",
    "[*]C(=O)c1cc(O)cc(O)c1",
    "[*]NC(C)CCC(C)C",
    "[*]C(C)CCC(=O)NCC(O)=O",
    "[*]C=CC(C)=CC",
    "[*]n1ccc(=N)nc1O",
    "[*]c1ccc(O)c(OC)c1O",
    "[*]OC(=O)NCCC",
    "[*]C(C)C(=O)N(CC)CC",
    "[*]C(CC(N)=O)NCC(N)C(N)=O",
    "[*]C(C)OC(=O)CC(C)C",
    "[*]COC1CCCC(NC)C1",
    "[*]C(=O)C1OC1(C)C",
    "[*]C(=O)C=Cc1cccc(O)c1",
    "[*]CC(NC)C(O)=O",
    "[*]C(C)C(=O)NC(C)C",
    "[*]c1ccc(N(=O)=O)cc1",
    "[*]C(C)=CC(C)C",
    "[*]Oc1cccc(C)c1",
    "[*]C(=O)NCCC(N)=O",
    "[*]CCCCCCCC(C)C",
    "[*]C=C1NC(=O)C(C)C1=CC",
    "[*]C=CC(=O)c1cc(C)ccc1O",
    "[*]C(C)C(CNC)NCC(C)O",
    "[*]C(C)C(=O)N1CCCC1",
    "[*]OC(=O)OC(C)(C)C",
    "[*]CCC(O)O",
    "[*]C(=O)OC1CCCC1",
    "[*]C=Cc1cccc(CCO)c1",
    "[*]OC(=O)CCCCCCC(O)=O",
    "[*]c1c(O)cccc1OC",
    "[*]c2ccc1OC(C)(C)CCc1c2",
    "[*]C1CCC(CCCO)CC1",
    "[*]CC(=O)NCC(O)=O",
    "[*]OC(=O)CCN1CCCCC1",
    "[*]C(C)CC(O)C(O)C(C)=C",
    "[*]C(=O)C(N)C(C)CC",
    "[*]CCC1(C)C(C)CCC=C1C",
    "[*]COC(=O)C(NC)C(O)=O",
    "[*]N1CCOCC1",
    "[*]OC(=O)C(C)(CC)OC(C)=O",
    "[*]COC(=O)c1ccccc1O",
    "[*]OC(=O)C(C)(O)CCl",
    "[*]C(=O)C(O)Cc1ccc(O)cc1",
    "[*]C(O)CC1CCCCC1",
    "[*]OC(=C)C(O)=O",
    "[*]OC(=O)C(=C)C(C)O",
    "[*]OC(=O)CCC(C)CO",
    "[*]C1(NC)CCCCC1",
    "[*]C(OC)C(C)C",
    "[*]C=C(CO)C(O)=O",
    "[*]CC=C(C)CCC(O)C(C)(C)O",
    "[*]C(C)c1c[nH]c2ccccc12",
    "[*]C1(CO)CO1",
    "[*]OC(=O)CCc1ccc(O)cc1",
    "[*]C(=O)c1c(C)cc(O)cc1O",
    "[*]c1ccc(C=CC(O)=O)cc1",
    "[*]Cc1ccc(C(N)N)cc1",
    "[*]C(=O)NC(C(C)C)C(=O)OC",
    "[*]C(CC(=O)OC)OC(C)=O",
    "[*]NC(=N)NCCc1ccccc1",
    "[*]CCCC(Cl)Cl",
    "[*]C(=O)NC(O)C(O)=O",
    "[*]C(C)C(CCCO)NCC(C)O",
    "[*]C=CCCCCCCC",
    "[*]OC(=O)CC(C)(O)CC(=O)OC",
    "[*]C1CCC(=O)N1",
    "[*]CCC(O)CC",
    "[*]Oc1cc(C)cc(O)c1C(O)=O",
    "[*]CNC1=NC=C[CH-]1",
    "[*]C(CO)CCO",
    "[*]CCC(C)C(C)=O",
    "[*]C1CCCC1c2cccc(O)c2",
    "[*]C=CC(=O)N1CCCCC1",
    "[*]C=CC(O)CO",
    "[*]C1CCC(C)(C=C)C(C(C)=C)C1",
    "[*]C(C)C(=O)NCCOC",
    "[*]Cc1c(O)cc(C)oc1=O",
    "[*]c1ccc(CO)o1",
    "[*]C(C)CC=Cc1ccccc1",
    "[*]CC(O)C(C)=O",
    "[*]CCC=C(C)COC(C)=O",
    "[*]OC(=O)C(OC(C)=O)C(C)CC",
    "[*]C(C)=CC(=O)CC(C)C",
    "[*]C1=N[CH+]N=C1",
    "[*]OC(C)(C)C",
    "[*]C=CCOC(=O)C(C)=CC",
    "[*]OCC(=O)OCCCC",
    "[*]CC(O)C(C)C",
    "[*]C(=O)C(C)N",
    "[*]C(CO)CCC(=C)C(C)C",
    "[*]c1ccc(N)cc1",
    "[*]OC(=O)c1cccc(O)c1",
    "[*]CC(=O)NC(CC)C(O)=O",
    "[*]CC(=O)NCCC",
    "[*]CC(O)C=C(C)C",
    "[*]C(O)C(O)C(C)(C)O",
    "[*]CNC(=O)CCC(O)=O",
    "[*]CC(=O)NCCCC(O)=O",
    "[*]C(C)(O)C1CC=C(C)C(=O)O1",
    "[*]C(=O)C(C)=C",
    "[*]OCCCNC",
    "[*]C1CCCN1",
    "[*]C1CCCCN1C",
    "[*]OCC(=O)NC(C)C(O)=O",
    "[*]C(=O)C(C)C=c1cc(C)c(=O)o1",
    "[*]C(=O)c1cc(Br)c(O)c(Br)c1",
    "[*]C(C)CC1CC(C)C(=O)O1",
    "[*]OC(=O)C(C)=C(C)C",
    "[*]c2coc1ccc(C)cc1c2=O",
    "[*]C(C)=CC(=O)C=C(C)C",
    "[*]C(O)(C(C)C)C(C)=O",
    "[*]c1c[nH]c(N)n1",
    "[*]NC(=O)COc1ccccc1",
    "[*]CCC(=O)OCCCC",
    "[*]OCC=C(C)CCC(O)C(C)=C",
    "[*]OC(=O)Cc1cccc(CO)c1",
    "[*]C(CO)C(O)CO",
    "[*]COC(=O)c1c(C)cccc1O",
    "[*]C(C)C1CC(CO)=C(C)C(=O)O1",
    "[*]c1ccc(O)c(OCC)c1",
    "[*]C(C)C=CC(CCO)C(C)C",
    "[*]NC1CCCC1",
    "[*]OC(=O)NC(C)C",
    "[*]CI",
    "[*]CCCCCc1ccccc1",
    "[*]C(O)(NCNC)C(O)=O",
    "[*]c2cc(O)c1OCOc1c2",
    "[*]C1(C)OCCO1",
    "[*]C=COC",
    "[*]CC(O)CC(O)=O",
    "[*]C(=O)OCc1ccc(O)cc1",
    "[*]C1(COC(=O)C(C)C)CO1",
    "[*]Oc1cc(C=O)ccc1OC",
    "[*]N1C(=O)CCC1=O",
    "[*]C1CC(=O)OC1(C)C",
    "[*]CC=Cc1ccc(O)cc1",
    "[*]C(Cc1ccccc1)C(O)=O",
    "[*]SSC",
    "[*]C1CCC(O)C(OC)C1",
    "[*]CCC(=O)CCC",
    "[*]C1CCCC1C(O)=O",
    "[*]OC(=O)c1ccccc1N",
    "[*]OCc1ccccc1OC",
    "[*]Cc1cccc(NCC(C)O)c1",
    "[*]C(C(C)C)C(O)=O",
    "[*]C(C)CCC(=C)C(C)C(=O)OC",
    "[*]OC(=O)C(C)C(C)OC(C)=O",
    "[*]OC(CNC)CNC",
    "[*]C(C)C2CC1(C)OC1(C)C(=O)O2",
    "[*]CCn1ccnc1",
    "[*]C1CCC(CCCCC)C1",
    "[*]C=CCNC(=O)c1cc(Br)c[nH]1",
    "[*]C(O)C1(C)OC1C(C)C=CC",
    "[*]C(C)C(=O)c1ccc[nH]1",
    "[*]C(O)(CO)CCC=C(C)C",
    "[*]C(C)C(O)C(O)C=C(C)C",
    "[*]COCC(C)C",
    "[*]CC(OO)C(C)=C",
    "[*]CN1CCCCCC1",
    "[*]CC[n+]1cc[nH]c1",
    "[*]Cc1ccc(Cl)cc1",
    "[*]C(=O)NCCCCN=C(N)N",
    "[*]Nc1ccccc1C(O)=O",
    "[*]OCC(=O)c1ccccc1",
    "[*]C(=O)COC(=O)c1ccccc1",
    "[*]CN2CCc1ccccc1C2",
    "[*]OCc1ccc(C)cc1",
    "[*]C(C)C1OC1c2ccccc2",
    "[*]C=CC(CCCCC)OC(C)=O",
    "[*]CCc1c[nH]c2ccccc12",
    "[*]C(CCC(=C)C(C)(C)O)C(O)=O",
    "[*]CCC(CO)OC",
    "[*]C(=O)OCC(O)COC(C)=O",
    "[*]NCCCOCC",
    "[*]CC(CC)C=CCC",
    "[*]CCCCCC(C)C",
    "[*]CC(O)C(C)(C)Cl",
    "[*]CC1CCNC(N)=N1",
    "[*]CCC#N",
    "[*]CC(=O)N1CCC(C)CC1",
    "[*]COc1cccc(OC)c1O",
    "[*]c1ccc(O)c(O)c1OC",
    "[*]OC(=O)CC(C)(C)OC(C)=O",
    "[*]C=Cc1cc(OC)cc(=O)o1",
    "[*]C(C)CCC(=C)C(C)(C)C",
    "[*]OC(=O)C(CO)=CCOC(C)=O",
    "[*]NCC=C",
    "[*]C(C)C(O)CCC(C)CO",
    "[*]NC(=O)CCC",
    "[*]C(C)CC=CC(C)C",
    "[*]C1OC(=O)C(C)C1C",
    "[*]CC(=C)C=O",
    "[*]OCC(=O)C(C)C",
    "[*]NC(=O)C(NC)C(C)C",
    "[*]CC(=O)c1c(O)cc(O)cc1O",
    "[*]C2COc1cc(OC)ccc1C2",
    "[*]CC(CC)CCCC",
    "[*]c2c(O)c1ccccc1oc2=O",
    "[*]C(O)CC1CCCC1",
    "[*]Cc1cccc(CNC)c1",
    "[*]C(CO)CCCC(C)C",
    "[*]C=C1CCCCC1",
    "[*]COC(=O)c1ccc(OC)cc1",
    "[*]OC(=O)CCCCCCCC",
    "[*]C=CCOC(=O)CC(C)C",
    "[*]CC(=O)c1ccco1",
    "[*]NC(=O)c1cccc(N)c1O",
    "[*]C(O)CCN=C(N)N",
    "[*]c1ccc(C(=O)OCC)o1",
    "[*]c2ccc1ccc(O)cc1c2",
    "[*]n1cc(C=O)[nH]c1=O",
    "[*]SCc1ccccc1",
    "[*]CCCc1ccoc1",
    "[*]Oc1cc(Br)cc(Br)c1OC",
    "[*]c1cc(C)cc(OC)c1O",
    "[*]OC1OC(CN)=CCC1N",
    "[*]CC(=O)N1CCC(O)C1",
    "[*]c1c(O)c(O)c(O)cc1C(O)=O",
    "[*]NC(=O)CCC(N)C(O)=O",
    "[*]C(CCC(C)=O)C(C)C",
    "[*]C(=O)c1c(O)cc(OC)cc1O",
    "[*]C(=C)CC=CC(C)(C)O",
    "[*]c1ccc(OCC(C)=C)cc1",
    "[*]CCC1COC(=O)C1",
    "[*]C(O)CC(N)O",
    "[*]C(C)CC(=O)CC(C)CC=O",
    "[*]CNC(=O)C1CCCC1",
    "[*]SCCN",
    "[*]C(C)=CC(O)C1OC1(C)C",
    "[*]c1c(O)ccc2ccc(=O)oc12",
    "[*]c1ccccc1Cl",
    "[*]c1ccc(OS(O)(=O)=O)cc1",
    "[*]OC(=C)c1ccccc1",
    "[*]SC#N",
    "[*]C1C(C)C=CC1=O",
    "[*]OC(=O)CCC(=O)OC",
    "[*]OCCC1(O)CCCC1",
    "[*]CC(C)CCC(C)C",
    "[*]c1cnc(N)[nH]1",
    "[*]CC(OC(C)=O)C1=CC(=O)OC1",
    "[*]CNC(C(C)C)C(O)=O",
    "[*]CC(=O)C(C)=CC",
    "[*]C(C)CCC(O)C(O)C(O)CO",
    "[*]C(C)CCC(O)(C=C)C(C)C",
    "[*]OC(=O)CC(O)CCCCC",
    "[*]C(=C)CCCO",
    "[*]CCC1CCC(N)NC1",
    "[*]CC(=O)NCCCCCC(O)=O",
    "[*]c2nc1c(C(O)=O)cccc1o2",
    "[*]NC(=O)C(C)C(O)C(C)C",
    "[*]CCC(C)C(=O)OC",
    "[*]c1ccccc1OCC(=O)OC",
    "[*]Cc1cnc(C)nc1N",
    "[*]CCC(C)=CC=C",
    "[*]COC(=O)NC1CCCCC1",
    "[*]OC1CCC(O)C(O)C1O",
    "[*]OC(=O)C(C)(O)CO",
    "[*]C1CCC(=O)c2c(O)cccc12",
    "[*]C1=C(O)C(=O)OC1O",
    "[*]c1ccc(OS(O)(=O)=O)c(O)c1",
    "[*]C(O)C(O)C(C)=C",
    "[*]CCCCCO",
    "[*]CCCCCCCCCCC=C",
    "[*]CCCCC(=O)OC",
    "[*]CCc1cc(OC)cc(OC)c1",
    "[*]c2cc(=O)c1cc(Br)ccc1o2",
    "[*]C(C)C(=O)CCC(C)C",
    "[*]OC(=O)C(=CC)C1(O)CCCC1",
    "[*]c1ccc(C)cc1C",
    "[*]CCC(=O)NC(C(C)C)C(O)=O",
    "[*]C(=O)NC(CCCCN)C(O)=O",
    "[*]CNCCCO",
    "[*]OC(=O)C=C(C)C(C)OC(C)=O",
    "[*]C(=O)OC(C)CC",
    "[*]NC(C)C",
    "[*]C(C)(O)CCC=C(C)C(O)=O",
    "[*]CC=C(C)C(=O)OC",
    "[*]C(=O)c1cc(Cl)c(Cl)[nH]1",
    "[*]CCC1CNC(=O)C12CCCC2",
    "[*]NC(=O)c1ccccc1N(C)C",
    "[*]Oc2ccc1OCOc1c2",
    "[*]C(C(C)CC)C(O)=O",
    "[*]C(O)C1CC2CCN1CC2C=C",
    "[*]C=c1[nH]c(=O)c(C)c1C=C",
    "[*]C1CC1",
    "[*]Cc1cc(C(N)N)ccc1CO",
    "[*]c1ccoc1C",
    "[*]C(=O)C(C)CCCCCCCC",
    "[*]Cc1ncc[nH]1",
    "[*]COC(=O)C=C(C)CCO",
    "[*]CCCCCC1CCCCC1",
    "[*]c1ccc(SC)cc1",
    "[*]CCC=C(C)Cc1cc(C)co1",
    "[*]C(=C)CCC(O)=O",
    "[*]c1ccc(O)c(N(=O)=O)c1",
    "[*]C1=C(C)CC(O)CC1=O",
    "[*]C(C)CCC(CC)C(C)CO",
    "[*]OCc1ccc(O)cc1",
    "[*]OC(=O)C(O)c1ccccc1",
    "[*]C(C)C1OC1C(C)C(C)C",
    "[*]Cc1c(Br)[nH]c2ccc(O)cc12",
    "[*]c1ccc(CO)cc1",
    "[*]OC(=O)CC=C(C)O",
    "[*]NC(=O)C(CC(C)C)N(C)C",
    "[*]C(O)CC(O)C(O)C(C)O",
    "[*]C(=O)CCC(=O)OC",
    "[*]C=CC(O)CC(O)=O",
    "[*]OC(O)c1cc(O)cc(CC)c1",
    "[*]CNC(C(C)CC)C(O)=O",
    "[*]OCC1CO1",
    "[*]C=Cc1cccc(O)c1",
    "[*]C(C)(O)C1CCC(=O)CC1",
    "[*]CC=CC(C)=O",
    "[*]C1OCC2C1COC2=O",
    "[*]OC(F)F",
    "[*]c1ccccc1OC(C)C",
    "[*]C(=O)OC(C)CC(O)=O",
    "[*]C1CCC(OC)CC1",
    "[*]c1cc(=O)c(OC)c(OC)c1=O",
    "[*]CC(O)C1=CC(=O)OC1O",
    "[*]C(=C)CC",
    "[*]C(OC)C(C)O",
    "[*]C(C)C=C(C)C",
    "[*]N1CCC(=O)NC1=O",
    "[*]C(=O)C(C)(C)C",
    "[*]C#CC#CC=CC",
    "[*]C(=O)CCOC(C)=O",
    "[*]CC(CO)Cc1ccccc1",
    "[*]C(=O)NC(CO)C(O)=O",
    "[*]C(=C)CCCC(C)C",
    "[*]c1ccc(C=O)cc1",
    "[*]Cc1ccc(Cl)cc1Cl",
    "[*]C(C)=CC=CC(C)C",
    "[*]C1OC1C(C)(C)O",
    "[*]C(C)CCC(=C)C(C)(C)O",
    "[*]OCc1ccccn1",
    "[*]Oc1ccc(Br)cc1",
    "[*]NCCCCC",
    "[*]C=CC(C)=CC=CC(C)=CC=O",
    "[*]C(=O)C(C)NC(=O)CN",
    "[*]CCCC(CCC(O)=O)NCC",
    "[*]NCCNC",
    "[*]C(C)N(C)C(C)=O",
    "[*]C(C)CCCC(C)C(=O)OC",
    "[*]CC(=O)N1CC(CNC)C1",
    "[*]C(=C)CCCC(C)=O",
    "[*]C(=O)c1ccc(OC)cc1O",
    "[*]C(C)C(O)C(=O)C=C(C)C",
    "[*]c1ccc(OCC)cc1",
    "[*]C(C)C(O)C1OC(=O)C(C)=C1",
    "[*]C(CCC)OC(C)=O",
    "[*]C(C)(O)C(O)CC=C(C)C",
    "[*]CCC(C)(C)OC",
    "[*]COC(=O)Cc1ccc(O)cc1",
    "[*]C(=O)Oc1ccc(C=C)cc1",
    "[*]CC#CC#CC",
    "[*]C=Cc1ccccc1O",
    "[*]C=CC(O)c1ccc(O)cc1",
    "[*]CNCNCC(C)O",
    "[*]C(C)CCC=O",
    "[*]C1C(O)C(N)C2COC1O2",
    "[*]OC(=O)C(C)(C)OC(C)=O",
    "[*]C(C)C1NCC(C)CC1O",
    "[*]C(C)OCC",
    "[*]OC(=O)C1CC(O)C(O)C(O)C1",
    "[*]CNc1ccccc1",
    "[*]C1CC(OC)C(O)C(OC)C1",
    "[*]NC(=O)OCc1ccccc1",
    "[*]CC(=O)OC(C)(C)C",
    "[*]Oc1c(Br)c(Br)c(Br)c(Br)c1Br",
    "[*]c1ccnc(N)n1",
    "[*]C(=C)CCC=C(C)C(O)=O",
    "[*]c1ccccc1N",
    "[*]c1ccc(Cl)c(Cl)c1",
    "[*]C1CC(C=C(C)C)OC1=O",
    "[*]C(O)OC",
    "[*]NC(=O)CCC(O)=O",
    "[*]OC(=O)C(C)(O)C(C)Cl",
    "[*]C(O)COC",
    "[*]OC=CC(C)C",
    "[*]C1(C)CCC(=O)C1(C)C",
    "[*]c1cc(OC)cc(O)c1C(O)=O",
    "[*]CS(C)=O",
    "[*]C1(C)OC(=O)C(C)=C1O",
    "[*]CN1CCN(C)CC1",
    "[*]C(O)C1OC1(C)C",
    "[*]C(C)CC(C)C(C)C(C)C",
    "[*]CC(C)CCCc1ccccc1",
    "[*]C(O)C(O)C(O)C(O)CO",
    "[*]COCCCC",
    "[*]C(=O)C(O)C(C)C",
    "[*]C=CC(=O)CC",
    "[*]c1ncccn1",
    "[*]N=C(O)c1ccccc1N",
    "[*]CC(N)C(=O)OC",
    "[*]C[N+](C)(C)C",
    "[*]c2cnc1NCCC(C)c1c2",
    "[*]C(=O)NC(CC(N)=O)C(O)=O",
    "[*]c1ccc(OC)c(OC(C)C)c1",
    "[*]C(C)C=CC(C(C)C)C(C)C",
    "[*]c1cccc(CCOCC)c1",
    "[*]CC(=O)N1CCCC1C(O)=O",
    "[*]c1cc(Br)c(C)cc1O",
    "[*]CC(C)(O)C=C",
    "[*]CCC(OO)C(C)=C",
    "[*]CCC(C)CNC(=N)NC",
    "[*]CC1=C(C)CC(C)(C)CC1",
    "[*]C=CCC(O)=O",
    "[*]C(C)(O)CCCC(C)=C",
    "[*]C(C)=CCCC(C)(C)O",
    "[*]C=CC=Cc1ccccc1",
    "[*]c2oc1ccc(C)cc1c2C",
    "[*]C(C)(C)CO",
    "[*]Cc1cccc(CO)c1C=O",
    "[*]c1c[nH]c2ccc(Cl)cc12",
]