"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateName = validateName;
exports.validateAirflowConfigurationOptions = validateAirflowConfigurationOptions;
exports.validateLoggingConfiguration = validateLoggingConfiguration;
exports.validateNetworkConfiguration = validateNetworkConfiguration;
exports.validateDagS3Path = validateDagS3Path;
exports.validateMaxWebservers = validateMaxWebservers;
exports.validateMaxWorkers = validateMaxWorkers;
exports.validateMinWebservers = validateMinWebservers;
exports.validateMinWorkers = validateMinWorkers;
exports.validatePluginsS3ObjectVersion = validatePluginsS3ObjectVersion;
exports.validatePluginsS3Path = validatePluginsS3Path;
exports.validateRequirementsS3ObjectVersion = validateRequirementsS3ObjectVersion;
exports.validateRequirementsS3Path = validateRequirementsS3Path;
exports.validateSchedulers = validateSchedulers;
exports.validateStartupScriptS3ObjectVersion = validateStartupScriptS3ObjectVersion;
exports.validateStartupScriptS3Path = validateStartupScriptS3Path;
exports.validateWeeklyMaintenanceWindowStart = validateWeeklyMaintenanceWindowStart;
const utils_1 = require("@robhan-cdk-lib/utils");
const environment_base_1 = require("../environment-base");
function validateName(name) {
    const errors = [];
    if (typeof name !== 'string') {
        errors.push('must be a string');
        return errors; // No need to check further if not a string
    }
    errors.push(...(0, utils_1.validateStringLength)({
        value: name,
        min: 1,
        max: 80,
    }));
    errors.push(...(0, utils_1.validateRegExp)({
        regExp: /^[a-zA-Z][0-9a-zA-Z-_]*$/,
        value: name,
    }));
    return errors;
}
function validateAirflowConfigurationOptions(airflowConfigurationOptions) {
    const errors = [];
    if (typeof airflowConfigurationOptions !== 'object') {
        errors.push('must be an object');
        return errors; // No need to check further if not an object
    }
    for (const [key, value] of Object.entries(airflowConfigurationOptions)) {
        errors.push(...(0, utils_1.validateStringLength)({
            value: key,
            min: 1,
            max: 64,
            message: `key "${key}" must be between 1 and 64 characters long`,
        }));
        errors.push(...(0, utils_1.validateRegExp)({
            regExp: /^[a-z]+([a-z0-9._]*[a-z0-9_]+)?$/,
            value: key,
            messagePrefix: `key "${key}" `,
        }));
        if (typeof value !== 'string') {
            errors.push(`value for key "${key}" must be a string`);
            continue; // No need to check further if not a string
        }
        errors.push(...(0, utils_1.validateStringLength)({
            value: value,
            min: 1,
            max: 65536,
            message: `value for key "${key}" must be between 1 and 65536 characters long`,
        }));
    }
    return errors;
}
function validateModuleLoggingConfiguration(moduleLoggingConfiguration) {
    const errors = [];
    if (typeof moduleLoggingConfiguration !== 'object') {
        errors.push('must be an object');
        return errors; // No need to check further if not an object
    }
    const moduleLoggingConfigurationObject = moduleLoggingConfiguration;
    errors.push(...(0, utils_1.validateObjectAttributes)({
        inputObject: moduleLoggingConfigurationObject,
        mandatoryAttributes: [],
        optionalAttributes: ['cloudWatchLogGroup', 'enabled', 'logLevel'],
    }));
    if (moduleLoggingConfigurationObject.enabled !== undefined) {
        if (typeof moduleLoggingConfigurationObject.enabled !== 'boolean') {
            errors.push('enabled: must be a boolean');
        }
    }
    return errors;
}
function validateLoggingConfiguration(loggingConfiguration) {
    const errors = [];
    if (typeof loggingConfiguration !== 'object') {
        errors.push('must be an object');
        return errors; // No need to check further if not an object
    }
    const loggingConfigurationObject = loggingConfiguration;
    errors.push(...(0, utils_1.validateObjectAttributes)({
        inputObject: loggingConfigurationObject,
        mandatoryAttributes: [],
        optionalAttributes: ['dagProcessingLogs', 'schedulerLogs', 'taskLogs', 'webServerLogs', 'workerLogs'],
    }));
    if (loggingConfigurationObject.dagProcessingLogs !== undefined) {
        errors.push(...validateModuleLoggingConfiguration(loggingConfigurationObject.dagProcessingLogs).map(error => `dagProcessingLogs: ${error}`));
    }
    if (loggingConfigurationObject.schedulerLogs !== undefined) {
        errors.push(...validateModuleLoggingConfiguration(loggingConfigurationObject.schedulerLogs).map(error => `schedulerLogs: ${error}`));
    }
    if (loggingConfigurationObject.taskLogs !== undefined) {
        errors.push(...validateModuleLoggingConfiguration(loggingConfigurationObject.taskLogs).map(error => `taskLogs: ${error}`));
    }
    if (loggingConfigurationObject.webServerLogs !== undefined) {
        errors.push(...validateModuleLoggingConfiguration(loggingConfigurationObject.webServerLogs).map(error => `webServerLogs: ${error}`));
    }
    if (loggingConfigurationObject.workerLogs !== undefined) {
        errors.push(...validateModuleLoggingConfiguration(loggingConfigurationObject.workerLogs).map(error => `workerLogs: ${error}`));
    }
    return errors;
}
function validateNetworkConfiguration(networkConfiguration) {
    const errors = [];
    if (typeof networkConfiguration !== 'object') {
        errors.push('must be an object');
        return errors; // No need to check further if not an object
    }
    const networkConfigurationObject = networkConfiguration;
    errors.push(...(0, utils_1.validateObjectAttributes)({
        inputObject: networkConfigurationObject,
        mandatoryAttributes: [],
        optionalAttributes: ['securityGroups', 'subnets'],
    }));
    if (networkConfigurationObject.securityGroups !== undefined) {
        if (!Array.isArray(networkConfigurationObject.securityGroups)) {
            errors.push('securityGroups: must be an array');
        }
        else {
            errors.push(...(0, utils_1.validateArrayLength)({
                value: networkConfigurationObject.securityGroups,
                min: 1,
                max: 5,
                messagePrefix: 'securityGroups: ',
            }));
        }
    }
    if (networkConfigurationObject.subnets !== undefined) {
        if (!Array.isArray(networkConfigurationObject.subnets)) {
            errors.push('subnets: must be an array');
        }
        else {
            errors.push(...(0, utils_1.validateArrayLength)({
                value: networkConfigurationObject.subnets,
                min: 2,
                max: 2,
                messagePrefix: 'subnets: ',
            }));
        }
    }
    return errors;
}
function validateDagS3Path(dagS3Path) {
    const errors = [];
    if (typeof dagS3Path !== 'string') {
        errors.push('must be a string');
        return errors; // No need to check further if not a string
    }
    errors.push(...(0, utils_1.validateStringLength)({
        value: dagS3Path,
        min: 1,
        max: 1024,
    }));
    return errors;
}
function validateMaxWebservers(props) {
    const errors = [];
    if (props.maxWebservers !== undefined) {
        if (typeof props.maxWebservers !== 'number') {
            return ['must be a number'];
        }
        if (props.environmentClass) {
            if (props.environmentClass === environment_base_1.EnvironmentClass.MW1_MICRO) {
                errors.push(...(0, utils_1.validateNumberRange)({
                    value: props.maxWebservers,
                    min: 1,
                    max: 1,
                }));
            }
            else {
                errors.push(...(0, utils_1.validateNumberRange)({
                    value: props.maxWebservers,
                    min: 2,
                    max: 5,
                }));
            }
        }
    }
    return errors;
}
function validateMaxWorkers(maxWorkers) {
    const errors = [];
    if (typeof maxWorkers !== 'number') {
        return ['must be a number'];
    }
    errors.push(...(0, utils_1.validateNumberRange)({
        value: maxWorkers,
        min: 1,
    }));
    return errors;
}
function validateMinWebservers(props) {
    const errors = [];
    if (props.minWebservers !== undefined) {
        if (typeof props.minWebservers !== 'number') {
            return ['must be a number'];
        }
        if (props.environmentClass) {
            if (props.environmentClass === environment_base_1.EnvironmentClass.MW1_MICRO) {
                errors.push(...(0, utils_1.validateNumberRange)({
                    value: props.minWebservers,
                    min: 1,
                    max: 1,
                }));
            }
            else {
                errors.push(...(0, utils_1.validateNumberRange)({
                    value: props.minWebservers,
                    min: 2,
                    max: 5,
                }));
            }
        }
    }
    return errors;
}
function validateMinWorkers(minWorkers) {
    const errors = [];
    if (typeof minWorkers !== 'number') {
        return ['must be a number'];
    }
    errors.push(...(0, utils_1.validateNumberRange)({
        value: minWorkers,
        min: 1,
    }));
    return errors;
}
function validatePluginsS3ObjectVersion(pluginsS3ObjectVersion) {
    const errors = [];
    if (typeof pluginsS3ObjectVersion !== 'string') {
        errors.push('must be a string');
        return errors; // No need to check further if not a string
    }
    errors.push(...(0, utils_1.validateStringLength)({
        value: pluginsS3ObjectVersion,
        min: 1,
        max: 1024,
    }));
    return errors;
}
function validatePluginsS3Path(pluginsS3Path) {
    const errors = [];
    if (typeof pluginsS3Path !== 'string') {
        errors.push('must be a string');
        return errors; // No need to check further if not a string
    }
    errors.push(...(0, utils_1.validateStringLength)({
        value: pluginsS3Path,
        min: 1,
        max: 1024,
    }));
    return errors;
}
function validateRequirementsS3ObjectVersion(requirementsS3ObjectVersion) {
    const errors = [];
    if (typeof requirementsS3ObjectVersion !== 'string') {
        errors.push('must be a string');
        return errors; // No need to check further if not a string
    }
    errors.push(...(0, utils_1.validateStringLength)({
        value: requirementsS3ObjectVersion,
        min: 1,
        max: 1024,
    }));
    return errors;
}
function validateRequirementsS3Path(requirementsS3Path) {
    const errors = [];
    if (typeof requirementsS3Path !== 'string') {
        errors.push('must be a string');
        return errors; // No need to check further if not a string
    }
    errors.push(...(0, utils_1.validateStringLength)({
        value: requirementsS3Path,
        min: 1,
        max: 1024,
    }));
    return errors;
}
function validateSchedulers(schedulers) {
    const errors = [];
    if (typeof schedulers !== 'number') {
        return ['must be a number'];
    }
    errors.push(...(0, utils_1.validateNumberRange)({
        value: schedulers,
        min: 1,
        max: 5,
    }));
    return errors;
}
function validateStartupScriptS3ObjectVersion(startupScriptS3ObjectVersion) {
    const errors = [];
    if (typeof startupScriptS3ObjectVersion !== 'string') {
        errors.push('must be a string');
        return errors; // No need to check further if not a string
    }
    errors.push(...(0, utils_1.validateStringLength)({
        value: startupScriptS3ObjectVersion,
        min: 1,
        max: 1024,
    }));
    return errors;
}
function validateStartupScriptS3Path(startupScriptS3Path) {
    const errors = [];
    if (typeof startupScriptS3Path !== 'string') {
        errors.push('must be a string');
        return errors; // No need to check further if not a string
    }
    errors.push(...(0, utils_1.validateStringLength)({
        value: startupScriptS3Path,
        min: 1,
        max: 1024,
    }));
    return errors;
}
function validateWeeklyMaintenanceWindowStart(weeklyMaintenanceWindowStart) {
    const errors = [];
    if (typeof weeklyMaintenanceWindowStart !== 'string') {
        errors.push('must be a string');
        return errors; // No need to check further if not a string
    }
    errors.push(...(0, utils_1.validateStringLength)({
        value: weeklyMaintenanceWindowStart,
        min: 1,
        max: 9,
    }));
    errors.push(...(0, utils_1.validateRegExp)({
        value: weeklyMaintenanceWindowStart,
        regExp: /^.*(MON|TUE|WED|THU|FRI|SAT|SUN):([01]\d|2[0-3]):(00|30).*$/,
    }));
    return errors;
}
//# sourceMappingURL=data:application/json;base64,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