"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitlabRunnerAutoscaling = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const asg = require("@aws-cdk/aws-autoscaling");
const aws_autoscaling_hooktargets_1 = require("@aws-cdk/aws-autoscaling-hooktargets");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const logs = require("@aws-cdk/aws-logs");
const assets = require("@aws-cdk/aws-s3-assets");
const sns = require("@aws-cdk/aws-sns");
const subscriptions = require("@aws-cdk/aws-sns-subscriptions");
const cdk = require("@aws-cdk/core");
const cr = require("@aws-cdk/custom-resources");
/**
 * GitlabRunnerAutoscaling Construct for create Autoscaling Gitlab Runner.
 */
class GitlabRunnerAutoscaling extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const defaultProps = {
            instanceType: 't3.micro',
            tags: ['gitlab', 'awscdk', 'runner'],
            gitlabUrl: 'https://gitlab.com/',
            gitlabRunnerImage: 'public.ecr.aws/gitlab/gitlab-runner:alpine',
            alarms: [
                {
                    AlarmName: 'GitlabRunnerDiskUsage',
                    MetricName: 'disk_used_percent',
                },
            ],
        };
        const runnerProps = { ...defaultProps, ...props };
        const asset = new assets.Asset(this, 'GitlabRunnerUserDataAsset', {
            path: path.join(__dirname, '../assets/userdata/amazon-cloudwatch-agent.json'),
        });
        const userData = ec2.UserData.forLinux();
        userData.addS3DownloadCommand({
            bucket: asset.bucket,
            bucketKey: asset.s3ObjectKey,
            localFile: '/opt/aws/amazon-cloudwatch-agent/etc/amazon-cloudwatch-agent.json',
        });
        userData.addCommands(...this.createUserData(runnerProps));
        this.instanceRole =
            runnerProps.instanceRole ??
                new iam.Role(this, 'GitlabRunnerInstanceRole', {
                    assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
                    description: 'For EC2 Instance (Gitlab Runner) Role',
                    managedPolicies: [
                        iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
                        iam.ManagedPolicy.fromAwsManagedPolicyName('CloudWatchAgentServerPolicy'),
                        iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonS3ReadOnlyAccess'),
                    ],
                });
        this.vpc = runnerProps.vpc ?? new ec2.Vpc(this, 'VPC');
        this.securityGroup = new ec2.SecurityGroup(this, 'GitlabRunnerSecurityGroup', {
            vpc: this.vpc,
        });
        const instanceProfile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [this.instanceRole.roleName],
        });
        const lt = new ec2.CfnLaunchTemplate(this, 'GitlabRunnerLaunchTemplate', {
            launchTemplateData: {
                imageId: ec2.MachineImage.latestAmazonLinux({
                    generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
                }).getImage(this).imageId,
                instanceType: runnerProps.instanceType,
                instanceMarketOptions: {
                    marketType: runnerProps.spotInstance ? 'spot' : undefined,
                    spotOptions: runnerProps.spotInstance ? {
                        spotInstanceType: 'one-time',
                    } : undefined,
                },
                userData: cdk.Fn.base64(userData.render()),
                blockDeviceMappings: [
                    {
                        deviceName: '/dev/xvda',
                        ebs: {
                            volumeSize: runnerProps.ebsSize ?? 60,
                        },
                    },
                ],
                iamInstanceProfile: {
                    arn: instanceProfile.attrArn,
                },
                securityGroupIds: this.securityGroup.connections.securityGroups.map((m) => m.securityGroupId),
            },
        });
        this.autoscalingGroup = new asg.AutoScalingGroup(this, 'GitlabRunnerAutoscalingGroup', {
            instanceType: new ec2.InstanceType(runnerProps.instanceType),
            autoScalingGroupName: `Gitlab Runners (${runnerProps.instanceType})`,
            vpc: this.vpc,
            vpcSubnets: runnerProps.vpcSubnet,
            machineImage: ec2.MachineImage.latestAmazonLinux({
                generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
            }),
            minCapacity: runnerProps.minCapacity,
            maxCapacity: runnerProps.maxCapacity,
            desiredCapacity: runnerProps.desiredCapacity,
        });
        const cfnAsg = this.autoscalingGroup.node.tryFindChild('ASG');
        cfnAsg.addPropertyDeletionOverride('LaunchConfigurationName');
        cfnAsg.addPropertyOverride('LaunchTemplate', {
            LaunchTemplateId: lt.ref,
            Version: lt.attrLatestVersionNumber,
        });
        this.autoscalingGroup.node.tryRemoveChild('LaunchConfig');
        this.topicAlarm = new sns.Topic(this, 'GitlabRunnerAlarm');
        const alarms = JSON.stringify(runnerProps.alarms);
        // Put alarms at launch
        const registerFunction = new lambda.Function(this, 'GitlabRunnerRegisterFunction', {
            code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
            handler: 'autoscaling_events.register',
            runtime: lambda.Runtime.PYTHON_3_8,
            timeout: cdk.Duration.seconds(60),
            logRetention: logs.RetentionDays.ONE_DAY,
            environment: {
                ALARMS: alarms,
                SNS_TOPIC_ARN: this.topicAlarm.topicArn,
            },
        });
        registerFunction.role?.addToPrincipalPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            resources: ['*'],
            actions: [
                'cloudwatch:PutMetricAlarm',
            ],
        }));
        this.autoscalingGroup.addLifecycleHook('GitlabRunnerLifeCycleHookLaunching', {
            lifecycleTransition: asg.LifecycleTransition.INSTANCE_LAUNCHING,
            notificationTarget: new aws_autoscaling_hooktargets_1.FunctionHook(registerFunction),
            defaultResult: asg.DefaultResult.CONTINUE,
            heartbeatTimeout: cdk.Duration.seconds(60),
        });
        // Add an alarm action to terminate invalid instances
        const alarmAction = new lambda.Function(this, 'GitlabRunnerAlarmAction', {
            code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
            handler: 'autoscaling_events.on_alarm',
            runtime: lambda.Runtime.PYTHON_3_8,
            timeout: cdk.Duration.seconds(60),
            logRetention: logs.RetentionDays.ONE_DAY,
        });
        alarmAction.role?.addToPrincipalPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            resources: ['*'],
            actions: [
                'autoscaling:SetInstanceHealth',
            ],
        }));
        const alarmSubscription = new subscriptions.LambdaSubscription(alarmAction);
        this.topicAlarm.addSubscription(alarmSubscription);
        // Unregister gitlab runners and remove alarms on instance termination or CFn stack deletion
        const unregisterRole = new iam.Role(this, 'GitlabRunnerUnregisterRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            description: 'For Gitlab Runner Unregistering Function Role',
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ],
        });
        unregisterRole.addToPrincipalPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            resources: ['*'],
            actions: [
                'ssm:SendCommand',
                'autoscaling:DescribeAutoScalingGroups',
                'cloudwatch:DeleteAlarms',
            ],
        }));
        const unregisterFunction = new lambda.Function(this, 'GitlabRunnerUnregisterFunction', {
            code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
            handler: 'autoscaling_events.unregister',
            runtime: lambda.Runtime.PYTHON_3_8,
            timeout: cdk.Duration.seconds(60),
            role: unregisterRole,
            logRetention: logs.RetentionDays.ONE_DAY,
            environment: {
                ALARMS: alarms,
            },
        });
        this.autoscalingGroup.addLifecycleHook('GitlabRunnerLifeCycleHookTerminating', {
            lifecycleTransition: asg.LifecycleTransition.INSTANCE_TERMINATING,
            notificationTarget: new aws_autoscaling_hooktargets_1.FunctionHook(unregisterFunction),
            defaultResult: asg.DefaultResult.CONTINUE,
            heartbeatTimeout: cdk.Duration.seconds(60),
        });
        const unregisterCustomResource = new lambda.Function(this, 'GitlabRunnerUnregisterCustomResource', {
            code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
            handler: 'autoscaling_events.on_event',
            runtime: lambda.Runtime.PYTHON_3_8,
            role: unregisterRole,
            logRetention: logs.RetentionDays.ONE_DAY,
            environment: {
                ALARMS: alarms,
            },
        });
        const unregisterProvider = new cr.Provider(this, 'GitlabRunnerUnregisterProvider', {
            onEventHandler: unregisterCustomResource,
        });
        const customResource = new cdk.CustomResource(this, 'GitlabRunnerCustomResource', {
            serviceToken: unregisterProvider.serviceToken,
            properties: {
                AutoScalingGroupNames: [this.autoscalingGroup.autoScalingGroupName],
            },
        });
        customResource.node.addDependency(unregisterProvider);
        new cdk.CfnOutput(this, 'GitlabRunnerAutoScalingGroupArn', {
            value: this.autoscalingGroup.autoScalingGroupArn,
        });
    }
    dockerVolumesList(dockerVolume) {
        let tempString = '--docker-volumes "/var/run/docker.sock:/var/run/docker.sock"';
        if (dockerVolume) {
            let tempList = [];
            dockerVolume.forEach(e => {
                tempList.push(`"${e.hostPath}:${e.containerPath}"`);
            });
            tempList.forEach(e => {
                tempString = `${tempString} --docker-volumes ${e}`;
            });
        }
        return tempString;
    }
    /**
     * @param props
     * @returns Array.
     */
    createUserData(props) {
        return [
            'yum update -y',
            'sleep 15 && amazon-linux-extras install docker && yum install -y amazon-cloudwatch-agent && systemctl start docker && usermod -aG docker ec2-user && chmod 777 /var/run/docker.sock',
            'systemctl restart docker && systemctl enable docker && systemctl start amazon-cloudwatch-agent && systemctl enable amazon-cloudwatch-agent',
            `docker run -d -v /home/ec2-user/.gitlab-runner:/etc/gitlab-runner -v /var/run/docker.sock:/var/run/docker.sock \
      --name gitlab-runner-register ${props.gitlabRunnerImage} register --non-interactive --url ${props.gitlabUrl} --registration-token ${props.gitlabToken} \
      --docker-pull-policy if-not-present ${this.dockerVolumesList(props?.dockerVolumes)} \
      --executor docker --docker-image "alpine:latest" --description "A Runner on EC2 Instance (${props.instanceType})" \
      --tag-list "${props.tags?.join(',')}" --docker-privileged`,
            `sleep 2 && docker run --restart always -d -v /home/ec2-user/.gitlab-runner:/etc/gitlab-runner -v /var/run/docker.sock:/var/run/docker.sock --name gitlab-runner ${props.gitlabRunnerImage}`,
        ];
    }
}
exports.GitlabRunnerAutoscaling = GitlabRunnerAutoscaling;
_a = JSII_RTTI_SYMBOL_1;
GitlabRunnerAutoscaling[_a] = { fqn: "cdk-gitlab-runner.GitlabRunnerAutoscaling", version: "1.115.305" };
//# sourceMappingURL=data:application/json;base64,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