"""
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
"""


import kfp.dsl
from kfp import dsl, components
import importlib
import os
from typing import List, Callable
import inspect
from disdat_kfp.cache_check import caching_check
from disdat_kfp.cache_push import caching_push
from disdat_kfp.gather_data import gather_data
import logging


def noop(var: str) -> None:
    """
    This no op container does nothing. However, it cannot be removed for the following reason:
        In order to use the output of caching_push outside the condition (which is represented as a group op in kfp),
        the parameter is referenced as task.condition.outputs.parameters.xxx, which again refers to the output of
        caching_push as task.caching_push.outputs.parameters.xxx. However, since caching_push is not used by another
        container, it does not have outputs.parameter but outputs.artifacts.
        Hence, a noop container is used to force the compiler to generate an outputs.parameter section for caching_push
    :param var: pipeline parameter passed in from another container
    :return: nothing
    """
    import logging
    logging.warning('{} - {}'.format('NO_OP',
                                     'this no op container is used to bypass a kfp compiler bug'))


class Caching:
    def __init__(self,
                 disdat_context: str,
                 disdat_repo_s3_url: str,
                 force_rerun_pipeline: bool = False,
                 use_verbose: bool = False,
                 caching_base_image: str = 'docker.intuit.com/docker-rmt/python:3.8',
                 generated_code_dir: str = 'temp_disdat_generated_code'):
        """
        :param disdat_context: context name in which disdat operates
        :param disdat_repo_s3_url: url to the remote s3 storage
        :param force_rerun_pipeline: pipeline-level rerun configuration, each step will use this variable if
            no force_rerun is specified
        :param use_verbose: pipeline-level verbose configuration, set true to see logging.info() messages
        :param caching_base_image: the base container for caching containers. Must be >= python3.8
        :param generated_code_dir: the directory where codes generated by this caching plugin reside
        """
        self.ctxt = disdat_context
        self.s3_url = disdat_repo_s3_url
        self.force_rerun = force_rerun_pipeline
        self.verbose = use_verbose
        self.base_image = caching_base_image
        self.generated_code_dir = generated_code_dir
        # self.aws_profile = aws_profile

        self.disdat_kwargs_signature = {'dsdt_bundle_name': 'str',  # prefix to avoid collision with user params
                                        'dsdt_context_name': 'str',  # passed to caching containers via cmd
                                        'dsdt_s3_url': 'str',
                                        'dsdt_force_rerun': 'bool',
                                        'dsdt_use_verbose': 'bool',
                                        'dsdt_container_used': 'str',
                                        'dsdt_container_cmd': 'str',
                                        # 'dsdt_s3_access_key_id': 'str',
                                        # 'dsdt_s3_secret_access_key_id': 'str',
                                        # 'dsdt_s3_session_token': 'str'
                                        }

        self.gather_data_signature = self.disdat_kwargs_signature.copy()  # gather data needs some more params
        self.gather_data_signature.update({'dsdt_caching_check_uuid': 'String',
                                           'dsdt_caching_push_uuid': 'String',
                                           'dsdt_output_var_name_list': 'list'})

    def enable_caching(self, *args, **kwargs) -> kfp.dsl.ContainerOp:
        """
        Enable caching for the given user container. The logic is straightforward, if parameters match, the container
        is skipped. Otherwise we rerun. This is based on the assumption that the container is idempotent - it should
        always give the same output given a fixed set of input parameters.

        :param args: should have length 1, include the user container op
        :param kwargs: all parameters should be passed in as key word arguments
        :return: an container op (you can think of it as the outputs of the given container )
        """
        assert len(args) == 1, 'All parameters must use in kwargs, not args; container itself uses arg'
        container = args[0]
        container_name = '_'.join(container.__name__.split(' ')).lower()
        container_name.replace('-', '_')  # '-' is illegal in func names
        bundle = kwargs.get('_disdat_bundle', container_name)  # by default use container name as bundle name
        force_rerun = kwargs.get('_disdat_force_rerun', self.force_rerun)
        verbose = kwargs.get('_disdat_verbose', self.verbose)  # component level configuration
        dependencies = kwargs.get('_after', [])  # force dependencies between components
        # aws_profile = kwargs.get('_disdat_aws_profile', self.aws_profile)

        for key in ['_disdat_bundle', '_disdat_force_rerun', '_disdat_verbose',
                    '_after']:
            if key in kwargs:
                kwargs.pop(key)  # no disdat args should be passed to container

        # if aws_profile is not None:
        #     for key in ['aws_access_key_id', 'aws_secret_access_key', 'aws_session_token']:
        #         if key not in aws_profile:
        #             logging.warning('{} not present in \'_disdat_aws_profile\' parameter'.format(key))

        disdat_params = {  # params passed to each caching container
            'dsdt_bundle_name': bundle,
            'dsdt_context_name': self.ctxt,
            'dsdt_s3_url': self.s3_url,
            'dsdt_force_rerun': force_rerun,
            'dsdt_use_verbose': verbose,
            'dsdt_container_used': container.component_spec.implementation.container.image,
            'dsdt_container_cmd': ' '.join(container.component_spec.implementation.container.command)
        }

        caching_check_dynamic = self._code_wrapper_generator(
            user_kwargs=container.component_spec.inputs,  # check parameters against existing records
            disdat_kwargs=self.disdat_kwargs_signature,
            core_code=caching_check,
            return_signature='''NamedTuple('Output', 
            [('use_cache', bool), ('bundle_id', str)])''',
            generated_func_name='caching_check_4_{}'.format(container_name))

        caching_push_dynamic = self._code_wrapper_generator(
            user_kwargs=container.component_spec.inputs,
            disdat_kwargs=self.disdat_kwargs_signature,
            input_artifact_list=container.component_spec.outputs,  # pass in those data as files to this container
            core_code=caching_push,
            return_signature='''NamedTuple('Output', [('bundle_id', str)])''',
            generated_func_name='caching_push_4_{}'.format(container_name))

        gather_data_dynamic = self._code_wrapper_generator(
            user_kwargs=[],
            disdat_kwargs=self.gather_data_signature,
            output_artifact_list=container.component_spec.outputs,  # what files to be put in /tmp/outputs folder
            core_code=gather_data,
            return_signature='None',
            generated_func_name='gather_data_4_{}'.format(container_name)
        )

        caching_check_op = components.create_component_from_func(caching_check_dynamic,
                                                                 base_image=self.base_image,
                                                                 packages_to_install=['disdat'])

        caching_push_op = components.create_component_from_func(caching_push_dynamic,
                                                                base_image=self.base_image,
                                                                packages_to_install=['disdat'])

        gather_data_op = components.create_component_from_func(gather_data_dynamic,
                                                               base_image=self.base_image,
                                                               packages_to_install=['disdat'])

        no_op_op = components.create_component_from_func(noop, base_image=self.base_image)
        # print(inspect.signature(caching_check_op))
        decision = caching_check_op(**kwargs, **disdat_params).after(*dependencies)
        use_cache, bundle_uuid = decision.outputs['use_cache'], decision.outputs['bundle_id']

        with dsl.Condition(use_cache == False):  # rerun the user container
            result = container(**kwargs)  # execute user container
            output_list = container.component_spec.outputs  # get the output variable names
            input_artifacts = {'reserve_disdat_{}'.format(
                self.remove_sensitive_var_name(var.name.lower())): result.outputs[var.name]
                               for var in output_list}  # add reserve_disdat to avoid naming collision
            push_result = caching_push_op(**kwargs, **input_artifacts, **disdat_params)
            caching_push_uuid = push_result.outputs['bundle_id']
            no_op_op(
                caching_push_uuid)  # must use caching_push_id once, otherwise submission will fail due to a compiler bug

        output_var_names = [self.remove_sensitive_var_name(var.name) for var in output_list]
        result = gather_data_op(dsdt_caching_check_uuid=bundle_uuid,
                                dsdt_caching_push_uuid=caching_push_uuid,
                                **disdat_params,
                                dsdt_output_var_name_list=output_var_names)
        return result

    @classmethod
    def _code_wrapper_generator(cls,
                                user_kwargs: List,
                                disdat_kwargs: dict,
                                core_code: Callable,
                                return_signature: str,
                                input_artifact_list: List = None,
                                output_artifact_list: List = None,
                                indentation: str = '    ',
                                generated_func_name: str = None,
                                temp_dir: str = 'dsdt_generated_code') -> Callable:
        """
        Dynamically generate a wrapper function that match user container's input/output signature.
        This function is critical as it ensures caching codes get access to user's parameters.
        We do this because kfp uses func annotation to generate YAML.

        :param user_kwargs: a list of kfp PipeLine parameters that contains name, type, and other info
        :param disdat_kwargs: dictionary {var name: var type}
        :param core_code: the code to inject
        :param return_signature:  the return signature of the core code, required by kfp compiler
        :param input_artifact_list: variables to read as inputs, kfp will copy them to /tmp/inputs/var name/data
        :param output_artifact_list: variables to outputs, them will be saved to /tmp/outputs/var name/data
        :param indentation: by default 4 spaces
        :param generated_func_name: the generated code will be in generated_func_name.py
        :param temp_dir: where the code resides
        :return: a wrapper function
        """
        core_code_name = core_code.__name__
        core_code_src = inspect.getsource(core_code)
        if generated_func_name is None:
            generated_func_name = core_code_name + '_generated'
        core_code_split = core_code_src.split('\n')
        # parameters passed in by value
        user_params = ['{}: \'{}\'={}'.format(param_spec.name, param_spec.type, param_spec.default)
                       for param_spec in user_kwargs if param_spec._passing_style is None]
        # parameters passed in by file
        user_by_file_params = ['{}: components.InputPath(None)'.format(param_spec.name)
                               for param_spec in user_kwargs if param_spec._passing_style is not None]
        user_params = user_by_file_params + user_params
        dsdt_params = ['{}: \'{}\' = None'.format(param, type_) for param, type_ in disdat_kwargs.items()]
        # signal to the compiler what artifacts should be passed by file using components.InputPath, not by value
        input_artifacts = [] if input_artifact_list is None else input_artifact_list
        input_artifacts = ['reserve_disdat_{}: components.InputPath(\'{}\') = None'.format(var.name, var.type)
                           for var in input_artifacts]
        # signal to the kfp compiler what artifacts to output
        output_artifacts = [] if output_artifact_list is None else output_artifact_list
        output_artifacts = ['{}: components.OutputPath(\'{}\')'.format(var.name, var.type) for var in output_artifacts]
        # the wrapper code with dynamic input and output signatures
        code_string = '''
from typing import *
from kfp import components
def {}({}) -> {}:
    import inspect, json
    from typing import NamedTuple
    frame = inspect.currentframe()
    # get the list of input parameters 
    args, _, _, _ = inspect.getargvalues(frame)
    dsdt_params = [{}]
    user_kwargs, dsdt_kwargs = [], []

    # grab all input parameters and pass them in as kwargs
    for key in args:                                    
        temp = key
        if key in dsdt_params:
            key = key.replace('dsdt_', '')                  # prefix is droped as dsdt params are passed in as dict, no need to worry about collision
            dsdt_kwargs.append((key, eval(temp)))
        else:
            user_kwargs.append((key, eval(temp)))
    user_kwargs = dict(user_kwargs)
    dsdt_kwargs = dict(dsdt_kwargs)                     
    # inject core code here !
{}
    # call core code here !
    result = {}(user_kwargs=user_kwargs, disdat_kwargs=dsdt_kwargs)
    return {}
        '''
        code_string = code_string.format(generated_func_name,
                                         ', '.join(output_artifacts + user_params + dsdt_params + input_artifacts),
                                         return_signature,
                                         ', '.join(['\'{}\''.format(key) for key in disdat_kwargs.keys()]),
                                         (indentation + '{}\n') * len(core_code_split),
                                         core_code_name,
                                         'None' if return_signature == 'None' else 'result')
        code_string = code_string.format(*core_code_split)  # inject user code
        output_file = 'dynamic_{}_wrapper.py'.format(generated_func_name)
        try:
            os.mkdir(temp_dir)
        except FileExistsError:
            pass
        with open(os.path.join(temp_dir, '__init__.py'), 'w') as fp:
            fp.write('')
        with open(os.path.join(temp_dir, output_file), 'w') as fp:
            fp.write(code_string)  # save generated file in a sub-folder
        module = importlib.import_module('dsdt_generated_code.' + output_file.split('.')[0],
                                         os.path.join(temp_dir, output_file))
        return getattr(module, generated_func_name)  # return the generate code

    @classmethod
    def remove_sensitive_var_name(cls, var_name: str) -> str:
        """
        This function mimics what kfp does to variable names - removing _file and _path suffix
        :param var_name: variable name
        :return: variable name cleaned
        """
        if var_name.endswith('_path'):
            var_name = var_name.replace('_path', '')
        if var_name.endswith('_file'):
            var_name = var_name.replace('_file', '')
        return var_name