#   coding: utf-8
#   This file is part of DancingCell.

#   DancingCell is free software: you can redistribute it and/or modify
#   it under the terms of the GNU Lesser General Public License.

__author__ = 'Guanjie Wang'
__monty_version = '3.0.0'
__version__ = 1.0
__maintainer__ = 'Guanjie Wang'
__email__ = "gjwang*buaa.edu.cn"
__date__ = "2020/06/19"

"""
copy from monty.json
"""

import os
import json
import datetime

from hashlib import sha1
from collections import OrderedDict, defaultdict
from enum import Enum

from importlib import import_module

from inspect import getfullargspec

try:
    import numpy as np
except ImportError:
    np = None  # type: ignore

try:
    import bson
except ImportError:
    bson = None

try:
    import ruamel.yaml as yaml
except ImportError:
    try:
        import yaml  # type: ignore
    except ImportError:
        yaml = None  # type: ignore


def _load_redirect(redirect_file):
    try:
        with open(redirect_file, "rt") as f:
            d = yaml.safe_load(f)
    except IOError:
        # If we can't find the file
        # Just use an empty redirect dict
        return {}

    # Convert the full paths to module/class
    redirect_dict = defaultdict(dict)
    for old_path, new_path in d.items():
        old_class = old_path.split(".")[-1]
        old_module = ".".join(old_path.split(".")[:-1])

        new_class = new_path.split(".")[-1]
        new_module = ".".join(new_path.split(".")[:-1])

        redirect_dict[old_module][old_class] = {
            "*mod": new_module,
            "*clas": new_class,
        }

    return dict(redirect_dict)


class DcJson:
    """
    This is a mix-in base class specifying an API for DcJson objects. MSON
    is Monty JSON. Essentially, DcJson objects must implement an as_dict
    method, which must return a json serializable dict and must also support
    no arguments (though optional arguments to finetune the output is ok),
    and a from_dict class method that regenerates the object from the dict
    generated by the as_dict method. The as_dict method should contain the
    "*mod" and "*clas" keys which will allow the DcEncoder to
    dynamically deserialize the class. E.g.::

        d["*mod"] = self.__class__.__module__
        d["*mod"] = self.__class__.__name__

    A default implementation is provided in DcJson, which automatically
    determines if the class already contains self.argname or self._argname
    attributes for every arg. If so, these will be used for serialization in
    the dict format. Similarly, the default from_dict will deserialization
    classes of such form. An example is given below::

        class MSONClass(DcJson):

        def __init__(self, a, b, c, d=1, **kwargs):
            self.a = a
            self.b = b
            self._c = c
            self._d = d
            self.kwargs = kwargs

    For such classes, you merely need to inherit from DcJson and you do not
    need to implement your own as_dict or from_dict protocol.

    New to Monty V2.0.6....
    Classes can be redirected to moved implementations by putting in the old
    fully qualified path and new fully qualified path into .monty.yaml in the
    home folder

    Example:
    old_module.old_class: new_module.new_class
    """

    REDIRECT = _load_redirect(os.path.join(os.path.expanduser("~"),
                                           ".monty.yaml"))

    def as_dict(self) -> dict:
        """
        A JSON serializable dict representation of an object.
        """
        d = {"*mod": self.__class__.__module__,
             "*clas": self.__class__.__name__}

        try:
            parent_module = self.__class__.__module__.split('.')[0]
            module_version = import_module(parent_module).__version__  # type: ignore
            d["*ver"] = u"{}".format(module_version)
        except (AttributeError, ImportError):
            d["*ver"] = None  # type: ignore

        args = getfullargspec(self.__class__.__init__).args

        def recursive_as_dict(obj):
            if isinstance(obj, (list, tuple)):
                return [recursive_as_dict(it) for it in obj]
            if isinstance(obj, dict):
                return {kk: recursive_as_dict(vv) for kk, vv in obj.items()}
            if hasattr(obj, "as_dict"):
                return obj.as_dict()
            return obj

        for c in args:
            if c != "self":
                try:
                    a = self.__getattribute__(c)
                except AttributeError:
                    try:
                        a = self.__getattribute__("_" + c)
                    except AttributeError:
                        raise NotImplementedError(
                            "Unable to automatically determine as_dict "
                            "format from class. DcJson requires all "
                            "args to be present as either self.argname or "
                            "self._argname, and kwargs to be present under"
                            "a self.kwargs variable to automatically "
                            "determine the dict format. Alternatively, "
                            "you can implement both as_dict and from_dict.")
                d[c] = recursive_as_dict(a)
        if hasattr(self, "kwargs"):
            # type: ignore
            d.update(**getattr(self, "kwargs"))  # pylint: disable=E1101
        if hasattr(self, "_kwargs"):
            d.update(**getattr(self, "_kwargs"))  # pylint: disable=E1101
        if isinstance(self, Enum):
            d.update({"value": self.value})  # pylint: disable=E1101
        return d

    @classmethod
    def from_dict(cls, d):
        """
        :param d: Dict representation.
        :return: DcJson class.
        """
        decoded = {k: DcDecoder().process_decoded(v) for k, v in d.items()
                   if not k.startswith("*")}
        return cls(**decoded)

    def to_json(self) -> str:
        """
        Returns a json string representation of the DcJson object.
        """
        return json.dumps(self, cls=DcEncoder)

    def unsafe_hash(self):
        """
        Returns an hash of the current object. This uses a generic but low
        performance method of converting the object to a dictionary, flattening
        any nested keys, and then performing a hash on the resulting object
        """

        def flatten(obj, seperator="."):
            # Flattens a dictionary

            flat_dict = {}
            for key, value in obj.items():
                if isinstance(value, dict):
                    flat_dict.update(
                        {
                            seperator.join([key, _key]): _value
                            for _key, _value in flatten(value).items()
                        }
                    )
                elif isinstance(value, list):
                    list_dict = {
                        "{}{}{}".format(key, seperator, num): item
                        for num, item in enumerate(value)
                    }
                    flat_dict.update(flatten(list_dict))
                else:
                    flat_dict[key] = value

            return flat_dict

        ordered_keys = sorted(flatten(jsanitize(self.as_dict())).items(),
                              key=lambda x: x[0])
        ordered_keys = [item for item in ordered_keys if "*" not in item[0]]
        return sha1(json.dumps(OrderedDict(ordered_keys)).encode("utf-8"))


class DcEncoder(json.JSONEncoder):
    """
    A Json Encoder which supports the DcJson API, plus adds support for
    numpy arrays, datetime objects, bson ObjectIds (requires bson) and array2d (DancingCell).

    Usage::

        # Add it as a *cls* keyword when using json.dump
        json.dumps(object, cls=DcEncoder)
    """

    def default(self, o) -> dict:  # pylint: disable=E0202
        """
        Overriding default method for JSON encoding. This method does two
        things: (a) If an object has a to_dict property, return the to_dict
        output. (b) If the *mod and *clas keys are not in the to_dict,
        add them to the output automatically. If the object has no to_dict
        property, the default Python json encoder default method is called.

        Args:
            o: Python object.

        Return:
            Python dict representation.
        """
        if isinstance(o, datetime.datetime):
            return {"*mod": "datetime", "*clas": "datetime",
                    "string": o.__str__()}
        if np is not None:
            if isinstance(o, np.ndarray):
                return {"*mod": "numpy",
                        "*clas": "array",
                        "dtype": o.dtype.__str__(),
                        "data": o.tolist()}
            if isinstance(o, np.generic):
                return o.item()
        if bson is not None:
            if isinstance(o, bson.objectid.ObjectId):
                return {"*mod": "bson.objectid",
                        "*clas": "ObjectId",
                        "oid": str(o)}

        if isinstance(o, array2d):
            return {"*mod": 'dancingcell',
                    '*clas': 'array2d',
                    'dtype': o.data.dtype.__str__(),
                    'data': o.data.tolist()}
        try:
            d = o.as_dict()
            if "*mod" not in d:
                d["*mod"] = u"{}".format(o.__class__.__module__)
            if "*clas" not in d:
                d["*clas"] = u"{}".format(o.__class__.__name__)
            if "*ver" not in d:
                try:
                    parent_module = o.__class__.__module__.split('.')[0]
                    module_version = import_module(parent_module).__version__  # type: ignore
                    d["*ver"] = u"{}".format(module_version)
                except (AttributeError, ImportError):
                    d["*ver"] = None
            return d
        except AttributeError:
            return json.JSONEncoder.default(self, o)


class DcDecoder(json.JSONDecoder):
    """
    A Json Decoder which supports the DcJson API. By default, the
    decoder attempts to find a module and name associated with a dict. If
    found, the decoder will generate a DancingCell as a priority.  If that fails,
    the original decoded dictionary from the string is returned.
    Note that nested lists and dicts containing DancingCell object will be decoded correctly
    as well.

    Usage:

        # Add it as a *cls* keyword when using json.load
        json.loads(json_string, cls=DcDecoder)
    """

    def process_decoded(self, d):
        """
        Recursive method to support decoding dicts and lists containing
        pymatgen objects.
        """
        if isinstance(d, dict):
            if "*mod" in d and "*clas" in d:
                modname = d["*mod"]
                classname = d["*clas"]
                if classname in DcJson.REDIRECT.get(modname, {}):
                    modname = DcJson.REDIRECT[modname][classname]["*mod"]
                    classname = DcJson.REDIRECT[modname][classname]["*clas"]
            else:
                modname = None
                classname = None
            if modname and modname not in ["bson.objectid", "numpy"]:
                if modname == "datetime" and classname == "datetime":
                    try:
                        dt = datetime.datetime.strptime(d["string"],
                                                        "%Y-%m-%d %H:%M:%S.%f")
                    except ValueError:
                        dt = datetime.datetime.strptime(d["string"],
                                                        "%Y-%m-%d %H:%M:%S")
                    return dt

                elif modname == "dancingcell" and classname == 'array2d':
                    return array2d(d['data'], tp=d['dtype'])

                mod = __import__(modname, globals(), locals(), [classname], 0)
                if hasattr(mod, classname):
                    cls_ = getattr(mod, classname)
                    data = {k: v for k, v in d.items()
                            if not k.startswith("*")}
                    if hasattr(cls_, "from_dict"):
                        return cls_.from_dict(data)
            elif np is not None and modname == "numpy" and classname == \
                    "array":
                return np.array(d["data"], dtype=d["dtype"])

            elif (bson is not None) and modname == "bson.objectid" and \
                    classname == "ObjectId":
                return bson.objectid.ObjectId(d["oid"])

            return {self.process_decoded(k): self.process_decoded(v)
                    for k, v in d.items()}

        if isinstance(d, list):
            return [self.process_decoded(x) for x in d]

        return d

    def decode(self, s):
        """
        Overrides decode from JSONDecoder.

        :param s: string
        :return: Object.
        """
        d = json.JSONDecoder.decode(self, s)
        return self.process_decoded(d)


def jsanitize(obj, strict=False, allow_bson=False):
    """
    This method cleans an input json-like object, either a list or a dict or
    some sequence, nested or otherwise, by converting all non-string
    dictionary keys (such as int and float) to strings, and also recursively
    encodes all objects using Monty's as_dict() protocol.

    Args:
        obj: input json-like object.
        strict (bool): This parameters sets the behavior when jsanitize
            encounters an object it does not understand. If strict is True,
            jsanitize will try to get the as_dict() attribute of the object. If
            no such attribute is found, an attribute error will be thrown. If
            strict is False, jsanitize will simply call str(object) to convert
            the object to a string representation.
        allow_bson (bool): This parameters sets the behavior when jsanitize
            encounters an bson supported type such as objectid and datetime. If
            True, such bson types will be ignored, allowing for proper
            insertion into MongoDb databases.

    Returns:
        Sanitized dict that can be json serialized.
    """
    if allow_bson and (isinstance(obj, (datetime.datetime, bytes)) or
                       (bson is not None and
                        isinstance(obj, bson.objectid.ObjectId))):
        return obj
    if isinstance(obj, (list, tuple)):
        return [jsanitize(i, strict=strict, allow_bson=allow_bson) for i in obj]
    if np is not None and isinstance(obj, np.ndarray):
        return [jsanitize(i, strict=strict, allow_bson=allow_bson) for i in
                obj.tolist()]
    if isinstance(obj, dict):
        return {k.__str__(): jsanitize(v, strict=strict, allow_bson=allow_bson)
                for k, v in obj.items()}
    if isinstance(obj, (int, float)):
        return obj
    if obj is None:
        return None

    if not strict:
        return obj.__str__()

    if isinstance(obj, str):
        return obj.__str__()

    return jsanitize(obj.as_dict(), strict=strict, allow_bson=allow_bson)


class array2d(DcJson):
    """
    array2d 是为了方便直接获取x y z数据，并且方便打印调用

    所有的数据都会被转换成3列，n行的数据
    get_item 返回str组成的列表
    str：返回self.data的字符串形式
    from_json： 从json字符串返回array2d，不许试用DcDecode解码器
    """

    def __init__(self, data, name='data', tp=None):

        self._data = np.array(data, dtype=tp)
        self._name = name
        self._tp = tp if tp else np.float
        if (len(self._data.shape) != 2) or (self._data.shape[1] != 3):
            self._data = self._data.reshape((-1, 3))

    @property
    def x(self):
        return self._data[:, 0]

    @property
    def y(self):
        return self._data[:, 1]

    @property
    def z(self):
        return self._data[:, 2]

    @property
    def data(self):
        return self._data

    @property
    def shape(self):
        return self.data.shape

    @property
    def row(self):
        return len(self)

    @property
    def column(self):
        return self._data.shape[1]

    def as_dict(self):
        d = {self._name: self.data} #, 'type': self._tp
        d['*mod'] = self.__class__.__module__
        d['*clas'] = self.__class__.__name__
        return d

    @classmethod
    def from_json(self, s):
        return json.loads(s, cls=DcDecoder)

    def append(self, i):
        return array2d(np.vstack((self.data, i)))

    def __repr__(self):
        return 'Dcjson: array2d: ' + str(self)

    def __str__(self):
        return ' '.join([' '.join(i) for i in self])

    def __getitem__(self, item):
        return list(self.data[item].astype(np.str))

    def __len__(self):
        return self.shape[0]

    def __call__(self, *args, **kwargs):
        return self.data