# -*- coding: utf-8 -*-
"""Configuration of Sphinx documentation generator.

This file does only contain a selection of the most common options. For a
full list see the documentation:
http://www.sphinx-doc.org/en/master/config
"""

from email import message_from_string
from itertools import chain
import pkg_resources


# -- Path setup --------------------------------------------------------------

# If extensions (or modules to document with autodoc) are in another directory,
# add these directories to sys.path here. If the directory is relative to the
# documentation root, use os.path.abspath to make it absolute, like shown here.
#
# import os
# import sys
# sys.path.insert(0, os.path.abspath('.'))


# -- Project information -----------------------------------------------------

def get_supported_pythons(classifiers):
    """Return min and max supported Python version from meta as tuples."""
    py_ver_classifier = 'Programming Language :: Python :: '
    vers = filter(lambda c: c.startswith(py_ver_classifier), classifiers)
    vers = map(lambda c: c[len(py_ver_classifier):], vers)
    vers = filter(lambda c: c[0].isdigit() and '.' in c, vers)
    vers = map(lambda c: tuple(c.split('.')), vers)
    vers = sorted(vers)
    del vers[1:-1]
    if len(vers) < 2:
        vers *= 2
    return vers


def get_github_data(project_urls):
    """Retrieve GitHub user/org and repo name from a bunch of links."""
    partitioned_urls = (p.partition(', ') for p in project_urls)
    for _url_type, _sep, url in partitioned_urls:
        proto, _gh, uri = url.partition('://github.com/')
        if proto not in ('http', 'https'):
            continue
        return uri.split('/')[:2]

    raise LookupError('There are no project URLs pointing to GitHub')


PYTHON_DISTRIBUTION_NAME = 'octomachinery'

PRJ_DIST = pkg_resources.get_distribution(PYTHON_DISTRIBUTION_NAME)
PRJ_PKG_INFO = PRJ_DIST.get_metadata(PRJ_DIST.PKG_INFO)
PRJ_META = message_from_string(PRJ_PKG_INFO)
PRJ_AUTHOR = PRJ_META['Author']
PRJ_LICENSE = PRJ_META['License']
PRJ_SUMMARY = PRJ_META['Summary']
PRJ_DESCRIPTION = PRJ_META['Description']
PRJ_PY_VER_RANGE = get_supported_pythons(PRJ_META.get_all('Classifier'))
PRJ_PY_MIN_SUPPORTED, PRJ_PY_MAX_SUPPORTED = map('.'.join, PRJ_PY_VER_RANGE)
PRJ_GITHUB_USER, PRJ_GITHUB_REPO = get_github_data(
    chain(
        (PRJ_META['Home-page'], ),
        PRJ_META.get_all('Project-URL'),
    ),
)

project = PRJ_DIST.project_name  # pylint: disable=invalid-name
author = PRJ_AUTHOR  # pylint: disable=invalid-name
copyright = f'2019, {author}'  # pylint: disable=invalid-name,redefined-builtin

# The full version, including alpha/beta/rc tags
release = PRJ_DIST.version  # pylint: disable=invalid-name
# The short X.Y version
# pylint: disable=invalid-name
version = pkg_resources.parse_version(release).base_version

rst_epilog = f"""
.. |project| replace:: {project}
.. |min_py_supported| replace:: {PRJ_PY_MIN_SUPPORTED}
.. |max_py_supported| replace:: {PRJ_PY_MAX_SUPPORTED}
"""  # pylint: disable=invalid-name


# -- General configuration ---------------------------------------------------

# If your documentation needs a minimal Sphinx version, state it here.
#
needs_sphinx = '1.7.5'

# Add any Sphinx extension module names here, as strings. They can be
# extensions coming with Sphinx (named 'sphinx.ext.*') or your custom
# ones.
extensions = [
    'sphinx.ext.autodoc',
    'sphinx.ext.doctest',
    'sphinx.ext.intersphinx',
    'sphinx.ext.todo',
    'sphinx.ext.coverage',
    'sphinx.ext.mathjax',
    'sphinx.ext.ifconfig',
    'sphinx.ext.viewcode',
    'sphinx.ext.githubpages',
    'sphinxcontrib.apidoc',
]

# sphinxcontrib.apidoc configuration options
apidoc_extra_args = ['--implicit-namespaces', '../octomachinery']
apidoc_module_dir = '.'
apidoc_output_dir = 'reference'
apidoc_separate_modules = True

# Add any paths that contain templates here, relative to this directory.
templates_path = ['_templates']

# The suffix(es) of source filenames.
# You can specify multiple suffix as a list of string:
#
# source_suffix = ['.rst', '.md']
source_suffix = '.rst'

# The master toctree document.
master_doc = 'index'

# The language for content autogenerated by Sphinx. Refer to documentation
# for a list of supported languages.
#
# This is also used if you do content translation via gettext catalogs.
# Usually you set "language" from the command line for these cases.
language = None

# List of patterns, relative to source directory, that match files and
# directories to ignore when looking for source files.
# This pattern also affects html_static_path and html_extra_path .
exclude_patterns = []

# The name of the Pygments (syntax highlighting) style to use.
# NOTE: These values are commented out so that Furo fall back to a nice style
# pygments_style = 'sphinx'
# pygments_dark_style = 'monokai'


# -- Options for HTML output -------------------------------------------------

# The theme to use for HTML and HTML Help pages.  See the documentation for
# a list of builtin themes.
#
html_theme = 'furo'

# Theme options are theme-specific and customize the look and feel of a theme
# further.  For a list of options available for each theme, see the
# documentation.
#
# html_theme_options = {}

# Add any paths that contain custom static files (such as style sheets) here,
# relative to this directory. They are copied after the builtin static files,
# so a file named "default.css" will overwrite the builtin "default.css".
html_static_path = ['_static']

# Custom sidebar templates, must be a dictionary that maps document names
# to template names.
#
# The default sidebars (for documents that don't match any pattern) are
# defined by theme itself.  Builtin themes are using these templates by
# default: ``['localtoc.html', 'relations.html', 'sourcelink.html',
# 'searchbox.html']``.
#
html_sidebars = {
    '**': (
        'sidebar/brand.html',
        'project-description.html',
        'sidebar/search.html',
        'sidebar/scroll-start.html',
        'sidebar/navigation.html',
        'github-sponsors.html',
        'sidebar/ethical-ads.html',
        'sidebar/scroll-end.html',
    ),
}


# -- Options for HTMLHelp output ---------------------------------------------

# Output file base name for HTML help builder.
htmlhelp_basename = f'{project}doc'


# -- Options for LaTeX output ------------------------------------------------

latex_elements = {
    # The paper size ('letterpaper' or 'a4paper').
    #
    # 'papersize': 'letterpaper',

    # The font size ('10pt', '11pt' or '12pt').
    #
    # 'pointsize': '10pt',

    # Additional stuff for the LaTeX preamble.
    #
    # 'preamble': '',

    # Latex figure (float) alignment
    #
    # 'figure_align': 'htbp',
}

# Grouping the document tree into LaTeX files. List of tuples
# (source start file, target name, title,
#  author, documentclass [howto, manual, or own class]).
latex_documents = [
    (
        master_doc, f'{project}.tex', f'{project} Documentation',
        author, 'manual',
    ),
]


# -- Options for manual page output ------------------------------------------

# One entry per manual page. List of tuples
# (source start file, name, description, authors, manual section).
man_pages = [
    (
        master_doc, project, f'{project} Documentation',
        [author], 1,
    ),
]


# -- Options for Texinfo output ----------------------------------------------

# Grouping the document tree into Texinfo files. List of tuples
# (source start file, target name, title, author,
#  dir menu entry, description, category)
texinfo_documents = [
    (
        master_doc, project, f'{project} Documentation',
        author, project, 'One line description of project.',
        'Miscellaneous',
    ),
]


# -- Options for Epub output -------------------------------------------------

# Bibliographic Dublin Core info.
epub_title = project
epub_author = author
epub_publisher = author
epub_copyright = copyright

# The unique identifier of the text. This can be a ISBN number
# or the project homepage.
#
# epub_identifier = ''

# A unique identification for the text.
#
# epub_uid = ''

# A list of files that should not be packed into the epub file.
epub_exclude_files = ['search.html']


# -- Extension configuration -------------------------------------------------

# -- Options for intersphinx extension ---------------------------------------

# Example configuration for intersphinx: refer to the Python standard library.
intersphinx_mapping = {
    'gidgethub': ('https://gidgethub.readthedocs.io/en/latest/', None),
    'python': ('https://docs.python.org/', None),
    'tutorial': ('https://tutorial.octomachinery.dev/en/latest/', None),
}

# -- Options for todo extension ----------------------------------------------

# If true, `todo` and `todoList` produce output, else they produce nothing.
todo_include_todos = True


def set_up_template_context(
        app, pagename, templatename,  # pylint: disable=unused-argument
        context,
        doctree,  # pylint: disable=unused-argument
):
    """Add a dist summary to Jinja2 context."""
    context['theme_prj_summary'] = PRJ_SUMMARY


def setup(app):
    """Patch the sphinx theme set up stage."""
    app.connect('html-page-context', set_up_template_context)


# Ref: https://github.com/python-attrs/attrs/pull/571/files\
#      #diff-85987f48f1258d9ee486e3191495582dR82
default_role = 'any'
