"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualServiceProvider = exports.VirtualService = void 0;
const cdk = require("@aws-cdk/core");
const appmesh_generated_1 = require("./appmesh.generated");
const mesh_1 = require("./mesh");
/**
 * (experimental) VirtualService represents a service inside an AppMesh.
 *
 * It routes traffic either to a Virtual Node or to a Virtual Router.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_services.html
 * @experimental
 */
class VirtualService extends cdk.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.virtualServiceName || cdk.Lazy.string({ produce: () => cdk.Names.uniqueId(this) }),
        });
        this.clientPolicy = props.clientPolicy;
        const providerConfig = props.virtualServiceProvider.bind(this);
        this.mesh = providerConfig.mesh;
        const svc = new appmesh_generated_1.CfnVirtualService(this, 'Resource', {
            meshName: this.mesh.meshName,
            virtualServiceName: this.physicalName,
            spec: {
                provider: providerConfig.virtualNodeProvider || providerConfig.virtualRouterProvider
                    ? {
                        virtualNode: providerConfig.virtualNodeProvider,
                        virtualRouter: providerConfig.virtualRouterProvider,
                    }
                    : undefined,
            },
        });
        this.virtualServiceName = this.getResourceNameAttribute(svc.attrVirtualServiceName);
        this.virtualServiceArn = this.getResourceArnAttribute(svc.ref, {
            service: 'appmesh',
            resource: `mesh/${this.mesh.meshName}/virtualService`,
            resourceName: this.physicalName,
        });
    }
    /**
     * (experimental) Import an existing VirtualService given an ARN.
     *
     * @experimental
     */
    static fromVirtualServiceArn(scope, id, virtualServiceArn) {
        return new class extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.virtualServiceArn = virtualServiceArn;
                this.parsedArn = cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(virtualServiceArn).resourceName);
                this.virtualServiceName = cdk.Fn.select(2, this.parsedArn);
                this.mesh = mesh_1.Mesh.fromMeshName(this, 'Mesh', cdk.Fn.select(0, this.parsedArn));
            }
        }(scope, id);
    }
    /**
     * (experimental) Import an existing VirtualService given its attributes.
     *
     * @experimental
     */
    static fromVirtualServiceAttributes(scope, id, attrs) {
        return new class extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.virtualServiceName = attrs.virtualServiceName;
                this.mesh = attrs.mesh;
                this.clientPolicy = attrs.clientPolicy;
                this.virtualServiceArn = cdk.Stack.of(this).formatArn({
                    service: 'appmesh',
                    resource: `mesh/${attrs.mesh.meshName}/virtualService`,
                    resourceName: this.virtualServiceName,
                });
            }
        }(scope, id);
    }
}
exports.VirtualService = VirtualService;
/**
 * (experimental) Represents the properties needed to define the provider for a VirtualService.
 *
 * @experimental
 */
class VirtualServiceProvider {
    /**
     * (experimental) Returns a VirtualNode based Provider for a VirtualService.
     *
     * @experimental
     */
    static virtualNode(virtualNode) {
        return new VirtualServiceProviderImpl(virtualNode, undefined);
    }
    /**
     * (experimental) Returns a VirtualRouter based Provider for a VirtualService.
     *
     * @experimental
     */
    static virtualRouter(virtualRouter) {
        return new VirtualServiceProviderImpl(undefined, virtualRouter);
    }
    /**
     * (experimental) Returns an Empty Provider for a VirtualService.
     *
     * This provides no routing capabilities
     * and should only be used as a placeholder
     *
     * @experimental
     */
    static none(mesh) {
        return new VirtualServiceProviderImpl(undefined, undefined, mesh);
    }
}
exports.VirtualServiceProvider = VirtualServiceProvider;
class VirtualServiceProviderImpl extends VirtualServiceProvider {
    constructor(virtualNode, virtualRouter, mesh) {
        var _a, _b, _c, _d;
        super();
        this.virtualNode = virtualNode;
        this.virtualRouter = virtualRouter;
        const providedMesh = (_d = (_b = (_a = this.virtualNode) === null || _a === void 0 ? void 0 : _a.mesh) !== null && _b !== void 0 ? _b : (_c = this.virtualRouter) === null || _c === void 0 ? void 0 : _c.mesh) !== null && _d !== void 0 ? _d : mesh;
        this.mesh = providedMesh;
    }
    bind(_construct) {
        return {
            mesh: this.mesh,
            virtualNodeProvider: this.virtualNode
                ? {
                    virtualNodeName: this.virtualNode.virtualNodeName,
                }
                : undefined,
            virtualRouterProvider: this.virtualRouter
                ? {
                    virtualRouterName: this.virtualRouter.virtualRouterName,
                }
                : undefined,
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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