"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RouteSpec = void 0;
const shared_interfaces_1 = require("./shared-interfaces");
/**
 * (experimental) Used to generate specs with different protocols for a RouteSpec.
 *
 * @experimental
 */
class RouteSpec {
    /**
     * (experimental) Creates an HTTP Based RouteSpec.
     *
     * @experimental
     */
    static http(options) {
        return new HttpRouteSpec(options, shared_interfaces_1.Protocol.HTTP);
    }
    /**
     * (experimental) Creates an HTTP2 Based RouteSpec.
     *
     * @experimental
     */
    static http2(options) {
        return new HttpRouteSpec(options, shared_interfaces_1.Protocol.HTTP2);
    }
    /**
     * (experimental) Creates a TCP Based RouteSpec.
     *
     * @experimental
     */
    static tcp(options) {
        return new TcpRouteSpec(options);
    }
    /**
     * (experimental) Creates a GRPC Based RouteSpec.
     *
     * @experimental
     */
    static grpc(options) {
        return new GrpcRouteSpec(options);
    }
}
exports.RouteSpec = RouteSpec;
class HttpRouteSpec extends RouteSpec {
    constructor(props, protocol) {
        super();
        this.protocol = protocol;
        this.match = props.match;
        this.weightedTargets = props.weightedTargets;
        this.timeout = props.timeout;
    }
    bind(_scope) {
        const prefixPath = this.match ? this.match.prefixPath : '/';
        if (prefixPath[0] != '/') {
            throw new Error(`Prefix Path must start with \'/\', got: ${prefixPath}`);
        }
        const httpConfig = {
            action: {
                weightedTargets: renderWeightedTargets(this.weightedTargets),
            },
            match: {
                prefix: prefixPath,
            },
            timeout: renderTimeout(this.timeout),
        };
        return {
            httpRouteSpec: this.protocol === shared_interfaces_1.Protocol.HTTP ? httpConfig : undefined,
            http2RouteSpec: this.protocol === shared_interfaces_1.Protocol.HTTP2 ? httpConfig : undefined,
        };
    }
}
class TcpRouteSpec extends RouteSpec {
    constructor(props) {
        super();
        this.weightedTargets = props.weightedTargets;
        this.timeout = props.timeout;
    }
    bind(_scope) {
        return {
            tcpRouteSpec: {
                action: {
                    weightedTargets: renderWeightedTargets(this.weightedTargets),
                },
                timeout: renderTimeout(this.timeout),
            },
        };
    }
}
class GrpcRouteSpec extends RouteSpec {
    constructor(props) {
        super();
        this.weightedTargets = props.weightedTargets;
        this.match = props.match;
        this.timeout = props.timeout;
    }
    bind(_scope) {
        return {
            grpcRouteSpec: {
                action: {
                    weightedTargets: renderWeightedTargets(this.weightedTargets),
                },
                match: {
                    serviceName: this.match.serviceName,
                },
                timeout: renderTimeout(this.timeout),
            },
        };
    }
}
/**
* Utility method to add weighted route targets to an existing route
*/
function renderWeightedTargets(weightedTargets) {
    const renderedTargets = [];
    for (const t of weightedTargets) {
        renderedTargets.push({
            virtualNode: t.virtualNode.virtualNodeName,
            weight: t.weight || 1,
        });
    }
    return renderedTargets;
}
/**
 * Utility method to construct a route timeout object
 */
function renderTimeout(timeout) {
    return timeout
        ? {
            idle: (timeout === null || timeout === void 0 ? void 0 : timeout.idle) !== undefined
                ? {
                    unit: 'ms',
                    value: timeout === null || timeout === void 0 ? void 0 : timeout.idle.toMilliseconds(),
                }
                : undefined,
            perRequest: (timeout === null || timeout === void 0 ? void 0 : timeout.perRequest) !== undefined
                ? {
                    unit: 'ms',
                    value: timeout === null || timeout === void 0 ? void 0 : timeout.perRequest.toMilliseconds(),
                }
                : undefined,
        }
        : undefined;
}
//# sourceMappingURL=data:application/json;base64,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