"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mesh = exports.MeshFilterType = void 0;
const cdk = require("@aws-cdk/core");
const appmesh_generated_1 = require("./appmesh.generated");
const virtual_gateway_1 = require("./virtual-gateway");
const virtual_node_1 = require("./virtual-node");
const virtual_router_1 = require("./virtual-router");
/**
 * (experimental) A utility enum defined for the egressFilter type property, the default of DROP_ALL, allows traffic only to other resources inside the mesh, or API calls to amazon resources.
 *
 * @default DROP_ALL
 * @experimental
 */
var MeshFilterType;
(function (MeshFilterType) {
    MeshFilterType["ALLOW_ALL"] = "ALLOW_ALL";
    MeshFilterType["DROP_ALL"] = "DROP_ALL";
})(MeshFilterType = exports.MeshFilterType || (exports.MeshFilterType = {}));
/**
 * Represents a new or imported AppMesh mesh
 */
class MeshBase extends cdk.Resource {
    /**
     * (experimental) Adds a VirtualRouter to the Mesh with the given id and props.
     *
     * @experimental
     */
    addVirtualRouter(id, props = {}) {
        return new virtual_router_1.VirtualRouter(this, id, {
            ...props,
            mesh: this,
        });
    }
    /**
     * (experimental) Adds a VirtualNode to the Mesh.
     *
     * @experimental
     */
    addVirtualNode(id, props = {}) {
        return new virtual_node_1.VirtualNode(this, id, {
            ...props,
            mesh: this,
        });
    }
    /**
     * (experimental) Adds a VirtualGateway to the Mesh.
     *
     * @experimental
     */
    addVirtualGateway(id, props) {
        return new virtual_gateway_1.VirtualGateway(this, id, {
            ...props,
            mesh: this,
        });
    }
}
/**
 * (experimental) Define a new AppMesh mesh.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/meshes.html
 * @experimental
 */
class Mesh extends MeshBase {
    /**
     * @experimental
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.meshName || cdk.Lazy.string({ produce: () => cdk.Names.uniqueId(this) }),
        });
        const mesh = new appmesh_generated_1.CfnMesh(this, 'Resource', {
            meshName: this.physicalName,
            spec: {
                egressFilter: props.egressFilter ? {
                    type: props.egressFilter,
                } : undefined,
            },
        });
        this.meshName = this.getResourceNameAttribute(mesh.attrMeshName);
        this.meshArn = this.getResourceArnAttribute(mesh.ref, {
            service: 'appmesh',
            resource: 'mesh',
            resourceName: this.physicalName,
        });
    }
    /**
     * (experimental) Import an existing mesh by arn.
     *
     * @experimental
     */
    static fromMeshArn(scope, id, meshArn) {
        const parts = cdk.Stack.of(scope).parseArn(meshArn);
        class Import extends MeshBase {
            constructor() {
                super(...arguments);
                this.meshName = parts.resourceName || '';
                this.meshArn = meshArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) Import an existing mesh by name.
     *
     * @experimental
     */
    static fromMeshName(scope, id, meshName) {
        const arn = cdk.Stack.of(scope).formatArn({
            service: 'appmesh',
            resource: 'mesh',
            resourceName: meshName,
        });
        class Import extends MeshBase {
            constructor() {
                super(...arguments);
                this.meshName = meshName;
                this.meshArn = arn;
            }
        }
        return new Import(scope, id);
    }
}
exports.Mesh = Mesh;
//# sourceMappingURL=data:application/json;base64,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