"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const cdk = require("aws-cdk-lib");
const lib_1 = require("../lib");
const cloudfront = require("aws-cdk-lib/aws-cloudfront");
const origins = require("aws-cdk-lib/aws-cloudfront-origins");
const s3 = require("aws-cdk-lib/aws-s3");
const waf = require("aws-cdk-lib/aws-wafv2");
const defaults = require("@aws-solutions-constructs/core");
const assertions_1 = require("aws-cdk-lib/assertions");
function deployConstruct(stack, constructProps) {
    const myBucket = new s3.Bucket(stack, 'myBucket', {
        removalPolicy: cdk.RemovalPolicy.DESTROY
    });
    const testCloudfrontDistribution = new cloudfront.Distribution(stack, 'myDist', {
        defaultBehavior: { origin: new origins.S3Origin(myBucket) },
    });
    const props = constructProps ?
        { webaclProps: constructProps, existingCloudFrontWebDistribution: testCloudfrontDistribution }
        : { existingCloudFrontWebDistribution: testCloudfrontDistribution };
    return new lib_1.WafwebaclToCloudFront(stack, 'test-wafwebacl-cloudfront', props);
}
test('Confirm CheckWafWebAclProps is called', () => {
    const stack = new cdk.Stack();
    const props = {
        defaultAction: {
            allow: {}
        },
        scope: 'CLOUDFRONT',
        visibilityConfig: {
            cloudWatchMetricsEnabled: false,
            metricName: 'webACL',
            sampledRequestsEnabled: true
        },
    };
    const myBucket = new s3.Bucket(stack, 'myBucket');
    const wafAcl = new waf.CfnWebACL(stack, 'test-waf', props);
    const testCloudfrontDistribution = new cloudfront.Distribution(stack, 'myDist', {
        defaultBehavior: { origin: new origins.S3Origin(myBucket) },
    });
    expect(() => {
        new lib_1.WafwebaclToCloudFront(stack, 'test-waf-cloudfront', {
            existingCloudFrontWebDistribution: testCloudfrontDistribution,
            existingWebaclObj: wafAcl,
            webaclProps: props
        });
    }).toThrowError('Error - Either provide existingWebaclObj or webaclProps, but not both.\n');
});
test('Test default deployment', () => {
    const stack = new cdk.Stack();
    const construct = deployConstruct(stack);
    expect(construct.webacl).toBeDefined();
    expect(construct.cloudFrontWebDistribution).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::WAFv2::WebACL", {
        Rules: [
            {
                Name: "AWS-AWSManagedRulesBotControlRuleSet",
                OverrideAction: {
                    None: {}
                },
                Priority: 0,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesBotControlRuleSet",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesBotControlRuleSet",
                    SampledRequestsEnabled: true
                }
            },
            {
                Name: "AWS-AWSManagedRulesKnownBadInputsRuleSet",
                OverrideAction: {
                    None: {}
                },
                Priority: 1,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesKnownBadInputsRuleSet",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesKnownBadInputsRuleSet",
                    SampledRequestsEnabled: true
                }
            },
            {
                Name: "AWS-AWSManagedRulesCommonRuleSet",
                OverrideAction: {
                    None: {}
                },
                Priority: 2,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesCommonRuleSet",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesCommonRuleSet",
                    SampledRequestsEnabled: true
                }
            },
            {
                Name: "AWS-AWSManagedRulesAnonymousIpList",
                OverrideAction: {
                    None: {}
                },
                Priority: 3,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesAnonymousIpList",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesAnonymousIpList",
                    SampledRequestsEnabled: true
                }
            },
            {
                Name: "AWS-AWSManagedRulesAmazonIpReputationList",
                OverrideAction: {
                    None: {}
                },
                Priority: 4,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesAmazonIpReputationList",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesAmazonIpReputationList",
                    SampledRequestsEnabled: true
                }
            },
            {
                Name: "AWS-AWSManagedRulesAdminProtectionRuleSet",
                OverrideAction: {
                    None: {}
                },
                Priority: 5,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesAdminProtectionRuleSet",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesAdminProtectionRuleSet",
                    SampledRequestsEnabled: true
                }
            },
            {
                Name: "AWS-AWSManagedRulesSQLiRuleSet",
                OverrideAction: {
                    None: {}
                },
                Priority: 6,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesSQLiRuleSet",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesSQLiRuleSet",
                    SampledRequestsEnabled: true
                }
            }
        ]
    });
});
test('Test user provided acl props', () => {
    const stack = new cdk.Stack();
    const webaclProps = {
        defaultAction: {
            allow: {}
        },
        scope: 'CLOUDFRONT',
        visibilityConfig: {
            cloudWatchMetricsEnabled: false,
            metricName: 'webACL',
            sampledRequestsEnabled: true
        },
        rules: [
            defaults.wrapManagedRuleSet("AWSManagedRulesCommonRuleSet", "AWS", 0),
            defaults.wrapManagedRuleSet("AWSManagedRulesWordPressRuleSet", "AWS", 1),
        ]
    };
    deployConstruct(stack, webaclProps);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::WAFv2::WebACL", {
        VisibilityConfig: {
            CloudWatchMetricsEnabled: false,
            MetricName: "webACL",
            SampledRequestsEnabled: true
        },
        Rules: [
            {
                Name: "AWS-AWSManagedRulesCommonRuleSet",
                OverrideAction: {
                    None: {}
                },
                Priority: 0,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesCommonRuleSet",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesCommonRuleSet",
                    SampledRequestsEnabled: true
                }
            },
            {
                Name: "AWS-AWSManagedRulesWordPressRuleSet",
                OverrideAction: {
                    None: {}
                },
                Priority: 1,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesWordPressRuleSet",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesWordPressRuleSet",
                    SampledRequestsEnabled: true
                }
            }
        ]
    });
});
test('Test user provided partial acl props', () => {
    const stack = new cdk.Stack();
    const testName = 'test-name';
    const webaclProps = {
        name: testName
    };
    deployConstruct(stack, webaclProps);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::WAFv2::WebACL", {
        Name: testName
    });
});
test('Test existing web ACL', () => {
    const stack = new cdk.Stack();
    const webacl = new waf.CfnWebACL(stack, 'test-webacl', {
        defaultAction: {
            allow: {}
        },
        scope: 'CLOUDFRONT',
        visibilityConfig: {
            cloudWatchMetricsEnabled: true,
            metricName: 'webACL',
            sampledRequestsEnabled: true
        },
    });
    const myBucket = new s3.Bucket(stack, 'myBucket');
    const testCloudfrontDistribution = new cloudfront.Distribution(stack, 'myDist', {
        defaultBehavior: { origin: new origins.S3Origin(myBucket) },
    });
    new lib_1.WafwebaclToCloudFront(stack, 'test-wafwebacl-cloudfront', {
        existingWebaclObj: webacl,
        existingCloudFrontWebDistribution: testCloudfrontDistribution
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::WAFv2::WebACL", {
        VisibilityConfig: {
            CloudWatchMetricsEnabled: true,
            MetricName: "webACL",
            SampledRequestsEnabled: true
        }
    });
    template.resourceCountIs("AWS::WAFv2::WebACL", 1);
});
//# sourceMappingURL=data:application/json;base64,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