"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TagType = exports.CfnResource = void 0;
const cxapi = require("../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
// import required to be here, otherwise causes a cycle when running the generated JavaScript
/* eslint-disable import/order */
const cfn_element_1 = require("./cfn-element");
const cfn_resource_policy_1 = require("./cfn-resource-policy");
const deps_1 = require("./deps");
const cfn_reference_1 = require("./private/cfn-reference");
const removal_policy_1 = require("./removal-policy");
const tag_manager_1 = require("./tag-manager");
const util_1 = require("./util");
/**
 * Represents a CloudFormation resource.
 */
class CfnResource extends cfn_element_1.CfnRefElement {
    /**
     * Creates a resource construct.
     * @param cfnResourceType The CloudFormation type of this resource (e.g. AWS::DynamoDB::Table)
     */
    constructor(scope, id, props) {
        super(scope, id);
        // MAINTAINERS NOTE: this class serves as the base class for the generated L1
        // ("CFN") resources (such as `s3.CfnBucket`). These resources will have a
        // property for each CloudFormation property of the resource. This means that
        // if at some point in the future a property is introduced with a name similar
        // to one of the properties here, it will be "masked" by the derived class. To
        // that end, we prefix all properties in this class with `cfnXxx` with the
        // hope to avoid those conflicts in the future.
        /**
         * Options for this resource, such as condition, update policy etc.
         */
        this.cfnOptions = {};
        /**
         * An object to be merged on top of the entire resource definition.
         */
        this.rawOverrides = {};
        /**
         * Logical IDs of dependencies.
         *
         * Is filled during prepare().
         */
        this.dependsOn = new Set();
        if (!props.type) {
            throw new Error('The `type` property is required');
        }
        this.cfnResourceType = props.type;
        this._cfnProperties = props.properties || {};
        // if aws:cdk:enable-path-metadata is set, embed the current construct's
        // path in the CloudFormation template, so it will be possible to trace
        // back to the actual construct path.
        if (this.node.tryGetContext(cxapi.PATH_METADATA_ENABLE_CONTEXT)) {
            this.addMetadata(cxapi.PATH_METADATA_KEY, this.node.path);
        }
    }
    /**
     * Check whether the given construct is a CfnResource
     */
    static isCfnResource(construct) {
        return construct.cfnResourceType !== undefined;
    }
    /**
     * Sets the deletion policy of the resource based on the removal policy specified.
     */
    applyRemovalPolicy(policy, options = {}) {
        policy = policy || options.default || removal_policy_1.RemovalPolicy.RETAIN;
        let deletionPolicy;
        switch (policy) {
            case removal_policy_1.RemovalPolicy.DESTROY:
                deletionPolicy = cfn_resource_policy_1.CfnDeletionPolicy.DELETE;
                break;
            case removal_policy_1.RemovalPolicy.RETAIN:
                deletionPolicy = cfn_resource_policy_1.CfnDeletionPolicy.RETAIN;
                break;
            case removal_policy_1.RemovalPolicy.SNAPSHOT:
                deletionPolicy = cfn_resource_policy_1.CfnDeletionPolicy.SNAPSHOT;
                break;
            default:
                throw new Error(`Invalid removal policy: ${policy}`);
        }
        this.cfnOptions.deletionPolicy = deletionPolicy;
        if (options.applyToUpdateReplacePolicy !== false) {
            this.cfnOptions.updateReplacePolicy = deletionPolicy;
        }
    }
    /**
     * Returns a token for an runtime attribute of this resource.
     * Ideally, use generated attribute accessors (e.g. `resource.arn`), but this can be used for future compatibility
     * in case there is no generated attribute.
     * @param attributeName The name of the attribute.
     */
    getAtt(attributeName) {
        return cfn_reference_1.CfnReference.for(this, attributeName);
    }
    /**
     * Adds an override to the synthesized CloudFormation resource. To add a
     * property override, either use `addPropertyOverride` or prefix `path` with
     * "Properties." (i.e. `Properties.TopicName`).
     *
     * If the override is nested, separate each nested level using a dot (.) in the path parameter.
     * If there is an array as part of the nesting, specify the index in the path.
     *
     * For example,
     * ```typescript
     * addOverride('Properties.GlobalSecondaryIndexes.0.Projection.NonKeyAttributes', ['myattribute'])
     * addOverride('Properties.GlobalSecondaryIndexes.1.ProjectionType', 'INCLUDE')
     * ```
     * would add the overrides
     * ```json
     * "Properties": {
     *   "GlobalSecondaryIndexes": [
     *     {
     *       "Projection": {
     *         "NonKeyAttributes": [ "myattribute" ]
     *         ...
     *       }
     *       ...
     *     },
     *     {
     *       "ProjectionType": "INCLUDE"
     *       ...
     *     },
     *   ]
     *   ...
     * }
     * ```
     *
     * @param path - The path of the property, you can use dot notation to
     *        override values in complex types. Any intermdediate keys
     *        will be created as needed.
     * @param value - The value. Could be primitive or complex.
     */
    addOverride(path, value) {
        const parts = path.split('.');
        let curr = this.rawOverrides;
        while (parts.length > 1) {
            const key = parts.shift();
            // if we can't recurse further or the previous value is not an
            // object overwrite it with an object.
            const isObject = curr[key] != null && typeof (curr[key]) === 'object' && !Array.isArray(curr[key]);
            if (!isObject) {
                curr[key] = {};
            }
            curr = curr[key];
        }
        const lastKey = parts.shift();
        curr[lastKey] = value;
    }
    /**
     * Syntactic sugar for `addOverride(path, undefined)`.
     * @param path The path of the value to delete
     */
    addDeletionOverride(path) {
        this.addOverride(path, undefined);
    }
    /**
     * Adds an override to a resource property.
     *
     * Syntactic sugar for `addOverride("Properties.<...>", value)`.
     *
     * @param propertyPath The path of the property
     * @param value The value
     */
    addPropertyOverride(propertyPath, value) {
        this.addOverride(`Properties.${propertyPath}`, value);
    }
    /**
     * Adds an override that deletes the value of a property from the resource definition.
     * @param propertyPath The path to the property.
     */
    addPropertyDeletionOverride(propertyPath) {
        this.addPropertyOverride(propertyPath, undefined);
    }
    /**
     * Indicates that this resource depends on another resource and cannot be
     * provisioned unless the other resource has been successfully provisioned.
     *
     * This can be used for resources across stacks (or nested stack) boundaries
     * and the dependency will automatically be transferred to the relevant scope.
     */
    addDependsOn(target) {
        deps_1.addDependency(this, target, `"${this.node.path}" depends on "${target.node.path}"`);
    }
    /**
     * Add a value to the CloudFormation Resource Metadata
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/metadata-section-structure.html
     *
     * Note that this is a different set of metadata from CDK node metadata; this
     * metadata ends up in the stack template under the resource, whereas CDK
     * node metadata ends up in the Cloud Assembly.
     */
    addMetadata(key, value) {
        if (!this.cfnOptions.metadata) {
            this.cfnOptions.metadata = {};
        }
        this.cfnOptions.metadata[key] = value;
    }
    /**
     * @returns a string representation of this resource
     */
    toString() {
        return `${super.toString()} [${this.cfnResourceType}]`;
    }
    /**
     * Called by the `addDependency` helper function in order to realize a direct
     * dependency between two resources that are directly defined in the same
     * stacks.
     *
     * Use `resource.addDependsOn` to define the dependency between two resources,
     * which also takes stack boundaries into account.
     *
     * @internal
     */
    _addResourceDependency(target) {
        this.dependsOn.add(target);
    }
    /**
     * Emits CloudFormation for this resource.
     * @internal
     */
    _toCloudFormation() {
        try {
            const ret = {
                Resources: {
                    // Post-Resolve operation since otherwise deepMerge is going to mix values into
                    // the Token objects returned by ignoreEmpty.
                    [this.logicalId]: new util_1.PostResolveToken({
                        Type: this.cfnResourceType,
                        Properties: util_1.ignoreEmpty(this.cfnProperties),
                        DependsOn: util_1.ignoreEmpty(renderDependsOn(this.dependsOn)),
                        CreationPolicy: util_1.capitalizePropertyNames(this, renderCreationPolicy(this.cfnOptions.creationPolicy)),
                        UpdatePolicy: util_1.capitalizePropertyNames(this, this.cfnOptions.updatePolicy),
                        UpdateReplacePolicy: util_1.capitalizePropertyNames(this, this.cfnOptions.updateReplacePolicy),
                        DeletionPolicy: util_1.capitalizePropertyNames(this, this.cfnOptions.deletionPolicy),
                        Metadata: util_1.ignoreEmpty(this.cfnOptions.metadata),
                        Condition: this.cfnOptions.condition && this.cfnOptions.condition.logicalId,
                    }, props => {
                        const renderedProps = this.renderProperties(props.Properties || {});
                        props.Properties = renderedProps && (Object.values(renderedProps).find(v => !!v) ? renderedProps : undefined);
                        return deepMerge(props, this.rawOverrides);
                    }),
                },
            };
            return ret;
        }
        catch (e) {
            // Change message
            e.message = `While synthesizing ${this.node.path}: ${e.message}`;
            // Adjust stack trace (make it look like node built it, too...)
            const trace = this.creationStack;
            if (trace) {
                const creationStack = ['--- resource created at ---', ...trace].join('\n  at ');
                const problemTrace = e.stack.substr(e.stack.indexOf(e.message) + e.message.length);
                e.stack = `${e.message}\n  ${creationStack}\n  --- problem discovered at ---${problemTrace}`;
            }
            // Re-throw
            throw e;
        }
        // returns the set of logical ID (tokens) this resource depends on
        // sorted by construct paths to ensure test determinism
        function renderDependsOn(dependsOn) {
            return Array
                .from(dependsOn)
                .sort((x, y) => x.node.path.localeCompare(y.node.path))
                .map(r => r.logicalId);
        }
        function renderCreationPolicy(policy) {
            if (!policy) {
                return undefined;
            }
            const result = { ...policy };
            if (policy.resourceSignal && policy.resourceSignal.timeout) {
                result.resourceSignal = policy.resourceSignal;
            }
            return result;
        }
    }
    get cfnProperties() {
        const props = this._cfnProperties || {};
        if (tag_manager_1.TagManager.isTaggable(this)) {
            const tagsProp = {};
            tagsProp[this.tags.tagPropertyName] = this.tags.renderTags();
            return deepMerge(props, tagsProp);
        }
        return props;
    }
    renderProperties(props) {
        return props;
    }
    /**
     * Return properties modified after initiation
     *
     * Resources that expose mutable properties should override this function to
     * collect and return the properties object for this resource.
     */
    get updatedProperites() {
        return this._cfnProperties;
    }
    validateProperties(_properties) {
        // Nothing
    }
}
exports.CfnResource = CfnResource;
var TagType;
(function (TagType) {
    TagType["STANDARD"] = "StandardTag";
    TagType["AUTOSCALING_GROUP"] = "AutoScalingGroupTag";
    TagType["MAP"] = "StringToStringMap";
    TagType["KEY_VALUE"] = "KeyValue";
    TagType["NOT_TAGGABLE"] = "NotTaggable";
})(TagType = exports.TagType || (exports.TagType = {}));
/**
 * Merges `source` into `target`, overriding any existing values.
 * `null`s will cause a value to be deleted.
 */
function deepMerge(target, ...sources) {
    for (const source of sources) {
        if (typeof (source) !== 'object' || typeof (target) !== 'object') {
            throw new Error(`Invalid usage. Both source (${JSON.stringify(source)}) and target (${JSON.stringify(target)}) must be objects`);
        }
        for (const key of Object.keys(source)) {
            const value = source[key];
            if (typeof (value) === 'object' && value != null && !Array.isArray(value)) {
                // if the value at the target is not an object, override it with an
                // object so we can continue the recursion
                if (typeof (target[key]) !== 'object') {
                    target[key] = {};
                }
                deepMerge(target[key], value);
                // if the result of the merge is an empty object, it's because the
                // eventual value we assigned is `undefined`, and there are no
                // sibling concrete values alongside, so we can delete this tree.
                const output = target[key];
                if (typeof (output) === 'object' && Object.keys(output).length === 0) {
                    delete target[key];
                }
            }
            else if (value === undefined) {
                delete target[key];
            }
            else {
                target[key] = value;
            }
        }
    }
    return target;
}
//# sourceMappingURL=data:application/json;base64,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