"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnInclude = void 0;
const core = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cfn_parse = require("../../core/lib/cfn-parse"); // Automatically re-written from '@aws-cdk/core/lib/cfn-parse'
const cfn_type_to_l1_mapping = require("./cfn-type-to-l1-mapping");
const futils = require("./file-utils");
/**
 * Construct to import an existing CloudFormation template file into a CDK application.
 * All resources defined in the template file can be retrieved by calling the {@link getResource} method.
 * Any modifications made on the returned resource objects will be reflected in the resulting CDK template.
 */
class CfnInclude extends core.CfnElement {
    constructor(scope, id, props) {
        super(scope, id);
        this.conditions = {};
        this.resources = {};
        this.parameters = {};
        this.outputs = {};
        this.nestedStacks = {};
        // read the template into a JS object
        this.template = futils.readYamlSync(props.templateFile);
        // ToDo implement preserveLogicalIds=false
        this.preserveLogicalIds = true;
        // instantiate all parameters
        for (const logicalId of Object.keys(this.template.Parameters || {})) {
            this.createParameter(logicalId);
        }
        // instantiate the conditions
        this.conditionsScope = new core.Construct(this, '$Conditions');
        for (const conditionName of Object.keys(this.template.Conditions || {})) {
            this.getOrCreateCondition(conditionName);
        }
        this.nestedStacksToInclude = props.nestedStacks || {};
        // instantiate all resources as CDK L1 objects
        for (const logicalId of Object.keys(this.template.Resources || {})) {
            this.getOrCreateResource(logicalId);
        }
        // verify that all nestedStacks have been instantiated
        for (const nestedStackId of Object.keys(props.nestedStacks || {})) {
            if (!(nestedStackId in this.resources)) {
                throw new Error(`Nested Stack with logical ID '${nestedStackId}' was not found in the template`);
            }
        }
        const outputScope = new core.Construct(this, '$Ouputs');
        for (const logicalId of Object.keys(this.template.Outputs || {})) {
            this.createOutput(logicalId, outputScope);
        }
    }
    /**
     * Returns the low-level CfnResource from the template with the given logical ID.
     * Any modifications performed on that resource will be reflected in the resulting CDK template.
     *
     * The returned object will be of the proper underlying class;
     * you can always cast it to the correct type in your code:
     *
     *     // assume the template contains an AWS::S3::Bucket with logical ID 'Bucket'
     *     const cfnBucket = cfnTemplate.getResource('Bucket') as s3.CfnBucket;
     *     // cfnBucket is of type s3.CfnBucket
     *
     * If the template does not contain a resource with the given logical ID,
     * an exception will be thrown.
     *
     * @param logicalId the logical ID of the resource in the CloudFormation template file
     */
    getResource(logicalId) {
        const ret = this.resources[logicalId];
        if (!ret) {
            throw new Error(`Resource with logical ID '${logicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnCondition object from the 'Conditions'
     * section of the CloudFormation template with the given name.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Condition with the given name is not present in the template,
     * throws an exception.
     *
     * @param conditionName the name of the Condition in the CloudFormation template file
     */
    getCondition(conditionName) {
        const ret = this.conditions[conditionName];
        if (!ret) {
            throw new Error(`Condition with name '${conditionName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnParameter object from the 'Parameters'
     * section of the included template
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Parameter with the given name is not present in the template,
     * throws an exception.
     *
     * @param parameterName the name of the parameter to retrieve
     */
    getParameter(parameterName) {
        const ret = this.parameters[parameterName];
        if (!ret) {
            throw new Error(`Parameter with name '${parameterName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnOutput object from the 'Outputs'
     * section of the included template
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If an Output with the given name is not present in the template,
     * throws an exception.
     *
     * @param logicalId the name of the output to retrieve
     */
    getOutput(logicalId) {
        const ret = this.outputs[logicalId];
        if (!ret) {
            throw new Error(`Output with logical ID '${logicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the NestedStack with name logicalId.
     * For a nested stack to be returned by this method, it must be specified in the {@link CfnIncludeProps.nestedStacks}
     * @param logicalId the ID of the stack to retrieve, as it appears in the template.
     */
    getNestedStack(logicalId) {
        if (!this.nestedStacks[logicalId]) {
            if (!this.template.Resources[logicalId]) {
                throw new Error(`Nested Stack with logical ID '${logicalId}' was not found in the template`);
            }
            else if (this.template.Resources[logicalId].Type !== 'AWS::CloudFormation::Stack') {
                throw new Error(`Resource with logical ID '${logicalId}' is not a CloudFormation Stack`);
            }
            else {
                throw new Error(`Nested Stack '${logicalId}' was not included in the nestedStacks property when including the parent template`);
            }
        }
        return this.nestedStacks[logicalId];
    }
    /** @internal */
    _toCloudFormation() {
        const ret = {};
        for (const section of Object.keys(this.template)) {
            // render all sections of the template unchanged,
            // except Conditions, Resources, Parameters, and Outputs which will be taken care of by the created L1s
            if (section !== 'Conditions' && section !== 'Resources' && section !== 'Parameters' && section !== 'Outputs') {
                ret[section] = this.template[section];
            }
        }
        return ret;
    }
    createParameter(logicalId) {
        const expression = new cfn_parse.CfnParser({
            finder: {
                findResource() { throw new Error('Using GetAtt expressions in Parameter definitions is not allowed'); },
                findRefTarget() { throw new Error('Using Ref expressions in Parameter definitions is not allowed'); },
                findCondition() { throw new Error('Referring to Conditions in Parameter definitions is not allowed'); },
            },
        }).parseValue(this.template.Parameters[logicalId]);
        const cfnParameter = new core.CfnParameter(this, logicalId, {
            type: expression.Type,
            default: expression.Default,
            allowedPattern: expression.AllowedPattern,
            constraintDescription: expression.ConstraintDescription,
            description: expression.Description,
            maxLength: expression.MaxLength,
            maxValue: expression.MaxValue,
            minLength: expression.MinLength,
            minValue: expression.MinValue,
            noEcho: expression.NoEcho,
        });
        cfnParameter.overrideLogicalId(logicalId);
        this.parameters[logicalId] = cfnParameter;
    }
    createOutput(logicalId, scope) {
        const self = this;
        const outputAttributes = new cfn_parse.CfnParser({
            finder: {
                findResource(lId) {
                    return self.resources[lId];
                },
                findRefTarget(elementName) {
                    var _a;
                    return (_a = self.resources[elementName]) !== null && _a !== void 0 ? _a : self.parameters[elementName];
                },
                findCondition() {
                    return undefined;
                },
            },
        }).parseValue(this.template.Outputs[logicalId]);
        const cfnOutput = new core.CfnOutput(scope, logicalId, {
            value: outputAttributes.Value,
            description: outputAttributes.Description,
            exportName: outputAttributes.Export ? outputAttributes.Export.Name : undefined,
            condition: (() => {
                if (!outputAttributes.Condition) {
                    return undefined;
                }
                else if (this.conditions[outputAttributes.Condition]) {
                    return self.getCondition(outputAttributes.Condition);
                }
                throw new Error(`Output with name '${logicalId}' refers to a Condition with name ` +
                    `'${outputAttributes.Condition}' which was not found in this template`);
            })(),
        });
        cfnOutput.overrideLogicalId(logicalId);
        this.outputs[logicalId] = cfnOutput;
    }
    getOrCreateCondition(conditionName) {
        if (conditionName in this.conditions) {
            return this.conditions[conditionName];
        }
        const self = this;
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findResource() { throw new Error('Using GetAtt in Condition definitions is not allowed'); },
                findRefTarget(elementName) {
                    // only Parameters can be referenced in the 'Conditions' section
                    return self.parameters[elementName];
                },
                findCondition(cName) {
                    return cName in (self.template.Conditions || {})
                        ? self.getOrCreateCondition(cName)
                        : undefined;
                },
            },
            context: cfn_parse.CfnParsingContext.CONDITIONS,
        });
        const cfnCondition = new core.CfnCondition(this.conditionsScope, conditionName, {
            expression: cfnParser.parseValue(this.template.Conditions[conditionName]),
        });
        // ToDo handle renaming of the logical IDs of the conditions
        cfnCondition.overrideLogicalId(conditionName);
        this.conditions[conditionName] = cfnCondition;
        return cfnCondition;
    }
    getOrCreateResource(logicalId) {
        const ret = this.resources[logicalId];
        if (ret) {
            return ret;
        }
        const resourceAttributes = this.template.Resources[logicalId];
        // fail early for resource attributes we don't support yet
        const knownAttributes = [
            'Type', 'Properties', 'Condition', 'DependsOn', 'Metadata',
            'CreationPolicy', 'UpdatePolicy', 'DeletionPolicy', 'UpdateReplacePolicy',
        ];
        for (const attribute of Object.keys(resourceAttributes)) {
            if (!knownAttributes.includes(attribute)) {
                throw new Error(`The ${attribute} resource attribute is not supported by cloudformation-include yet. ` +
                    'Either remove it from the template, or use the CdkInclude class from the core package instead.');
            }
        }
        const self = this;
        const finder = {
            findCondition(conditionName) {
                return self.conditions[conditionName];
            },
            findResource(lId) {
                if (!(lId in (self.template.Resources || {}))) {
                    return undefined;
                }
                return self.getOrCreateResource(lId);
            },
            findRefTarget(elementName) {
                if (elementName in self.parameters) {
                    return self.parameters[elementName];
                }
                return this.findResource(elementName);
            },
        };
        const cfnParser = new cfn_parse.CfnParser({
            finder,
        });
        let l1Instance;
        if (this.nestedStacksToInclude[logicalId]) {
            l1Instance = this.createNestedStack(logicalId, cfnParser);
        }
        else {
            const l1ClassFqn = cfn_type_to_l1_mapping.lookup(resourceAttributes.Type);
            if (l1ClassFqn) {
                const options = {
                    finder,
                };
                const [moduleName, ...className] = l1ClassFqn.split('.');
                const module = require(moduleName); // eslint-disable-line @typescript-eslint/no-require-imports
                const jsClassFromModule = module[className.join('.')];
                l1Instance = jsClassFromModule.fromCloudFormation(this, logicalId, resourceAttributes, options);
            }
            else {
                l1Instance = new core.CfnResource(this, logicalId, {
                    type: resourceAttributes.Type,
                    properties: cfnParser.parseValue(resourceAttributes.Properties),
                });
                cfnParser.handleAttributes(l1Instance, resourceAttributes, logicalId);
            }
        }
        if (this.preserveLogicalIds) {
            // override the logical ID to match the original template
            l1Instance.overrideLogicalId(logicalId);
        }
        this.resources[logicalId] = l1Instance;
        return l1Instance;
    }
    createNestedStack(nestedStackId, cfnParser) {
        const templateResources = this.template.Resources || {};
        const nestedStackAttributes = templateResources[nestedStackId] || {};
        if (nestedStackAttributes.Type !== 'AWS::CloudFormation::Stack') {
            throw new Error(`Nested Stack with logical ID '${nestedStackId}' is not an AWS::CloudFormation::Stack resource`);
        }
        if (nestedStackAttributes.CreationPolicy) {
            throw new Error('CreationPolicy is not supported by the AWS::CloudFormation::Stack resource');
        }
        if (nestedStackAttributes.UpdatePolicy) {
            throw new Error('UpdatePolicy is not supported by the AWS::CloudFormation::Stack resource');
        }
        const nestedStackProps = cfnParser.parseValue(nestedStackAttributes.Properties);
        const nestedStack = new core.NestedStack(this, nestedStackId, {
            parameters: nestedStackProps.Parameters,
            notificationArns: nestedStackProps.NotificationArns,
            timeout: nestedStackProps.Timeout,
        });
        // we know this is never undefined for nested stacks
        const nestedStackResource = nestedStack.nestedStackResource;
        cfnParser.handleAttributes(nestedStackResource, nestedStackAttributes, nestedStackId);
        const propStack = this.nestedStacksToInclude[nestedStackId];
        const template = new CfnInclude(nestedStack, nestedStackId, {
            templateFile: propStack.templateFile,
            nestedStacks: propStack.nestedStacks,
        });
        const includedStack = { stack: nestedStack, includedTemplate: template };
        this.nestedStacks[nestedStackId] = includedStack;
        return nestedStackResource;
    }
}
exports.CfnInclude = CfnInclude;
//# sourceMappingURL=data:application/json;base64,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