"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Condition = void 0;
/**
 * A Condition for use in a Choice state branch
 */
class Condition {
    /**
     * Matches if a boolean field has the given value
     */
    static booleanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.BooleanEquals, value);
    }
    /**
     * Matches if a string field has the given value
     */
    static stringEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringEquals, value);
    }
    /**
     * Matches if a string field sorts before a given value
     */
    static stringLessThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringLessThan, value);
    }
    /**
     * Matches if a string field sorts equal to or before a given value
     */
    static stringLessThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringLessThanEquals, value);
    }
    /**
     * Matches if a string field sorts after a given value
     */
    static stringGreaterThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringGreaterThan, value);
    }
    /**
     * Matches if a string field sorts after or equal to a given value
     */
    static stringGreaterThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringGreaterThanEquals, value);
    }
    /**
     * Matches if a numeric field has the given value
     */
    static numberEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericEquals, value);
    }
    /**
     * Matches if a numeric field is less than the given value
     */
    static numberLessThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericLessThan, value);
    }
    /**
     * Matches if a numeric field is less than or equal to the given value
     */
    static numberLessThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericLessThanEquals, value);
    }
    /**
     * Matches if a numeric field is greater than the given value
     */
    static numberGreaterThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericGreaterThan, value);
    }
    /**
     * Matches if a numeric field is greater than or equal to the given value
     */
    static numberGreaterThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericGreaterThanEquals, value);
    }
    /**
     * Matches if a timestamp field is the same time as the given timestamp
     */
    static timestampEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampEquals, value);
    }
    /**
     * Matches if a timestamp field is before the given timestamp
     */
    static timestampLessThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampLessThan, value);
    }
    /**
     * Matches if a timestamp field is before or equal to the given timestamp
     */
    static timestampLessThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampLessThanEquals, value);
    }
    /**
     * Matches if a timestamp field is after the given timestamp
     */
    static timestampGreaterThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampGreaterThan, value);
    }
    /**
     * Matches if a timestamp field is after or equal to the given timestamp
     */
    static timestampGreaterThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampGreaterThanEquals, value);
    }
    /**
     * Combine two or more conditions with a logical AND
     */
    static and(...conditions) {
        return new CompoundCondition(CompoundOperator.And, ...conditions);
    }
    /**
     * Combine two or more conditions with a logical OR
     */
    static or(...conditions) {
        return new CompoundCondition(CompoundOperator.Or, ...conditions);
    }
    /**
     * Negate a condition
     */
    static not(condition) {
        return new NotCondition(condition);
    }
}
exports.Condition = Condition;
/**
 * Comparison Operator types
 */
var ComparisonOperator;
(function (ComparisonOperator) {
    ComparisonOperator[ComparisonOperator["StringEquals"] = 0] = "StringEquals";
    ComparisonOperator[ComparisonOperator["StringLessThan"] = 1] = "StringLessThan";
    ComparisonOperator[ComparisonOperator["StringGreaterThan"] = 2] = "StringGreaterThan";
    ComparisonOperator[ComparisonOperator["StringLessThanEquals"] = 3] = "StringLessThanEquals";
    ComparisonOperator[ComparisonOperator["StringGreaterThanEquals"] = 4] = "StringGreaterThanEquals";
    ComparisonOperator[ComparisonOperator["NumericEquals"] = 5] = "NumericEquals";
    ComparisonOperator[ComparisonOperator["NumericLessThan"] = 6] = "NumericLessThan";
    ComparisonOperator[ComparisonOperator["NumericGreaterThan"] = 7] = "NumericGreaterThan";
    ComparisonOperator[ComparisonOperator["NumericLessThanEquals"] = 8] = "NumericLessThanEquals";
    ComparisonOperator[ComparisonOperator["NumericGreaterThanEquals"] = 9] = "NumericGreaterThanEquals";
    ComparisonOperator[ComparisonOperator["BooleanEquals"] = 10] = "BooleanEquals";
    ComparisonOperator[ComparisonOperator["TimestampEquals"] = 11] = "TimestampEquals";
    ComparisonOperator[ComparisonOperator["TimestampLessThan"] = 12] = "TimestampLessThan";
    ComparisonOperator[ComparisonOperator["TimestampGreaterThan"] = 13] = "TimestampGreaterThan";
    ComparisonOperator[ComparisonOperator["TimestampLessThanEquals"] = 14] = "TimestampLessThanEquals";
    ComparisonOperator[ComparisonOperator["TimestampGreaterThanEquals"] = 15] = "TimestampGreaterThanEquals";
})(ComparisonOperator || (ComparisonOperator = {}));
/**
 * Compound Operator types
 */
var CompoundOperator;
(function (CompoundOperator) {
    CompoundOperator[CompoundOperator["And"] = 0] = "And";
    CompoundOperator[CompoundOperator["Or"] = 1] = "Or";
})(CompoundOperator || (CompoundOperator = {}));
/**
 * Scalar comparison
 */
class VariableComparison extends Condition {
    constructor(variable, comparisonOperator, value) {
        super();
        this.variable = variable;
        this.comparisonOperator = comparisonOperator;
        this.value = value;
        if (!/^\$|(\$[.[])/.test(variable)) {
            throw new Error(`Variable reference must be '$', start with '$.', or start with '$[', got '${variable}'`);
        }
    }
    renderCondition() {
        return {
            Variable: this.variable,
            [ComparisonOperator[this.comparisonOperator]]: this.value,
        };
    }
}
/**
 * Logical compound condition
 */
class CompoundCondition extends Condition {
    constructor(operator, ...conditions) {
        super();
        this.operator = operator;
        this.conditions = conditions;
        if (conditions.length === 0) {
            throw new Error('Must supply at least one inner condition for a logical combination');
        }
    }
    renderCondition() {
        return {
            [CompoundOperator[this.operator]]: this.conditions.map(c => c.renderCondition()),
        };
    }
}
/**
 * Logical unary condition
 */
class NotCondition extends Condition {
    constructor(comparisonOperation) {
        super();
        this.comparisonOperation = comparisonOperation;
    }
    renderCondition() {
        return {
            Not: this.comparisonOperation.renderCondition(),
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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