"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Runtime = exports.RuntimeFamily = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
var RuntimeFamily;
(function (RuntimeFamily) {
    RuntimeFamily[RuntimeFamily["NODEJS"] = 0] = "NODEJS";
    RuntimeFamily[RuntimeFamily["JAVA"] = 1] = "JAVA";
    RuntimeFamily[RuntimeFamily["PYTHON"] = 2] = "PYTHON";
    RuntimeFamily[RuntimeFamily["DOTNET_CORE"] = 3] = "DOTNET_CORE";
    RuntimeFamily[RuntimeFamily["GO"] = 4] = "GO";
    RuntimeFamily[RuntimeFamily["RUBY"] = 5] = "RUBY";
    RuntimeFamily[RuntimeFamily["OTHER"] = 6] = "OTHER";
})(RuntimeFamily = exports.RuntimeFamily || (exports.RuntimeFamily = {}));
/**
 * Lambda function runtime environment.
 *
 * If you need to use a runtime name that doesn't exist as a static member, you
 * can instantiate a `Runtime` object, e.g: `new Runtime('nodejs99.99')`.
 */
class Runtime {
    constructor(name, family, props = {}) {
        this.name = name;
        this.supportsInlineCode = !!props.supportsInlineCode;
        this.family = family;
        this.bundlingDockerImage = core_1.BundlingDockerImage.fromRegistry(`amazon/aws-sam-cli-build-image-${name}`);
        Runtime.ALL.push(this);
    }
    toString() {
        return this.name;
    }
    runtimeEquals(other) {
        return other.name === this.name &&
            other.family === this.family &&
            other.supportsInlineCode === this.supportsInlineCode;
    }
}
exports.Runtime = Runtime;
/** A list of all known `Runtime`'s. */
Runtime.ALL = new Array();
/**
 * The NodeJS runtime (nodejs)
 *
 * @deprecated Use {@link NODEJS_10_X}
 */
Runtime.NODEJS = new Runtime('nodejs', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The NodeJS 4.3 runtime (nodejs4.3)
 *
 * @deprecated Use {@link NODEJS_10_X}
 */
Runtime.NODEJS_4_3 = new Runtime('nodejs4.3', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The NodeJS 6.10 runtime (nodejs6.10)
 *
 * @deprecated Use {@link NODEJS_10_X}
 */
Runtime.NODEJS_6_10 = new Runtime('nodejs6.10', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The NodeJS 8.10 runtime (nodejs8.10)
 *
 * @deprecated Use {@link NODEJS_10_X}
 */
Runtime.NODEJS_8_10 = new Runtime('nodejs8.10', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The NodeJS 10.x runtime (nodejs10.x)
 */
Runtime.NODEJS_10_X = new Runtime('nodejs10.x', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The NodeJS 12.x runtime (nodejs12.x)
 */
Runtime.NODEJS_12_X = new Runtime('nodejs12.x', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The Python 2.7 runtime (python2.7)
 */
Runtime.PYTHON_2_7 = new Runtime('python2.7', RuntimeFamily.PYTHON, { supportsInlineCode: true });
/**
 * The Python 3.6 runtime (python3.6)
 */
Runtime.PYTHON_3_6 = new Runtime('python3.6', RuntimeFamily.PYTHON, { supportsInlineCode: true });
/**
 * The Python 3.7 runtime (python3.7)
 */
Runtime.PYTHON_3_7 = new Runtime('python3.7', RuntimeFamily.PYTHON, { supportsInlineCode: true });
/**
 * The Python 3.8 runtime (python3.8)
 */
Runtime.PYTHON_3_8 = new Runtime('python3.8', RuntimeFamily.PYTHON);
/**
 * The Java 8 runtime (java8)
 */
Runtime.JAVA_8 = new Runtime('java8', RuntimeFamily.JAVA);
/**
 * The Java 11 runtime (java11)
 */
Runtime.JAVA_11 = new Runtime('java11', RuntimeFamily.JAVA);
/**
 * The .NET Core 1.0 runtime (dotnetcore1.0)
 *
 * @deprecated Use {@link DOTNET_CORE_2_1}
 */
Runtime.DOTNET_CORE_1 = new Runtime('dotnetcore1.0', RuntimeFamily.DOTNET_CORE);
/**
 * The .NET Core 2.0 runtime (dotnetcore2.0)
 *
 * @deprecated Use {@link DOTNET_CORE_2_1}
 */
Runtime.DOTNET_CORE_2 = new Runtime('dotnetcore2.0', RuntimeFamily.DOTNET_CORE);
/**
 * The .NET Core 2.1 runtime (dotnetcore2.1)
 */
Runtime.DOTNET_CORE_2_1 = new Runtime('dotnetcore2.1', RuntimeFamily.DOTNET_CORE);
/**
 * The .NET Core 3.1 runtime (dotnetcore3.1)
 */
Runtime.DOTNET_CORE_3_1 = new Runtime('dotnetcore3.1', RuntimeFamily.DOTNET_CORE);
/**
 * The Go 1.x runtime (go1.x)
 */
Runtime.GO_1_X = new Runtime('go1.x', RuntimeFamily.GO);
/**
 * The Ruby 2.5 runtime (ruby2.5)
 */
Runtime.RUBY_2_5 = new Runtime('ruby2.5', RuntimeFamily.RUBY);
/**
 * The Ruby 2.7 runtime (ruby2.7)
 */
Runtime.RUBY_2_7 = new Runtime('ruby2.7', RuntimeFamily.RUBY);
/**
 * The custom provided runtime (provided)
 */
Runtime.PROVIDED = new Runtime('provided', RuntimeFamily.OTHER);
//# sourceMappingURL=data:application/json;base64,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