"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaDestination = void 0;
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const targets = require("../../aws-events-targets"); // Automatically re-written from '@aws-cdk/aws-events-targets'
const event_bridge_1 = require("./event-bridge");
/**
 * Use a Lambda function as a Lambda destination
 */
class LambdaDestination {
    constructor(fn, options = {}) {
        this.fn = fn;
        this.options = options;
    }
    /**
     * Returns a destination configuration
     */
    bind(scope, fn, options) {
        // Normal Lambda destination (full invocation record)
        if (!this.options.responseOnly) {
            // deduplicated automatically
            this.fn.grantInvoke(fn);
            return {
                destination: this.fn.functionArn,
            };
        }
        // Otherwise add rule to extract the response payload and use EventBridge
        // as destination
        if (!options) { // `options` added to bind() as optionnal to avoid breaking change
            throw new Error('Options must be defined when using `responseOnly`.');
        }
        // Match invocation result of the source function (`fn`) and use it
        // to trigger the destination function (`this.fn`).
        new events.Rule(scope, options.type, {
            eventPattern: {
                detailType: [`Lambda Function Invocation Result - ${options.type}`],
                resources: [`${fn.functionArn}:$LATEST`],
                source: ['lambda'],
            },
            targets: [
                new targets.LambdaFunction(this.fn, {
                    event: events.RuleTargetInput.fromEventPath('$.detail.responsePayload'),
                }),
            ],
        });
        const destination = new event_bridge_1.EventBridgeDestination(); // Use default event bus here
        return destination.bind(scope, fn);
    }
}
exports.LambdaDestination = LambdaDestination;
//# sourceMappingURL=data:application/json;base64,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