"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DataFormat = exports.SerializationLibrary = exports.OutputFormat = exports.InputFormat = void 0;
/**
 * Absolute class name of the Hadoop `InputFormat` to use when reading table files.
 */
class InputFormat {
    constructor(className) {
        this.className = className;
    }
}
exports.InputFormat = InputFormat;
/**
 * InputFormat for Avro files.
 *
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/avro/AvroContainerInputFormat.html
 */
InputFormat.AVRO = new InputFormat('org.apache.hadoop.hive.ql.io.avro.AvroContainerInputFormat');
/**
 * InputFormat for Cloudtrail Logs.
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/cloudtrail.html
 */
InputFormat.CLOUDTRAIL = new InputFormat('com.amazon.emr.cloudtrail.CloudTrailInputFormat');
/**
 * InputFormat for Orc files.
 *
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/orc/OrcInputFormat.html
 */
InputFormat.ORC = new InputFormat('org.apache.hadoop.hive.ql.io.orc.OrcInputFormat');
/**
 * InputFormat for Parquet files.
 *
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/parquet/MapredParquetInputFormat.html
 */
InputFormat.PARQUET = new InputFormat('org.apache.hadoop.hive.ql.io.parquet.MapredParquetInputFormat');
/**
 * An InputFormat for plain text files. Files are broken into lines. Either linefeed or
 * carriage-return are used to signal end of line. Keys are the position in the file, and
 * values are the line of text.
 * JSON & CSV files are examples of this InputFormat
 *
 * @see https://hadoop.apache.org/docs/stable/api/org/apache/hadoop/mapred/TextInputFormat.html
 */
InputFormat.TEXT = new InputFormat('org.apache.hadoop.mapred.TextInputFormat');
/**
 * Absolute class name of the Hadoop `OutputFormat` to use when writing table files.
 */
class OutputFormat {
    constructor(className) {
        this.className = className;
    }
}
exports.OutputFormat = OutputFormat;
/**
 * Writes text data with a null key (value only).
 *
 * @see https://hive.apache.org/javadocs/r2.2.0/api/org/apache/hadoop/hive/ql/io/HiveIgnoreKeyTextOutputFormat.html
 */
OutputFormat.HIVE_IGNORE_KEY_TEXT = new OutputFormat('org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat');
/**
 * OutputFormat for Avro files.
 *
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/avro/AvroContainerOutputFormat.html
 */
OutputFormat.AVRO = new InputFormat('org.apache.hadoop.hive.ql.io.avro.AvroContainerOutputFormat');
/**
 * OutputFormat for Orc files.
 *
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/orc/OrcOutputFormat.html
 */
OutputFormat.ORC = new InputFormat('org.apache.hadoop.hive.ql.io.orc.OrcOutputFormat');
/**
 * OutputFormat for Parquet files.
 *
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/parquet/MapredParquetOutputFormat.html
 */
OutputFormat.PARQUET = new OutputFormat('org.apache.hadoop.hive.ql.io.parquet.MapredParquetOutputFormat');
/**
 * Serialization library to use when serializing/deserializing (SerDe) table records.
 *
 * @see https://cwiki.apache.org/confluence/display/Hive/SerDe
 */
class SerializationLibrary {
    constructor(className) {
        this.className = className;
    }
}
exports.SerializationLibrary = SerializationLibrary;
/**
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/serde2/avro/AvroSerDe.html
 */
SerializationLibrary.AVRO = new SerializationLibrary('org.apache.hadoop.hive.serde2.avro.AvroSerDe');
/**
 * @see https://docs.aws.amazon.com/athena/latest/ug/cloudtrail.html
 */
SerializationLibrary.CLOUDTRAIL = new SerializationLibrary('com.amazon.emr.hive.serde.CloudTrailSerde');
/**
 * @see https://docs.aws.amazon.com/athena/latest/ug/grok.html
 */
SerializationLibrary.GROK = new SerializationLibrary('com.amazonaws.glue.serde.GrokSerDe');
/**
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hive/hcatalog/data/JsonSerDe.html
 */
SerializationLibrary.HIVE_JSON = new SerializationLibrary('org.apache.hive.hcatalog.data.JsonSerDe');
/**
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/serde2/lazy/LazySimpleSerDe.html
 */
SerializationLibrary.LAZY_SIMPLE = new SerializationLibrary('org.apache.hadoop.hive.serde2.lazy.LazySimpleSerDe');
/**
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/serde2/OpenCSVSerde.html
 */
SerializationLibrary.OPEN_CSV = new SerializationLibrary('org.apache.hadoop.hive.serde2.OpenCSVSerde');
/**
 * @see https://github.com/rcongiu/Hive-JSON-Serde
 */
SerializationLibrary.OPENX_JSON = new SerializationLibrary('org.openx.data.jsonserde.JsonSerDe');
/**
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/orc/OrcSerde.html
 */
SerializationLibrary.ORC = new SerializationLibrary('org.apache.hadoop.hive.ql.io.orc.OrcSerde');
/**
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/parquet/serde/ParquetHiveSerDe.html
 */
SerializationLibrary.PARQUET = new SerializationLibrary('org.apache.hadoop.hive.ql.io.parquet.serde.ParquetHiveSerDe');
/**
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/serde2/RegexSerDe.html
 */
SerializationLibrary.REGEXP = new SerializationLibrary('org.apache.hadoop.hive.serde2.RegexSerDe');
/**
 * Defines the input/output formats and ser/de for a single DataFormat.
 */
class DataFormat {
    constructor(props) {
        this.inputFormat = props.inputFormat;
        this.outputFormat = props.outputFormat;
        this.serializationLibrary = props.serializationLibrary;
    }
}
exports.DataFormat = DataFormat;
/**
 * DataFormat for Apache Web Server Logs. Also works for CloudFront logs
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/apache.html
 */
DataFormat.APACHE_LOGS = new DataFormat({
    inputFormat: InputFormat.TEXT,
    outputFormat: OutputFormat.HIVE_IGNORE_KEY_TEXT,
    serializationLibrary: SerializationLibrary.REGEXP,
});
/**
 * DataFormat for Apache Avro
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/avro.html
 */
DataFormat.AVRO = new DataFormat({
    inputFormat: InputFormat.AVRO,
    outputFormat: OutputFormat.AVRO,
    serializationLibrary: SerializationLibrary.AVRO,
});
/**
 * DataFormat for CloudTrail logs stored on S3
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/cloudtrail.html
 */
DataFormat.CLOUDTRAIL_LOGS = new DataFormat({
    inputFormat: InputFormat.CLOUDTRAIL,
    outputFormat: OutputFormat.HIVE_IGNORE_KEY_TEXT,
    serializationLibrary: SerializationLibrary.CLOUDTRAIL,
});
/**
 * DataFormat for CSV Files
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/csv.html
 */
DataFormat.CSV = new DataFormat({
    inputFormat: InputFormat.TEXT,
    outputFormat: OutputFormat.HIVE_IGNORE_KEY_TEXT,
    serializationLibrary: SerializationLibrary.OPEN_CSV,
});
/**
 * Stored as plain text files in JSON format.
 * Uses OpenX Json SerDe for serialization and deseralization.
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/json.html
 */
DataFormat.JSON = new DataFormat({
    inputFormat: InputFormat.TEXT,
    outputFormat: OutputFormat.HIVE_IGNORE_KEY_TEXT,
    serializationLibrary: SerializationLibrary.OPENX_JSON,
});
/**
 * DataFormat for Logstash Logs, using the GROK SerDe
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/grok.html
 */
DataFormat.LOGSTASH = new DataFormat({
    inputFormat: InputFormat.TEXT,
    outputFormat: OutputFormat.HIVE_IGNORE_KEY_TEXT,
    serializationLibrary: SerializationLibrary.GROK,
});
/**
 * DataFormat for Apache ORC (Optimized Row Columnar)
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/orc.html
 */
DataFormat.ORC = new DataFormat({
    inputFormat: InputFormat.ORC,
    outputFormat: OutputFormat.ORC,
    serializationLibrary: SerializationLibrary.ORC,
});
/**
 * DataFormat for Apache Parquet
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/parquet.html
 */
DataFormat.PARQUET = new DataFormat({
    inputFormat: InputFormat.PARQUET,
    outputFormat: OutputFormat.PARQUET,
    serializationLibrary: SerializationLibrary.PARQUET,
});
/**
 * DataFormat for TSV (Tab-Separated Values)
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/lazy-simple-serde.html
 */
DataFormat.TSV = new DataFormat({
    inputFormat: InputFormat.TEXT,
    outputFormat: OutputFormat.HIVE_IGNORE_KEY_TEXT,
    serializationLibrary: SerializationLibrary.LAZY_SIMPLE,
});
//# sourceMappingURL=data:application/json;base64,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