"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RepositoryImage = void 0;
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const container_image_1 = require("../container-image");
/**
 * Regex pattern to check if it is an ECR image URL.
 *
 * @experimental
 */
const ECR_IMAGE_REGEX = /(^[a-zA-Z0-9][a-zA-Z0-9-_]*).dkr.ecr.([a-zA-Z0-9][a-zA-Z0-9-_]*).amazonaws.com(.cn)?\/.*/;
/**
 * An image hosted in a public or private repository. For images hosted in Amazon ECR, see
 * [EcrImage](https://docs.aws.amazon.com/AmazonECR/latest/userguide/images.html).
 */
class RepositoryImage extends container_image_1.ContainerImage {
    /**
     * Constructs a new instance of the RepositoryImage class.
     */
    constructor(imageName, props = {}) {
        super();
        this.imageName = imageName;
        this.props = props;
    }
    bind(scope, containerDefinition) {
        // name could be a Token - in that case, skip validation altogether
        if (!core_1.Token.isUnresolved(this.imageName) && ECR_IMAGE_REGEX.test(this.imageName)) {
            scope.node.addWarning("Proper policies need to be attached before pulling from ECR repository, or use 'fromEcrRepository'.");
        }
        if (this.props.credentials) {
            this.props.credentials.grantRead(containerDefinition.taskDefinition.obtainExecutionRole());
        }
        return {
            imageName: this.imageName,
            repositoryCredentials: this.props.credentials && {
                credentialsParameter: this.props.credentials.secretArn,
            },
        };
    }
}
exports.RepositoryImage = RepositoryImage;
//# sourceMappingURL=data:application/json;base64,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