"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cache = exports.LocalCacheMode = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Local cache modes to enable for the CodeBuild Project
 */
var LocalCacheMode;
(function (LocalCacheMode) {
    /**
     * Caches Git metadata for primary and secondary sources
     */
    LocalCacheMode["SOURCE"] = "LOCAL_SOURCE_CACHE";
    /**
     * Caches existing Docker layers
     */
    LocalCacheMode["DOCKER_LAYER"] = "LOCAL_DOCKER_LAYER_CACHE";
    /**
     * Caches directories you specify in the buildspec file
     */
    LocalCacheMode["CUSTOM"] = "LOCAL_CUSTOM_CACHE";
})(LocalCacheMode = exports.LocalCacheMode || (exports.LocalCacheMode = {}));
/**
 * Cache options for CodeBuild Project.
 * A cache can store reusable pieces of your build environment and use them across multiple builds.
 * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-caching.html
 */
class Cache {
    static none() {
        return { _toCloudFormation: () => undefined, _bind: () => { return; } };
    }
    /**
     * Create a local caching strategy.
     * @param modes the mode(s) to enable for local caching
     */
    static local(...modes) {
        return {
            _toCloudFormation: () => ({
                type: 'LOCAL',
                modes,
            }),
            _bind: () => { return; },
        };
    }
    /**
     * Create an S3 caching strategy.
     * @param bucket the S3 bucket to use for caching
     * @param options additional options to pass to the S3 caching
     */
    static bucket(bucket, options) {
        return {
            _toCloudFormation: () => ({
                type: 'S3',
                location: core_1.Fn.join('/', [bucket.bucketName, options && options.prefix || core_1.Aws.NO_VALUE]),
            }),
            _bind: (project) => {
                bucket.grantReadWrite(project);
            },
        };
    }
}
exports.Cache = Cache;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2FjaGUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJjYWNoZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFDQSxxQ0FBcUMsQ0FBQyxnREFBZ0Q7QUFTdEY7O0dBRUc7QUFDSCxJQUFZLGNBYVg7QUFiRCxXQUFZLGNBQWM7SUFDdEI7O09BRUc7SUFDSCwrQ0FBNkIsQ0FBQTtJQUM3Qjs7T0FFRztJQUNILDJEQUF5QyxDQUFBO0lBQ3pDOztPQUVHO0lBQ0gsK0NBQTZCLENBQUE7QUFDakMsQ0FBQyxFQWJXLGNBQWMsR0FBZCxzQkFBYyxLQUFkLHNCQUFjLFFBYXpCO0FBQ0Q7Ozs7R0FJRztBQUNILE1BQXNCLEtBQUs7SUFDaEIsTUFBTSxDQUFDLElBQUk7UUFDZCxPQUFPLEVBQUUsaUJBQWlCLEVBQUUsR0FBRyxFQUFFLENBQUMsU0FBUyxFQUFFLEtBQUssRUFBRSxHQUFHLEVBQUUsR0FBRyxPQUFPLENBQUMsQ0FBQyxFQUFFLENBQUM7SUFDNUUsQ0FBQztJQUNEOzs7T0FHRztJQUNJLE1BQU0sQ0FBQyxLQUFLLENBQUMsR0FBRyxLQUF1QjtRQUMxQyxPQUFPO1lBQ0gsaUJBQWlCLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQztnQkFDdEIsSUFBSSxFQUFFLE9BQU87Z0JBQ2IsS0FBSzthQUNSLENBQUM7WUFDRixLQUFLLEVBQUUsR0FBRyxFQUFFLEdBQUcsT0FBTyxDQUFDLENBQUM7U0FDM0IsQ0FBQztJQUNOLENBQUM7SUFDRDs7OztPQUlHO0lBQ0ksTUFBTSxDQUFDLE1BQU0sQ0FBQyxNQUFlLEVBQUUsT0FBNEI7UUFDOUQsT0FBTztZQUNILGlCQUFpQixFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUM7Z0JBQ3RCLElBQUksRUFBRSxJQUFJO2dCQUNWLFFBQVEsRUFBRSxTQUFFLENBQUMsSUFBSSxDQUFDLEdBQUcsRUFBRSxDQUFDLE1BQU0sQ0FBQyxVQUFVLEVBQUUsT0FBTyxJQUFJLE9BQU8sQ0FBQyxNQUFNLElBQUksVUFBRyxDQUFDLFFBQVEsQ0FBQyxDQUFDO2FBQ3pGLENBQUM7WUFDRixLQUFLLEVBQUUsQ0FBQyxPQUFPLEVBQUUsRUFBRTtnQkFDZixNQUFNLENBQUMsY0FBYyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1lBQ25DLENBQUM7U0FDSixDQUFDO0lBQ04sQ0FBQztDQVNKO0FBekNELHNCQXlDQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IElCdWNrZXQgfSBmcm9tIFwiLi4vLi4vYXdzLXMzXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3MtczMnXG5pbXBvcnQgeyBBd3MsIEZuIH0gZnJvbSBcIi4uLy4uL2NvcmVcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2NvcmUnXG5pbXBvcnQgeyBDZm5Qcm9qZWN0IH0gZnJvbSAnLi9jb2RlYnVpbGQuZ2VuZXJhdGVkJztcbmltcG9ydCB7IElQcm9qZWN0IH0gZnJvbSAnLi9wcm9qZWN0JztcbmV4cG9ydCBpbnRlcmZhY2UgQnVja2V0Q2FjaGVPcHRpb25zIHtcbiAgICAvKipcbiAgICAgKiBUaGUgcHJlZml4IHRvIHVzZSB0byBzdG9yZSB0aGUgY2FjaGUgaW4gdGhlIGJ1Y2tldFxuICAgICAqL1xuICAgIHJlYWRvbmx5IHByZWZpeD86IHN0cmluZztcbn1cbi8qKlxuICogTG9jYWwgY2FjaGUgbW9kZXMgdG8gZW5hYmxlIGZvciB0aGUgQ29kZUJ1aWxkIFByb2plY3RcbiAqL1xuZXhwb3J0IGVudW0gTG9jYWxDYWNoZU1vZGUge1xuICAgIC8qKlxuICAgICAqIENhY2hlcyBHaXQgbWV0YWRhdGEgZm9yIHByaW1hcnkgYW5kIHNlY29uZGFyeSBzb3VyY2VzXG4gICAgICovXG4gICAgU09VUkNFID0gJ0xPQ0FMX1NPVVJDRV9DQUNIRScsXG4gICAgLyoqXG4gICAgICogQ2FjaGVzIGV4aXN0aW5nIERvY2tlciBsYXllcnNcbiAgICAgKi9cbiAgICBET0NLRVJfTEFZRVIgPSAnTE9DQUxfRE9DS0VSX0xBWUVSX0NBQ0hFJyxcbiAgICAvKipcbiAgICAgKiBDYWNoZXMgZGlyZWN0b3JpZXMgeW91IHNwZWNpZnkgaW4gdGhlIGJ1aWxkc3BlYyBmaWxlXG4gICAgICovXG4gICAgQ1VTVE9NID0gJ0xPQ0FMX0NVU1RPTV9DQUNIRSdcbn1cbi8qKlxuICogQ2FjaGUgb3B0aW9ucyBmb3IgQ29kZUJ1aWxkIFByb2plY3QuXG4gKiBBIGNhY2hlIGNhbiBzdG9yZSByZXVzYWJsZSBwaWVjZXMgb2YgeW91ciBidWlsZCBlbnZpcm9ubWVudCBhbmQgdXNlIHRoZW0gYWNyb3NzIG11bHRpcGxlIGJ1aWxkcy5cbiAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZGVidWlsZC9sYXRlc3QvdXNlcmd1aWRlL2J1aWxkLWNhY2hpbmcuaHRtbFxuICovXG5leHBvcnQgYWJzdHJhY3QgY2xhc3MgQ2FjaGUge1xuICAgIHB1YmxpYyBzdGF0aWMgbm9uZSgpOiBDYWNoZSB7XG4gICAgICAgIHJldHVybiB7IF90b0Nsb3VkRm9ybWF0aW9uOiAoKSA9PiB1bmRlZmluZWQsIF9iaW5kOiAoKSA9PiB7IHJldHVybjsgfSB9O1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBDcmVhdGUgYSBsb2NhbCBjYWNoaW5nIHN0cmF0ZWd5LlxuICAgICAqIEBwYXJhbSBtb2RlcyB0aGUgbW9kZShzKSB0byBlbmFibGUgZm9yIGxvY2FsIGNhY2hpbmdcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGxvY2FsKC4uLm1vZGVzOiBMb2NhbENhY2hlTW9kZVtdKTogQ2FjaGUge1xuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgICAgX3RvQ2xvdWRGb3JtYXRpb246ICgpID0+ICh7XG4gICAgICAgICAgICAgICAgdHlwZTogJ0xPQ0FMJyxcbiAgICAgICAgICAgICAgICBtb2RlcyxcbiAgICAgICAgICAgIH0pLFxuICAgICAgICAgICAgX2JpbmQ6ICgpID0+IHsgcmV0dXJuOyB9LFxuICAgICAgICB9O1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBDcmVhdGUgYW4gUzMgY2FjaGluZyBzdHJhdGVneS5cbiAgICAgKiBAcGFyYW0gYnVja2V0IHRoZSBTMyBidWNrZXQgdG8gdXNlIGZvciBjYWNoaW5nXG4gICAgICogQHBhcmFtIG9wdGlvbnMgYWRkaXRpb25hbCBvcHRpb25zIHRvIHBhc3MgdG8gdGhlIFMzIGNhY2hpbmdcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGJ1Y2tldChidWNrZXQ6IElCdWNrZXQsIG9wdGlvbnM/OiBCdWNrZXRDYWNoZU9wdGlvbnMpOiBDYWNoZSB7XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICBfdG9DbG91ZEZvcm1hdGlvbjogKCkgPT4gKHtcbiAgICAgICAgICAgICAgICB0eXBlOiAnUzMnLFxuICAgICAgICAgICAgICAgIGxvY2F0aW9uOiBGbi5qb2luKCcvJywgW2J1Y2tldC5idWNrZXROYW1lLCBvcHRpb25zICYmIG9wdGlvbnMucHJlZml4IHx8IEF3cy5OT19WQUxVRV0pLFxuICAgICAgICAgICAgfSksXG4gICAgICAgICAgICBfYmluZDogKHByb2plY3QpID0+IHtcbiAgICAgICAgICAgICAgICBidWNrZXQuZ3JhbnRSZWFkV3JpdGUocHJvamVjdCk7XG4gICAgICAgICAgICB9LFxuICAgICAgICB9O1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBAaW50ZXJuYWxcbiAgICAgKi9cbiAgICBwdWJsaWMgYWJzdHJhY3QgX3RvQ2xvdWRGb3JtYXRpb24oKTogQ2ZuUHJvamVjdC5Qcm9qZWN0Q2FjaGVQcm9wZXJ0eSB8IHVuZGVmaW5lZDtcbiAgICAvKipcbiAgICAgKiBAaW50ZXJuYWxcbiAgICAgKi9cbiAgICBwdWJsaWMgYWJzdHJhY3QgX2JpbmQocHJvamVjdDogSVByb2plY3QpOiB2b2lkO1xufVxuIl19