"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OriginBase = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Represents a distribution origin, that describes the Amazon S3 bucket, HTTP server (for example, a web server),
 * Amazon MediaStore, or other server from which CloudFront gets your files.
 *
 * @experimental
 */
class OriginBase {
    constructor(domainName, props = {}) {
        var _a;
        validateIntInRangeOrUndefined('connectionTimeout', 1, 10, (_a = props.connectionTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds());
        validateIntInRangeOrUndefined('connectionAttempts', 1, 3, props.connectionAttempts, false);
        this.domainName = domainName;
        this.originPath = this.validateOriginPath(props.originPath);
        this.connectionTimeout = props.connectionTimeout;
        this.connectionAttempts = props.connectionAttempts;
        this.customHeaders = props.customHeaders;
    }
    /**
     * The unique id for this origin.
     *
     * Cannot be accesed until bind() is called.
     */
    get id() {
        if (!this.originId) {
            throw new Error('Cannot access originId until `bind` is called.');
        }
        return this.originId;
    }
    /**
     * Binds the origin to the associated Distribution. Can be used to grant permissions, create dependent resources, etc.
     */
    bind(_scope, options) {
        var _a;
        this.originId = options.originId;
        const s3OriginConfig = this.renderS3OriginConfig();
        const customOriginConfig = this.renderCustomOriginConfig();
        if (!s3OriginConfig && !customOriginConfig) {
            throw new Error('Subclass must override and provide either s3OriginConfig or customOriginConfig');
        }
        return { originProperty: {
                domainName: this.domainName,
                id: this.id,
                originPath: this.originPath,
                connectionAttempts: this.connectionAttempts,
                connectionTimeout: (_a = this.connectionTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds(),
                originCustomHeaders: this.renderCustomHeaders(),
                s3OriginConfig,
                customOriginConfig,
            } };
    }
    // Overridden by sub-classes to provide S3 origin config.
    renderS3OriginConfig() {
        return undefined;
    }
    // Overridden by sub-classes to provide custom origin config.
    renderCustomOriginConfig() {
        return undefined;
    }
    renderCustomHeaders() {
        if (!this.customHeaders || Object.entries(this.customHeaders).length === 0) {
            return undefined;
        }
        return Object.entries(this.customHeaders).map(([headerName, headerValue]) => {
            return { headerName, headerValue };
        });
    }
    /**
     * If the path is defined, it must start with a '/' and not end with a '/'.
     * This method takes in the originPath, and returns it back (if undefined) or adds/removes the '/' as appropriate.
     */
    validateOriginPath(originPath) {
        if (core_1.Token.isUnresolved(originPath)) {
            return originPath;
        }
        if (originPath === undefined) {
            return undefined;
        }
        let path = originPath;
        if (!path.startsWith('/')) {
            path = '/' + path;
        }
        if (path.endsWith('/')) {
            path = path.substr(0, path.length - 1);
        }
        return path;
    }
}
exports.OriginBase = OriginBase;
/**
 * Throws an error if a value is defined and not an integer or not in a range.
 */
function validateIntInRangeOrUndefined(name, min, max, value, isDuration = true) {
    if (value === undefined) {
        return;
    }
    if (!Number.isInteger(value) || value < min || value > max) {
        const seconds = isDuration ? ' seconds' : '';
        throw new Error(`${name}: Must be an int between ${min} and ${max}${seconds} (inclusive); received ${value}.`);
    }
}
//# sourceMappingURL=data:application/json;base64,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