"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaEdgeEventType = exports.AllowedMethods = exports.SecurityPolicyProtocol = exports.SSLMethod = exports.OriginProtocolPolicy = exports.ViewerProtocolPolicy = exports.PriceClass = exports.Distribution = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cloudfront_generated_1 = require("./cloudfront.generated");
const cache_behavior_1 = require("./private/cache-behavior");
/**
 * A CloudFront distribution with associated origin(s) and caching behavior(s).
 *
 * @experimental
 */
class Distribution extends core_1.Resource {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        this.additionalBehaviors = [];
        this.boundOrigins = [];
        if (props.certificate) {
            const certificateRegion = core_1.Stack.of(this).parseArn(props.certificate.certificateArn).region;
            if (!core_1.Token.isUnresolved(certificateRegion) && certificateRegion !== 'us-east-1') {
                throw new Error(`Distribution certificates must be in the us-east-1 region and the certificate you provided is in ${certificateRegion}.`);
            }
        }
        const originId = this.addOrigin(props.defaultBehavior.origin);
        this.defaultBehavior = new cache_behavior_1.CacheBehavior(originId, { pathPattern: '*', ...props.defaultBehavior });
        if (props.additionalBehaviors) {
            Object.entries(props.additionalBehaviors).forEach(([pathPattern, behaviorOptions]) => {
                this.addBehavior(pathPattern, behaviorOptions.origin, behaviorOptions);
            });
        }
        this.certificate = props.certificate;
        this.errorResponses = (_a = props.errorResponses) !== null && _a !== void 0 ? _a : [];
        const distribution = new cloudfront_generated_1.CfnDistribution(this, 'CFDistribution', { distributionConfig: {
                enabled: true,
                origins: core_1.Lazy.anyValue({ produce: () => this.renderOrigins() }),
                defaultCacheBehavior: this.defaultBehavior._renderBehavior(),
                cacheBehaviors: core_1.Lazy.anyValue({ produce: () => this.renderCacheBehaviors() }),
                viewerCertificate: this.certificate ? this.renderViewerCertificate(this.certificate) : undefined,
                customErrorResponses: this.renderErrorResponses(),
                priceClass: (_b = props.priceClass) !== null && _b !== void 0 ? _b : undefined,
            } });
        this.domainName = distribution.attrDomainName;
        this.distributionDomainName = distribution.attrDomainName;
        this.distributionId = distribution.ref;
    }
    /**
     * Creates a Distribution construct that represents an external (imported) distribution.
     */
    static fromDistributionAttributes(scope, id, attrs) {
        return new class extends core_1.Resource {
            constructor() {
                super(scope, id);
                this.domainName = attrs.domainName;
                this.distributionDomainName = attrs.domainName;
                this.distributionId = attrs.distributionId;
            }
        }();
    }
    /**
     * Adds a new behavior to this distribution for the given pathPattern.
     *
     * @param pathPattern the path pattern (e.g., 'images/*') that specifies which requests to apply the behavior to.
     * @param origin the origin to use for this behavior
     * @param behaviorOptions the options for the behavior at this path.
     */
    addBehavior(pathPattern, origin, behaviorOptions = {}) {
        if (pathPattern === '*') {
            throw new Error('Only the default behavior can have a path pattern of \'*\'');
        }
        const originId = this.addOrigin(origin);
        this.additionalBehaviors.push(new cache_behavior_1.CacheBehavior(originId, { pathPattern, ...behaviorOptions }));
    }
    addOrigin(origin) {
        const existingOrigin = this.boundOrigins.find(boundOrigin => boundOrigin.origin === origin);
        if (existingOrigin) {
            return existingOrigin.originId;
        }
        else {
            const originIndex = this.boundOrigins.length + 1;
            const scope = new core_1.Construct(this, `Origin${originIndex}`);
            const originId = scope.node.uniqueId;
            const originBindConfig = origin.bind(scope, { originId });
            this.boundOrigins.push({ origin, originId, ...originBindConfig });
            return originId;
        }
    }
    renderOrigins() {
        const renderedOrigins = [];
        this.boundOrigins.forEach(boundOrigin => {
            if (boundOrigin.originProperty) {
                renderedOrigins.push(boundOrigin.originProperty);
            }
        });
        return renderedOrigins;
    }
    renderCacheBehaviors() {
        if (this.additionalBehaviors.length === 0) {
            return undefined;
        }
        return this.additionalBehaviors.map(behavior => behavior._renderBehavior());
    }
    renderErrorResponses() {
        if (this.errorResponses.length === 0) {
            return undefined;
        }
        function validateCustomErrorResponse(errorResponse) {
            if (errorResponse.responsePagePath && !errorResponse.responseHttpStatus) {
                throw new Error('\'responseCode\' must be provided if \'responsePagePath\' is defined');
            }
            if (!errorResponse.responseHttpStatus && !errorResponse.ttl) {
                throw new Error('A custom error response without either a \'responseCode\' or \'errorCachingMinTtl\' is not valid.');
            }
        }
        this.errorResponses.forEach(e => validateCustomErrorResponse(e));
        return this.errorResponses.map(errorConfig => {
            var _a;
            return {
                errorCachingMinTtl: (_a = errorConfig.ttl) === null || _a === void 0 ? void 0 : _a.toSeconds(),
                errorCode: errorConfig.httpStatus,
                responseCode: errorConfig.responseHttpStatus,
                responsePagePath: errorConfig.responsePagePath,
            };
        });
    }
    renderViewerCertificate(certificate) {
        return {
            acmCertificateArn: certificate.certificateArn,
            sslSupportMethod: SSLMethod.SNI,
            minimumProtocolVersion: SecurityPolicyProtocol.TLS_V1_2_2018,
        };
    }
}
exports.Distribution = Distribution;
/**
 * The price class determines how many edge locations CloudFront will use for your distribution.
 * See https://aws.amazon.com/cloudfront/pricing/ for full list of supported regions.
 */
var PriceClass;
(function (PriceClass) {
    /** USA, Canada, Europe, & Israel */
    PriceClass["PRICE_CLASS_100"] = "PriceClass_100";
    /** PRICE_CLASS_100 + South Africa, Kenya, Middle East, Japan, Singapore, South Korea, Taiwan, Hong Kong, & Philippines */
    PriceClass["PRICE_CLASS_200"] = "PriceClass_200";
    /** All locations */
    PriceClass["PRICE_CLASS_ALL"] = "PriceClass_All";
})(PriceClass = exports.PriceClass || (exports.PriceClass = {}));
/**
 * How HTTPs should be handled with your distribution.
 */
var ViewerProtocolPolicy;
(function (ViewerProtocolPolicy) {
    /** HTTPS only */
    ViewerProtocolPolicy["HTTPS_ONLY"] = "https-only";
    /** Will redirect HTTP requests to HTTPS */
    ViewerProtocolPolicy["REDIRECT_TO_HTTPS"] = "redirect-to-https";
    /** Both HTTP and HTTPS supported */
    ViewerProtocolPolicy["ALLOW_ALL"] = "allow-all";
})(ViewerProtocolPolicy = exports.ViewerProtocolPolicy || (exports.ViewerProtocolPolicy = {}));
/**
 * Defines what protocols CloudFront will use to connect to an origin.
 */
var OriginProtocolPolicy;
(function (OriginProtocolPolicy) {
    /** Connect on HTTP only */
    OriginProtocolPolicy["HTTP_ONLY"] = "http-only";
    /** Connect with the same protocol as the viewer */
    OriginProtocolPolicy["MATCH_VIEWER"] = "match-viewer";
    /** Connect on HTTPS only */
    OriginProtocolPolicy["HTTPS_ONLY"] = "https-only";
})(OriginProtocolPolicy = exports.OriginProtocolPolicy || (exports.OriginProtocolPolicy = {}));
/**
 * The SSL method CloudFront will use for your distribution.
 *
 * Server Name Indication (SNI) - is an extension to the TLS computer networking protocol by which a client indicates
 *  which hostname it is attempting to connect to at the start of the handshaking process. This allows a server to present
 *  multiple certificates on the same IP address and TCP port number and hence allows multiple secure (HTTPS) websites
 * (or any other service over TLS) to be served by the same IP address without requiring all those sites to use the same certificate.
 *
 * CloudFront can use SNI to host multiple distributions on the same IP - which a large majority of clients will support.
 *
 * If your clients cannot support SNI however - CloudFront can use dedicated IPs for your distribution - but there is a prorated monthly charge for
 * using this feature. By default, we use SNI - but you can optionally enable dedicated IPs (VIP).
 *
 * See the CloudFront SSL for more details about pricing : https://aws.amazon.com/cloudfront/custom-ssl-domains/
 *
 */
var SSLMethod;
(function (SSLMethod) {
    SSLMethod["SNI"] = "sni-only";
    SSLMethod["VIP"] = "vip";
})(SSLMethod = exports.SSLMethod || (exports.SSLMethod = {}));
/**
 * The minimum version of the SSL protocol that you want CloudFront to use for HTTPS connections.
 * CloudFront serves your objects only to browsers or devices that support at least the SSL version that you specify.
 */
var SecurityPolicyProtocol;
(function (SecurityPolicyProtocol) {
    SecurityPolicyProtocol["SSL_V3"] = "SSLv3";
    SecurityPolicyProtocol["TLS_V1"] = "TLSv1";
    SecurityPolicyProtocol["TLS_V1_2016"] = "TLSv1_2016";
    SecurityPolicyProtocol["TLS_V1_1_2016"] = "TLSv1.1_2016";
    SecurityPolicyProtocol["TLS_V1_2_2018"] = "TLSv1.2_2018";
})(SecurityPolicyProtocol = exports.SecurityPolicyProtocol || (exports.SecurityPolicyProtocol = {}));
/**
 * The HTTP methods that the Behavior will accept requests on.
 */
class AllowedMethods {
    constructor(methods) { this.methods = methods; }
}
exports.AllowedMethods = AllowedMethods;
/** HEAD and GET */
AllowedMethods.ALLOW_GET_HEAD = new AllowedMethods(['GET', 'HEAD']);
/** HEAD, GET, and OPTIONS */
AllowedMethods.ALLOW_GET_HEAD_OPTIONS = new AllowedMethods(['GET', 'HEAD', 'OPTIONS']);
/** All supported HTTP methods */
AllowedMethods.ALLOW_ALL = new AllowedMethods(['GET', 'HEAD', 'OPTIONS', 'PUT', 'PATCH', 'POST', 'DELETE']);
/**
 * The type of events that a Lambda@Edge function can be invoked in response to.
 */
var LambdaEdgeEventType;
(function (LambdaEdgeEventType) {
    /**
     * The origin-request specifies the request to the
     * origin location (e.g. S3)
     */
    LambdaEdgeEventType["ORIGIN_REQUEST"] = "origin-request";
    /**
     * The origin-response specifies the response from the
     * origin location (e.g. S3)
     */
    LambdaEdgeEventType["ORIGIN_RESPONSE"] = "origin-response";
    /**
     * The viewer-request specifies the incoming request
     */
    LambdaEdgeEventType["VIEWER_REQUEST"] = "viewer-request";
    /**
     * The viewer-response specifies the outgoing reponse
     */
    LambdaEdgeEventType["VIEWER_RESPONSE"] = "viewer-response";
})(LambdaEdgeEventType = exports.LambdaEdgeEventType || (exports.LambdaEdgeEventType = {}));
//# sourceMappingURL=data:application/json;base64,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