import * as acm from "../../aws-certificatemanager";
import * as lambda from "../../aws-lambda";
import { Construct, IResource, Resource, Duration } from "../../core";
import { IOrigin } from './origin';
/**
 * Interface for CloudFront distributions
 */
export interface IDistribution extends IResource {
    /**
     * The domain name of the Distribution, such as d111111abcdef8.cloudfront.net.
     *
     * @attribute
     * @deprecated - Use `distributionDomainName` instead.
     */
    readonly domainName: string;
    /**
     * The domain name of the Distribution, such as d111111abcdef8.cloudfront.net.
     *
     * @attribute
     */
    readonly distributionDomainName: string;
    /**
     * The distribution ID for this distribution.
     *
     * @attribute
     */
    readonly distributionId: string;
}
/**
 * Attributes used to import a Distribution.
 *
 * @experimental
 */
export interface DistributionAttributes {
    /**
     * The generated domain name of the Distribution, such as d111111abcdef8.cloudfront.net.
     *
     * @attribute
     */
    readonly domainName: string;
    /**
     * The distribution ID for this distribution.
     *
     * @attribute
     */
    readonly distributionId: string;
}
/**
 * Properties for a Distribution
 *
 * @experimental
 */
export interface DistributionProps {
    /**
     * The default behavior for the distribution.
     */
    readonly defaultBehavior: BehaviorOptions;
    /**
     * Additional behaviors for the distribution, mapped by the pathPattern that specifies which requests to apply the behavior to.
     *
     * @default - no additional behaviors are added.
     */
    readonly additionalBehaviors?: Record<string, BehaviorOptions>;
    /**
     * A certificate to associate with the distribution. The certificate must be located in N. Virginia (us-east-1).
     *
     * @default - the CloudFront wildcard certificate (*.cloudfront.net) will be used.
     */
    readonly certificate?: acm.ICertificate;
    /**
     * The price class that corresponds with the maximum price that you want to pay for CloudFront service.
     * If you specify PriceClass_All, CloudFront responds to requests for your objects from all CloudFront edge locations.
     * If you specify a price class other than PriceClass_All, CloudFront serves your objects from the CloudFront edge location
     * that has the lowest latency among the edge locations in your price class.
     *
     * @default PriceClass.PRICE_CLASS_ALL
     */
    readonly priceClass?: PriceClass;
    /**
     * How CloudFront should handle requests that are not successful (e.g., PageNotFound).
     *
     * @default - No custom error responses.
     */
    readonly errorResponses?: ErrorResponse[];
}
/**
 * A CloudFront distribution with associated origin(s) and caching behavior(s).
 *
 * @experimental
 */
export declare class Distribution extends Resource implements IDistribution {
    /**
     * Creates a Distribution construct that represents an external (imported) distribution.
     */
    static fromDistributionAttributes(scope: Construct, id: string, attrs: DistributionAttributes): IDistribution;
    readonly domainName: string;
    readonly distributionDomainName: string;
    readonly distributionId: string;
    private readonly defaultBehavior;
    private readonly additionalBehaviors;
    private readonly boundOrigins;
    private readonly errorResponses;
    private readonly certificate?;
    constructor(scope: Construct, id: string, props: DistributionProps);
    /**
     * Adds a new behavior to this distribution for the given pathPattern.
     *
     * @param pathPattern the path pattern (e.g., 'images/*') that specifies which requests to apply the behavior to.
     * @param origin the origin to use for this behavior
     * @param behaviorOptions the options for the behavior at this path.
     */
    addBehavior(pathPattern: string, origin: IOrigin, behaviorOptions?: AddBehaviorOptions): void;
    private addOrigin;
    private renderOrigins;
    private renderCacheBehaviors;
    private renderErrorResponses;
    private renderViewerCertificate;
}
/**
 * The price class determines how many edge locations CloudFront will use for your distribution.
 * See https://aws.amazon.com/cloudfront/pricing/ for full list of supported regions.
 */
export declare enum PriceClass {
    /** USA, Canada, Europe, & Israel */
    PRICE_CLASS_100 = "PriceClass_100",
    /** PRICE_CLASS_100 + South Africa, Kenya, Middle East, Japan, Singapore, South Korea, Taiwan, Hong Kong, & Philippines */
    PRICE_CLASS_200 = "PriceClass_200",
    /** All locations */
    PRICE_CLASS_ALL = "PriceClass_All"
}
/**
 * How HTTPs should be handled with your distribution.
 */
export declare enum ViewerProtocolPolicy {
    /** HTTPS only */
    HTTPS_ONLY = "https-only",
    /** Will redirect HTTP requests to HTTPS */
    REDIRECT_TO_HTTPS = "redirect-to-https",
    /** Both HTTP and HTTPS supported */
    ALLOW_ALL = "allow-all"
}
/**
 * Defines what protocols CloudFront will use to connect to an origin.
 */
export declare enum OriginProtocolPolicy {
    /** Connect on HTTP only */
    HTTP_ONLY = "http-only",
    /** Connect with the same protocol as the viewer */
    MATCH_VIEWER = "match-viewer",
    /** Connect on HTTPS only */
    HTTPS_ONLY = "https-only"
}
/**
 * The SSL method CloudFront will use for your distribution.
 *
 * Server Name Indication (SNI) - is an extension to the TLS computer networking protocol by which a client indicates
 *  which hostname it is attempting to connect to at the start of the handshaking process. This allows a server to present
 *  multiple certificates on the same IP address and TCP port number and hence allows multiple secure (HTTPS) websites
 * (or any other service over TLS) to be served by the same IP address without requiring all those sites to use the same certificate.
 *
 * CloudFront can use SNI to host multiple distributions on the same IP - which a large majority of clients will support.
 *
 * If your clients cannot support SNI however - CloudFront can use dedicated IPs for your distribution - but there is a prorated monthly charge for
 * using this feature. By default, we use SNI - but you can optionally enable dedicated IPs (VIP).
 *
 * See the CloudFront SSL for more details about pricing : https://aws.amazon.com/cloudfront/custom-ssl-domains/
 *
 */
export declare enum SSLMethod {
    SNI = "sni-only",
    VIP = "vip"
}
/**
 * The minimum version of the SSL protocol that you want CloudFront to use for HTTPS connections.
 * CloudFront serves your objects only to browsers or devices that support at least the SSL version that you specify.
 */
export declare enum SecurityPolicyProtocol {
    SSL_V3 = "SSLv3",
    TLS_V1 = "TLSv1",
    TLS_V1_2016 = "TLSv1_2016",
    TLS_V1_1_2016 = "TLSv1.1_2016",
    TLS_V1_2_2018 = "TLSv1.2_2018"
}
/**
 * The HTTP methods that the Behavior will accept requests on.
 */
export declare class AllowedMethods {
    /** HEAD and GET */
    static readonly ALLOW_GET_HEAD: AllowedMethods;
    /** HEAD, GET, and OPTIONS */
    static readonly ALLOW_GET_HEAD_OPTIONS: AllowedMethods;
    /** All supported HTTP methods */
    static readonly ALLOW_ALL: AllowedMethods;
    /** HTTP methods supported */
    readonly methods: string[];
    private constructor();
}
/**
 * Options for configuring custom error responses.
 *
 * @experimental
 */
export interface ErrorResponse {
    /**
     * The minimum amount of time, in seconds, that you want CloudFront to cache the HTTP status code specified in ErrorCode.
     *
     * @default - the default caching TTL behavior applies
     */
    readonly ttl?: Duration;
    /**
     * The HTTP status code for which you want to specify a custom error page and/or a caching duration.
     */
    readonly httpStatus: number;
    /**
     * The HTTP status code that you want CloudFront to return to the viewer along with the custom error page.
     *
     * If you specify a value for `responseHttpStatus`, you must also specify a value for `responsePagePath`.
     *
     * @default - not set, the error code will be returned as the response code.
     */
    readonly responseHttpStatus?: number;
    /**
     * The path to the custom error page that you want CloudFront to return to a viewer when your origin returns the
     * `httpStatus`, for example, /4xx-errors/403-forbidden.html
     *
     * @default - the default CloudFront response is shown.
     */
    readonly responsePagePath?: string;
}
/**
 * The type of events that a Lambda@Edge function can be invoked in response to.
 */
export declare enum LambdaEdgeEventType {
    /**
     * The origin-request specifies the request to the
     * origin location (e.g. S3)
     */
    ORIGIN_REQUEST = "origin-request",
    /**
     * The origin-response specifies the response from the
     * origin location (e.g. S3)
     */
    ORIGIN_RESPONSE = "origin-response",
    /**
     * The viewer-request specifies the incoming request
     */
    VIEWER_REQUEST = "viewer-request",
    /**
     * The viewer-response specifies the outgoing reponse
     */
    VIEWER_RESPONSE = "viewer-response"
}
/**
 * Represents a Lambda function version and event type when using Lambda@Edge.
 * The type of the {@link AddBehaviorOptions.edgeLambdas} property.
 */
export interface EdgeLambda {
    /**
     * The version of the Lambda function that will be invoked.
     *
     * **Note**: it's not possible to use the '$LATEST' function version for Lambda@Edge!
     */
    readonly functionVersion: lambda.IVersion;
    /** The type of event in response to which should the function be invoked. */
    readonly eventType: LambdaEdgeEventType;
}
/**
 * Options for adding a new behavior to a Distribution.
 *
 * @experimental
 */
export interface AddBehaviorOptions {
    /**
     * HTTP methods to allow for this behavior.
     *
     * @default - GET and HEAD
     */
    readonly allowedMethods?: AllowedMethods;
    /**
     * Whether CloudFront will forward query strings to the origin.
     * If this is set to true, CloudFront will forward all query parameters to the origin, and cache
     * based on all parameters. See `forwardQueryStringCacheKeys` for a way to limit the query parameters
     * CloudFront caches on.
     *
     * @default false
     */
    readonly forwardQueryString?: boolean;
    /**
     * A set of query string parameter names to use for caching if `forwardQueryString` is set to true.
     *
     * @default []
     */
    readonly forwardQueryStringCacheKeys?: string[];
    /**
     * The Lambda@Edge functions to invoke before serving the contents.
     *
     * @default - no Lambda functions will be invoked
     * @see https://aws.amazon.com/lambda/edge
     */
    readonly edgeLambdas?: EdgeLambda[];
}
/**
 * Options for creating a new behavior.
 *
 * @experimental
 */
export interface BehaviorOptions extends AddBehaviorOptions {
    /**
     * The origin that you want CloudFront to route requests to when they match this behavior.
     */
    readonly origin: IOrigin;
}
