"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GraphQLApi = exports.IamResource = exports.SchemaDefinition = exports.FieldLogLevel = exports.UserPoolDefaultAction = exports.AuthorizationType = void 0;
const fs_1 = require("fs");
const aws_iam_1 = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const appsync_generated_1 = require("./appsync.generated");
const data_source_1 = require("./data-source");
/**
 * enum with all possible values for AppSync authorization type
 */
var AuthorizationType;
(function (AuthorizationType) {
    /**
     * API Key authorization type
     */
    AuthorizationType["API_KEY"] = "API_KEY";
    /**
     * AWS IAM authorization type. Can be used with Cognito Identity Pool federated credentials
     */
    AuthorizationType["IAM"] = "AWS_IAM";
    /**
     * Cognito User Pool authorization type
     */
    AuthorizationType["USER_POOL"] = "AMAZON_COGNITO_USER_POOLS";
    /**
     * OpenID Connect authorization type
     */
    AuthorizationType["OIDC"] = "OPENID_CONNECT";
})(AuthorizationType = exports.AuthorizationType || (exports.AuthorizationType = {}));
/**
 * enum with all possible values for Cognito user-pool default actions
 */
var UserPoolDefaultAction;
(function (UserPoolDefaultAction) {
    /**
     * ALLOW access to API
     */
    UserPoolDefaultAction["ALLOW"] = "ALLOW";
    /**
     * DENY access to API
     */
    UserPoolDefaultAction["DENY"] = "DENY";
})(UserPoolDefaultAction = exports.UserPoolDefaultAction || (exports.UserPoolDefaultAction = {}));
/**
 * log-level for fields in AppSync
 */
var FieldLogLevel;
(function (FieldLogLevel) {
    /**
     * No logging
     */
    FieldLogLevel["NONE"] = "NONE";
    /**
     * Error logging
     */
    FieldLogLevel["ERROR"] = "ERROR";
    /**
     * All logging
     */
    FieldLogLevel["ALL"] = "ALL";
})(FieldLogLevel = exports.FieldLogLevel || (exports.FieldLogLevel = {}));
/**
 * Enum containing the different modes of schema definition
 */
var SchemaDefinition;
(function (SchemaDefinition) {
    /**
     * Define schema through functions like addType, addQuery, etc.
     */
    SchemaDefinition["CODE"] = "CODE";
    /**
     * Define schema in a file, i.e. schema.graphql
     */
    SchemaDefinition["FILE"] = "FILE";
})(SchemaDefinition = exports.SchemaDefinition || (exports.SchemaDefinition = {}));
/**
 * A class used to generate resource arns for AppSync
 */
class IamResource {
    constructor(arns) {
        this.arns = arns;
    }
    /**
     * Generate the resource names given custom arns
     *
     * @param arns The custom arns that need to be permissioned
     *
     * Example: custom('/types/Query/fields/getExample')
     */
    static custom(...arns) {
        if (arns.length === 0) {
            throw new Error('At least 1 custom ARN must be provided.');
        }
        return new IamResource(arns);
    }
    /**
     * Generate the resource names given a type and fields
     *
     * @param type The type that needs to be allowed
     * @param fields The fields that need to be allowed, if empty grant permissions to ALL fields
     *
     * Example: ofType('Query', 'GetExample')
     */
    static ofType(type, ...fields) {
        const arns = fields.length ? fields.map((field) => `types/${type}/fields/${field}`) : [`types/${type}/*`];
        return new IamResource(arns);
    }
    /**
     * Generate the resource names that accepts all types: `*`
     */
    static all() {
        return new IamResource(['*']);
    }
    /**
     * Return the Resource ARN
     *
     * @param api The GraphQL API to give permissions
     */
    resourceArns(api) {
        return this.arns.map((arn) => core_1.Stack.of(api).formatArn({
            service: 'appsync',
            resource: `apis/${api.apiId}`,
            sep: '/',
            resourceName: `${arn}`,
        }));
    }
}
exports.IamResource = IamResource;
/**
 * An AppSync GraphQL API
 */
class GraphQLApi extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
        super(scope, id);
        this.validateAuthorizationProps(props);
        const defaultAuthorizationType = ((_b = (_a = props.authorizationConfig) === null || _a === void 0 ? void 0 : _a.defaultAuthorization) === null || _b === void 0 ? void 0 : _b.authorizationType) ||
            AuthorizationType.API_KEY;
        let apiLogsRole;
        if (props.logConfig) {
            apiLogsRole = new aws_iam_1.Role(this, 'ApiLogsRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('appsync'),
            });
            apiLogsRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSAppSyncPushToCloudWatchLogs'));
        }
        this.api = new appsync_generated_1.CfnGraphQLApi(this, 'Resource', {
            name: props.name,
            authenticationType: defaultAuthorizationType,
            ...(props.logConfig && {
                logConfig: {
                    cloudWatchLogsRoleArn: apiLogsRole ? apiLogsRole.roleArn : undefined,
                    excludeVerboseContent: props.logConfig.excludeVerboseContent,
                    fieldLogLevel: props.logConfig.fieldLogLevel
                        ? props.logConfig.fieldLogLevel.toString()
                        : undefined,
                },
            }),
            openIdConnectConfig: ((_d = (_c = props.authorizationConfig) === null || _c === void 0 ? void 0 : _c.defaultAuthorization) === null || _d === void 0 ? void 0 : _d.authorizationType) ===
                AuthorizationType.OIDC
                ? this.formatOpenIdConnectConfig(props.authorizationConfig.defaultAuthorization
                    .openIdConnectConfig)
                : undefined,
            userPoolConfig: ((_f = (_e = props.authorizationConfig) === null || _e === void 0 ? void 0 : _e.defaultAuthorization) === null || _f === void 0 ? void 0 : _f.authorizationType) ===
                AuthorizationType.USER_POOL
                ? this.formatUserPoolConfig(props.authorizationConfig.defaultAuthorization.userPoolConfig)
                : undefined,
            additionalAuthenticationProviders: this.formatAdditionalAuthenticationProviders(props),
        });
        this.apiId = this.api.attrApiId;
        this.arn = this.api.attrArn;
        this.graphQlUrl = this.api.attrGraphQlUrl;
        this.name = this.api.name;
        this.schemaMode = props.schemaDefinition;
        if (defaultAuthorizationType === AuthorizationType.API_KEY || ((_h = (_g = props.authorizationConfig) === null || _g === void 0 ? void 0 : _g.additionalAuthorizationModes) === null || _h === void 0 ? void 0 : _h.some((authMode) => authMode.authorizationType === AuthorizationType.API_KEY))) {
            const apiKeyConfig = ((_k = (_j = props.authorizationConfig) === null || _j === void 0 ? void 0 : _j.defaultAuthorization) === null || _k === void 0 ? void 0 : _k.apiKeyConfig) || {
                name: 'DefaultAPIKey',
                description: 'Default API Key created by CDK',
            };
            this._apiKey = this.createAPIKey(apiKeyConfig);
        }
        this.schema = this.defineSchema(props.schemaDefinitionFile);
    }
    /**
     * the configured API key, if present
     */
    get apiKey() {
        return this._apiKey;
    }
    /**
     * add a new dummy data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     */
    addNoneDataSource(name, description) {
        return new data_source_1.NoneDataSource(this, `${name}DS`, {
            api: this,
            description,
            name,
        });
    }
    /**
     * add a new DynamoDB data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     * @param table The DynamoDB table backing this data source [disable-awslint:ref-via-interface]
     */
    addDynamoDbDataSource(name, description, table) {
        return new data_source_1.DynamoDbDataSource(this, `${name}DS`, {
            api: this,
            description,
            name,
            table,
        });
    }
    /**
     * add a new http data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     * @param endpoint The http endpoint
     */
    addHttpDataSource(name, description, endpoint) {
        return new data_source_1.HttpDataSource(this, `${name}DS`, {
            api: this,
            description,
            endpoint,
            name,
        });
    }
    /**
     * add a new Lambda data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     * @param lambdaFunction The Lambda function to call to interact with this data source
     */
    addLambdaDataSource(name, description, lambdaFunction) {
        return new data_source_1.LambdaDataSource(this, `${name}DS`, {
            api: this,
            description,
            name,
            lambdaFunction,
        });
    }
    /**
     * Adds an IAM policy statement associated with this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param resources The set of resources to allow (i.e. ...:[region]:[accountId]:apis/GraphQLId/...)
     * @param actions The actions that should be granted to the principal (i.e. appsync:graphql )
     */
    grant(grantee, resources, ...actions) {
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: resources.resourceArns(this),
            scope: this,
        });
    }
    /**
     * Adds an IAM policy statement for Mutation access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Mutations (leave blank for all)
     */
    grantMutation(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Mutation', ...fields), 'appsync:GraphQL');
    }
    /**
     * Adds an IAM policy statement for Query access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Queries (leave blank for all)
     */
    grantQuery(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Query', ...fields), 'appsync:GraphQL');
    }
    /**
     * Adds an IAM policy statement for Subscription access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Subscriptions (leave blank for all)
     */
    grantSubscription(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Subscription', ...fields), 'appsync:GraphQL');
    }
    validateAuthorizationProps(props) {
        var _a, _b, _c, _d, _e, _f, _g;
        const defaultAuthorizationType = ((_b = (_a = props.authorizationConfig) === null || _a === void 0 ? void 0 : _a.defaultAuthorization) === null || _b === void 0 ? void 0 : _b.authorizationType) ||
            AuthorizationType.API_KEY;
        if (defaultAuthorizationType === AuthorizationType.OIDC &&
            !((_d = (_c = props.authorizationConfig) === null || _c === void 0 ? void 0 : _c.defaultAuthorization) === null || _d === void 0 ? void 0 : _d.openIdConnectConfig)) {
            throw new Error('Missing default OIDC Configuration');
        }
        if (defaultAuthorizationType === AuthorizationType.USER_POOL &&
            !((_f = (_e = props.authorizationConfig) === null || _e === void 0 ? void 0 : _e.defaultAuthorization) === null || _f === void 0 ? void 0 : _f.userPoolConfig)) {
            throw new Error('Missing default User Pool Configuration');
        }
        if ((_g = props.authorizationConfig) === null || _g === void 0 ? void 0 : _g.additionalAuthorizationModes) {
            props.authorizationConfig.additionalAuthorizationModes.forEach((authorizationMode) => {
                if (authorizationMode.authorizationType === AuthorizationType.API_KEY &&
                    defaultAuthorizationType === AuthorizationType.API_KEY) {
                    throw new Error("You can't duplicate API_KEY in additional authorization config. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html");
                }
                if (authorizationMode.authorizationType === AuthorizationType.IAM &&
                    defaultAuthorizationType === AuthorizationType.IAM) {
                    throw new Error("You can't duplicate IAM in additional authorization config. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html");
                }
                if (authorizationMode.authorizationType === AuthorizationType.OIDC &&
                    !authorizationMode.openIdConnectConfig) {
                    throw new Error('Missing OIDC Configuration inside an additional authorization mode');
                }
                if (authorizationMode.authorizationType ===
                    AuthorizationType.USER_POOL &&
                    !authorizationMode.userPoolConfig) {
                    throw new Error('Missing User Pool Configuration inside an additional authorization mode');
                }
            });
        }
    }
    formatOpenIdConnectConfig(config) {
        return {
            authTtl: config.tokenExpiryFromAuth,
            clientId: config.clientId,
            iatTtl: config.tokenExpiryFromIssue,
            issuer: config.oidcProvider,
        };
    }
    formatUserPoolConfig(config) {
        return {
            userPoolId: config.userPool.userPoolId,
            awsRegion: config.userPool.stack.region,
            appIdClientRegex: config.appIdClientRegex,
            defaultAction: config.defaultAction || 'ALLOW',
        };
    }
    createAPIKey(config) {
        let expires;
        if (config.expires) {
            expires = new Date(config.expires).valueOf();
            const days = (d) => Date.now() + core_1.Duration.days(d).toMilliseconds();
            if (expires < days(1) || expires > days(365)) {
                throw Error('API key expiration must be between 1 and 365 days.');
            }
            expires = Math.round(expires / 1000);
        }
        const key = new appsync_generated_1.CfnApiKey(this, `${config.name || 'DefaultAPIKey'}ApiKey`, {
            expires,
            description: config.description || 'Default API Key created by CDK',
            apiId: this.apiId,
        });
        return key.attrApiKey;
    }
    formatAdditionalAuthorizationModes(authModes) {
        return authModes.reduce((acc, authMode) => [
            ...acc,
            {
                authenticationType: authMode.authorizationType,
                userPoolConfig: authMode.authorizationType === AuthorizationType.USER_POOL
                    ? this.formatUserPoolConfig(authMode.userPoolConfig)
                    : undefined,
                openIdConnectConfig: authMode.authorizationType === AuthorizationType.OIDC
                    ? this.formatOpenIdConnectConfig(authMode.openIdConnectConfig)
                    : undefined,
            },
        ], []);
    }
    formatAdditionalAuthenticationProviders(props) {
        var _a;
        const authModes = (_a = props.authorizationConfig) === null || _a === void 0 ? void 0 : _a.additionalAuthorizationModes;
        return authModes ? this.formatAdditionalAuthorizationModes(authModes) : undefined;
    }
    /**
     * Sets schema defintiion to input if schema mode is configured with SchemaDefinition.CODE
     *
     * @param definition string that is the graphql representation of schema
     * @experimental temporary
     */
    updateDefinition(definition) {
        if (this.schemaMode != SchemaDefinition.CODE) {
            throw new Error('API cannot add type because schema definition mode is not configured as CODE.');
        }
        this.schema.definition = definition;
    }
    /**
     * Define schema based on props configuration
     * @param file the file name/s3 location of Schema
     */
    defineSchema(file) {
        let definition;
        if (this.schemaMode == SchemaDefinition.FILE && !file) {
            throw new Error('schemaDefinitionFile must be configured if using FILE definition mode.');
        }
        else if (this.schemaMode == SchemaDefinition.FILE && file) {
            definition = fs_1.readFileSync(file).toString('UTF-8');
        }
        else if (this.schemaMode == SchemaDefinition.CODE && !file) {
            definition = '';
        }
        else if (this.schemaMode == SchemaDefinition.CODE && file) {
            throw new Error('definition mode CODE is incompatible with file definition. Change mode to FILE/S3 or unconfigure schemaDefinitionFile');
        }
        return new appsync_generated_1.CfnGraphQLSchema(this, 'Schema', {
            apiId: this.apiId,
            definition,
        });
    }
}
exports.GraphQLApi = GraphQLApi;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZ3JhcGhxbGFwaS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImdyYXBocWxhcGkudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsMkJBQWtDO0FBR2xDLDJDQUF5RixDQUFDLG1EQUFtRDtBQUU3SSxxQ0FBcUUsQ0FBQyxnREFBZ0Q7QUFDdEgsMkRBQWlGO0FBQ2pGLCtDQUFxRztBQUNyRzs7R0FFRztBQUNILElBQVksaUJBaUJYO0FBakJELFdBQVksaUJBQWlCO0lBQ3pCOztPQUVHO0lBQ0gsd0NBQW1CLENBQUE7SUFDbkI7O09BRUc7SUFDSCxvQ0FBZSxDQUFBO0lBQ2Y7O09BRUc7SUFDSCw0REFBdUMsQ0FBQTtJQUN2Qzs7T0FFRztJQUNILDRDQUF1QixDQUFBO0FBQzNCLENBQUMsRUFqQlcsaUJBQWlCLEdBQWpCLHlCQUFpQixLQUFqQix5QkFBaUIsUUFpQjVCO0FBNkJEOztHQUVHO0FBQ0gsSUFBWSxxQkFTWDtBQVRELFdBQVkscUJBQXFCO0lBQzdCOztPQUVHO0lBQ0gsd0NBQWUsQ0FBQTtJQUNmOztPQUVHO0lBQ0gsc0NBQWEsQ0FBQTtBQUNqQixDQUFDLEVBVFcscUJBQXFCLEdBQXJCLDZCQUFxQixLQUFyQiw2QkFBcUIsUUFTaEM7QUF5RkQ7O0dBRUc7QUFDSCxJQUFZLGFBYVg7QUFiRCxXQUFZLGFBQWE7SUFDckI7O09BRUc7SUFDSCw4QkFBYSxDQUFBO0lBQ2I7O09BRUc7SUFDSCxnQ0FBZSxDQUFBO0lBQ2Y7O09BRUc7SUFDSCw0QkFBVyxDQUFBO0FBQ2YsQ0FBQyxFQWJXLGFBQWEsR0FBYixxQkFBYSxLQUFiLHFCQUFhLFFBYXhCO0FBa0JEOztHQUVHO0FBQ0gsSUFBWSxnQkFTWDtBQVRELFdBQVksZ0JBQWdCO0lBQ3hCOztPQUVHO0lBQ0gsaUNBQWEsQ0FBQTtJQUNiOztPQUVHO0lBQ0gsaUNBQWEsQ0FBQTtBQUNqQixDQUFDLEVBVFcsZ0JBQWdCLEdBQWhCLHdCQUFnQixLQUFoQix3QkFBZ0IsUUFTM0I7QUFxQ0Q7O0dBRUc7QUFDSCxNQUFhLFdBQVc7SUFpQ3BCLFlBQW9CLElBQWM7UUFDOUIsSUFBSSxDQUFDLElBQUksR0FBRyxJQUFJLENBQUM7SUFDckIsQ0FBQztJQWxDRDs7Ozs7O09BTUc7SUFDSSxNQUFNLENBQUMsTUFBTSxDQUFDLEdBQUcsSUFBYztRQUNsQyxJQUFJLElBQUksQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQ25CLE1BQU0sSUFBSSxLQUFLLENBQUMseUNBQXlDLENBQUMsQ0FBQztTQUM5RDtRQUNELE9BQU8sSUFBSSxXQUFXLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDakMsQ0FBQztJQUNEOzs7Ozs7O09BT0c7SUFDSSxNQUFNLENBQUMsTUFBTSxDQUFDLElBQVksRUFBRSxHQUFHLE1BQWdCO1FBQ2xELE1BQU0sSUFBSSxHQUFHLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLFNBQVMsSUFBSSxXQUFXLEtBQUssRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUyxJQUFJLElBQUksQ0FBQyxDQUFDO1FBQzFHLE9BQU8sSUFBSSxXQUFXLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDakMsQ0FBQztJQUNEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLEdBQUc7UUFDYixPQUFPLElBQUksV0FBVyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBS0Q7Ozs7T0FJRztJQUNJLFlBQVksQ0FBQyxHQUFlO1FBQy9CLE9BQU8sSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxHQUFHLEVBQUUsRUFBRSxDQUFDLFlBQUssQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUMsU0FBUyxDQUFDO1lBQ2xELE9BQU8sRUFBRSxTQUFTO1lBQ2xCLFFBQVEsRUFBRSxRQUFRLEdBQUcsQ0FBQyxLQUFLLEVBQUU7WUFDN0IsR0FBRyxFQUFFLEdBQUc7WUFDUixZQUFZLEVBQUUsR0FBRyxHQUFHLEVBQUU7U0FDekIsQ0FBQyxDQUFDLENBQUM7SUFDUixDQUFDO0NBQ0o7QUFqREQsa0NBaURDO0FBQ0Q7O0dBRUc7QUFDSCxNQUFhLFVBQVcsU0FBUSxnQkFBUztJQThCckMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFzQjs7UUFDNUQsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUNqQixJQUFJLENBQUMsMEJBQTBCLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDdkMsTUFBTSx3QkFBd0IsR0FBRyxhQUFBLEtBQUssQ0FBQyxtQkFBbUIsMENBQUUsb0JBQW9CLDBDQUFFLGlCQUFpQjtZQUMvRixpQkFBaUIsQ0FBQyxPQUFPLENBQUM7UUFDOUIsSUFBSSxXQUFXLENBQUM7UUFDaEIsSUFBSSxLQUFLLENBQUMsU0FBUyxFQUFFO1lBQ2pCLFdBQVcsR0FBRyxJQUFJLGNBQUksQ0FBQyxJQUFJLEVBQUUsYUFBYSxFQUFFO2dCQUN4QyxTQUFTLEVBQUUsSUFBSSwwQkFBZ0IsQ0FBQyxTQUFTLENBQUM7YUFDN0MsQ0FBQyxDQUFDO1lBQ0gsV0FBVyxDQUFDLGdCQUFnQixDQUFDLHVCQUFhLENBQUMsd0JBQXdCLENBQUMsNkNBQTZDLENBQUMsQ0FBQyxDQUFDO1NBQ3ZIO1FBQ0QsSUFBSSxDQUFDLEdBQUcsR0FBRyxJQUFJLGlDQUFhLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUMzQyxJQUFJLEVBQUUsS0FBSyxDQUFDLElBQUk7WUFDaEIsa0JBQWtCLEVBQUUsd0JBQXdCO1lBQzVDLEdBQUcsQ0FBQyxLQUFLLENBQUMsU0FBUyxJQUFJO2dCQUNuQixTQUFTLEVBQUU7b0JBQ1AscUJBQXFCLEVBQUUsV0FBVyxDQUFDLENBQUMsQ0FBQyxXQUFXLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxTQUFTO29CQUNwRSxxQkFBcUIsRUFBRSxLQUFLLENBQUMsU0FBUyxDQUFDLHFCQUFxQjtvQkFDNUQsYUFBYSxFQUFFLEtBQUssQ0FBQyxTQUFTLENBQUMsYUFBYTt3QkFDeEMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsYUFBYSxDQUFDLFFBQVEsRUFBRTt3QkFDMUMsQ0FBQyxDQUFDLFNBQVM7aUJBQ2xCO2FBQ0osQ0FBQztZQUNGLG1CQUFtQixFQUFFLGFBQUEsS0FBSyxDQUFDLG1CQUFtQiwwQ0FBRSxvQkFBb0IsMENBQUUsaUJBQWlCO2dCQUNuRixpQkFBaUIsQ0FBQyxJQUFJO2dCQUN0QixDQUFDLENBQUMsSUFBSSxDQUFDLHlCQUF5QixDQUFDLEtBQUssQ0FBQyxtQkFBbUIsQ0FBQyxvQkFBb0I7cUJBQzFFLG1CQUFvQixDQUFDO2dCQUMxQixDQUFDLENBQUMsU0FBUztZQUNmLGNBQWMsRUFBRSxhQUFBLEtBQUssQ0FBQyxtQkFBbUIsMENBQUUsb0JBQW9CLDBDQUFFLGlCQUFpQjtnQkFDOUUsaUJBQWlCLENBQUMsU0FBUztnQkFDM0IsQ0FBQyxDQUFDLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxLQUFLLENBQUMsbUJBQW1CLENBQUMsb0JBQW9CLENBQUMsY0FBZSxDQUFDO2dCQUMzRixDQUFDLENBQUMsU0FBUztZQUNmLGlDQUFpQyxFQUFFLElBQUksQ0FBQyx1Q0FBdUMsQ0FBQyxLQUFLLENBQUM7U0FDekYsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLEtBQUssR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLFNBQVMsQ0FBQztRQUNoQyxJQUFJLENBQUMsR0FBRyxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsT0FBTyxDQUFDO1FBQzVCLElBQUksQ0FBQyxVQUFVLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxjQUFjLENBQUM7UUFDMUMsSUFBSSxDQUFDLElBQUksR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQztRQUMxQixJQUFJLENBQUMsVUFBVSxHQUFHLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQztRQUN6QyxJQUFJLHdCQUF3QixLQUFLLGlCQUFpQixDQUFDLE9BQU8saUJBQ3RELEtBQUssQ0FBQyxtQkFBbUIsMENBQUUsNEJBQTRCLDBDQUFFLElBQUksQ0FBQyxDQUFDLFFBQVEsRUFBRSxFQUFFLENBQUMsUUFBUSxDQUFDLGlCQUFpQixLQUFLLGlCQUFpQixDQUFDLE9BQU8sRUFBQyxFQUFFO1lBQ3ZJLE1BQU0sWUFBWSxHQUFpQixhQUFBLEtBQUssQ0FBQyxtQkFBbUIsMENBQ3RELG9CQUFvQiwwQ0FBRSxZQUFZLEtBQUk7Z0JBQ3hDLElBQUksRUFBRSxlQUFlO2dCQUNyQixXQUFXLEVBQUUsZ0NBQWdDO2FBQ2hELENBQUM7WUFDRixJQUFJLENBQUMsT0FBTyxHQUFHLElBQUksQ0FBQyxZQUFZLENBQUMsWUFBWSxDQUFDLENBQUM7U0FDbEQ7UUFDRCxJQUFJLENBQUMsTUFBTSxHQUFHLElBQUksQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLG9CQUFvQixDQUFDLENBQUM7SUFDaEUsQ0FBQztJQTNERDs7T0FFRztJQUNILElBQVcsTUFBTTtRQUNiLE9BQU8sSUFBSSxDQUFDLE9BQU8sQ0FBQztJQUN4QixDQUFDO0lBdUREOzs7O09BSUc7SUFDSSxpQkFBaUIsQ0FBQyxJQUFZLEVBQUUsV0FBbUI7UUFDdEQsT0FBTyxJQUFJLDRCQUFjLENBQUMsSUFBSSxFQUFFLEdBQUcsSUFBSSxJQUFJLEVBQUU7WUFDekMsR0FBRyxFQUFFLElBQUk7WUFDVCxXQUFXO1lBQ1gsSUFBSTtTQUNQLENBQUMsQ0FBQztJQUNQLENBQUM7SUFDRDs7Ozs7T0FLRztJQUNJLHFCQUFxQixDQUFDLElBQVksRUFBRSxXQUFtQixFQUFFLEtBQWE7UUFDekUsT0FBTyxJQUFJLGdDQUFrQixDQUFDLElBQUksRUFBRSxHQUFHLElBQUksSUFBSSxFQUFFO1lBQzdDLEdBQUcsRUFBRSxJQUFJO1lBQ1QsV0FBVztZQUNYLElBQUk7WUFDSixLQUFLO1NBQ1IsQ0FBQyxDQUFDO0lBQ1AsQ0FBQztJQUNEOzs7OztPQUtHO0lBQ0ksaUJBQWlCLENBQUMsSUFBWSxFQUFFLFdBQW1CLEVBQUUsUUFBZ0I7UUFDeEUsT0FBTyxJQUFJLDRCQUFjLENBQUMsSUFBSSxFQUFFLEdBQUcsSUFBSSxJQUFJLEVBQUU7WUFDekMsR0FBRyxFQUFFLElBQUk7WUFDVCxXQUFXO1lBQ1gsUUFBUTtZQUNSLElBQUk7U0FDUCxDQUFDLENBQUM7SUFDUCxDQUFDO0lBQ0Q7Ozs7O09BS0c7SUFDSSxtQkFBbUIsQ0FBQyxJQUFZLEVBQUUsV0FBbUIsRUFBRSxjQUF5QjtRQUNuRixPQUFPLElBQUksOEJBQWdCLENBQUMsSUFBSSxFQUFFLEdBQUcsSUFBSSxJQUFJLEVBQUU7WUFDM0MsR0FBRyxFQUFFLElBQUk7WUFDVCxXQUFXO1lBQ1gsSUFBSTtZQUNKLGNBQWM7U0FDakIsQ0FBQyxDQUFDO0lBQ1AsQ0FBQztJQUNEOzs7Ozs7O09BT0c7SUFDSSxLQUFLLENBQUMsT0FBbUIsRUFBRSxTQUFzQixFQUFFLEdBQUcsT0FBaUI7UUFDMUUsT0FBTyxlQUFLLENBQUMsY0FBYyxDQUFDO1lBQ3hCLE9BQU87WUFDUCxPQUFPO1lBQ1AsWUFBWSxFQUFFLFNBQVMsQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDO1lBQzFDLEtBQUssRUFBRSxJQUFJO1NBQ2QsQ0FBQyxDQUFDO0lBQ1AsQ0FBQztJQUNEOzs7Ozs7T0FNRztJQUNJLGFBQWEsQ0FBQyxPQUFtQixFQUFFLEdBQUcsTUFBZ0I7UUFDekQsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sRUFBRSxXQUFXLENBQUMsTUFBTSxDQUFDLFVBQVUsRUFBRSxHQUFHLE1BQU0sQ0FBQyxFQUFFLGlCQUFpQixDQUFDLENBQUM7SUFDN0YsQ0FBQztJQUNEOzs7Ozs7T0FNRztJQUNJLFVBQVUsQ0FBQyxPQUFtQixFQUFFLEdBQUcsTUFBZ0I7UUFDdEQsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sRUFBRSxXQUFXLENBQUMsTUFBTSxDQUFDLE9BQU8sRUFBRSxHQUFHLE1BQU0sQ0FBQyxFQUFFLGlCQUFpQixDQUFDLENBQUM7SUFDMUYsQ0FBQztJQUNEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQixDQUFDLE9BQW1CLEVBQUUsR0FBRyxNQUFnQjtRQUM3RCxPQUFPLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxFQUFFLFdBQVcsQ0FBQyxNQUFNLENBQUMsY0FBYyxFQUFFLEdBQUcsTUFBTSxDQUFDLEVBQUUsaUJBQWlCLENBQUMsQ0FBQztJQUNqRyxDQUFDO0lBQ08sMEJBQTBCLENBQUMsS0FBc0I7O1FBQ3JELE1BQU0sd0JBQXdCLEdBQUcsYUFBQSxLQUFLLENBQUMsbUJBQW1CLDBDQUFFLG9CQUFvQiwwQ0FBRSxpQkFBaUI7WUFDL0YsaUJBQWlCLENBQUMsT0FBTyxDQUFDO1FBQzlCLElBQUksd0JBQXdCLEtBQUssaUJBQWlCLENBQUMsSUFBSTtZQUNuRCxjQUFDLEtBQUssQ0FBQyxtQkFBbUIsMENBQUUsb0JBQW9CLDBDQUFFLG1CQUFtQixDQUFBLEVBQUU7WUFDdkUsTUFBTSxJQUFJLEtBQUssQ0FBQyxvQ0FBb0MsQ0FBQyxDQUFDO1NBQ3pEO1FBQ0QsSUFBSSx3QkFBd0IsS0FBSyxpQkFBaUIsQ0FBQyxTQUFTO1lBQ3hELGNBQUMsS0FBSyxDQUFDLG1CQUFtQiwwQ0FBRSxvQkFBb0IsMENBQUUsY0FBYyxDQUFBLEVBQUU7WUFDbEUsTUFBTSxJQUFJLEtBQUssQ0FBQyx5Q0FBeUMsQ0FBQyxDQUFDO1NBQzlEO1FBQ0QsVUFBSSxLQUFLLENBQUMsbUJBQW1CLDBDQUFFLDRCQUE0QixFQUFFO1lBQ3pELEtBQUssQ0FBQyxtQkFBbUIsQ0FBQyw0QkFBNEIsQ0FBQyxPQUFPLENBQUMsQ0FBQyxpQkFBaUIsRUFBRSxFQUFFO2dCQUNqRixJQUFJLGlCQUFpQixDQUFDLGlCQUFpQixLQUFLLGlCQUFpQixDQUFDLE9BQU87b0JBQ2pFLHdCQUF3QixLQUFLLGlCQUFpQixDQUFDLE9BQU8sRUFBRTtvQkFDeEQsTUFBTSxJQUFJLEtBQUssQ0FBQyx1SUFBdUksQ0FBQyxDQUFDO2lCQUM1SjtnQkFDRCxJQUFJLGlCQUFpQixDQUFDLGlCQUFpQixLQUFLLGlCQUFpQixDQUFDLEdBQUc7b0JBQzdELHdCQUF3QixLQUFLLGlCQUFpQixDQUFDLEdBQUcsRUFBRTtvQkFDcEQsTUFBTSxJQUFJLEtBQUssQ0FBQyxtSUFBbUksQ0FBQyxDQUFDO2lCQUN4SjtnQkFDRCxJQUFJLGlCQUFpQixDQUFDLGlCQUFpQixLQUFLLGlCQUFpQixDQUFDLElBQUk7b0JBQzlELENBQUMsaUJBQWlCLENBQUMsbUJBQW1CLEVBQUU7b0JBQ3hDLE1BQU0sSUFBSSxLQUFLLENBQUMsb0VBQW9FLENBQUMsQ0FBQztpQkFDekY7Z0JBQ0QsSUFBSSxpQkFBaUIsQ0FBQyxpQkFBaUI7b0JBQ25DLGlCQUFpQixDQUFDLFNBQVM7b0JBQzNCLENBQUMsaUJBQWlCLENBQUMsY0FBYyxFQUFFO29CQUNuQyxNQUFNLElBQUksS0FBSyxDQUFDLHlFQUF5RSxDQUFDLENBQUM7aUJBQzlGO1lBQ0wsQ0FBQyxDQUFDLENBQUM7U0FDTjtJQUNMLENBQUM7SUFDTyx5QkFBeUIsQ0FBQyxNQUEyQjtRQUN6RCxPQUFPO1lBQ0gsT0FBTyxFQUFFLE1BQU0sQ0FBQyxtQkFBbUI7WUFDbkMsUUFBUSxFQUFFLE1BQU0sQ0FBQyxRQUFRO1lBQ3pCLE1BQU0sRUFBRSxNQUFNLENBQUMsb0JBQW9CO1lBQ25DLE1BQU0sRUFBRSxNQUFNLENBQUMsWUFBWTtTQUM5QixDQUFDO0lBQ04sQ0FBQztJQUNPLG9CQUFvQixDQUFDLE1BQXNCO1FBQy9DLE9BQU87WUFDSCxVQUFVLEVBQUUsTUFBTSxDQUFDLFFBQVEsQ0FBQyxVQUFVO1lBQ3RDLFNBQVMsRUFBRSxNQUFNLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxNQUFNO1lBQ3ZDLGdCQUFnQixFQUFFLE1BQU0sQ0FBQyxnQkFBZ0I7WUFDekMsYUFBYSxFQUFFLE1BQU0sQ0FBQyxhQUFhLElBQUksT0FBTztTQUNqRCxDQUFDO0lBQ04sQ0FBQztJQUNPLFlBQVksQ0FBQyxNQUFvQjtRQUNyQyxJQUFJLE9BQTJCLENBQUM7UUFDaEMsSUFBSSxNQUFNLENBQUMsT0FBTyxFQUFFO1lBQ2hCLE9BQU8sR0FBRyxJQUFJLElBQUksQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUFDLENBQUMsT0FBTyxFQUFFLENBQUM7WUFDN0MsTUFBTSxJQUFJLEdBQUcsQ0FBQyxDQUFTLEVBQUUsRUFBRSxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUUsR0FBRyxlQUFRLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLGNBQWMsRUFBRSxDQUFDO1lBQzNFLElBQUksT0FBTyxHQUFHLElBQUksQ0FBQyxDQUFDLENBQUMsSUFBSSxPQUFPLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFO2dCQUMxQyxNQUFNLEtBQUssQ0FBQyxvREFBb0QsQ0FBQyxDQUFDO2FBQ3JFO1lBQ0QsT0FBTyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxHQUFHLElBQUksQ0FBQyxDQUFDO1NBQ3hDO1FBQ0QsTUFBTSxHQUFHLEdBQUcsSUFBSSw2QkFBUyxDQUFDLElBQUksRUFBRSxHQUFHLE1BQU0sQ0FBQyxJQUFJLElBQUksZUFBZSxRQUFRLEVBQUU7WUFDdkUsT0FBTztZQUNQLFdBQVcsRUFBRSxNQUFNLENBQUMsV0FBVyxJQUFJLGdDQUFnQztZQUNuRSxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUs7U0FDcEIsQ0FBQyxDQUFDO1FBQ0gsT0FBTyxHQUFHLENBQUMsVUFBVSxDQUFDO0lBQzFCLENBQUM7SUFDTyxrQ0FBa0MsQ0FBQyxTQUE4QjtRQUNyRSxPQUFPLFNBQVMsQ0FBQyxNQUFNLENBQTJELENBQUMsR0FBRyxFQUFFLFFBQVEsRUFBRSxFQUFFLENBQUM7WUFDakcsR0FBRyxHQUFHO1lBQ047Z0JBQ0ksa0JBQWtCLEVBQUUsUUFBUSxDQUFDLGlCQUFpQjtnQkFDOUMsY0FBYyxFQUFFLFFBQVEsQ0FBQyxpQkFBaUIsS0FBSyxpQkFBaUIsQ0FBQyxTQUFTO29CQUN0RSxDQUFDLENBQUMsSUFBSSxDQUFDLG9CQUFvQixDQUFDLFFBQVEsQ0FBQyxjQUFlLENBQUM7b0JBQ3JELENBQUMsQ0FBQyxTQUFTO2dCQUNmLG1CQUFtQixFQUFFLFFBQVEsQ0FBQyxpQkFBaUIsS0FBSyxpQkFBaUIsQ0FBQyxJQUFJO29CQUN0RSxDQUFDLENBQUMsSUFBSSxDQUFDLHlCQUF5QixDQUFDLFFBQVEsQ0FBQyxtQkFBb0IsQ0FBQztvQkFDL0QsQ0FBQyxDQUFDLFNBQVM7YUFDbEI7U0FDSixFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQ1gsQ0FBQztJQUNPLHVDQUF1QyxDQUFDLEtBQXNCOztRQUNsRSxNQUFNLFNBQVMsU0FBRyxLQUFLLENBQUMsbUJBQW1CLDBDQUFFLDRCQUE0QixDQUFDO1FBQzFFLE9BQU8sU0FBUyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsa0NBQWtDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQztJQUN0RixDQUFDO0lBQ0Q7Ozs7O09BS0c7SUFDSSxnQkFBZ0IsQ0FBQyxVQUFrQjtRQUN0QyxJQUFJLElBQUksQ0FBQyxVQUFVLElBQUksZ0JBQWdCLENBQUMsSUFBSSxFQUFFO1lBQzFDLE1BQU0sSUFBSSxLQUFLLENBQUMsK0VBQStFLENBQUMsQ0FBQztTQUNwRztRQUNELElBQUksQ0FBQyxNQUFNLENBQUMsVUFBVSxHQUFHLFVBQVUsQ0FBQztJQUN4QyxDQUFDO0lBQ0Q7OztPQUdHO0lBQ0ssWUFBWSxDQUFDLElBQWE7UUFDOUIsSUFBSSxVQUFVLENBQUM7UUFDZixJQUFJLElBQUksQ0FBQyxVQUFVLElBQUksZ0JBQWdCLENBQUMsSUFBSSxJQUFJLENBQUMsSUFBSSxFQUFFO1lBQ25ELE1BQU0sSUFBSSxLQUFLLENBQUMsd0VBQXdFLENBQUMsQ0FBQztTQUM3RjthQUNJLElBQUksSUFBSSxDQUFDLFVBQVUsSUFBSSxnQkFBZ0IsQ0FBQyxJQUFJLElBQUksSUFBSSxFQUFFO1lBQ3ZELFVBQVUsR0FBRyxpQkFBWSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQztTQUNyRDthQUNJLElBQUksSUFBSSxDQUFDLFVBQVUsSUFBSSxnQkFBZ0IsQ0FBQyxJQUFJLElBQUksQ0FBQyxJQUFJLEVBQUU7WUFDeEQsVUFBVSxHQUFHLEVBQUUsQ0FBQztTQUNuQjthQUNJLElBQUksSUFBSSxDQUFDLFVBQVUsSUFBSSxnQkFBZ0IsQ0FBQyxJQUFJLElBQUksSUFBSSxFQUFFO1lBQ3ZELE1BQU0sSUFBSSxLQUFLLENBQUMsdUhBQXVILENBQUMsQ0FBQztTQUM1STtRQUNELE9BQU8sSUFBSSxvQ0FBZ0IsQ0FBQyxJQUFJLEVBQUUsUUFBUSxFQUFFO1lBQ3hDLEtBQUssRUFBRSxJQUFJLENBQUMsS0FBSztZQUNqQixVQUFVO1NBQ2IsQ0FBQyxDQUFDO0lBQ1AsQ0FBQztDQUNKO0FBNVNELGdDQTRTQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IHJlYWRGaWxlU3luYyB9IGZyb20gJ2ZzJztcbmltcG9ydCB7IElVc2VyUG9vbCB9IGZyb20gXCIuLi8uLi9hd3MtY29nbml0b1wiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWNvZ25pdG8nXG5pbXBvcnQgeyBJVGFibGUgfSBmcm9tIFwiLi4vLi4vYXdzLWR5bmFtb2RiXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3MtZHluYW1vZGInXG5pbXBvcnQgeyBHcmFudCwgSUdyYW50YWJsZSwgTWFuYWdlZFBvbGljeSwgUm9sZSwgU2VydmljZVByaW5jaXBhbCB9IGZyb20gXCIuLi8uLi9hd3MtaWFtXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3MtaWFtJ1xuaW1wb3J0IHsgSUZ1bmN0aW9uIH0gZnJvbSBcIi4uLy4uL2F3cy1sYW1iZGFcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1sYW1iZGEnXG5pbXBvcnQgeyBDb25zdHJ1Y3QsIER1cmF0aW9uLCBJUmVzb2x2YWJsZSwgU3RhY2sgfSBmcm9tIFwiLi4vLi4vY29yZVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvY29yZSdcbmltcG9ydCB7IENmbkFwaUtleSwgQ2ZuR3JhcGhRTEFwaSwgQ2ZuR3JhcGhRTFNjaGVtYSB9IGZyb20gJy4vYXBwc3luYy5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgRHluYW1vRGJEYXRhU291cmNlLCBIdHRwRGF0YVNvdXJjZSwgTGFtYmRhRGF0YVNvdXJjZSwgTm9uZURhdGFTb3VyY2UgfSBmcm9tICcuL2RhdGEtc291cmNlJztcbi8qKlxuICogZW51bSB3aXRoIGFsbCBwb3NzaWJsZSB2YWx1ZXMgZm9yIEFwcFN5bmMgYXV0aG9yaXphdGlvbiB0eXBlXG4gKi9cbmV4cG9ydCBlbnVtIEF1dGhvcml6YXRpb25UeXBlIHtcbiAgICAvKipcbiAgICAgKiBBUEkgS2V5IGF1dGhvcml6YXRpb24gdHlwZVxuICAgICAqL1xuICAgIEFQSV9LRVkgPSAnQVBJX0tFWScsXG4gICAgLyoqXG4gICAgICogQVdTIElBTSBhdXRob3JpemF0aW9uIHR5cGUuIENhbiBiZSB1c2VkIHdpdGggQ29nbml0byBJZGVudGl0eSBQb29sIGZlZGVyYXRlZCBjcmVkZW50aWFsc1xuICAgICAqL1xuICAgIElBTSA9ICdBV1NfSUFNJyxcbiAgICAvKipcbiAgICAgKiBDb2duaXRvIFVzZXIgUG9vbCBhdXRob3JpemF0aW9uIHR5cGVcbiAgICAgKi9cbiAgICBVU0VSX1BPT0wgPSAnQU1BWk9OX0NPR05JVE9fVVNFUl9QT09MUycsXG4gICAgLyoqXG4gICAgICogT3BlbklEIENvbm5lY3QgYXV0aG9yaXphdGlvbiB0eXBlXG4gICAgICovXG4gICAgT0lEQyA9ICdPUEVOSURfQ09OTkVDVCdcbn1cbi8qKlxuICogSW50ZXJmYWNlIHRvIHNwZWNpZnkgZGVmYXVsdCBvciBhZGRpdGlvbmFsIGF1dGhvcml6YXRpb24ocylcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBBdXRob3JpemF0aW9uTW9kZSB7XG4gICAgLyoqXG4gICAgICogT25lIG9mIHBvc3NpYmxlIGZvdXIgdmFsdWVzIEFwcFN5bmMgc3VwcG9ydHNcbiAgICAgKlxuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN5bmMvbGF0ZXN0L2Rldmd1aWRlL3NlY3VyaXR5Lmh0bWxcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gYEF1dGhvcml6YXRpb25UeXBlLkFQSV9LRVlgXG4gICAgICovXG4gICAgcmVhZG9ubHkgYXV0aG9yaXphdGlvblR5cGU6IEF1dGhvcml6YXRpb25UeXBlO1xuICAgIC8qKlxuICAgICAqIElmIGF1dGhvcml6YXRpb25UeXBlIGlzIGBBdXRob3JpemF0aW9uVHlwZS5VU0VSX1BPT0xgLCB0aGlzIG9wdGlvbiBpcyByZXF1aXJlZC5cbiAgICAgKiBAZGVmYXVsdCAtIG5vbmVcbiAgICAgKi9cbiAgICByZWFkb25seSB1c2VyUG9vbENvbmZpZz86IFVzZXJQb29sQ29uZmlnO1xuICAgIC8qKlxuICAgICAqIElmIGF1dGhvcml6YXRpb25UeXBlIGlzIGBBdXRob3JpemF0aW9uVHlwZS5BUElfS0VZYCwgdGhpcyBvcHRpb24gY2FuIGJlIGNvbmZpZ3VyZWQuXG4gICAgICogQGRlZmF1bHQgLSBuYW1lOiAnRGVmYXVsdEFQSUtleScgfCBkZXNjcmlwdGlvbjogJ0RlZmF1bHQgQVBJIEtleSBjcmVhdGVkIGJ5IENESydcbiAgICAgKi9cbiAgICByZWFkb25seSBhcGlLZXlDb25maWc/OiBBcGlLZXlDb25maWc7XG4gICAgLyoqXG4gICAgICogSWYgYXV0aG9yaXphdGlvblR5cGUgaXMgYEF1dGhvcml6YXRpb25UeXBlLk9JRENgLCB0aGlzIG9wdGlvbiBpcyByZXF1aXJlZC5cbiAgICAgKiBAZGVmYXVsdCAtIG5vbmVcbiAgICAgKi9cbiAgICByZWFkb25seSBvcGVuSWRDb25uZWN0Q29uZmlnPzogT3BlbklkQ29ubmVjdENvbmZpZztcbn1cbi8qKlxuICogZW51bSB3aXRoIGFsbCBwb3NzaWJsZSB2YWx1ZXMgZm9yIENvZ25pdG8gdXNlci1wb29sIGRlZmF1bHQgYWN0aW9uc1xuICovXG5leHBvcnQgZW51bSBVc2VyUG9vbERlZmF1bHRBY3Rpb24ge1xuICAgIC8qKlxuICAgICAqIEFMTE9XIGFjY2VzcyB0byBBUElcbiAgICAgKi9cbiAgICBBTExPVyA9ICdBTExPVycsXG4gICAgLyoqXG4gICAgICogREVOWSBhY2Nlc3MgdG8gQVBJXG4gICAgICovXG4gICAgREVOWSA9ICdERU5ZJ1xufVxuLyoqXG4gKiBDb25maWd1cmF0aW9uIGZvciBDb2duaXRvIHVzZXItcG9vbHMgaW4gQXBwU3luY1xuICovXG5leHBvcnQgaW50ZXJmYWNlIFVzZXJQb29sQ29uZmlnIHtcbiAgICAvKipcbiAgICAgKiBUaGUgQ29nbml0byB1c2VyIHBvb2wgdG8gdXNlIGFzIGlkZW50aXR5IHNvdXJjZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHVzZXJQb29sOiBJVXNlclBvb2w7XG4gICAgLyoqXG4gICAgICogdGhlIG9wdGlvbmFsIGFwcCBpZCByZWdleFxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSAgTm9uZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGFwcElkQ2xpZW50UmVnZXg/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogRGVmYXVsdCBhdXRoIGFjdGlvblxuICAgICAqXG4gICAgICogQGRlZmF1bHQgQUxMT1dcbiAgICAgKi9cbiAgICByZWFkb25seSBkZWZhdWx0QWN0aW9uPzogVXNlclBvb2xEZWZhdWx0QWN0aW9uO1xufVxuLyoqXG4gKiBDb25maWd1cmF0aW9uIGZvciBBUEkgS2V5IGF1dGhvcml6YXRpb24gaW4gQXBwU3luY1xuICovXG5leHBvcnQgaW50ZXJmYWNlIEFwaUtleUNvbmZpZyB7XG4gICAgLyoqXG4gICAgICogVW5pcXVlIG5hbWUgb2YgdGhlIEFQSSBLZXlcbiAgICAgKiBAZGVmYXVsdCAtICdEZWZhdWx0QVBJS2V5J1xuICAgICAqL1xuICAgIHJlYWRvbmx5IG5hbWU/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogRGVzY3JpcHRpb24gb2YgQVBJIGtleVxuICAgICAqIEBkZWZhdWx0IC0gJ0RlZmF1bHQgQVBJIEtleSBjcmVhdGVkIGJ5IENESydcbiAgICAgKi9cbiAgICByZWFkb25seSBkZXNjcmlwdGlvbj86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgdGltZSBmcm9tIGNyZWF0aW9uIHRpbWUgYWZ0ZXIgd2hpY2ggdGhlIEFQSSBrZXkgZXhwaXJlcywgdXNpbmcgUkZDMzMzOSByZXByZXNlbnRhdGlvbi5cbiAgICAgKiBJdCBtdXN0IGJlIGEgbWluaW11bSBvZiAxIGRheSBhbmQgYSBtYXhpbXVtIG9mIDM2NSBkYXlzIGZyb20gZGF0ZSBvZiBjcmVhdGlvbi5cbiAgICAgKiBSb3VuZGVkIGRvd24gdG8gdGhlIG5lYXJlc3QgaG91ci5cbiAgICAgKiBAZGVmYXVsdCAtIDcgZGF5cyBmcm9tIGNyZWF0aW9uIHRpbWVcbiAgICAgKi9cbiAgICByZWFkb25seSBleHBpcmVzPzogc3RyaW5nO1xufVxuLyoqXG4gKiBDb25maWd1cmF0aW9uIGZvciBPcGVuSUQgQ29ubmVjdCBhdXRob3JpemF0aW9uIGluIEFwcFN5bmNcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBPcGVuSWRDb25uZWN0Q29uZmlnIHtcbiAgICAvKipcbiAgICAgKiBUaGUgbnVtYmVyIG9mIG1pbGxpc2Vjb25kcyBhbiBPSURDIHRva2VuIGlzIHZhbGlkIGFmdGVyIGJlaW5nIGF1dGhlbnRpY2F0ZWQgYnkgT0lEQyBwcm92aWRlci5cbiAgICAgKiBgYXV0aF90aW1lYCBjbGFpbSBpbiBPSURDIHRva2VuIGlzIHJlcXVpcmVkIGZvciB0aGlzIHZhbGlkYXRpb24gdG8gd29yay5cbiAgICAgKiBAZGVmYXVsdCAtIG5vIHZhbGlkYXRpb25cbiAgICAgKi9cbiAgICByZWFkb25seSB0b2tlbkV4cGlyeUZyb21BdXRoPzogbnVtYmVyO1xuICAgIC8qKlxuICAgICAqIFRoZSBudW1iZXIgb2YgbWlsbGlzZWNvbmRzIGFuIE9JREMgdG9rZW4gaXMgdmFsaWQgYWZ0ZXIgYmVpbmcgaXNzdWVkIHRvIGEgdXNlci5cbiAgICAgKiBUaGlzIHZhbGlkYXRpb24gdXNlcyBgaWF0YCBjbGFpbSBvZiBPSURDIHRva2VuLlxuICAgICAqIEBkZWZhdWx0IC0gbm8gdmFsaWRhdGlvblxuICAgICAqL1xuICAgIHJlYWRvbmx5IHRva2VuRXhwaXJ5RnJvbUlzc3VlPzogbnVtYmVyO1xuICAgIC8qKlxuICAgICAqIFRoZSBjbGllbnQgaWRlbnRpZmllciBvZiB0aGUgUmVseWluZyBwYXJ0eSBhdCB0aGUgT3BlbklEIGlkZW50aXR5IHByb3ZpZGVyLlxuICAgICAqIEEgcmVndWxhciBleHByZXNzaW9uIGNhbiBiZSBzcGVjaWZpZWQgc28gQXBwU3luYyBjYW4gdmFsaWRhdGUgYWdhaW5zdCBtdWx0aXBsZSBjbGllbnQgaWRlbnRpZmllcnMgYXQgYSB0aW1lLlxuICAgICAqIEBleGFtcGxlIC0gJ0FCQ0R8Q0RFRicgd2hlcmUgQUJDRCBhbmQgQ0RFRiBhcmUgdHdvIGRpZmZlcmVudCBjbGllbnRJZFxuICAgICAqIEBkZWZhdWx0IC0gKiAoQWxsKVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGNsaWVudElkPzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSBpc3N1ZXIgZm9yIHRoZSBPSURDIGNvbmZpZ3VyYXRpb24uIFRoZSBpc3N1ZXIgcmV0dXJuZWQgYnkgZGlzY292ZXJ5IG11c3QgZXhhY3RseSBtYXRjaCB0aGUgdmFsdWUgb2YgYGlzc2AgaW4gdGhlIE9JREMgdG9rZW4uXG4gICAgICovXG4gICAgcmVhZG9ubHkgb2lkY1Byb3ZpZGVyOiBzdHJpbmc7XG59XG4vKipcbiAqIENvbmZpZ3VyYXRpb24gb2YgdGhlIEFQSSBhdXRob3JpemF0aW9uIG1vZGVzLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEF1dGhvcml6YXRpb25Db25maWcge1xuICAgIC8qKlxuICAgICAqIE9wdGlvbmFsIGF1dGhvcml6YXRpb24gY29uZmlndXJhdGlvblxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBBUEkgS2V5IGF1dGhvcml6YXRpb25cbiAgICAgKi9cbiAgICByZWFkb25seSBkZWZhdWx0QXV0aG9yaXphdGlvbj86IEF1dGhvcml6YXRpb25Nb2RlO1xuICAgIC8qKlxuICAgICAqIEFkZGl0aW9uYWwgYXV0aG9yaXphdGlvbiBtb2Rlc1xuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBObyBvdGhlciBtb2Rlc1xuICAgICAqL1xuICAgIHJlYWRvbmx5IGFkZGl0aW9uYWxBdXRob3JpemF0aW9uTW9kZXM/OiBBdXRob3JpemF0aW9uTW9kZVtdO1xufVxuLyoqXG4gKiBsb2ctbGV2ZWwgZm9yIGZpZWxkcyBpbiBBcHBTeW5jXG4gKi9cbmV4cG9ydCBlbnVtIEZpZWxkTG9nTGV2ZWwge1xuICAgIC8qKlxuICAgICAqIE5vIGxvZ2dpbmdcbiAgICAgKi9cbiAgICBOT05FID0gJ05PTkUnLFxuICAgIC8qKlxuICAgICAqIEVycm9yIGxvZ2dpbmdcbiAgICAgKi9cbiAgICBFUlJPUiA9ICdFUlJPUicsXG4gICAgLyoqXG4gICAgICogQWxsIGxvZ2dpbmdcbiAgICAgKi9cbiAgICBBTEwgPSAnQUxMJ1xufVxuLyoqXG4gKiBMb2dnaW5nIGNvbmZpZ3VyYXRpb24gZm9yIEFwcFN5bmNcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBMb2dDb25maWcge1xuICAgIC8qKlxuICAgICAqIGV4Y2x1ZGUgdmVyYm9zZSBjb250ZW50XG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGV4Y2x1ZGVWZXJib3NlQ29udGVudD86IGJvb2xlYW4gfCBJUmVzb2x2YWJsZTtcbiAgICAvKipcbiAgICAgKiBsb2cgbGV2ZWwgZm9yIGZpZWxkc1xuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBVc2UgQXBwU3luYyBkZWZhdWx0XG4gICAgICovXG4gICAgcmVhZG9ubHkgZmllbGRMb2dMZXZlbD86IEZpZWxkTG9nTGV2ZWw7XG59XG4vKipcbiAqIEVudW0gY29udGFpbmluZyB0aGUgZGlmZmVyZW50IG1vZGVzIG9mIHNjaGVtYSBkZWZpbml0aW9uXG4gKi9cbmV4cG9ydCBlbnVtIFNjaGVtYURlZmluaXRpb24ge1xuICAgIC8qKlxuICAgICAqIERlZmluZSBzY2hlbWEgdGhyb3VnaCBmdW5jdGlvbnMgbGlrZSBhZGRUeXBlLCBhZGRRdWVyeSwgZXRjLlxuICAgICAqL1xuICAgIENPREUgPSAnQ09ERScsXG4gICAgLyoqXG4gICAgICogRGVmaW5lIHNjaGVtYSBpbiBhIGZpbGUsIGkuZS4gc2NoZW1hLmdyYXBocWxcbiAgICAgKi9cbiAgICBGSUxFID0gJ0ZJTEUnXG59XG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGFuIEFwcFN5bmMgR3JhcGhRTCBBUElcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBHcmFwaFFMQXBpUHJvcHMge1xuICAgIC8qKlxuICAgICAqIHRoZSBuYW1lIG9mIHRoZSBHcmFwaFFMIEFQSVxuICAgICAqL1xuICAgIHJlYWRvbmx5IG5hbWU6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBPcHRpb25hbCBhdXRob3JpemF0aW9uIGNvbmZpZ3VyYXRpb25cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gQVBJIEtleSBhdXRob3JpemF0aW9uXG4gICAgICovXG4gICAgcmVhZG9ubHkgYXV0aG9yaXphdGlvbkNvbmZpZz86IEF1dGhvcml6YXRpb25Db25maWc7XG4gICAgLyoqXG4gICAgICogTG9nZ2luZyBjb25maWd1cmF0aW9uIGZvciB0aGlzIGFwaVxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBOb25lXG4gICAgICovXG4gICAgcmVhZG9ubHkgbG9nQ29uZmlnPzogTG9nQ29uZmlnO1xuICAgIC8qKlxuICAgICAqIEdyYXBoUUwgc2NoZW1hIGRlZmluaXRpb24uIFNwZWNpZnkgaG93IHlvdSB3YW50IHRvIGRlZmluZSB5b3VyIHNjaGVtYS5cbiAgICAgKlxuICAgICAqIFNjaGVtYURlZmluaXRpb24uQ09ERSBhbGxvd3Mgc2NoZW1hIGRlZmluaXRpb24gdGhyb3VnaCBDREtcbiAgICAgKiBTY2hlbWFEZWZpbml0aW9uLkZJTEUgYWxsb3dzIHNjaGVtYSBkZWZpbml0aW9uIHRocm91Z2ggc2NoZW1hLmdyYXBocWwgZmlsZVxuICAgICAqXG4gICAgICogQGV4cGVyaW1lbnRhbFxuICAgICAqL1xuICAgIHJlYWRvbmx5IHNjaGVtYURlZmluaXRpb246IFNjaGVtYURlZmluaXRpb247XG4gICAgLyoqXG4gICAgICogRmlsZSBjb250YWluaW5nIHRoZSBHcmFwaFFMIHNjaGVtYSBkZWZpbml0aW9uLiBZb3UgaGF2ZSB0byBzcGVjaWZ5IGEgZGVmaW5pdGlvbiBvciBhIGZpbGUgY29udGFpbmluZyBvbmUuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIFVzZSBzY2hlbWFEZWZpbml0aW9uXG4gICAgICovXG4gICAgcmVhZG9ubHkgc2NoZW1hRGVmaW5pdGlvbkZpbGU/OiBzdHJpbmc7XG59XG4vKipcbiAqIEEgY2xhc3MgdXNlZCB0byBnZW5lcmF0ZSByZXNvdXJjZSBhcm5zIGZvciBBcHBTeW5jXG4gKi9cbmV4cG9ydCBjbGFzcyBJYW1SZXNvdXJjZSB7XG4gICAgLyoqXG4gICAgICogR2VuZXJhdGUgdGhlIHJlc291cmNlIG5hbWVzIGdpdmVuIGN1c3RvbSBhcm5zXG4gICAgICpcbiAgICAgKiBAcGFyYW0gYXJucyBUaGUgY3VzdG9tIGFybnMgdGhhdCBuZWVkIHRvIGJlIHBlcm1pc3Npb25lZFxuICAgICAqXG4gICAgICogRXhhbXBsZTogY3VzdG9tKCcvdHlwZXMvUXVlcnkvZmllbGRzL2dldEV4YW1wbGUnKVxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgY3VzdG9tKC4uLmFybnM6IHN0cmluZ1tdKTogSWFtUmVzb3VyY2Uge1xuICAgICAgICBpZiAoYXJucy5sZW5ndGggPT09IDApIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignQXQgbGVhc3QgMSBjdXN0b20gQVJOIG11c3QgYmUgcHJvdmlkZWQuJyk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIG5ldyBJYW1SZXNvdXJjZShhcm5zKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogR2VuZXJhdGUgdGhlIHJlc291cmNlIG5hbWVzIGdpdmVuIGEgdHlwZSBhbmQgZmllbGRzXG4gICAgICpcbiAgICAgKiBAcGFyYW0gdHlwZSBUaGUgdHlwZSB0aGF0IG5lZWRzIHRvIGJlIGFsbG93ZWRcbiAgICAgKiBAcGFyYW0gZmllbGRzIFRoZSBmaWVsZHMgdGhhdCBuZWVkIHRvIGJlIGFsbG93ZWQsIGlmIGVtcHR5IGdyYW50IHBlcm1pc3Npb25zIHRvIEFMTCBmaWVsZHNcbiAgICAgKlxuICAgICAqIEV4YW1wbGU6IG9mVHlwZSgnUXVlcnknLCAnR2V0RXhhbXBsZScpXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyBvZlR5cGUodHlwZTogc3RyaW5nLCAuLi5maWVsZHM6IHN0cmluZ1tdKTogSWFtUmVzb3VyY2Uge1xuICAgICAgICBjb25zdCBhcm5zID0gZmllbGRzLmxlbmd0aCA/IGZpZWxkcy5tYXAoKGZpZWxkKSA9PiBgdHlwZXMvJHt0eXBlfS9maWVsZHMvJHtmaWVsZH1gKSA6IFtgdHlwZXMvJHt0eXBlfS8qYF07XG4gICAgICAgIHJldHVybiBuZXcgSWFtUmVzb3VyY2UoYXJucyk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEdlbmVyYXRlIHRoZSByZXNvdXJjZSBuYW1lcyB0aGF0IGFjY2VwdHMgYWxsIHR5cGVzOiBgKmBcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGFsbCgpOiBJYW1SZXNvdXJjZSB7XG4gICAgICAgIHJldHVybiBuZXcgSWFtUmVzb3VyY2UoWycqJ10pO1xuICAgIH1cbiAgICBwcml2YXRlIGFybnM6IHN0cmluZ1tdO1xuICAgIHByaXZhdGUgY29uc3RydWN0b3IoYXJuczogc3RyaW5nW10pIHtcbiAgICAgICAgdGhpcy5hcm5zID0gYXJucztcbiAgICB9XG4gICAgLyoqXG4gICAgICogUmV0dXJuIHRoZSBSZXNvdXJjZSBBUk5cbiAgICAgKlxuICAgICAqIEBwYXJhbSBhcGkgVGhlIEdyYXBoUUwgQVBJIHRvIGdpdmUgcGVybWlzc2lvbnNcbiAgICAgKi9cbiAgICBwdWJsaWMgcmVzb3VyY2VBcm5zKGFwaTogR3JhcGhRTEFwaSk6IHN0cmluZ1tdIHtcbiAgICAgICAgcmV0dXJuIHRoaXMuYXJucy5tYXAoKGFybikgPT4gU3RhY2sub2YoYXBpKS5mb3JtYXRBcm4oe1xuICAgICAgICAgICAgc2VydmljZTogJ2FwcHN5bmMnLFxuICAgICAgICAgICAgcmVzb3VyY2U6IGBhcGlzLyR7YXBpLmFwaUlkfWAsXG4gICAgICAgICAgICBzZXA6ICcvJyxcbiAgICAgICAgICAgIHJlc291cmNlTmFtZTogYCR7YXJufWAsXG4gICAgICAgIH0pKTtcbiAgICB9XG59XG4vKipcbiAqIEFuIEFwcFN5bmMgR3JhcGhRTCBBUElcbiAqL1xuZXhwb3J0IGNsYXNzIEdyYXBoUUxBcGkgZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuICAgIC8qKlxuICAgICAqIHRoZSBpZCBvZiB0aGUgR3JhcGhRTCBBUElcbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgYXBpSWQ6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiB0aGUgQVJOIG9mIHRoZSBBUElcbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgYXJuOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogdGhlIFVSTCBvZiB0aGUgZW5kcG9pbnQgY3JlYXRlZCBieSBBcHBTeW5jXG4gICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IGdyYXBoUWxVcmw6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiB0aGUgbmFtZSBvZiB0aGUgQVBJXG4gICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IG5hbWU6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiB1bmRlcmx5aW5nIENGTiBzY2hlbWEgcmVzb3VyY2VcbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgc2NoZW1hOiBDZm5HcmFwaFFMU2NoZW1hO1xuICAgIC8qKlxuICAgICAqIHRoZSBjb25maWd1cmVkIEFQSSBrZXksIGlmIHByZXNlbnRcbiAgICAgKi9cbiAgICBwdWJsaWMgZ2V0IGFwaUtleSgpOiBzdHJpbmcgfCB1bmRlZmluZWQge1xuICAgICAgICByZXR1cm4gdGhpcy5fYXBpS2V5O1xuICAgIH1cbiAgICBwcml2YXRlIHNjaGVtYU1vZGU6IFNjaGVtYURlZmluaXRpb247XG4gICAgcHJpdmF0ZSBhcGk6IENmbkdyYXBoUUxBcGk7XG4gICAgcHJpdmF0ZSBfYXBpS2V5Pzogc3RyaW5nO1xuICAgIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBHcmFwaFFMQXBpUHJvcHMpIHtcbiAgICAgICAgc3VwZXIoc2NvcGUsIGlkKTtcbiAgICAgICAgdGhpcy52YWxpZGF0ZUF1dGhvcml6YXRpb25Qcm9wcyhwcm9wcyk7XG4gICAgICAgIGNvbnN0IGRlZmF1bHRBdXRob3JpemF0aW9uVHlwZSA9IHByb3BzLmF1dGhvcml6YXRpb25Db25maWc/LmRlZmF1bHRBdXRob3JpemF0aW9uPy5hdXRob3JpemF0aW9uVHlwZSB8fFxuICAgICAgICAgICAgQXV0aG9yaXphdGlvblR5cGUuQVBJX0tFWTtcbiAgICAgICAgbGV0IGFwaUxvZ3NSb2xlO1xuICAgICAgICBpZiAocHJvcHMubG9nQ29uZmlnKSB7XG4gICAgICAgICAgICBhcGlMb2dzUm9sZSA9IG5ldyBSb2xlKHRoaXMsICdBcGlMb2dzUm9sZScsIHtcbiAgICAgICAgICAgICAgICBhc3N1bWVkQnk6IG5ldyBTZXJ2aWNlUHJpbmNpcGFsKCdhcHBzeW5jJyksXG4gICAgICAgICAgICB9KTtcbiAgICAgICAgICAgIGFwaUxvZ3NSb2xlLmFkZE1hbmFnZWRQb2xpY3koTWFuYWdlZFBvbGljeS5mcm9tQXdzTWFuYWdlZFBvbGljeU5hbWUoJ3NlcnZpY2Utcm9sZS9BV1NBcHBTeW5jUHVzaFRvQ2xvdWRXYXRjaExvZ3MnKSk7XG4gICAgICAgIH1cbiAgICAgICAgdGhpcy5hcGkgPSBuZXcgQ2ZuR3JhcGhRTEFwaSh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICAgICAgICBuYW1lOiBwcm9wcy5uYW1lLFxuICAgICAgICAgICAgYXV0aGVudGljYXRpb25UeXBlOiBkZWZhdWx0QXV0aG9yaXphdGlvblR5cGUsXG4gICAgICAgICAgICAuLi4ocHJvcHMubG9nQ29uZmlnICYmIHtcbiAgICAgICAgICAgICAgICBsb2dDb25maWc6IHtcbiAgICAgICAgICAgICAgICAgICAgY2xvdWRXYXRjaExvZ3NSb2xlQXJuOiBhcGlMb2dzUm9sZSA/IGFwaUxvZ3NSb2xlLnJvbGVBcm4gOiB1bmRlZmluZWQsXG4gICAgICAgICAgICAgICAgICAgIGV4Y2x1ZGVWZXJib3NlQ29udGVudDogcHJvcHMubG9nQ29uZmlnLmV4Y2x1ZGVWZXJib3NlQ29udGVudCxcbiAgICAgICAgICAgICAgICAgICAgZmllbGRMb2dMZXZlbDogcHJvcHMubG9nQ29uZmlnLmZpZWxkTG9nTGV2ZWxcbiAgICAgICAgICAgICAgICAgICAgICAgID8gcHJvcHMubG9nQ29uZmlnLmZpZWxkTG9nTGV2ZWwudG9TdHJpbmcoKVxuICAgICAgICAgICAgICAgICAgICAgICAgOiB1bmRlZmluZWQsXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0pLFxuICAgICAgICAgICAgb3BlbklkQ29ubmVjdENvbmZpZzogcHJvcHMuYXV0aG9yaXphdGlvbkNvbmZpZz8uZGVmYXVsdEF1dGhvcml6YXRpb24/LmF1dGhvcml6YXRpb25UeXBlID09PVxuICAgICAgICAgICAgICAgIEF1dGhvcml6YXRpb25UeXBlLk9JRENcbiAgICAgICAgICAgICAgICA/IHRoaXMuZm9ybWF0T3BlbklkQ29ubmVjdENvbmZpZyhwcm9wcy5hdXRob3JpemF0aW9uQ29uZmlnLmRlZmF1bHRBdXRob3JpemF0aW9uXG4gICAgICAgICAgICAgICAgICAgIC5vcGVuSWRDb25uZWN0Q29uZmlnISlcbiAgICAgICAgICAgICAgICA6IHVuZGVmaW5lZCxcbiAgICAgICAgICAgIHVzZXJQb29sQ29uZmlnOiBwcm9wcy5hdXRob3JpemF0aW9uQ29uZmlnPy5kZWZhdWx0QXV0aG9yaXphdGlvbj8uYXV0aG9yaXphdGlvblR5cGUgPT09XG4gICAgICAgICAgICAgICAgQXV0aG9yaXphdGlvblR5cGUuVVNFUl9QT09MXG4gICAgICAgICAgICAgICAgPyB0aGlzLmZvcm1hdFVzZXJQb29sQ29uZmlnKHByb3BzLmF1dGhvcml6YXRpb25Db25maWcuZGVmYXVsdEF1dGhvcml6YXRpb24udXNlclBvb2xDb25maWchKVxuICAgICAgICAgICAgICAgIDogdW5kZWZpbmVkLFxuICAgICAgICAgICAgYWRkaXRpb25hbEF1dGhlbnRpY2F0aW9uUHJvdmlkZXJzOiB0aGlzLmZvcm1hdEFkZGl0aW9uYWxBdXRoZW50aWNhdGlvblByb3ZpZGVycyhwcm9wcyksXG4gICAgICAgIH0pO1xuICAgICAgICB0aGlzLmFwaUlkID0gdGhpcy5hcGkuYXR0ckFwaUlkO1xuICAgICAgICB0aGlzLmFybiA9IHRoaXMuYXBpLmF0dHJBcm47XG4gICAgICAgIHRoaXMuZ3JhcGhRbFVybCA9IHRoaXMuYXBpLmF0dHJHcmFwaFFsVXJsO1xuICAgICAgICB0aGlzLm5hbWUgPSB0aGlzLmFwaS5uYW1lO1xuICAgICAgICB0aGlzLnNjaGVtYU1vZGUgPSBwcm9wcy5zY2hlbWFEZWZpbml0aW9uO1xuICAgICAgICBpZiAoZGVmYXVsdEF1dGhvcml6YXRpb25UeXBlID09PSBBdXRob3JpemF0aW9uVHlwZS5BUElfS0VZIHx8XG4gICAgICAgICAgICBwcm9wcy5hdXRob3JpemF0aW9uQ29uZmlnPy5hZGRpdGlvbmFsQXV0aG9yaXphdGlvbk1vZGVzPy5zb21lKChhdXRoTW9kZSkgPT4gYXV0aE1vZGUuYXV0aG9yaXphdGlvblR5cGUgPT09IEF1dGhvcml6YXRpb25UeXBlLkFQSV9LRVkpKSB7XG4gICAgICAgICAgICBjb25zdCBhcGlLZXlDb25maWc6IEFwaUtleUNvbmZpZyA9IHByb3BzLmF1dGhvcml6YXRpb25Db25maWdcbiAgICAgICAgICAgICAgICA/LmRlZmF1bHRBdXRob3JpemF0aW9uPy5hcGlLZXlDb25maWcgfHwge1xuICAgICAgICAgICAgICAgIG5hbWU6ICdEZWZhdWx0QVBJS2V5JyxcbiAgICAgICAgICAgICAgICBkZXNjcmlwdGlvbjogJ0RlZmF1bHQgQVBJIEtleSBjcmVhdGVkIGJ5IENESycsXG4gICAgICAgICAgICB9O1xuICAgICAgICAgICAgdGhpcy5fYXBpS2V5ID0gdGhpcy5jcmVhdGVBUElLZXkoYXBpS2V5Q29uZmlnKTtcbiAgICAgICAgfVxuICAgICAgICB0aGlzLnNjaGVtYSA9IHRoaXMuZGVmaW5lU2NoZW1hKHByb3BzLnNjaGVtYURlZmluaXRpb25GaWxlKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogYWRkIGEgbmV3IGR1bW15IGRhdGEgc291cmNlIHRvIHRoaXMgQVBJXG4gICAgICogQHBhcmFtIG5hbWUgVGhlIG5hbWUgb2YgdGhlIGRhdGEgc291cmNlXG4gICAgICogQHBhcmFtIGRlc2NyaXB0aW9uIFRoZSBkZXNjcmlwdGlvbiBvZiB0aGUgZGF0YSBzb3VyY2VcbiAgICAgKi9cbiAgICBwdWJsaWMgYWRkTm9uZURhdGFTb3VyY2UobmFtZTogc3RyaW5nLCBkZXNjcmlwdGlvbjogc3RyaW5nKTogTm9uZURhdGFTb3VyY2Uge1xuICAgICAgICByZXR1cm4gbmV3IE5vbmVEYXRhU291cmNlKHRoaXMsIGAke25hbWV9RFNgLCB7XG4gICAgICAgICAgICBhcGk6IHRoaXMsXG4gICAgICAgICAgICBkZXNjcmlwdGlvbixcbiAgICAgICAgICAgIG5hbWUsXG4gICAgICAgIH0pO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBhZGQgYSBuZXcgRHluYW1vREIgZGF0YSBzb3VyY2UgdG8gdGhpcyBBUElcbiAgICAgKiBAcGFyYW0gbmFtZSBUaGUgbmFtZSBvZiB0aGUgZGF0YSBzb3VyY2VcbiAgICAgKiBAcGFyYW0gZGVzY3JpcHRpb24gVGhlIGRlc2NyaXB0aW9uIG9mIHRoZSBkYXRhIHNvdXJjZVxuICAgICAqIEBwYXJhbSB0YWJsZSBUaGUgRHluYW1vREIgdGFibGUgYmFja2luZyB0aGlzIGRhdGEgc291cmNlIFtkaXNhYmxlLWF3c2xpbnQ6cmVmLXZpYS1pbnRlcmZhY2VdXG4gICAgICovXG4gICAgcHVibGljIGFkZER5bmFtb0RiRGF0YVNvdXJjZShuYW1lOiBzdHJpbmcsIGRlc2NyaXB0aW9uOiBzdHJpbmcsIHRhYmxlOiBJVGFibGUpOiBEeW5hbW9EYkRhdGFTb3VyY2Uge1xuICAgICAgICByZXR1cm4gbmV3IER5bmFtb0RiRGF0YVNvdXJjZSh0aGlzLCBgJHtuYW1lfURTYCwge1xuICAgICAgICAgICAgYXBpOiB0aGlzLFxuICAgICAgICAgICAgZGVzY3JpcHRpb24sXG4gICAgICAgICAgICBuYW1lLFxuICAgICAgICAgICAgdGFibGUsXG4gICAgICAgIH0pO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBhZGQgYSBuZXcgaHR0cCBkYXRhIHNvdXJjZSB0byB0aGlzIEFQSVxuICAgICAqIEBwYXJhbSBuYW1lIFRoZSBuYW1lIG9mIHRoZSBkYXRhIHNvdXJjZVxuICAgICAqIEBwYXJhbSBkZXNjcmlwdGlvbiBUaGUgZGVzY3JpcHRpb24gb2YgdGhlIGRhdGEgc291cmNlXG4gICAgICogQHBhcmFtIGVuZHBvaW50IFRoZSBodHRwIGVuZHBvaW50XG4gICAgICovXG4gICAgcHVibGljIGFkZEh0dHBEYXRhU291cmNlKG5hbWU6IHN0cmluZywgZGVzY3JpcHRpb246IHN0cmluZywgZW5kcG9pbnQ6IHN0cmluZyk6IEh0dHBEYXRhU291cmNlIHtcbiAgICAgICAgcmV0dXJuIG5ldyBIdHRwRGF0YVNvdXJjZSh0aGlzLCBgJHtuYW1lfURTYCwge1xuICAgICAgICAgICAgYXBpOiB0aGlzLFxuICAgICAgICAgICAgZGVzY3JpcHRpb24sXG4gICAgICAgICAgICBlbmRwb2ludCxcbiAgICAgICAgICAgIG5hbWUsXG4gICAgICAgIH0pO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBhZGQgYSBuZXcgTGFtYmRhIGRhdGEgc291cmNlIHRvIHRoaXMgQVBJXG4gICAgICogQHBhcmFtIG5hbWUgVGhlIG5hbWUgb2YgdGhlIGRhdGEgc291cmNlXG4gICAgICogQHBhcmFtIGRlc2NyaXB0aW9uIFRoZSBkZXNjcmlwdGlvbiBvZiB0aGUgZGF0YSBzb3VyY2VcbiAgICAgKiBAcGFyYW0gbGFtYmRhRnVuY3Rpb24gVGhlIExhbWJkYSBmdW5jdGlvbiB0byBjYWxsIHRvIGludGVyYWN0IHdpdGggdGhpcyBkYXRhIHNvdXJjZVxuICAgICAqL1xuICAgIHB1YmxpYyBhZGRMYW1iZGFEYXRhU291cmNlKG5hbWU6IHN0cmluZywgZGVzY3JpcHRpb246IHN0cmluZywgbGFtYmRhRnVuY3Rpb246IElGdW5jdGlvbik6IExhbWJkYURhdGFTb3VyY2Uge1xuICAgICAgICByZXR1cm4gbmV3IExhbWJkYURhdGFTb3VyY2UodGhpcywgYCR7bmFtZX1EU2AsIHtcbiAgICAgICAgICAgIGFwaTogdGhpcyxcbiAgICAgICAgICAgIGRlc2NyaXB0aW9uLFxuICAgICAgICAgICAgbmFtZSxcbiAgICAgICAgICAgIGxhbWJkYUZ1bmN0aW9uLFxuICAgICAgICB9KTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQWRkcyBhbiBJQU0gcG9saWN5IHN0YXRlbWVudCBhc3NvY2lhdGVkIHdpdGggdGhpcyBHcmFwaFFMQXBpIHRvIGFuIElBTVxuICAgICAqIHByaW5jaXBhbCdzIHBvbGljeS5cbiAgICAgKlxuICAgICAqIEBwYXJhbSBncmFudGVlIFRoZSBwcmluY2lwYWxcbiAgICAgKiBAcGFyYW0gcmVzb3VyY2VzIFRoZSBzZXQgb2YgcmVzb3VyY2VzIHRvIGFsbG93IChpLmUuIC4uLjpbcmVnaW9uXTpbYWNjb3VudElkXTphcGlzL0dyYXBoUUxJZC8uLi4pXG4gICAgICogQHBhcmFtIGFjdGlvbnMgVGhlIGFjdGlvbnMgdGhhdCBzaG91bGQgYmUgZ3JhbnRlZCB0byB0aGUgcHJpbmNpcGFsIChpLmUuIGFwcHN5bmM6Z3JhcGhxbCApXG4gICAgICovXG4gICAgcHVibGljIGdyYW50KGdyYW50ZWU6IElHcmFudGFibGUsIHJlc291cmNlczogSWFtUmVzb3VyY2UsIC4uLmFjdGlvbnM6IHN0cmluZ1tdKTogR3JhbnQge1xuICAgICAgICByZXR1cm4gR3JhbnQuYWRkVG9QcmluY2lwYWwoe1xuICAgICAgICAgICAgZ3JhbnRlZSxcbiAgICAgICAgICAgIGFjdGlvbnMsXG4gICAgICAgICAgICByZXNvdXJjZUFybnM6IHJlc291cmNlcy5yZXNvdXJjZUFybnModGhpcyksXG4gICAgICAgICAgICBzY29wZTogdGhpcyxcbiAgICAgICAgfSk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEFkZHMgYW4gSUFNIHBvbGljeSBzdGF0ZW1lbnQgZm9yIE11dGF0aW9uIGFjY2VzcyB0byB0aGlzIEdyYXBoUUxBcGkgdG8gYW4gSUFNXG4gICAgICogcHJpbmNpcGFsJ3MgcG9saWN5LlxuICAgICAqXG4gICAgICogQHBhcmFtIGdyYW50ZWUgVGhlIHByaW5jaXBhbFxuICAgICAqIEBwYXJhbSBmaWVsZHMgVGhlIGZpZWxkcyB0byBncmFudCBhY2Nlc3MgdG8gdGhhdCBhcmUgTXV0YXRpb25zIChsZWF2ZSBibGFuayBmb3IgYWxsKVxuICAgICAqL1xuICAgIHB1YmxpYyBncmFudE11dGF0aW9uKGdyYW50ZWU6IElHcmFudGFibGUsIC4uLmZpZWxkczogc3RyaW5nW10pOiBHcmFudCB7XG4gICAgICAgIHJldHVybiB0aGlzLmdyYW50KGdyYW50ZWUsIElhbVJlc291cmNlLm9mVHlwZSgnTXV0YXRpb24nLCAuLi5maWVsZHMpLCAnYXBwc3luYzpHcmFwaFFMJyk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEFkZHMgYW4gSUFNIHBvbGljeSBzdGF0ZW1lbnQgZm9yIFF1ZXJ5IGFjY2VzcyB0byB0aGlzIEdyYXBoUUxBcGkgdG8gYW4gSUFNXG4gICAgICogcHJpbmNpcGFsJ3MgcG9saWN5LlxuICAgICAqXG4gICAgICogQHBhcmFtIGdyYW50ZWUgVGhlIHByaW5jaXBhbFxuICAgICAqIEBwYXJhbSBmaWVsZHMgVGhlIGZpZWxkcyB0byBncmFudCBhY2Nlc3MgdG8gdGhhdCBhcmUgUXVlcmllcyAobGVhdmUgYmxhbmsgZm9yIGFsbClcbiAgICAgKi9cbiAgICBwdWJsaWMgZ3JhbnRRdWVyeShncmFudGVlOiBJR3JhbnRhYmxlLCAuLi5maWVsZHM6IHN0cmluZ1tdKTogR3JhbnQge1xuICAgICAgICByZXR1cm4gdGhpcy5ncmFudChncmFudGVlLCBJYW1SZXNvdXJjZS5vZlR5cGUoJ1F1ZXJ5JywgLi4uZmllbGRzKSwgJ2FwcHN5bmM6R3JhcGhRTCcpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBBZGRzIGFuIElBTSBwb2xpY3kgc3RhdGVtZW50IGZvciBTdWJzY3JpcHRpb24gYWNjZXNzIHRvIHRoaXMgR3JhcGhRTEFwaSB0byBhbiBJQU1cbiAgICAgKiBwcmluY2lwYWwncyBwb2xpY3kuXG4gICAgICpcbiAgICAgKiBAcGFyYW0gZ3JhbnRlZSBUaGUgcHJpbmNpcGFsXG4gICAgICogQHBhcmFtIGZpZWxkcyBUaGUgZmllbGRzIHRvIGdyYW50IGFjY2VzcyB0byB0aGF0IGFyZSBTdWJzY3JpcHRpb25zIChsZWF2ZSBibGFuayBmb3IgYWxsKVxuICAgICAqL1xuICAgIHB1YmxpYyBncmFudFN1YnNjcmlwdGlvbihncmFudGVlOiBJR3JhbnRhYmxlLCAuLi5maWVsZHM6IHN0cmluZ1tdKTogR3JhbnQge1xuICAgICAgICByZXR1cm4gdGhpcy5ncmFudChncmFudGVlLCBJYW1SZXNvdXJjZS5vZlR5cGUoJ1N1YnNjcmlwdGlvbicsIC4uLmZpZWxkcyksICdhcHBzeW5jOkdyYXBoUUwnKTtcbiAgICB9XG4gICAgcHJpdmF0ZSB2YWxpZGF0ZUF1dGhvcml6YXRpb25Qcm9wcyhwcm9wczogR3JhcGhRTEFwaVByb3BzKSB7XG4gICAgICAgIGNvbnN0IGRlZmF1bHRBdXRob3JpemF0aW9uVHlwZSA9IHByb3BzLmF1dGhvcml6YXRpb25Db25maWc/LmRlZmF1bHRBdXRob3JpemF0aW9uPy5hdXRob3JpemF0aW9uVHlwZSB8fFxuICAgICAgICAgICAgQXV0aG9yaXphdGlvblR5cGUuQVBJX0tFWTtcbiAgICAgICAgaWYgKGRlZmF1bHRBdXRob3JpemF0aW9uVHlwZSA9PT0gQXV0aG9yaXphdGlvblR5cGUuT0lEQyAmJlxuICAgICAgICAgICAgIXByb3BzLmF1dGhvcml6YXRpb25Db25maWc/LmRlZmF1bHRBdXRob3JpemF0aW9uPy5vcGVuSWRDb25uZWN0Q29uZmlnKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ01pc3NpbmcgZGVmYXVsdCBPSURDIENvbmZpZ3VyYXRpb24nKTtcbiAgICAgICAgfVxuICAgICAgICBpZiAoZGVmYXVsdEF1dGhvcml6YXRpb25UeXBlID09PSBBdXRob3JpemF0aW9uVHlwZS5VU0VSX1BPT0wgJiZcbiAgICAgICAgICAgICFwcm9wcy5hdXRob3JpemF0aW9uQ29uZmlnPy5kZWZhdWx0QXV0aG9yaXphdGlvbj8udXNlclBvb2xDb25maWcpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignTWlzc2luZyBkZWZhdWx0IFVzZXIgUG9vbCBDb25maWd1cmF0aW9uJyk7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKHByb3BzLmF1dGhvcml6YXRpb25Db25maWc/LmFkZGl0aW9uYWxBdXRob3JpemF0aW9uTW9kZXMpIHtcbiAgICAgICAgICAgIHByb3BzLmF1dGhvcml6YXRpb25Db25maWcuYWRkaXRpb25hbEF1dGhvcml6YXRpb25Nb2Rlcy5mb3JFYWNoKChhdXRob3JpemF0aW9uTW9kZSkgPT4ge1xuICAgICAgICAgICAgICAgIGlmIChhdXRob3JpemF0aW9uTW9kZS5hdXRob3JpemF0aW9uVHlwZSA9PT0gQXV0aG9yaXphdGlvblR5cGUuQVBJX0tFWSAmJlxuICAgICAgICAgICAgICAgICAgICBkZWZhdWx0QXV0aG9yaXphdGlvblR5cGUgPT09IEF1dGhvcml6YXRpb25UeXBlLkFQSV9LRVkpIHtcbiAgICAgICAgICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKFwiWW91IGNhbid0IGR1cGxpY2F0ZSBBUElfS0VZIGluIGFkZGl0aW9uYWwgYXV0aG9yaXphdGlvbiBjb25maWcuIFNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3luYy9sYXRlc3QvZGV2Z3VpZGUvc2VjdXJpdHkuaHRtbFwiKTtcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgaWYgKGF1dGhvcml6YXRpb25Nb2RlLmF1dGhvcml6YXRpb25UeXBlID09PSBBdXRob3JpemF0aW9uVHlwZS5JQU0gJiZcbiAgICAgICAgICAgICAgICAgICAgZGVmYXVsdEF1dGhvcml6YXRpb25UeXBlID09PSBBdXRob3JpemF0aW9uVHlwZS5JQU0pIHtcbiAgICAgICAgICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKFwiWW91IGNhbid0IGR1cGxpY2F0ZSBJQU0gaW4gYWRkaXRpb25hbCBhdXRob3JpemF0aW9uIGNvbmZpZy4gU2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzeW5jL2xhdGVzdC9kZXZndWlkZS9zZWN1cml0eS5odG1sXCIpO1xuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICBpZiAoYXV0aG9yaXphdGlvbk1vZGUuYXV0aG9yaXphdGlvblR5cGUgPT09IEF1dGhvcml6YXRpb25UeXBlLk9JREMgJiZcbiAgICAgICAgICAgICAgICAgICAgIWF1dGhvcml6YXRpb25Nb2RlLm9wZW5JZENvbm5lY3RDb25maWcpIHtcbiAgICAgICAgICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdNaXNzaW5nIE9JREMgQ29uZmlndXJhdGlvbiBpbnNpZGUgYW4gYWRkaXRpb25hbCBhdXRob3JpemF0aW9uIG1vZGUnKTtcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgaWYgKGF1dGhvcml6YXRpb25Nb2RlLmF1dGhvcml6YXRpb25UeXBlID09PVxuICAgICAgICAgICAgICAgICAgICBBdXRob3JpemF0aW9uVHlwZS5VU0VSX1BPT0wgJiZcbiAgICAgICAgICAgICAgICAgICAgIWF1dGhvcml6YXRpb25Nb2RlLnVzZXJQb29sQ29uZmlnKSB7XG4gICAgICAgICAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignTWlzc2luZyBVc2VyIFBvb2wgQ29uZmlndXJhdGlvbiBpbnNpZGUgYW4gYWRkaXRpb25hbCBhdXRob3JpemF0aW9uIG1vZGUnKTtcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICB9KTtcbiAgICAgICAgfVxuICAgIH1cbiAgICBwcml2YXRlIGZvcm1hdE9wZW5JZENvbm5lY3RDb25maWcoY29uZmlnOiBPcGVuSWRDb25uZWN0Q29uZmlnKTogQ2ZuR3JhcGhRTEFwaS5PcGVuSURDb25uZWN0Q29uZmlnUHJvcGVydHkge1xuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgICAgYXV0aFR0bDogY29uZmlnLnRva2VuRXhwaXJ5RnJvbUF1dGgsXG4gICAgICAgICAgICBjbGllbnRJZDogY29uZmlnLmNsaWVudElkLFxuICAgICAgICAgICAgaWF0VHRsOiBjb25maWcudG9rZW5FeHBpcnlGcm9tSXNzdWUsXG4gICAgICAgICAgICBpc3N1ZXI6IGNvbmZpZy5vaWRjUHJvdmlkZXIsXG4gICAgICAgIH07XG4gICAgfVxuICAgIHByaXZhdGUgZm9ybWF0VXNlclBvb2xDb25maWcoY29uZmlnOiBVc2VyUG9vbENvbmZpZyk6IENmbkdyYXBoUUxBcGkuVXNlclBvb2xDb25maWdQcm9wZXJ0eSB7XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICB1c2VyUG9vbElkOiBjb25maWcudXNlclBvb2wudXNlclBvb2xJZCxcbiAgICAgICAgICAgIGF3c1JlZ2lvbjogY29uZmlnLnVzZXJQb29sLnN0YWNrLnJlZ2lvbixcbiAgICAgICAgICAgIGFwcElkQ2xpZW50UmVnZXg6IGNvbmZpZy5hcHBJZENsaWVudFJlZ2V4LFxuICAgICAgICAgICAgZGVmYXVsdEFjdGlvbjogY29uZmlnLmRlZmF1bHRBY3Rpb24gfHwgJ0FMTE9XJyxcbiAgICAgICAgfTtcbiAgICB9XG4gICAgcHJpdmF0ZSBjcmVhdGVBUElLZXkoY29uZmlnOiBBcGlLZXlDb25maWcpIHtcbiAgICAgICAgbGV0IGV4cGlyZXM6IG51bWJlciB8IHVuZGVmaW5lZDtcbiAgICAgICAgaWYgKGNvbmZpZy5leHBpcmVzKSB7XG4gICAgICAgICAgICBleHBpcmVzID0gbmV3IERhdGUoY29uZmlnLmV4cGlyZXMpLnZhbHVlT2YoKTtcbiAgICAgICAgICAgIGNvbnN0IGRheXMgPSAoZDogbnVtYmVyKSA9PiBEYXRlLm5vdygpICsgRHVyYXRpb24uZGF5cyhkKS50b01pbGxpc2Vjb25kcygpO1xuICAgICAgICAgICAgaWYgKGV4cGlyZXMgPCBkYXlzKDEpIHx8IGV4cGlyZXMgPiBkYXlzKDM2NSkpIHtcbiAgICAgICAgICAgICAgICB0aHJvdyBFcnJvcignQVBJIGtleSBleHBpcmF0aW9uIG11c3QgYmUgYmV0d2VlbiAxIGFuZCAzNjUgZGF5cy4nKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIGV4cGlyZXMgPSBNYXRoLnJvdW5kKGV4cGlyZXMgLyAxMDAwKTtcbiAgICAgICAgfVxuICAgICAgICBjb25zdCBrZXkgPSBuZXcgQ2ZuQXBpS2V5KHRoaXMsIGAke2NvbmZpZy5uYW1lIHx8ICdEZWZhdWx0QVBJS2V5J31BcGlLZXlgLCB7XG4gICAgICAgICAgICBleHBpcmVzLFxuICAgICAgICAgICAgZGVzY3JpcHRpb246IGNvbmZpZy5kZXNjcmlwdGlvbiB8fCAnRGVmYXVsdCBBUEkgS2V5IGNyZWF0ZWQgYnkgQ0RLJyxcbiAgICAgICAgICAgIGFwaUlkOiB0aGlzLmFwaUlkLFxuICAgICAgICB9KTtcbiAgICAgICAgcmV0dXJuIGtleS5hdHRyQXBpS2V5O1xuICAgIH1cbiAgICBwcml2YXRlIGZvcm1hdEFkZGl0aW9uYWxBdXRob3JpemF0aW9uTW9kZXMoYXV0aE1vZGVzOiBBdXRob3JpemF0aW9uTW9kZVtdKTogQ2ZuR3JhcGhRTEFwaS5BZGRpdGlvbmFsQXV0aGVudGljYXRpb25Qcm92aWRlclByb3BlcnR5W10ge1xuICAgICAgICByZXR1cm4gYXV0aE1vZGVzLnJlZHVjZTxDZm5HcmFwaFFMQXBpLkFkZGl0aW9uYWxBdXRoZW50aWNhdGlvblByb3ZpZGVyUHJvcGVydHlbXT4oKGFjYywgYXV0aE1vZGUpID0+IFtcbiAgICAgICAgICAgIC4uLmFjYyxcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBhdXRoZW50aWNhdGlvblR5cGU6IGF1dGhNb2RlLmF1dGhvcml6YXRpb25UeXBlLFxuICAgICAgICAgICAgICAgIHVzZXJQb29sQ29uZmlnOiBhdXRoTW9kZS5hdXRob3JpemF0aW9uVHlwZSA9PT0gQXV0aG9yaXphdGlvblR5cGUuVVNFUl9QT09MXG4gICAgICAgICAgICAgICAgICAgID8gdGhpcy5mb3JtYXRVc2VyUG9vbENvbmZpZyhhdXRoTW9kZS51c2VyUG9vbENvbmZpZyEpXG4gICAgICAgICAgICAgICAgICAgIDogdW5kZWZpbmVkLFxuICAgICAgICAgICAgICAgIG9wZW5JZENvbm5lY3RDb25maWc6IGF1dGhNb2RlLmF1dGhvcml6YXRpb25UeXBlID09PSBBdXRob3JpemF0aW9uVHlwZS5PSURDXG4gICAgICAgICAgICAgICAgICAgID8gdGhpcy5mb3JtYXRPcGVuSWRDb25uZWN0Q29uZmlnKGF1dGhNb2RlLm9wZW5JZENvbm5lY3RDb25maWchKVxuICAgICAgICAgICAgICAgICAgICA6IHVuZGVmaW5lZCxcbiAgICAgICAgICAgIH0sXG4gICAgICAgIF0sIFtdKTtcbiAgICB9XG4gICAgcHJpdmF0ZSBmb3JtYXRBZGRpdGlvbmFsQXV0aGVudGljYXRpb25Qcm92aWRlcnMocHJvcHM6IEdyYXBoUUxBcGlQcm9wcyk6IENmbkdyYXBoUUxBcGkuQWRkaXRpb25hbEF1dGhlbnRpY2F0aW9uUHJvdmlkZXJQcm9wZXJ0eVtdIHwgdW5kZWZpbmVkIHtcbiAgICAgICAgY29uc3QgYXV0aE1vZGVzID0gcHJvcHMuYXV0aG9yaXphdGlvbkNvbmZpZz8uYWRkaXRpb25hbEF1dGhvcml6YXRpb25Nb2RlcztcbiAgICAgICAgcmV0dXJuIGF1dGhNb2RlcyA/IHRoaXMuZm9ybWF0QWRkaXRpb25hbEF1dGhvcml6YXRpb25Nb2RlcyhhdXRoTW9kZXMpIDogdW5kZWZpbmVkO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBTZXRzIHNjaGVtYSBkZWZpbnRpaW9uIHRvIGlucHV0IGlmIHNjaGVtYSBtb2RlIGlzIGNvbmZpZ3VyZWQgd2l0aCBTY2hlbWFEZWZpbml0aW9uLkNPREVcbiAgICAgKlxuICAgICAqIEBwYXJhbSBkZWZpbml0aW9uIHN0cmluZyB0aGF0IGlzIHRoZSBncmFwaHFsIHJlcHJlc2VudGF0aW9uIG9mIHNjaGVtYVxuICAgICAqIEBleHBlcmltZW50YWwgdGVtcG9yYXJ5XG4gICAgICovXG4gICAgcHVibGljIHVwZGF0ZURlZmluaXRpb24oZGVmaW5pdGlvbjogc3RyaW5nKTogdm9pZCB7XG4gICAgICAgIGlmICh0aGlzLnNjaGVtYU1vZGUgIT0gU2NoZW1hRGVmaW5pdGlvbi5DT0RFKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0FQSSBjYW5ub3QgYWRkIHR5cGUgYmVjYXVzZSBzY2hlbWEgZGVmaW5pdGlvbiBtb2RlIGlzIG5vdCBjb25maWd1cmVkIGFzIENPREUuJyk7XG4gICAgICAgIH1cbiAgICAgICAgdGhpcy5zY2hlbWEuZGVmaW5pdGlvbiA9IGRlZmluaXRpb247XG4gICAgfVxuICAgIC8qKlxuICAgICAqIERlZmluZSBzY2hlbWEgYmFzZWQgb24gcHJvcHMgY29uZmlndXJhdGlvblxuICAgICAqIEBwYXJhbSBmaWxlIHRoZSBmaWxlIG5hbWUvczMgbG9jYXRpb24gb2YgU2NoZW1hXG4gICAgICovXG4gICAgcHJpdmF0ZSBkZWZpbmVTY2hlbWEoZmlsZT86IHN0cmluZyk6IENmbkdyYXBoUUxTY2hlbWEge1xuICAgICAgICBsZXQgZGVmaW5pdGlvbjtcbiAgICAgICAgaWYgKHRoaXMuc2NoZW1hTW9kZSA9PSBTY2hlbWFEZWZpbml0aW9uLkZJTEUgJiYgIWZpbGUpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignc2NoZW1hRGVmaW5pdGlvbkZpbGUgbXVzdCBiZSBjb25maWd1cmVkIGlmIHVzaW5nIEZJTEUgZGVmaW5pdGlvbiBtb2RlLicpO1xuICAgICAgICB9XG4gICAgICAgIGVsc2UgaWYgKHRoaXMuc2NoZW1hTW9kZSA9PSBTY2hlbWFEZWZpbml0aW9uLkZJTEUgJiYgZmlsZSkge1xuICAgICAgICAgICAgZGVmaW5pdGlvbiA9IHJlYWRGaWxlU3luYyhmaWxlKS50b1N0cmluZygnVVRGLTgnKTtcbiAgICAgICAgfVxuICAgICAgICBlbHNlIGlmICh0aGlzLnNjaGVtYU1vZGUgPT0gU2NoZW1hRGVmaW5pdGlvbi5DT0RFICYmICFmaWxlKSB7XG4gICAgICAgICAgICBkZWZpbml0aW9uID0gJyc7XG4gICAgICAgIH1cbiAgICAgICAgZWxzZSBpZiAodGhpcy5zY2hlbWFNb2RlID09IFNjaGVtYURlZmluaXRpb24uQ09ERSAmJiBmaWxlKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ2RlZmluaXRpb24gbW9kZSBDT0RFIGlzIGluY29tcGF0aWJsZSB3aXRoIGZpbGUgZGVmaW5pdGlvbi4gQ2hhbmdlIG1vZGUgdG8gRklMRS9TMyBvciB1bmNvbmZpZ3VyZSBzY2hlbWFEZWZpbml0aW9uRmlsZScpO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiBuZXcgQ2ZuR3JhcGhRTFNjaGVtYSh0aGlzLCAnU2NoZW1hJywge1xuICAgICAgICAgICAgYXBpSWQ6IHRoaXMuYXBpSWQsXG4gICAgICAgICAgICBkZWZpbml0aW9uLFxuICAgICAgICB9KTtcbiAgICB9XG59XG4iXX0=