import typing
from typing import Any, Callable, List, Tuple, Union

from interpretdl.interpreter.abc_interpreter import Interpreter
from interpretdl.data_processor.readers import preprocess_image, read_image
from interpretdl.data_processor.visualizer import visualize_ig

import IPython.display as display
import cv2
import numpy as np
import paddle.fluid as fluid
import os, sys
from PIL import Image


class SmoothGradInterpreter(Interpreter):
    """
    Integrated Gradients Interpreter.

    More details regarding the Integrated Gradients method can be found in the original paper:
    http://proceedings.mlr.press/v70/sundararajan17a/sundararajan17a.pdf
    """

    def __init__(self,
                 predict_fn,
                 trained_model_path,
                 class_num,
                 use_cuda,
                 model_input_shape=[3, 224, 224]) -> None:
        """
        Initialize the IntGradInterpreter

        Args:
            predict_fn: A user-defined function that gives access to model predictions. It takes the following arguments:
                    - data: Data input.
                    example:
                        def predict_fn(data):
                            import paddle.fluid as fluid
                            class_num = 1000
                            model = ResNet50()
                            logits = model.net(input=data, class_dim=class_num)

                            probs = fluid.layers.softmax(logits, axis=-1)
                            return probs
            trained_model_path: The pretrained model directory.
            class_num: Number of classes for the model.
            use_cuda: Whether or not to use cuda.
            model_input_shape: The input shape of the model

        Returns:
        """
        Interpreter.__init__(self)
        self.predict_fn = predict_fn
        self.trained_model_path = trained_model_path
        self.class_num = class_num
        self.use_cuda = use_cuda
        self.model_input_shape = model_input_shape

    def set_params(self):
        """
        Set parameters for the interpreter.
        """
        pass

    def interpret(self,
                  data,
                  label=None,
                  noise=0.1,
                  n_samples=50,
                  visual=True,
                  save_path=None):
        """
        Main function of the interpreter.

        Args:
            data: The image filepath or a numpy array.
            label: The target label to analyze. If None, the most likely label will be used.
            noise: The amount of noise to add to the original image.
            n_samples: The number of new images generated by adding noise.
            visual: Whether or not to visualize the processed image.
            save_path: The filepath to save the processed image. If None, the image will not be saved.

        Returns:
        """
        startup_prog = fluid.Program()
        main_program = fluid.Program()

        # Read in image
        if isinstance(data, str):
            with open(data, 'rb') as f:
                org = Image.open(f)
                org = org.convert('RGB')
                org = np.array(org)
                img = read_image(data, crop_size=self.model_input_shape[1])
                data = preprocess_image(img)

        data_type = np.array(data).dtype

        startup_prog = fluid.Program()
        main_program = fluid.Program()
        with fluid.program_guard(main_program, startup_prog):
            with fluid.unique_name.guard():

                data_op = fluid.data(
                    name='data',
                    shape=[1] + self.model_input_shape,
                    dtype=data_type)
                label_op = fluid.layers.data(
                    name='label', shape=[1], dtype='int64')

                x_noise = fluid.layers.gaussian_random(
                    [1] + self.model_input_shape, std=0, dtype=data_type)

                x_plus_noise = data_op + x_noise

                probs = self.predict_fn(x_plus_noise)

                for op in main_program.global_block().ops:
                    if op.type == 'batch_norm':
                        op._set_attr('use_global_stats', True)
                    elif op.type == 'dropout':
                        op._set_attr('is_test', True)

                one_hot = fluid.layers.one_hot(label_op, self.class_num)
                one_hot = fluid.layers.elementwise_mul(probs, one_hot)
                target_category_loss = fluid.layers.reduce_sum(one_hot)

                p_g_list = fluid.backward.append_backward(target_category_loss)

                gradients_map = fluid.gradients(one_hot, x_plus_noise)[0]

        if self.use_cuda:
            gpu_id = int(os.environ.get('FLAGS_selected_gpus', 0))
            place = fluid.CUDAPlace(gpu_id)
        else:
            place = fluid.CPUPlace()
        exe = fluid.Executor(place)

        fluid.io.load_persistables(exe, self.trained_model_path, main_program)

        # if label is None, let it be the most likely label
        gradients, out = exe.run(main_program,
                                 feed={
                                     'data': data,
                                     'label': np.array([[0]]),
                                 },
                                 fetch_list=[gradients_map, probs],
                                 return_numpy=False)

        if label is None:
            label = np.argmax(out[0])

        std = noise * (np.max(data) - np.min(data))

        startup_prog = fluid.Program()
        main_program = fluid.Program()
        with fluid.program_guard(main_program, startup_prog):
            with fluid.unique_name.guard():

                data_op = fluid.data(
                    name='data',
                    shape=[1] + self.model_input_shape,
                    dtype=data_type)
                label_op = fluid.layers.data(
                    name='label', shape=[1], dtype='int64')

                x_noise = fluid.layers.gaussian_random(
                    [1] + self.model_input_shape, std=std, dtype=data_type)

                x_plus_noise = data_op + x_noise

                probs = self.predict_fn(x_plus_noise)

                for op in main_program.global_block().ops:
                    if op.type == 'batch_norm':
                        op._set_attr('use_global_stats', True)
                    elif op.type == 'dropout':
                        op._set_attr('is_test', True)

                one_hot = fluid.layers.one_hot(label_op, self.class_num)
                one_hot = fluid.layers.elementwise_mul(probs, one_hot)
                target_category_loss = fluid.layers.reduce_sum(one_hot)

                p_g_list = fluid.backward.append_backward(target_category_loss)

                gradients_map = fluid.gradients(one_hot, x_plus_noise)[0]

        fluid.io.load_persistables(exe, self.trained_model_path, main_program)
        total_gradients = np.zeros_like(gradients)
        for i in range(n_samples):
            [gradients] = exe.run(
                main_program,
                feed={'data': data,
                      'label': np.array([[label]])},
                fetch_list=[gradients_map],
                return_numpy=False)
            total_gradients += np.array(gradients)

        avg_gradients = total_gradients / n_samples

        visualize_ig(avg_gradients, img, visual, save_path)

        return avg_gradients
