from typing import Any, List, Optional
from xpander_sdk import ExecutionStatus, XpanderClient, Agent

class SDKAdapter:
    """
    Adapter class for interacting with xpander.ai SDK.

    This class provides an interface to manage an xpander.ai agent, 
    including handling tasks, execution results, and memory.

    Attributes:
        agent (Agent): The xpander.ai agent instance.
        xpander_client (XpanderClient): The xpander.ai client instance.
    """

    agent: Agent

    def __init__(self, api_key: str, agent_id: str, base_url: Optional[str] = None, organization_id: Optional[str] = None):
        """
        Initialize the SDKAdapter with the provided API key and agent ID.

        Args:
            api_key (str): The API key for authentication with xpander.ai.
            agent_id (str): The ID of the agent to interact with.
            base_url (Optional[str], optional): The base URL for the xpander.ai API. Defaults to None.
            organization_id (Optional[str], optional): The organization ID, if applicable. Defaults to None.
        """
        self.xpander_client = XpanderClient(api_key=api_key, base_url=base_url, organization_id=organization_id)
        self.agent = self.xpander_client.agents.get(agent_id=agent_id)

    def get_system_prompt(self) -> str:
        """
        Retrieve the system prompt for the agent.

        Raises:
            NotImplementedError: This method is not implemented.

        Returns:
            str: The system prompt.
        """
        raise NotImplementedError('method "get_system_prompt" not implemented')

    def get_tools(self) -> List[Any]:
        """
        Retrieve the tools available for the agent.

        Raises:
            NotImplementedError: This method is not implemented.

        Returns:
            List[Any]: A list of tools.
        """
        raise NotImplementedError('method "get_tools" not implemented')

    def init_memory(self, agent: Any):
        """
        Initialize memory for the given agent.

        Args:
            agent (Any): The agent instance.

        Raises:
            NotImplementedError: This method is not implemented.
        """
        raise NotImplementedError('method "init_memory" not implemented')

    def set_system_prompt(self, agent: Any):
        """
        Set the system prompt for the given agent.

        Args:
            agent (Any): The agent instance.

        Raises:
            NotImplementedError: This method is not implemented.
        """
        raise NotImplementedError('method "set_system_prompt" not implemented')

    def add_task(self, input: str, files: Optional[List[str]] = None, use_worker: Optional[bool] = False, thread_id: Optional[str] = None):
        """
        Add a task to the agent and initialize memory with the input and instructions.

        Args:
            input (str): The task input.
            files (Optional[List[str]], optional): A list of file paths associated with the task. Defaults to None.
            use_worker (Optional[bool], optional): Whether to use a worker for execution. Defaults to False.
            thread_id (Optional[str], optional): The thread ID for task execution. Defaults to None.
        """
        files = files or []
        self.agent.add_task(input=input, files=files, use_worker=use_worker, thread_id=thread_id)
        self.agent.memory.init_messages(input=self.agent.execution.input_message, instructions=self.agent.instructions)

    def update_execution_result(self, is_success: bool, result: str):
        """
        Update the execution result of the agent.

        Args:
            is_success (bool): Whether the execution was successful.
            result (str): The execution result.
        """
        self.agent.execution.update(
            agent=self.agent,
            execution_id=self.agent.execution.id,
            delta={
                "status": ExecutionStatus.COMPLETED if is_success else ExecutionStatus.ERROR,
                "result": result
            }
        )

    def sync_memory(self, agent: Any):
        """
        Synchronize memory for the given agent.

        Args:
            agent (Any): The agent instance.

        Raises:
            NotImplementedError: This method is not implemented.
        """
        raise NotImplementedError('method "sync_memory" not implemented')
