from __future__ import annotations

from typing import TYPE_CHECKING

import pytest

from dissect.target.helpers.regutil import VirtualHive, VirtualKey
from dissect.target.plugins.os.windows.productkey import WindowsProductKeyPlugin, decode_cd_key

if TYPE_CHECKING:
    from dissect.target.target import Target


@pytest.mark.parametrize(
    ("encoded_key", "decoded_key"),
    [
        pytest.param(
            "ec0c00000000a8d27b6e89814f6d0900",
            "VK7JG-NPHTM-C97JM-9MPGT-3V66T",
            id="windows-11-24h2-eval-unregistered",
        ),
        pytest.param(
            "d71000000000288b1b9eaf44f3d70800",
            "J4WNC-H9BG3-6XRX4-3XD8K-Y7XRX",
            id="windows-server-2019-eval-unregistered",
        ),
    ],
)
def test_windows_product_key_base24_decode(encoded_key: str, decoded_key: str) -> None:
    """Test if we decode cd keys correctly."""
    assert decode_cd_key(bytes.fromhex(encoded_key)) == decoded_key


def test_windows_product_key_plugin(target_win: Target, hive_hklm: VirtualHive) -> None:
    """Test the Windows Product Key plugin on discovery of ProductId values from a Windows 11 24H2 VM."""

    key_name = "SOFTWARE\\Microsoft\\Windows NT\\CurrentVersion"
    key = VirtualKey(hive_hklm, key_name)
    key.add_value(
        "DigitalProductId4",
        bytes.fromhex(
            "f804000004000000300033003600310032002d00300033003300300038002d003000300030002d00300030003000300030003000"
            "2d00300030002d0031003000330033002d00320036003100300030002e0030003000300030002d00310031003800320030003200"
            "3500000000000000000000000000000000000000000000000000000000000000340064006500370063006200360035002d006300"
            "6400660031002d0034006400650039002d0038006100650038002d00650033006300630065003200370062003900660032006300"
            "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000500072006f00660065007300730069006f006e0061006c000000000000000000"
            "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
            "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
            "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
            "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
            "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
            "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
            "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
            "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
            "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
            "00000000000000000000000000000000000000000000000000000000ec0c00000000a8d27b6e89814f6d090058ed3820fbf01d2c"
            "f44d615860ce274a3cf5af8470d167f54b2113f4a7dfc119fe24621376018d3ff2ba9520a3c013777e49f7afb55bfb494131b337"
            "47b925b85b00540048005d005800310039002d003900380038003400310000000000000000000000000000000000000000000000"
            "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
            "00000000000000000000000000000000000000000000000000000000520065007400610069006c00000000000000000000000000"
            "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
            "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
            "520065007400610069006c0000000000000000000000000000000000000000000000000000000000000000000000000000000000"
            "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
            "000000000000000000000000000000000000000000000000"
        ),
    )
    key.add_value(
        "DigitalProductId",
        bytes.fromhex(
            "a40000000300000030303333302d38303030302d30303030302d414139363500ec0c00005b54485d5831392d3938383431000000"
            "ec0c00000000a8d27b6e89814f6d0900000000004d700f685d032237000000000000000000000000000000000000000000000000"
            "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
            "00000000d7b3eb3a"
        ),
    )
    hive_hklm.map_key(key_name, key)

    target_win.add_plugin(WindowsProductKeyPlugin)

    records = list(target_win.license())

    assert len(records) == 1
    assert records[0].key == "VK7JG-NPHTM-C97JM-9MPGT-3V66T"
    assert records[0].name == "Professional [TH]X19-98841"
    assert records[0].type == "Retail"
    assert records[0].source == "HKLM\\SOFTWARE\\Microsoft\\Windows NT\\CurrentVersion\\DigitalProductId4"
