# Documentation Generation Reference

## API Reference

### DocumentationGenerator Class

Core class for automated documentation generation.

Initialization:
```python
from moai_docs_generation import DocumentationGenerator

doc_gen = DocumentationGenerator(
    output_dir="./docs",           # Output directory for generated docs
    template_dir="./templates",    # Custom templates directory
    config_file="docs.config.yaml" # Configuration file path
)
```

Methods:

generate_api_docs(source_path, format):
- Parameters: source_path (str) - Path to source code, format (str) - Output format
- Returns: Dict containing generated documentation
- Formats: openapi, swagger, markdown, html

generate_user_guide(project_info, template):
- Parameters: project_info (dict) - Project metadata, template (str) - Guide template
- Returns: Dict containing user guide content
- Templates: getting-started, tutorial, reference, cookbook

generate_comprehensive_docs(source_directory, options):
- Parameters: source_directory (str), options (dict) with keys include_api_docs, include_user_guides, output_formats
- Returns: Dict containing all generated documentation

export_to_formats(docs, formats):
- Parameters: docs (dict) - Generated documentation, formats (list) - Target formats
- Returns: Dict with paths to exported files
- Supported formats: html, pdf, markdown, json, static-site

update_documentation_for_changed_files():
- Analyzes git changes and updates affected documentation
- Returns: Dict with updated files and status

validate_documentation(options):
- Parameters: options (dict) with keys completeness_threshold, include_example_validation, check_link_integrity
- Returns: Dict containing quality report

### APIDocumentationGenerator Class

Specialized generator for API documentation.

Methods:

generate_openapi_spec(app):
- Parameters: app - FastAPI/Flask application instance
- Returns: Dict containing OpenAPI 3.1 specification

generate_graphql_docs(schema):
- Parameters: schema - GraphQL schema object
- Returns: Dict containing GraphQL documentation

generate_grpc_docs(proto_files):
- Parameters: proto_files (list) - List of .proto file paths
- Returns: Dict containing gRPC documentation

### CodeDocumentationEnhancer Class

AI-powered code documentation enhancement.

Methods:

analyze_and_enhance(source_directory, enhancement_level):
- Parameters: source_directory (str), enhancement_level (str) - basic/comprehensive/expert
- Returns: Dict containing enhanced documentation

generate_docstrings(code_file, style):
- Parameters: code_file (str), style (str) - google/numpy/sphinx
- Returns: Dict containing generated docstrings

---

## Configuration Options

### Main Configuration File (docs.config.yaml)

```yaml
documentation:
  output_directory: "./docs"
  default_format: "markdown"
  include_private_members: false
  include_inherited_members: true

api_docs:
  enabled: true
  format: "openapi"
  version: "3.1.0"
  include_examples: true
  include_schemas: true

user_guides:
  enabled: true
  templates:
    - getting-started
    - api-reference
    - tutorials
  auto_generate_toc: true

export:
  html:
    theme: "default"
    syntax_highlighting: true
    search_enabled: true
  pdf:
    page_size: "A4"
    include_toc: true
    header_footer: true
  static_site:
    generator: "mkdocs"
    theme: "material"

quality:
  completeness_threshold: 0.85
  validate_examples: true
  check_links: true
  spell_check: false

ai_enhancement:
  enabled: true
  model: "gpt-4"
  enhancement_level: "comprehensive"
  max_tokens_per_file: 4000
```

### Environment Variables

MOAI_DOCS_OUTPUT_DIR - Override output directory
MOAI_DOCS_FORMAT - Default output format
MOAI_DOCS_THEME - Documentation theme
MOAI_DOCS_AI_ENABLED - Enable/disable AI enhancement
MOAI_DOCS_VALIDATE - Enable validation on generation

---

## Integration Patterns

### FastAPI Integration

```python
from fastapi import FastAPI
from moai_docs_generation import APIDocumentationGenerator

app = FastAPI(
    title="My API",
    description="API documentation generated by MoAI",
    version="1.0.0"
)

# Generate OpenAPI documentation
api_gen = APIDocumentationGenerator()
openapi_spec = api_gen.generate_openapi_spec(app)

# Export to static HTML
api_gen.export_to_html(openapi_spec, output_dir="./docs/api")
```

### Sphinx Integration

```python
from moai_docs_generation import SphinxIntegration

sphinx = SphinxIntegration(
    source_dir="./src",
    build_dir="./docs/_build"
)

# Generate RST files from source code
sphinx.generate_rst_files()

# Build HTML documentation
sphinx.build_html()
```

### MkDocs Integration

```python
from moai_docs_generation import MkDocsIntegration

mkdocs = MkDocsIntegration(
    docs_dir="./docs",
    config_file="mkdocs.yml"
)

# Generate navigation from structure
mkdocs.generate_nav_from_structure()

# Deploy to GitHub Pages
mkdocs.deploy_gh_pages()
```

### CI/CD Pipeline Integration

GitHub Actions:
```yaml
name: Generate Documentation
on:
  push:
    branches: [main]
    paths:
      - 'src/**'
      - 'docs/**'

jobs:
  docs:
    runs-on: ubuntu-latest
    steps:
      - uses: actions/checkout@v4

      - name: Setup Python
        uses: actions/setup-python@v5
        with:
          python-version: '3.11'

      - name: Install dependencies
        run: pip install moai-docs-generation

      - name: Generate Documentation
        run: |
          moai generate-docs --source ./src --output ./docs
          moai validate-docs --threshold 0.9

      - name: Deploy to Pages
        uses: peaceiris/actions-gh-pages@v3
        with:
          github_token: ${{ secrets.GITHUB_TOKEN }}
          publish_dir: ./docs/_site
```

---

## Troubleshooting

### Common Issues

Issue: Documentation generation fails with "Module not found"
Solution:
- Ensure all dependencies are installed: pip install -e .
- Check PYTHONPATH includes source directory
- Verify __init__.py files exist in all packages

Issue: OpenAPI spec generation incomplete
Solution:
- Add proper type hints to all endpoints
- Include response_model in route decorators
- Use Pydantic models for request/response schemas

Issue: AI enhancement timeout
Solution:
- Reduce enhancement_level to "basic"
- Process files in smaller batches
- Increase max_tokens_per_file limit

Issue: PDF generation fails
Solution:
- Install WeasyPrint dependencies: apt-get install libpango-1.0-0
- Check font availability for specified styles
- Verify page size and margin settings

Issue: Links validation fails
Solution:
- Check for outdated external links
- Ensure internal links use relative paths
- Run link checker with --ignore-patterns for dynamic links

### Debug Mode

Enable debug logging:
```python
import logging
logging.getLogger('moai_docs_generation').setLevel(logging.DEBUG)

# Or via environment variable
# export MOAI_DOCS_DEBUG=true
```

### Performance Optimization

For large codebases:
- Use incremental generation with --changed-only flag
- Enable caching with --cache-dir ./docs/.cache
- Process in parallel with --workers 4
- Exclude test files with --exclude "tests/**"

---

## External Resources

### Official Documentation
- MoAI-ADK Documentation: https://moai-adk.dev/docs
- Claude Code Skills: https://docs.anthropic.com/claude-code/skills

### Related Tools
- Sphinx: https://www.sphinx-doc.org/
- MkDocs: https://www.mkdocs.org/
- MkDocs Material: https://squidfunk.github.io/mkdocs-material/
- Docusaurus: https://docusaurus.io/
- GitBook: https://www.gitbook.com/

### API Documentation Tools
- OpenAPI Specification: https://swagger.io/specification/
- Swagger UI: https://swagger.io/tools/swagger-ui/
- Redoc: https://redocly.github.io/redoc/
- Stoplight: https://stoplight.io/

### Standards and Best Practices
- Google Developer Documentation Style Guide: https://developers.google.com/style
- Microsoft Writing Style Guide: https://docs.microsoft.com/style-guide/
- Write the Docs: https://www.writethedocs.org/

### PDF Generation
- WeasyPrint: https://weasyprint.org/
- wkhtmltopdf: https://wkhtmltopdf.org/
- Puppeteer: https://pptr.dev/

---

Version: 1.0.0
Last Updated: 2025-12-06
