"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DefaultTokenResolver = exports.StringConcat = void 0;
const encoding_1 = require("./private/encoding");
const token_map_1 = require("./private/token-map");
/**
 * Converts all fragments to strings and concats those
 *
 * Drops 'undefined's.
 */
class StringConcat {
    join(left, right) {
        if (left === undefined) {
            return right !== undefined ? `${right}` : undefined;
        }
        if (right === undefined) {
            return `${left}`;
        }
        return `${left}${right}`;
    }
}
exports.StringConcat = StringConcat;
/**
 * Default resolver implementation
 *
 * @experimental
 */
class DefaultTokenResolver {
    constructor(concat) {
        this.concat = concat;
    }
    /**
     * Default Token resolution
     *
     * Resolve the Token, recurse into whatever it returns,
     * then finally post-process it.
     */
    resolveToken(t, context, postProcessor) {
        try {
            let resolved = t.resolve(context);
            // The token might have returned more values that need resolving, recurse
            resolved = context.resolve(resolved);
            resolved = postProcessor.postProcess(resolved, context);
            return resolved;
        }
        catch (e) {
            let message = `Resolution error: ${e.message}.`;
            if (t.creationStack && t.creationStack.length > 0) {
                message += `\nObject creation stack:\n  at ${t.creationStack.join("\n  at ")}`;
            }
            e.message = message;
            throw e;
        }
    }
    /**
     * Resolve string fragments to Tokens
     */
    resolveString(fragments, context) {
        return fragments.mapTokens({ mapToken: context.resolve }).join(this.concat);
    }
    resolveList(xs, context) {
        // Must be a singleton list token, because concatenation is not allowed.
        if (xs.length !== 1) {
            throw new Error(`Cannot add elements to list token, got: ${xs}`);
        }
        const str = encoding_1.TokenString.forListToken(xs[0]);
        const tokenMap = token_map_1.TokenMap.instance();
        const fragments = str.split(tokenMap.lookupToken.bind(tokenMap));
        if (fragments.length !== 1) {
            throw new Error(`Cannot concatenate strings in a tokenized string array, got: ${xs[0]}`);
        }
        return fragments.mapTokens({ mapToken: context.resolve }).firstValue;
    }
}
exports.DefaultTokenResolver = DefaultTokenResolver;
//# sourceMappingURL=data:application/json;base64,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