import os
import shutil
import numpy as np
import xarray as xr
import pandas as pd
from ..convert import Convert

# raw_path = './echopype/data/azfp/17031001.01A'     # Canada (Different ranges)
# xml_path = './echopype/data/azfp/17030815.XML'     # Canada (Different ranges)
raw_path = './echopype/test_data/azfp/17082117.01A'     # Standard test
xml_path = './echopype/test_data/azfp/17041823.XML'     # Standard test
test_path = './echopype/test_data/azfp/from_matlab/17082117.nc'
# raw_path = ['./echopype/test_data/azfp/set1/17033000.01A',     # Multiple files
#                  './echopype/test_data/azfp/set1/17033001.01A',
#                  './echopype/test_data/azfp/set1/17033002.01A',
#                  './echopype/test_data/azfp/set1/17033003.01A']
# xml_path = './echopype/test_data/azfp/set1/17033000.XML'       # Multiple files
csv_paths = ['./echopype/test_data/azfp/from_echoview/17082117-raw38.csv',    # EchoView exports
             './echopype/test_data/azfp/from_echoview/17082117-raw125.csv',
             './echopype/test_data/azfp/from_echoview/17082117-raw200.csv',
             './echopype/test_data/azfp/from_echoview/17082117-raw455.csv']


def test_convert_raw_matlab():

    # Unpacking data
    tmp = Convert(raw_path, xml_path)
    tmp.raw2nc()

    # Read in the dataset that will be used to confirm working conversions. (Generated by Matlab)
    ds_test = xr.open_dataset(test_path)

    # Test beam group
    with xr.open_dataset(tmp.nc_path, group='Beam') as ds_beam:
        # Test frequency
        assert np.array_equal(ds_test.frequency, ds_beam.frequency)
        # Test sea absorption
        # assert np.array_equal(ds_test.sea_abs, ds_beam.sea_abs)
        # Test ping time
        assert np.array_equal(ds_test.ping_time, ds_beam.ping_time)
        # Test tilt x and y
        assert np.array_equal(ds_test.tilt_x, ds_beam.tilt_x)
        assert np.array_equal(ds_test.tilt_y, ds_beam.tilt_y)
        # Test backscatter_r
        assert np.array_equal(ds_test.backscatter, ds_beam.backscatter_r)

    # Test environment group
    with xr.open_dataset(tmp.nc_path, group='Environment') as ds_env:
        # Test temperature
        assert np.array_equal(ds_test.temperature, ds_env.temperature)
        # Test sound speed. 1 value is used because sound speed is the same across frequencies
        # assert ds_test.sound_speed == ds_env.sound_speed_indicative.values[0]

    # with xr.open_dataset(tmp.nc_path, group="Vendor") as ds_vend:
    #     # Test battery values
    #     assert np.array_equal(ds_test.battery_main, ds_vend.battery_main)
    #     assert np.array_equal(ds_test.battery_tx, ds_vend.battery_tx)

    ds_test.close()
    os.remove(tmp.nc_path)
    del tmp


def test_convert_raw_echoview():
    # Compare parsed backscatter data with EchoView generated csv files.

    # Unpacking data
    tmp = Convert(raw_path, xml_path)
    tmp.raw2nc()

    # Read in csv files that will be used to confirm working conversions.
    channels = []
    for file in csv_paths:
        channels.append(pd.read_csv(file, header=None, skiprows=[0]).iloc[:, 6:])
    test_power = np.stack(channels)
    with xr.open_dataset(tmp.nc_path, group='Beam') as ds_beam:
        assert np.array_equal(test_power, ds_beam.backscatter_r)

    os.remove(tmp.nc_path)


def test_convert_zarr():
    # Test saving zarr file. Compare with EchoView generated csv files.
    tmp = Convert(raw_path, xml_path)
    tmp.raw2zarr()
    # Read in csv files that will be used to confirm working conversions.
    channels = []
    for file in csv_paths:
        channels.append(pd.read_csv(file, header=None, skiprows=[0]).iloc[:, 6:])
    test_power = np.stack(channels)
    with xr.open_zarr(tmp.zarr_path, group='Beam') as ds_beam:
        assert np.array_equal(test_power, ds_beam.backscatter_r)

    shutil.rmtree(tmp.zarr_path, ignore_errors=True)


def test_validate_path():
    tmp = Convert(raw_path, xml_path)
    directory = './echopype/test_data/azfp/temp_test_folder/'
    filename = 'test.nc'
    file = os.path.join(directory, filename)
    # Tests cases involving only 1 file
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    # Combining and path is a file
    tmp.validate_path(save_path=file, file_format='.nc', combine_opt=True)
    assert os.path.exists(directory)
    os.rmdir(directory)

    # Combining and path is a folder
    tmp.validate_path(save_path=directory, file_format='.nc', combine_opt=True)
    assert os.path.exists(directory)
    os.rmdir(directory)

    # Not combining and path is a file
    tmp.validate_path(save_path=file, file_format='.nc', combine_opt=False)
    assert os.path.exists(directory)
    os.rmdir(directory)

    # Not combining and path is a folder
    tmp.validate_path(save_path=directory, file_format='.nc', combine_opt=False)
    assert os.path.exists(directory)
    os.rmdir(directory)

    # Tests cases involving 2 files
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    tmp.filename = [raw_path, raw_path]
    tmp_path = './echopype/test_data/azfp/temp_test_folder/.echopype_tmp'       # Path where temp files are stored
    # Combining and path is a file
    tmp.validate_path(save_path=file, file_format='.nc', combine_opt=True)
    assert os.path.exists(directory)
    assert os.path.exists(tmp_path)
    os.rmdir(tmp_path)
    os.rmdir(directory)

    # Combining and path is a folder (should error)
    try:
        tmp.validate_path(save_path=directory, file_format='.nc', combine_opt=True)
    except ValueError:
        pass
    else:
        raise AssertionError("Expected error not thrown. combine_opt=True, nfiles=2, path=directory")

    # Not combining and path is a file (should error)
    try:
        tmp.validate_path(save_path=file, file_format='.nc', combine_opt=False)
    except ValueError:
        pass
    else:
        raise AssertionError("Expected error not thrown. combine_opt=False, nfiles=2, path=file")
    # Not combining and path is a folder
    tmp.validate_path(save_path=directory, file_format='.nc', combine_opt=False)
    assert os.path.exists(directory)
    os.rmdir(directory)
