"use strict";
var _a, _b, _c, _d, _e, _f, _g;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LookupMachineImage = exports.OperatingSystemType = exports.GenericWindowsImage = exports.GenericLinuxImage = exports.AmazonLinuxStorage = exports.AmazonLinuxVirt = exports.AmazonLinuxEdition = exports.AmazonLinuxKernel = exports.AmazonLinuxGeneration = exports.AmazonLinuxImage = exports.AmazonLinuxCpuType = exports.WindowsImage = exports.GenericSSMParameterImage = exports.MachineImage = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ssm = require("@aws-cdk/aws-ssm");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const user_data_1 = require("./user-data");
const windows_versions_1 = require("./windows-versions");
/**
 * Factory functions for standard Amazon Machine Image objects.
 */
class MachineImage {
    /**
     * A Windows image that is automatically kept up-to-date
     *
     * This Machine Image automatically updates to the latest version on every
     * deployment. Be aware this will cause your instances to be replaced when a
     * new version of the image becomes available. Do not store stateful information
     * on the instance if you are using this image.
     */
    static latestWindows(version, props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_WindowsVersion(version);
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_WindowsImageProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.latestWindows);
            }
            throw error;
        }
        return new WindowsImage(version, props);
    }
    /**
     * An Amazon Linux image that is automatically kept up-to-date
     *
     * This Machine Image automatically updates to the latest version on every
     * deployment. Be aware this will cause your instances to be replaced when a
     * new version of the image becomes available. Do not store stateful information
     * on the instance if you are using this image.
     */
    static latestAmazonLinux(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_AmazonLinuxImageProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.latestAmazonLinux);
            }
            throw error;
        }
        return new AmazonLinuxImage(props);
    }
    /**
     * A Linux image where you specify the AMI ID for every region
     *
     * @param amiMap For every region where you are deploying the stack,
     * specify the AMI ID for that region.
     * @param props Customize the image by supplying additional props
     */
    static genericLinux(amiMap, props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_GenericLinuxImageProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.genericLinux);
            }
            throw error;
        }
        return new GenericLinuxImage(amiMap, props);
    }
    /**
     * A Windows image where you specify the AMI ID for every region
     *
     * @param amiMap For every region where you are deploying the stack,
     * specify the AMI ID for that region.
     * @param props Customize the image by supplying additional props
     */
    static genericWindows(amiMap, props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_GenericWindowsImageProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.genericWindows);
            }
            throw error;
        }
        return new GenericWindowsImage(amiMap, props);
    }
    /**
     * An image specified in SSM parameter store that is automatically kept up-to-date
     *
     * This Machine Image automatically updates to the latest version on every
     * deployment. Be aware this will cause your instances to be replaced when a
     * new version of the image becomes available. Do not store stateful information
     * on the instance if you are using this image.
     *
     * @param parameterName The name of SSM parameter containing the AMi id
     * @param os The operating system type of the AMI
     * @param userData optional user data for the given image
     * @deprecated Use `MachineImage.fromSsmParameter()` instead
     */
    static fromSSMParameter(parameterName, os, userData) {
        try {
            jsiiDeprecationWarnings.print("@aws-cdk/aws-ec2.MachineImage#fromSSMParameter", "Use `MachineImage.fromSsmParameter()` instead");
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_OperatingSystemType(os);
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_UserData(userData);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromSSMParameter);
            }
            throw error;
        }
        return new GenericSSMParameterImage(parameterName, os, userData);
    }
    /**
     * An image specified in SSM parameter store
     *
     * By default, the SSM parameter is refreshed at every deployment,
     * causing your instances to be replaced whenever a new version of the AMI
     * is released.
     *
     * Pass `{ cachedInContext: true }` to keep the AMI ID stable. If you do, you
     * will have to remember to periodically invalidate the context to refresh
     * to the newest AMI ID.
     */
    static fromSsmParameter(parameterName, options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_SsmParameterImageOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromSsmParameter);
            }
            throw error;
        }
        return new GenericSsmParameterImage(parameterName, options);
    }
    /**
     * Look up a shared Machine Image using DescribeImages
     *
     * The most recent, available, launchable image matching the given filter
     * criteria will be used. Looking up AMIs may take a long time; specify
     * as many filter criteria as possible to narrow down the search.
     *
     * The AMI selected will be cached in `cdk.context.json` and the same value
     * will be used on future runs. To refresh the AMI lookup, you will have to
     * evict the value from the cache using the `cdk context` command. See
     * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
     *
     * This function can not be used in environment-agnostic stacks.
     */
    static lookup(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_LookupMachineImageProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.lookup);
            }
            throw error;
        }
        return new LookupMachineImage(props);
    }
}
exports.MachineImage = MachineImage;
_a = JSII_RTTI_SYMBOL_1;
MachineImage[_a] = { fqn: "@aws-cdk/aws-ec2.MachineImage", version: "1.174.0" };
/**
 * Select the image based on a given SSM parameter
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 */
class GenericSSMParameterImage {
    constructor(parameterName, os, userData) {
        this.os = os;
        this.userData = userData;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_OperatingSystemType(os);
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_UserData(userData);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, GenericSSMParameterImage);
            }
            throw error;
        }
        this.parameterName = parameterName;
    }
    /**
     * Return the image to use in the given context
     */
    getImage(scope) {
        const ami = ssm.StringParameter.valueForTypedStringParameter(scope, this.parameterName, ssm.ParameterType.AWS_EC2_IMAGE_ID);
        return {
            imageId: ami,
            osType: this.os,
            userData: this.userData ?? (this.os === OperatingSystemType.WINDOWS ? user_data_1.UserData.forWindows() : user_data_1.UserData.forLinux()),
        };
    }
}
exports.GenericSSMParameterImage = GenericSSMParameterImage;
_b = JSII_RTTI_SYMBOL_1;
GenericSSMParameterImage[_b] = { fqn: "@aws-cdk/aws-ec2.GenericSSMParameterImage", version: "1.174.0" };
/**
 * Select the image based on a given SSM parameter
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 */
class GenericSsmParameterImage {
    constructor(parameterName, props = {}) {
        this.parameterName = parameterName;
        this.props = props;
    }
    /**
     * Return the image to use in the given context
     */
    getImage(scope) {
        const imageId = lookupImage(scope, this.props.cachedInContext, this.parameterName);
        const osType = this.props.os ?? OperatingSystemType.LINUX;
        return {
            imageId,
            osType,
            userData: this.props.userData ?? (osType === OperatingSystemType.WINDOWS ? user_data_1.UserData.forWindows() : user_data_1.UserData.forLinux()),
        };
    }
}
/**
 * Select the latest version of the indicated Windows version
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 *
 * https://aws.amazon.com/blogs/mt/query-for-the-latest-windows-ami-using-systems-manager-parameter-store/
 */
class WindowsImage extends GenericSSMParameterImage {
    constructor(version, props = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_WindowsVersion(version);
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_WindowsImageProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, WindowsImage);
            }
            throw error;
        }
        const nonDeprecatedVersionName = WindowsImage.DEPRECATED_VERSION_NAME_MAP[version] ?? version;
        super('/aws/service/ami-windows-latest/' + nonDeprecatedVersionName, OperatingSystemType.WINDOWS, props.userData);
    }
}
exports.WindowsImage = WindowsImage;
_c = JSII_RTTI_SYMBOL_1;
WindowsImage[_c] = { fqn: "@aws-cdk/aws-ec2.WindowsImage", version: "1.174.0" };
WindowsImage.DEPRECATED_VERSION_NAME_MAP = {
    [windows_versions_1.WindowsVersion.WINDOWS_SERVER_2016_GERMAL_FULL_BASE]: windows_versions_1.WindowsVersion.WINDOWS_SERVER_2016_GERMAN_FULL_BASE,
    [windows_versions_1.WindowsVersion.WINDOWS_SERVER_2012_R2_SP1_PORTUGESE_BRAZIL_64BIT_CORE]: windows_versions_1.WindowsVersion.WINDOWS_SERVER_2012_R2_SP1_PORTUGUESE_BRAZIL_64BIT_CORE,
    [windows_versions_1.WindowsVersion.WINDOWS_SERVER_2016_PORTUGESE_PORTUGAL_FULL_BASE]: windows_versions_1.WindowsVersion.WINDOWS_SERVER_2016_PORTUGUESE_PORTUGAL_FULL_BASE,
    [windows_versions_1.WindowsVersion.WINDOWS_SERVER_2012_R2_RTM_PORTUGESE_BRAZIL_64BIT_BASE]: windows_versions_1.WindowsVersion.WINDOWS_SERVER_2012_R2_RTM_PORTUGUESE_BRAZIL_64BIT_BASE,
    [windows_versions_1.WindowsVersion.WINDOWS_SERVER_2012_R2_RTM_PORTUGESE_PORTUGAL_64BIT_BASE]: windows_versions_1.WindowsVersion.WINDOWS_SERVER_2012_R2_RTM_PORTUGUESE_PORTUGAL_64BIT_BASE,
    [windows_versions_1.WindowsVersion.WINDOWS_SERVER_2016_PORTUGESE_BRAZIL_FULL_BASE]: windows_versions_1.WindowsVersion.WINDOWS_SERVER_2016_PORTUGUESE_BRAZIL_FULL_BASE,
    [windows_versions_1.WindowsVersion.WINDOWS_SERVER_2012_SP2_PORTUGESE_BRAZIL_64BIT_BASE]: windows_versions_1.WindowsVersion.WINDOWS_SERVER_2012_SP2_PORTUGUESE_BRAZIL_64BIT_BASE,
    [windows_versions_1.WindowsVersion.WINDOWS_SERVER_2012_RTM_PORTUGESE_BRAZIL_64BIT_BASE]: windows_versions_1.WindowsVersion.WINDOWS_SERVER_2012_RTM_PORTUGUESE_BRAZIL_64BIT_BASE,
    [windows_versions_1.WindowsVersion.WINDOWS_SERVER_2008_R2_SP1_PORTUGESE_BRAZIL_64BIT_BASE]: windows_versions_1.WindowsVersion.WINDOWS_SERVER_2008_R2_SP1_PORTUGUESE_BRAZIL_64BIT_BASE,
    [windows_versions_1.WindowsVersion.WINDOWS_SERVER_2008_SP2_PORTUGESE_BRAZIL_32BIT_BASE]: windows_versions_1.WindowsVersion.WINDOWS_SERVER_2008_SP2_PORTUGUESE_BRAZIL_32BIT_BASE,
    [windows_versions_1.WindowsVersion.WINDOWS_SERVER_2012_RTM_PORTUGESE_PORTUGAL_64BIT_BASE]: windows_versions_1.WindowsVersion.WINDOWS_SERVER_2012_RTM_PORTUGUESE_PORTUGAL_64BIT_BASE,
    [windows_versions_1.WindowsVersion.WINDOWS_SERVER_2019_PORTUGESE_BRAZIL_FULL_BASE]: windows_versions_1.WindowsVersion.WINDOWS_SERVER_2019_PORTUGUESE_BRAZIL_FULL_BASE,
    [windows_versions_1.WindowsVersion.WINDOWS_SERVER_2019_PORTUGESE_PORTUGAL_FULL_BASE]: windows_versions_1.WindowsVersion.WINDOWS_SERVER_2019_PORTUGUESE_PORTUGAL_FULL_BASE,
};
/**
 * CPU type
 */
var AmazonLinuxCpuType;
(function (AmazonLinuxCpuType) {
    /**
     * arm64 CPU type
     */
    AmazonLinuxCpuType["ARM_64"] = "arm64";
    /**
     * x86_64 CPU type
     */
    AmazonLinuxCpuType["X86_64"] = "x86_64";
})(AmazonLinuxCpuType = exports.AmazonLinuxCpuType || (exports.AmazonLinuxCpuType = {}));
/**
 * Selects the latest version of Amazon Linux
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 */
class AmazonLinuxImage extends GenericSSMParameterImage {
    constructor(props = {}) {
        super(AmazonLinuxImage.ssmParameterName(props), OperatingSystemType.LINUX, props.userData);
        this.props = props;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_AmazonLinuxImageProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, AmazonLinuxImage);
            }
            throw error;
        }
        this.cachedInContext = props.cachedInContext ?? false;
    }
    /**
     * Return the SSM parameter name that will contain the Amazon Linux image with the given attributes
     */
    static ssmParameterName(props = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_AmazonLinuxImageProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.ssmParameterName);
            }
            throw error;
        }
        const generation = (props && props.generation) || AmazonLinuxGeneration.AMAZON_LINUX;
        const edition = (props && props.edition) || AmazonLinuxEdition.STANDARD;
        const cpu = (props && props.cpuType) || AmazonLinuxCpuType.X86_64;
        let kernel = (props && props.kernel) || undefined;
        let virtualization;
        let storage;
        if (generation === AmazonLinuxGeneration.AMAZON_LINUX_2022) {
            kernel = AmazonLinuxKernel.KERNEL5_X;
            if (props && props.storage) {
                throw new Error('Storage parameter does not exist in smm parameter name for Amazon Linux 2022.');
            }
            if (props && props.virtualization) {
                throw new Error('Virtualization parameter does not exist in smm parameter name for Amazon Linux 2022.');
            }
        }
        else {
            virtualization = (props && props.virtualization) || AmazonLinuxVirt.HVM;
            storage = (props && props.storage) || AmazonLinuxStorage.GENERAL_PURPOSE;
        }
        const parts = [
            generation,
            'ami',
            edition !== AmazonLinuxEdition.STANDARD ? edition : undefined,
            kernel,
            virtualization,
            cpu,
            storage,
        ].filter(x => x !== undefined); // Get rid of undefineds
        return '/aws/service/ami-amazon-linux-latest/' + parts.join('-');
    }
    /**
     * Return the image to use in the given context
     */
    getImage(scope) {
        const imageId = lookupImage(scope, this.cachedInContext, this.parameterName);
        const osType = OperatingSystemType.LINUX;
        return {
            imageId,
            osType,
            userData: this.props.userData ?? user_data_1.UserData.forLinux(),
        };
    }
}
exports.AmazonLinuxImage = AmazonLinuxImage;
_d = JSII_RTTI_SYMBOL_1;
AmazonLinuxImage[_d] = { fqn: "@aws-cdk/aws-ec2.AmazonLinuxImage", version: "1.174.0" };
/**
 * What generation of Amazon Linux to use
 */
var AmazonLinuxGeneration;
(function (AmazonLinuxGeneration) {
    /**
     * Amazon Linux
     */
    AmazonLinuxGeneration["AMAZON_LINUX"] = "amzn";
    /**
     * Amazon Linux 2
     */
    AmazonLinuxGeneration["AMAZON_LINUX_2"] = "amzn2";
    /**
     * Amazon Linux 2022
     */
    AmazonLinuxGeneration["AMAZON_LINUX_2022"] = "al2022";
})(AmazonLinuxGeneration = exports.AmazonLinuxGeneration || (exports.AmazonLinuxGeneration = {}));
/**
 * Amazon Linux Kernel
 */
var AmazonLinuxKernel;
(function (AmazonLinuxKernel) {
    /**
     * Standard edition
     */
    AmazonLinuxKernel["KERNEL5_X"] = "kernel-5.10";
})(AmazonLinuxKernel = exports.AmazonLinuxKernel || (exports.AmazonLinuxKernel = {}));
/**
 * Amazon Linux edition
 */
var AmazonLinuxEdition;
(function (AmazonLinuxEdition) {
    /**
     * Standard edition
     */
    AmazonLinuxEdition["STANDARD"] = "standard";
    /**
     * Minimal edition
     */
    AmazonLinuxEdition["MINIMAL"] = "minimal";
})(AmazonLinuxEdition = exports.AmazonLinuxEdition || (exports.AmazonLinuxEdition = {}));
/**
 * Virtualization type for Amazon Linux
 */
var AmazonLinuxVirt;
(function (AmazonLinuxVirt) {
    /**
     * HVM virtualization (recommended)
     */
    AmazonLinuxVirt["HVM"] = "hvm";
    /**
     * PV virtualization
     */
    AmazonLinuxVirt["PV"] = "pv";
})(AmazonLinuxVirt = exports.AmazonLinuxVirt || (exports.AmazonLinuxVirt = {}));
var AmazonLinuxStorage;
(function (AmazonLinuxStorage) {
    /**
     * EBS-backed storage
     */
    AmazonLinuxStorage["EBS"] = "ebs";
    /**
     * S3-backed storage
     */
    AmazonLinuxStorage["S3"] = "s3";
    /**
     * General Purpose-based storage (recommended)
     */
    AmazonLinuxStorage["GENERAL_PURPOSE"] = "gp2";
})(AmazonLinuxStorage = exports.AmazonLinuxStorage || (exports.AmazonLinuxStorage = {}));
/**
 * Construct a Linux machine image from an AMI map
 *
 * Linux images IDs are not published to SSM parameter store yet, so you'll have to
 * manually specify an AMI map.
 */
class GenericLinuxImage {
    constructor(amiMap, props = {}) {
        this.amiMap = amiMap;
        this.props = props;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_GenericLinuxImageProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, GenericLinuxImage);
            }
            throw error;
        }
    }
    getImage(scope) {
        const userData = this.props.userData ?? user_data_1.UserData.forLinux();
        const osType = OperatingSystemType.LINUX;
        const region = core_1.Stack.of(scope).region;
        if (core_1.Token.isUnresolved(region)) {
            const mapping = {};
            for (const [rgn, ami] of Object.entries(this.amiMap)) {
                mapping[rgn] = { ami };
            }
            const amiMap = new core_1.CfnMapping(scope, 'AmiMap', { mapping });
            return {
                imageId: amiMap.findInMap(core_1.Aws.REGION, 'ami'),
                userData,
                osType,
            };
        }
        const imageId = region !== 'test-region' ? this.amiMap[region] : 'ami-12345';
        if (!imageId) {
            throw new Error(`Unable to find AMI in AMI map: no AMI specified for region '${region}'`);
        }
        return {
            imageId,
            userData,
            osType,
        };
    }
}
exports.GenericLinuxImage = GenericLinuxImage;
_e = JSII_RTTI_SYMBOL_1;
GenericLinuxImage[_e] = { fqn: "@aws-cdk/aws-ec2.GenericLinuxImage", version: "1.174.0" };
/**
 * Construct a Windows machine image from an AMI map
 *
 * Allows you to create a generic Windows EC2 , manually specify an AMI map.
 */
class GenericWindowsImage {
    constructor(amiMap, props = {}) {
        this.amiMap = amiMap;
        this.props = props;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_GenericWindowsImageProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, GenericWindowsImage);
            }
            throw error;
        }
    }
    getImage(scope) {
        const userData = this.props.userData ?? user_data_1.UserData.forWindows();
        const osType = OperatingSystemType.WINDOWS;
        const region = core_1.Stack.of(scope).region;
        if (core_1.Token.isUnresolved(region)) {
            const mapping = {};
            for (const [rgn, ami] of Object.entries(this.amiMap)) {
                mapping[rgn] = { ami };
            }
            const amiMap = new core_1.CfnMapping(scope, 'AmiMap', { mapping });
            return {
                imageId: amiMap.findInMap(core_1.Aws.REGION, 'ami'),
                userData,
                osType,
            };
        }
        const imageId = region !== 'test-region' ? this.amiMap[region] : 'ami-12345';
        if (!imageId) {
            throw new Error(`Unable to find AMI in AMI map: no AMI specified for region '${region}'`);
        }
        return {
            imageId,
            userData,
            osType,
        };
    }
}
exports.GenericWindowsImage = GenericWindowsImage;
_f = JSII_RTTI_SYMBOL_1;
GenericWindowsImage[_f] = { fqn: "@aws-cdk/aws-ec2.GenericWindowsImage", version: "1.174.0" };
/**
 * The OS type of a particular image
 */
var OperatingSystemType;
(function (OperatingSystemType) {
    OperatingSystemType[OperatingSystemType["LINUX"] = 0] = "LINUX";
    OperatingSystemType[OperatingSystemType["WINDOWS"] = 1] = "WINDOWS";
    /**
     * Used when the type of the operating system is not known
     * (for example, for imported Auto-Scaling Groups).
     */
    OperatingSystemType[OperatingSystemType["UNKNOWN"] = 2] = "UNKNOWN";
})(OperatingSystemType = exports.OperatingSystemType || (exports.OperatingSystemType = {}));
/**
 * A machine image whose AMI ID will be searched using DescribeImages.
 *
 * The most recent, available, launchable image matching the given filter
 * criteria will be used. Looking up AMIs may take a long time; specify
 * as many filter criteria as possible to narrow down the search.
 *
 * The AMI selected will be cached in `cdk.context.json` and the same value
 * will be used on future runs. To refresh the AMI lookup, you will have to
 * evict the value from the cache using the `cdk context` command. See
 * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
 */
class LookupMachineImage {
    constructor(props) {
        this.props = props;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_LookupMachineImageProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, LookupMachineImage);
            }
            throw error;
        }
    }
    getImage(scope) {
        // Need to know 'windows' or not before doing the query to return the right
        // osType for the dummy value, so might as well add it to the filter.
        const filters = {
            'name': [this.props.name],
            'state': ['available'],
            'image-type': ['machine'],
            'platform': this.props.windows ? ['windows'] : undefined,
        };
        Object.assign(filters, this.props.filters);
        const value = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.AMI_PROVIDER,
            props: {
                owners: this.props.owners,
                filters,
            },
            dummyValue: 'ami-1234',
        }).value;
        if (typeof value !== 'string') {
            throw new Error(`Response to AMI lookup invalid, got: ${value}`);
        }
        const osType = this.props.windows ? OperatingSystemType.WINDOWS : OperatingSystemType.LINUX;
        return {
            imageId: value,
            osType,
            userData: this.props.userData ?? user_data_1.UserData.forOperatingSystem(osType),
        };
    }
}
exports.LookupMachineImage = LookupMachineImage;
_g = JSII_RTTI_SYMBOL_1;
LookupMachineImage[_g] = { fqn: "@aws-cdk/aws-ec2.LookupMachineImage", version: "1.174.0" };
function lookupImage(scope, cachedInContext, parameterName) {
    return cachedInContext
        ? ssm.StringParameter.valueFromLookup(scope, parameterName)
        : ssm.StringParameter.valueForTypedStringParameter(scope, parameterName, ssm.ParameterType.AWS_EC2_IMAGE_ID);
}
//# sourceMappingURL=data:application/json;base64,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