from sympy import Min, Max, Set, Lambda, symbols, S, oo
from sympy.core import Basic, Expr, Integer
from sympy.core.numbers import Infinity, NegativeInfinity, Zero
from sympy.multipledispatch import dispatch
from sympy.sets import Interval, FiniteSet, Union, ImageSet


_x, _y = symbols("x y")


@dispatch(Basic, Basic)
def _set_pow(x, y):
    return None

@dispatch(Set, Set)
def _set_pow(x, y):
    return ImageSet(Lambda((_x, _y), (_x ** _y)), x, y)

@dispatch(Expr, Expr)
def _set_pow(x, y):
    return x**y

@dispatch(Interval, Zero)
def _set_pow(x, z):
    return FiniteSet(S.One)

@dispatch(Interval, Integer)
def _set_pow(x, exponent):
    """
    Powers in interval arithmetic
    https://en.wikipedia.org/wiki/Interval_arithmetic
    """
    s1 = x.start**exponent
    s2 = x.stop**exponent
    if ((s2 > s1) if exponent > 0 else (x.stop > -x.start)) == True:
        left_open = x.left_open
        right_open = x.right_open
        # TODO: handle unevaluated condition.
        sleft = s2
    else:
        # TODO: `s2 > s1` could be unevaluated.
        left_open = x.right_open
        right_open = x.left_open
        sleft = s1

    if x.start.is_positive:
        return Interval(Min(s1, s2), Max(s1, s2), left_open=left_open, right_open=right_open)
    elif x.stop.is_negative:
        return Interval(Min(s1, s2), Max(s1, s2), left_open=left_open, right_open=right_open)

    # Case where x.start < 0 and x.stop > 0:
    if exponent.is_odd:
        if exponent.is_negative:
            if x.start.is_zero:
                return Interval(s2, oo, left_open=x.right_open)
            if x.stop.is_zero:
                return Interval(-oo, s1, left_open=True, right_open=x.left_open)
            return Union(Interval(-oo, s1, left_open=True, right_open=x.left_open), Interval(s2, oo, left_open=x.right_open))
        else:
            return Interval(s1, s2, left_open=x.left_open, right_open=x.right_open)
    elif exponent.is_even:
        if exponent.is_negative:
            if x.start.is_zero:
                return Interval(s2, oo, left_open=x.right_open)
            if x.stop.is_zero:
                return Interval(s1, oo, left_open=x.left_open)
            return Interval(0, oo)
        else:
            return Interval(S.Zero, sleft, left_open=S.Zero not in x, right_open=left_open)

@dispatch(Interval, Infinity)
def _set_pow(b, e):
    # TODO: add logic for open intervals?
    if b.start.is_nonnegative:
        if b.stop < 1:
            return FiniteSet(S.Zero)
        if b.start > 1:
            return FiniteSet(S.Infinity)
        return Interval(0, oo)
    elif b.stop.is_negative:
        if b.start > -1:
            return FiniteSet(S.Zero)
        if b.stop < -1:
            return FiniteSet(-oo, oo)
        return Interval(-oo, oo)
    else:
        if b.start > -1:
            if b.stop < 1:
                return FiniteSet(S.Zero)
            return Interval(0, oo)
        return Interval(-oo, oo)

@dispatch(Interval, NegativeInfinity)
def _set_pow(b, e):
    from sympy.sets.setexpr import set_div
    return _set_pow(set_div(S.One, b), oo)
