"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SqlServerSeeder = void 0;
const cdk = require("@aws-cdk/core");
const ec2 = require("@aws-cdk/aws-ec2");
const lambda = require("@aws-cdk/aws-lambda");
const cr = require("@aws-cdk/custom-resources");
const s3 = require("@aws-cdk/aws-s3");
const aws_s3_deployment_1 = require("@aws-cdk/aws-s3-deployment");
const tmp = require("tmp");
const fs = require("fs");
const path = require("path");
class SqlServerSeeder extends cdk.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        if (!props.database.secret) {
            throw new Error("Database does not have secret value assigned");
        }
        if (!fs.existsSync(props.createScriptPath)) {
            throw new Error("Create script does not exist: " + props.createScriptPath);
        }
        if (props.deleteScriptPath && !fs.existsSync(props.deleteScriptPath)) {
            throw new Error("Create script does not exist: " + props.deleteScriptPath);
        }
        const destinationBucket = new s3.Bucket(this, 'bucket', { removalPolicy: cdk.RemovalPolicy.DESTROY });
        this.prepareSqlScripts(id, props, destinationBucket);
        const sqlSeederLambda = new lambda.Function(this, 'lambda', {
            code: new lambda.AssetCode(__dirname + '/resources/sqlserver-seeder.zip'),
            handler: 'seed::seed.Bootstrap::ExecuteFunction',
            timeout: cdk.Duration.seconds(300),
            runtime: lambda.Runtime.DOTNET_CORE_3_1,
            memorySize: props.memorySize,
            vpc: props.vpc,
            vpcSubnets: {
                subnetType: ec2.SubnetType.PRIVATE
            },
            environment: {
                "DbEndpoint": props.database.dbInstanceEndpointAddress,
                "SecretArn": (_a = props.database.secret) === null || _a === void 0 ? void 0 : _a.secretArn,
                "ScriptsBucket": destinationBucket.bucketName,
                "RunOnDelete": `${!!props.deleteScriptPath}`
            }
        });
        const sqlSeederProvider = new cr.Provider(this, 'provider', {
            onEventHandler: sqlSeederLambda
        });
        const sqlSeederResource = new cdk.CustomResource(this, 'resource', {
            serviceToken: sqlSeederProvider.serviceToken,
            properties: {
                "IgnoreSqlErrors": !!props.ignoreSqlErrors
            }
        });
        sqlSeederResource.node.addDependency(props.database);
        // allow access
        destinationBucket.grantRead(sqlSeederLambda);
        (_b = props.database.secret) === null || _b === void 0 ? void 0 : _b.grantRead(sqlSeederLambda);
        // enable connection to RDS instance
        sqlSeederLambda.connections.allowTo(props.database, ec2.Port.tcp(props.port));
    }
    prepareSqlScripts(id, props, destinationBucket) {
        tmp.setGracefulCleanup();
        tmp.dir((err, dir) => {
            if (err)
                throw err;
            fs.copyFileSync(props.createScriptPath, path.join(dir, 'create.sql'));
            if (props.deleteScriptPath) {
                fs.copyFileSync(props.deleteScriptPath, path.join(dir, 'delete.sql'));
            }
            new aws_s3_deployment_1.BucketDeployment(this, id, {
                sources: [aws_s3_deployment_1.Source.asset(dir)],
                destinationBucket,
                retainOnDelete: false
            });
        });
    }
}
exports.SqlServerSeeder = SqlServerSeeder;
//# sourceMappingURL=data:application/json;base64,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