import inspect
import logging
import os
import time
from collections import defaultdict
from datetime import datetime
from pathlib import Path
from typing import Any, Callable, Dict, List, Optional, Set, Union

from ploomber import DAG
from ploomber.executors import Serial
from ploomber.io import serializer_pickle, unserializer_pickle
from ploomber.products import File
from ploomber.tasks import PythonCallable

from cf_pipelines.base.helper_classes import FunctionDetails, ProductLineage
from cf_pipelines.base.utils import get_return_keys_from_function, remove_extension, wrap_preserving_signature


class Pipeline:
    """
    A class used to build a code-first pipeline

    Attributes
    ---------
    name: str
        The name of the pipeline
    location: str or Path
        The pipelines store data locally for each run, with this parameter one can specify where this data is stored
    extra_args: dict
        Any extra arguments that should be passed to the functions upon request
    dag_clients: dict
        Any Ploomber clients, used to back up the data produced by the pipeline.
        See https://docs.ploomber.io/en/latest/user-guide/faq_index.html#why-do-products-have-clients for more info
    serializer: Callable
        The method used to serialise data generated by the pipelines.
        See https://docs.ploomber.io/en/latest/user-guide/serialization.html#Serialization for more info
    unserializer: Callable
        The method used to unserializer data generated by the pipelines.
        See https://docs.ploomber.io/en/latest/user-guide/serialization.html#Serialization for more info
    track_all: bool
        A flag that specifies whether each run should be tracked independently. When set to false, all the artifacts
        are saved to a "default" folder. If true, each run is saved to a unique folder identified by the time it ran
    current_run_id: str
        A unique identifier for a given run, this value changes depending on the value of `track_all`.
    """

    def __init__(
        self,
        name: str,
        location: Union[str, Path] = ".pipelines",
        extra_args: Dict[str, Any] = None,
        dag_clients: Dict = None,
        serializer: Callable = None,
        unserializer: Callable = None,
        track_all: bool = False,
    ):
        self.name = name
        self.location = Path(location)
        self.product_lineages: Dict[str, ProductLineage] = {}
        self.function_details: Dict[str, FunctionDetails] = {}
        self.extra_arguments = extra_args or dict()
        self.dag_clients = dag_clients or dict()
        self.serializer = serializer or serializer_pickle
        self.unserializer = unserializer or unserializer_pickle
        self.track_all = track_all
        self.current_run_id = "default"
        self.before_function: Optional[Callable[[str], None]] = None
        self.after_function: Optional[Callable[[str, Dict[str, Any], float], None]] = None
        self.exception_handler: Optional[Callable[[str, Exception, float], None]] = None
        self.meta_logger = logging.getLogger(__name__)

    def set_before_step(self, before_function: Callable[[str], None]):
        self.before_function = before_function

    def set_after_function(self, after_function: Callable[[str, Dict[str, Any], float], None]):
        self.after_function = after_function

    def set_exception_handler(self, exception_handler: Callable[[str, Exception, float], None]):
        self.exception_handler = exception_handler

    def clear_function_data(self, function_name: str) -> None:
        """
        Removes from the pipeline all the data related to a function identified by `function_name`
        :param function_name: The name of the function to be deleted
        """
        function_details = self.function_details.pop(function_name)
        for product in function_details.produces:
            self.product_lineages.pop(product)

    def step(self, group: str) -> Callable:
        """
        A decorator that registers the decorated function into a Ploomber pipeline

        :param group: The group the pipeline belongs to
        :return:
        """

        def _wrapper(original_fn: Callable):
            original_name = original_fn.__name__

            if original_name in self.function_details:
                self.meta_logger.warning(f"The function {original_name} was already in {self.name}, replacing it")
                self.clear_function_data(original_name)

            def decorated_function_replacement(upstream=None, **kwargs):
                # Work is done here to understand from where the function was called.
                # `upstream` is set when the function is called by Plomber, but when the user executes the function
                # directly, its value is none and `kwargs` contains the arguments to the function.

                # When `upstream` is equals to `{"kwargs": None}` means that the function has been by Ploomber otherwise,
                if kwargs == {"kwargs": None}:
                    called_from_ploomber = True
                    kwargs.pop("kwargs")
                else:
                    called_from_ploomber = False

                if called_from_ploomber:
                    # Thanks to the serialisers Upstream now contains the read data in a nested dictionary where
                    # the first level contains the name of the function that produced the product, second level contains
                    # the name of the product and the value is the data for that product.
                    # However the scientist's function expects the data at the first level, this bit of code flattens the
                    # dictionary
                    kwargs = {}
                    current_details = self.function_details[original_name]
                    for needed_artifact in current_details.needs:
                        try:
                            lineage = self.product_lineages[needed_artifact]
                            kwargs[needed_artifact] = upstream[lineage.produced_by][needed_artifact]
                        except KeyError:
                            # The required artifact was not available from any other source, try getting it from the
                            # extra_arguments passed on to the pipeline

                            environment_variable = f"CF_{needed_artifact.upper()}"
                            if environment_variable in os.environ:
                                kwargs[needed_artifact] = os.environ[environment_variable]
                            elif needed_artifact in self.extra_arguments:
                                kwargs[needed_artifact] = self.extra_arguments[needed_artifact]
                            else:
                                raise KeyError(
                                    f"The product {needed_artifact} requested by {original_name} is "
                                    "not generated by another step, "
                                    "nor does it exist as an environment variable,"
                                    "nor is it passed as an extra argument to the pipeline."
                                )

                    # When called from Ploomber it is also necessary to call the `after_function`
                    if self.before_function:
                        self.before_function(original_name)

                start_time = time.time()
                # Execution of the actual function
                try:
                    returns = original_fn(**kwargs)
                except Exception as exception:
                    end_time = time.time()
                    if self.exception_handler:
                        self.exception_handler(original_name, exception, end_time - start_time)
                    raise exception

                end_time = time.time()

                # Results contains a mapping "filename.extension": value, we must remove the file extension
                returns = {remove_extension(filename): value for filename, value in returns.items()}

                if called_from_ploomber:
                    if self.after_function:
                        self.after_function(original_name, returns, end_time - start_time)

                return returns

            # Get information about what the original function generates and what it needs
            returnable_products = get_return_keys_from_function(original_fn)
            function_args = set(inspect.getfullargspec(original_fn).kwonlyargs)

            wrap_preserving_signature(decorated_function_replacement, original_fn)

            # Fill in the metadata of the pipeline
            self.add_product_lineages(original_name, returnable_products, group)
            self.add_function_details(
                decorated_function_replacement, function_args, original_name, returnable_products, group
            )

            return decorated_function_replacement

        return _wrapper

    def add_product_lineages(self, original_name: str, returnable_products: List[str], group: str) -> None:
        for product_filename in returnable_products:
            self.product_lineages[remove_extension(product_filename)] = ProductLineage(
                group, product_filename, original_name
            )

    def add_function_details(
        self,
        replacement_fn: Callable,
        function_args: Set[str],
        original_name: str,
        returnable_products: List[str],
        group: str,
    ):
        returnable_arguments = {remove_extension(product) for product in returnable_products}
        function_details = FunctionDetails(
            replacement_fn, produces=returnable_arguments, needs=function_args, group=group
        )
        self.function_details[original_name] = function_details

    def get_local_artifact_path(self, product_name: str) -> Path:
        """
        Get the local path for a given product given its product name

        :param product_name:
        :return:
        """
        product_lineage = self.product_lineages[product_name]
        return Path(self.location, self.current_run_id, product_lineage.group, product_lineage.file_name)

    def solve_dependencies(self) -> Dict[str, Set[str]]:
        """
        This method tries to solve the graph dependencies based on what each function needs

        :return: A dictionary whose keys are the names of the registered functions and its value is a set of
        the function names it depends upon
        """
        dependencies = defaultdict(set)
        for function_name, function_details in self.function_details.items():
            for needed_product in function_details.needs:
                try:
                    product_lineage = self.product_lineages[needed_product]
                    dependencies[function_name].add(product_lineage.produced_by)
                except KeyError:
                    # The argument needed by the function is not available as a product from another function, so
                    # we need to check in the extra arguments passed by the client.
                    environment_variable = f"CF_{needed_product.upper()}"
                    if (needed_product not in self.extra_arguments) and (environment_variable not in os.environ):
                        raise KeyError(
                            f"The product {needed_product} requested by {function_name} is not generated by another step, "
                            "nor does it exist as an environment variable,"
                            "nor is it passed as an extra argument to the pipeline."
                        )
        return dict(dependencies)

    def create_ploomber_callables(self, dag) -> Dict[str, PythonCallable]:
        """
        Creates the corresponding `PythonCallables` for each one of the functions in the `Pipeline`

        :param dag: The DAG the task should be added to
        :return: A dictionary where the keys are the function names and the values are
        their corresponding `PythonCallables`
        """
        callables = {}
        for function_name, function_details in self.function_details.items():
            products = {
                product_name: File(self.get_local_artifact_path(product_name))
                for product_name in function_details.produces
            }

            callable_function = PythonCallable(
                function_details.python_function,
                products,
                dag,
                function_name,
                # We need to pass `kwargs` here since the replacement function requires `kwargs` to work when called
                # directly. This value will be then passed on to the function when called via Ploomber but not when
                # the function is called directly.
                params={"kwargs": None},
                serializer=self.serializer,
                unserializer=self.unserializer,
            )
            callables[function_name] = callable_function
        return callables

    def make_dag(self) -> DAG:
        """
        Build the Ploomber DAG from the dependencies added vía the `step` decorator.

        :return: A fully generated Ploomber dag
        """
        executor = Serial(build_in_subprocess=False)
        dag = DAG(executor=executor, name=self.name, clients=self.dag_clients)

        # Create ploomber's `PythonCallables` for the functions in the pipeliene
        callables = self.create_ploomber_callables(dag)

        # Create Ploomber dependencies
        solved_dependencies = self.solve_dependencies()
        for function_name, dependencies in solved_dependencies.items():
            for dependency in dependencies:
                callables[dependency] >> callables[function_name]
        return dag

    def generate_run_id(self) -> str:
        """
        Generates a new run identifier for a DAG run (and sets it to the value)

        :return: The generated run id
        """
        if self.track_all:
            self.current_run_id = datetime.now().strftime("%Y%m%d%H%M%S%f")
        else:
            self.current_run_id = "default"
        return self.current_run_id

    def run(self) -> None:
        self.generate_run_id()
        dag = self.make_dag()
        dag.build()
