/*
 * Decompiled with CFR 0.152.
 */
package org.planit.geo;

import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.geotools.geometry.jts.JTS;
import org.geotools.geometry.jts.JTSFactoryFinder;
import org.geotools.referencing.crs.DefaultGeographicCRS;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.MultiLineString;
import org.locationtech.jts.geom.Point;
import org.locationtech.jts.geom.Polygon;
import org.opengis.geometry.DirectPosition;
import org.opengis.geometry.coordinate.PointArray;
import org.opengis.referencing.crs.CoordinateReferenceSystem;
import org.planit.geo.PlanitOpenGisUtils;
import org.planit.utils.exceptions.PlanItException;
import org.planit.utils.graph.Vertex;

public class PlanitJtsUtils {
    private static final Logger LOGGER = Logger.getLogger(PlanitJtsUtils.class.getCanonicalName());
    public static final DefaultGeographicCRS DEFAULT_GEOGRAPHIC_CRS = PlanitOpenGisUtils.DEFAULT_GEOGRAPHIC_CRS;
    public static final CoordinateReferenceSystem CARTESIANCRS = PlanitOpenGisUtils.CARTESIANCRS;
    private final CoordinateReferenceSystem crs;
    private static final GeometryFactory jtsGeometryFactory = JTSFactoryFinder.getGeometryFactory();

    public PlanitJtsUtils() {
        this(DEFAULT_GEOGRAPHIC_CRS);
    }

    public PlanitJtsUtils(CoordinateReferenceSystem coordinateReferenceSystem) {
        this.crs = coordinateReferenceSystem;
    }

    public double getDistanceInMetres(Point startPosition, Point endPosition) throws PlanItException {
        return this.getDistanceInMetres(startPosition.getCoordinate(), endPosition.getCoordinate());
    }

    public double getDistanceInMetres(Coordinate startCoordinate, Coordinate endCoordinate) throws PlanItException {
        try {
            if (this.crs.equals(CARTESIANCRS)) {
                double deltaCoordinate0 = startCoordinate.x - endCoordinate.x;
                double deltaCoordinate1 = startCoordinate.y - endCoordinate.y;
                return Math.sqrt(Math.pow(deltaCoordinate0, 2.0) + Math.pow(deltaCoordinate1, 2.0));
            }
            return JTS.orthodromicDistance(startCoordinate, endCoordinate, this.crs);
        }
        catch (Exception e) {
            LOGGER.severe(e.getMessage());
            throw new PlanItException("Error when computing distance in meters between two Positions in JtsUtils", e);
        }
    }

    public double getDistanceInKilometres(Point startPosition, Point endPosition) throws PlanItException {
        return this.getDistanceInMetres(startPosition, endPosition) / 1000.0;
    }

    public double getDistanceInKilometres(Vertex vertex1, Vertex vertex2) throws PlanItException {
        return this.getDistanceInKilometres(vertex1.getPosition(), vertex2.getPosition());
    }

    public static Coordinate createCoordinate(DirectPosition position) {
        return new Coordinate(position.getOrdinate(0), position.getOrdinate(1));
    }

    public static Point createPoint(double xCoordinate, double yCoordinate) throws PlanItException {
        Coordinate coordinate = new Coordinate(xCoordinate, yCoordinate);
        Point newPoint = jtsGeometryFactory.createPoint(coordinate);
        return newPoint;
    }

    public static LineString convertToJtsLineString(org.opengis.geometry.coordinate.LineString openGisLineString) throws PlanItException {
        PointArray samplePoints = openGisLineString.getSamplePoints();
        List coordinates = samplePoints.stream().map(point -> PlanitJtsUtils.createCoordinate(point.getDirectPosition())).collect(Collectors.toList());
        return jtsGeometryFactory.createLineString((Coordinate[])coordinates.toArray());
    }

    public static LineString convertToLineString(MultiLineString jtsMultiLineString) throws PlanItException {
        PlanItException.throwIf(jtsMultiLineString.getNumGeometries() > 1, "MultiLineString contains multiple LineStrings");
        return (LineString)jtsMultiLineString.getGeometryN(0);
    }

    public static LineString createLineString(List<Double> coordinateList) throws PlanItException {
        PlanItException.throwIf(coordinateList.size() % 2 != 0, "coordinate list must contain an even number of entries to correctly identify (x,y) pairs");
        Iterator<Double> iter = coordinateList.iterator();
        Coordinate[] coordinateArray = new Coordinate[coordinateList.size() / 2];
        int index = 0;
        while (iter.hasNext()) {
            coordinateArray[index++] = new Coordinate(iter.next(), iter.next());
        }
        return PlanitJtsUtils.createLineStringFromCoordinates(coordinateArray);
    }

    public static LineString createLineString(String value, char ts, char cs) throws PlanItException {
        ArrayList<Double> coordinateDoubleList = new ArrayList<Double>();
        String[] tupleString = value.split("[" + ts + "]");
        for (int index = 0; index < tupleString.length; ++index) {
            String xyCoordinateString = tupleString[index];
            String[] coordinateString = xyCoordinateString.split("[" + cs + "]");
            if (coordinateString.length != 2) {
                throw new PlanItException(String.format("invalid coordinate encountered, expected two coordinates in tuple, but found %d", coordinateString.length));
            }
            coordinateDoubleList.add(Double.parseDouble(coordinateString[0]));
            coordinateDoubleList.add(Double.parseDouble(coordinateString[1]));
        }
        return PlanitJtsUtils.createLineString(coordinateDoubleList);
    }

    public static LineString createLineStringFromCoordinates(Coordinate[] coordinates) throws PlanItException {
        return jtsGeometryFactory.createLineString(coordinates);
    }

    public static LineString createLineStringFromCsvString(String value, String ts, String cs) throws PlanItException {
        if (ts.length() > 1 || cs.length() > 1) {
            PlanItException.throwIf(ts.length() > 1, String.format("tuple separating string to create LineString is not a single character but %s", ts));
            PlanItException.throwIf(cs.length() > 1, String.format("comma separating string to create LineString is not a single character but %s", cs));
        }
        return PlanitJtsUtils.createLineString(value, ts.charAt(0), cs.charAt(0));
    }

    public static String createCsvStringFromLineString(LineString geometry, Character ts, Character cs, DecimalFormat df) {
        Coordinate[] coordinates = geometry.getCoordinates();
        StringBuilder csvStringBuilder = new StringBuilder();
        int lastIndex = coordinates.length - 1;
        for (int index = 0; index < coordinates.length; ++index) {
            Coordinate coordinate = coordinates[index];
            csvStringBuilder.append(df.format(coordinate.x)).append(cs).append(df.format(coordinate.y));
            if (index == lastIndex) break;
            csvStringBuilder.append(ts);
        }
        return csvStringBuilder.toString();
    }

    public static Polygon createPolygon() {
        Polygon polygon = jtsGeometryFactory.createPolygon();
        return polygon;
    }

    public static Polygon create2DPolygon(List<Double> coordinateList2D) {
        return jtsGeometryFactory.createPolygon(PlanitJtsUtils.listTo2DCoordinates(coordinateList2D));
    }

    public static Polygon createPolygon(Coordinate[] coords) {
        return jtsGeometryFactory.createPolygon(jtsGeometryFactory.createLinearRing(coords));
    }

    public static Coordinate[] directPositionsToCoordinates(List<DirectPosition> positions) throws PlanItException {
        Coordinate[] coordinates = new Coordinate[positions.size()];
        for (int index = 0; index < coordinates.length; ++index) {
            coordinates[index] = PlanitJtsUtils.createCoordinate(positions.get(index));
        }
        return coordinates;
    }

    public static Coordinate[] listTo2DCoordinates(List<?> posList) {
        int dimensions = 2;
        Coordinate[] coordinates = new Coordinate[posList.size() / dimensions];
        int index = 0;
        while (index + dimensions - 1 < posList.size()) {
            coordinates[index / dimensions] = new Coordinate(Double.parseDouble(posList.get(index).toString()), Double.parseDouble(posList.get(index + 1).toString()));
            index += dimensions;
        }
        return coordinates;
    }

    public double getDistanceInKilometres(LineString geometry) throws PlanItException {
        Coordinate[] coordinates = geometry.getCoordinates();
        int numberOfCoords = coordinates.length;
        if (numberOfCoords > 1) {
            double computedLengthInMetres = 0.0;
            Coordinate previousCoordinate = coordinates[0];
            for (int index = 1; index < numberOfCoords; ++index) {
                Coordinate currentCoordinate = coordinates[index];
                computedLengthInMetres += this.getDistanceInMetres(previousCoordinate, currentCoordinate);
                previousCoordinate = currentCoordinate;
            }
            return computedLengthInMetres / 1000.0;
        }
        throw new PlanItException("unable to compute distance for less than two points");
    }

    public static LineString createCopyWithoutCoordinatesBefore(Point position, LineString geometry) throws PlanItException {
        Optional<Integer> offset = PlanitJtsUtils.findFirstCoordinatePosition(position.getCoordinate(), geometry);
        if (!offset.isPresent()) {
            throw new PlanItException(String.format("point (%s) does not exist on line string (%s), unable to create copy from this location", position.toString(), geometry.toString()));
        }
        Coordinate[] coordinates = PlanitJtsUtils.copyCoordinatesFrom(offset.get(), geometry);
        return PlanitJtsUtils.createLineStringFromCoordinates(coordinates);
    }

    public static LineString createCopyWithoutCoordinatesBefore(int startIndex, LineString geometry) throws PlanItException {
        if (startIndex >= geometry.getNumPoints() || startIndex < 0) {
            throw new PlanItException("invalid start index for extracting coordinates from line string geometry");
        }
        return PlanitJtsUtils.createLineStringFromCoordinates(PlanitJtsUtils.copyCoordinatesFrom(startIndex, geometry));
    }

    public static LineString createCopyWithoutCoordinatesAfter(Point position, LineString geometry) throws PlanItException {
        Optional<Integer> offset = PlanitJtsUtils.findFirstCoordinatePosition(position.getCoordinate(), geometry);
        if (!offset.isPresent()) {
            throw new PlanItException(String.format("point (%s) does not exist on line string %s, unable to create copy from this location", position.toString(), geometry.toString()));
        }
        Coordinate[] coordinates = PlanitJtsUtils.copyCoordinatesUpToNotIncluding(offset.get() + 1, geometry);
        return PlanitJtsUtils.createLineStringFromCoordinates(coordinates);
    }

    public static LineString createCopyWithoutCoordinatesAfter(int endIndex, LineString geometry) throws PlanItException {
        if (geometry == null) {
            return null;
        }
        if (endIndex >= geometry.getNumPoints() || endIndex < 0) {
            throw new PlanItException("invalid end index for extracting coordinates from line string geometry");
        }
        return PlanitJtsUtils.createLineStringFromCoordinates(PlanitJtsUtils.copyCoordinatesUpToNotIncluding(endIndex + 1, geometry));
    }

    public static LineString createCopyWithoutAdjacentDuplicateCoordinates(LineString geometry) {
        if (geometry == null) {
            return null;
        }
        ArrayList<Coordinate> coordinateList = new ArrayList<Coordinate>(geometry.getNumPoints());
        int numCoordinates = geometry.getNumPoints();
        int index = 0;
        int nextIndex = index + 1;
        while (index < numCoordinates) {
            Coordinate coordinate = geometry.getCoordinateN(index);
            boolean isAdjacentDuplicate = false;
            if (nextIndex < numCoordinates) {
                Coordinate adjacentCoordinate = geometry.getCoordinateN(index + 1);
                isAdjacentDuplicate = coordinate.equals2D(adjacentCoordinate);
            }
            if (!isAdjacentDuplicate) {
                coordinateList.add(coordinate);
            }
            ++index;
            ++nextIndex;
        }
        return jtsGeometryFactory.createLineString((Coordinate[])coordinateList.stream().toArray(Coordinate[]::new));
    }

    public static Optional<Integer> findFirstCoordinatePosition(Coordinate coordinateToLocate, int offset, LineString geometry) {
        if (geometry == null || coordinateToLocate == null) {
            return Optional.empty();
        }
        int numCoordinates = geometry.getNumPoints();
        for (int index = offset; index < numCoordinates; ++index) {
            Coordinate coordinate = geometry.getCoordinateN(index);
            if (!coordinate.equals2D(coordinateToLocate)) continue;
            return Optional.of(index);
        }
        return Optional.empty();
    }

    public static Optional<Integer> findFirstCoordinatePosition(Coordinate coordinateToLocate, LineString geometry) {
        return PlanitJtsUtils.findFirstCoordinatePosition(coordinateToLocate, 0, geometry);
    }

    public static Coordinate[] copyCoordinatesFrom(int offset, LineString geometry) throws PlanItException {
        return PlanitJtsUtils.copyCoordinatesFromUpToNotIncluding(offset, geometry.getNumPoints(), geometry);
    }

    public static Coordinate[] copyCoordinatesUpToNotIncluding(int untilPoint, LineString geometry) throws PlanItException {
        return PlanitJtsUtils.copyCoordinatesFromUpToNotIncluding(0, untilPoint, geometry);
    }

    public static Coordinate[] copyCoordinatesFromUpToNotIncluding(int offset, int untilPoint, LineString geometry) throws PlanItException {
        PlanItException.throwIfNull(geometry, "provided geometry to copy coordinates from is null");
        int numCoordinates = geometry.getNumPoints();
        if (offset > untilPoint || untilPoint > numCoordinates) {
            LOGGER.severe("unable to extract coordinates from line string, offset is larger than final point, and/or final point exceeds number of coordinates in geometry");
        }
        Coordinate[] coordinates = new Coordinate[untilPoint - offset];
        for (int index = offset; index < untilPoint; ++index) {
            Coordinate coordinate;
            coordinates[index - offset] = coordinate = geometry.getCoordinateN(index);
        }
        return coordinates;
    }

    public static LineString concatenate(LineString ... geometries) {
        MultiLineString theMultiLineString = jtsGeometryFactory.createMultiLineString(geometries);
        return jtsGeometryFactory.createLineString(theMultiLineString.getCoordinates());
    }

    public CoordinateReferenceSystem getCoordinateReferenceSystem() {
        return this.crs;
    }
}

