"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PrivateCA = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_eks_1 = require("aws-cdk-lib/aws-eks");
const constructs_1 = require("constructs");
const kong_core = require("kong-core");
class PrivateCA extends constructs_1.Construct {
    // public cluster_issuer: aws_eks.KubernetesManifest;
    constructor(scope, id, props) {
        super(scope, id);
        this.route53_zone = new aws_cdk_lib_1.aws_route53.PrivateHostedZone(this, 'CpInternalHostedZone', {
            vpc: props.cluster.vpc,
            zoneName: props.hostedZoneName,
        });
        //TODO : Customize this and further restrict access to aws pca issuer to scoped cert only
        const cfnCertificateAuthority = new aws_cdk_lib_1.aws_acmpca.CfnCertificateAuthority(this, 'CA', {
            type: 'ROOT',
            keyAlgorithm: 'RSA_2048',
            signingAlgorithm: 'SHA256WITHRSA',
            subject: {
                country: 'US',
                organization: 'Amazon',
                organizationalUnit: 'AWS',
                state: 'WA',
                commonName: props.hostedZoneName,
                locality: 'Seattle',
            },
        });
        this.private_ca_arn = cfnCertificateAuthority.attrArn;
        const internalRootCaCert = new aws_cdk_lib_1.aws_acmpca.CfnCertificate(this, 'PrivCaCert', {
            certificateAuthorityArn: this.private_ca_arn,
            certificateSigningRequest: cfnCertificateAuthority.attrCertificateSigningRequest,
            templateArn: 'arn:aws:acm-pca:::template/RootCACertificate/V1',
            signingAlgorithm: 'SHA256WITHRSA',
            validity: {
                value: 10,
                type: 'YEARS',
            },
        });
        const internalRootCaActivation = new aws_cdk_lib_1.aws_acmpca.CfnCertificateAuthorityActivation(this, 'PrivCaActivation', {
            certificateAuthorityArn: this.private_ca_arn,
            certificate: internalRootCaCert.attrCertificate,
        });
        const wild_card_cert = new aws_cdk_lib_1.aws_certificatemanager.CfnCertificate(this, 'CpCert', {
            domainName: `*.${props.hostedZoneName}`,
            certificateAuthorityArn: this.private_ca_arn,
        });
        wild_card_cert.node.addDependency(internalRootCaActivation);
        const cert_manager_namespace = new aws_eks_1.KubernetesManifest(this, 'CertManagerNamespace', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'v1',
                    kind: 'Namespace',
                    metadata: { name: kong_core.Namespace.CERT_MANAGER },
                },
            ],
        });
        const deploy_cert_manager = new aws_cdk_lib_1.aws_eks.HelmChart(this, 'CertManagerHelm', {
            cluster: props.cluster,
            repository: 'https://charts.jetstack.io',
            chart: 'cert-manager',
            createNamespace: false,
            namespace: kong_core.Namespace.CERT_MANAGER,
            release: 'cert-manager',
            wait: true,
            values: {
                installCRDs: true,
                webhook: {
                    timeoutSeconds: 30,
                },
            },
        });
        deploy_cert_manager.node.addDependency(cert_manager_namespace);
        const aws_pca_issuer_namespace = new aws_eks_1.KubernetesManifest(this, 'AwsPcaIssuerNamespace', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'v1',
                    kind: 'Namespace',
                    metadata: { name: kong_core.Namespace.AWS_PCA_ISSUER },
                },
            ],
        });
        const aws_pca_issuer_service_account = new aws_cdk_lib_1.aws_eks.ServiceAccount(this, 'AwsPcaIssuerSa', {
            cluster: props.cluster,
            name: 'aws-pca-issuer-service-account',
            namespace: kong_core.Namespace.AWS_PCA_ISSUER,
        });
        aws_pca_issuer_service_account.node.addDependency(aws_pca_issuer_namespace);
        aws_pca_issuer_service_account.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            resources: [this.private_ca_arn],
            actions: [
                'acm-pca:Get*',
                'acm-pca:Issue*',
                'acm-pca:Describe*',
            ],
        }));
        const deploy_pca_issuer = new aws_cdk_lib_1.aws_eks.HelmChart(this, 'AwsPcaIssuerHelm', {
            cluster: props.cluster,
            repository: 'https://cert-manager.github.io/aws-privateca-issuer',
            chart: 'aws-pca-issuer',
            namespace: aws_pca_issuer_service_account.serviceAccountNamespace,
            release: 'aws-pca-issuer',
            wait: true,
            createNamespace: false,
            values: {
                serviceAccount: {
                    create: false,
                    name: aws_pca_issuer_service_account.serviceAccountName,
                },
            },
        });
        deploy_pca_issuer.node.addDependency(aws_pca_issuer_service_account, aws_pca_issuer_namespace, internalRootCaActivation, deploy_cert_manager);
        const helm_waiter = new aws_cdk_lib_1.aws_eks.HelmChart(this, 'DummyWaiter', {
            cluster: props.cluster,
            repository: 'https://anshrma.github.io/helm-charts',
            chart: 'helm-waiter',
            release: 'helm-waiter',
            wait: true,
        });
        helm_waiter.node.addDependency(deploy_pca_issuer);
        const aws_pca_cluster_issuer = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'ClusterIssuerManifest', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'awspca.cert-manager.io/v1beta1',
                    kind: 'AWSPCAClusterIssuer',
                    metadata: { name: props.clusterIssuerName },
                    spec: {
                        arn: this.private_ca_arn,
                        region: aws_cdk_lib_1.Stack.of(this).region,
                    },
                },
            ],
        });
        aws_pca_cluster_issuer.node.addDependency(helm_waiter, deploy_pca_issuer);
        // this.cluster_issuer = props.cluster.addManifest('ClusterIssuerManifest', cluster_issuer_manifest);
        // this.cluster_issuer.node.addDependency(helm_waiter, priv_ca_issuer_helm);
        // new CfnOutput(this, 'ExtDnsRoleArn', { value: service_account_external_dns.role.roleArn });
        // new CfnOutput(this, 'CertIssuerRoleArnOutput', { value: service_account.serviceAccountName });
    }
}
exports.PrivateCA = PrivateCA;
//# sourceMappingURL=data:application/json;base64,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