"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KongControlPlane = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const kong_core = require("kong-core");
class KongControlPlane extends constructs_1.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        // TODO : Use IRSA , Scope down the secret values and use new external secrets helm
        props.nodegroup.role.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            resources: ['*'],
            actions: [
                'secretsmanager:GetResourcePolicy',
                'secretsmanager:GetSecretValue',
                'secretsmanager:DescribeSecret',
                'secretsmanager:ListSecretVersionIds',
            ],
        }));
        const kong_namespace = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'KongNamespace', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'v1',
                    kind: 'Namespace',
                    metadata: { name: props.namespace },
                },
            ],
        });
        const kong_certificate = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'KongCert', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'cert-manager.io/v1',
                    kind: 'Certificate',
                    metadata: {
                        name: props.cacertname,
                        namespace: props.namespace,
                    },
                    spec: {
                        commonName: props.hostedZoneName,
                        dnsNames: props.dnsNames,
                        duration: '2160h0m0s',
                        issuerRef: {
                            group: 'awspca.cert-manager.io',
                            kind: 'AWSPCAClusterIssuer',
                            name: props.clusterIssuerName,
                        },
                        renewBefore: '360h0m0s',
                        secretName: `${props.clusterIssuerName}-secret`,
                        usages: [
                            'server auth',
                            'client auth',
                        ],
                        privateKey: {
                            algorithm: 'RSA',
                            size: 2048,
                        },
                    },
                },
            ],
        });
        kong_certificate.node.addDependency(kong_namespace);
        const deploy_external_secrets_helm = new aws_cdk_lib_1.aws_eks.HelmChart(this, 'ExternalSecrets', {
            cluster: props.cluster,
            repository: 'https://external-secrets.github.io/kubernetes-external-secrets/',
            chart: 'kubernetes-external-secrets',
            release: 'kubernetes-external-secrets',
            namespace: props.namespace,
            wait: true,
            values: {
                securityContext: {
                    fsGroup: 65534,
                },
                env: {
                    AWS_REGION: aws_cdk_lib_1.Stack.of(this).region,
                },
            },
        });
        deploy_external_secrets_helm.node.addDependency(kong_namespace);
        const secretNameParts = aws_cdk_lib_1.Fn.split('-', (_a = props.rds.secret) === null || _a === void 0 ? void 0 : _a.secretName);
        const secretNameWithoutSuffix = aws_cdk_lib_1.Fn.join('-', [aws_cdk_lib_1.Fn.select(0, secretNameParts), aws_cdk_lib_1.Fn.select(1, secretNameParts)]);
        const secrets_crd_mount = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'SecretsMount', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'kubernetes-client.io/v1',
                    kind: 'ExternalSecret',
                    metadata: {
                        name: 'rds-secrets',
                        namespace: props.namespace,
                    },
                    spec: {
                        backendType: 'secretsManager',
                        data: [
                            {
                                key: secretNameWithoutSuffix,
                                name: 'password',
                                property: 'password',
                            },
                            {
                                key: secretNameWithoutSuffix,
                                name: 'host',
                                property: 'host',
                            },
                            {
                                key: secretNameWithoutSuffix,
                                name: 'username',
                                property: 'username',
                            },
                            {
                                key: secretNameWithoutSuffix,
                                name: 'dbname',
                                property: 'dbname',
                            },
                        ],
                    },
                },
                {
                    apiVersion: 'kubernetes-client.io/v1',
                    kind: 'ExternalSecret',
                    metadata: {
                        name: 'kong-license',
                        namespace: props.namespace,
                    },
                    spec: {
                        backendType: 'secretsManager',
                        data: [
                            {
                                key: props.license_secret_name,
                                name: 'license',
                            },
                        ],
                    },
                },
            ],
        });
        secrets_crd_mount.node.addDependency(deploy_external_secrets_helm);
        const external_dns = new kong_core.ExternalDns(this, 'KongCpDns', { cluster: props.cluster });
        const deploy_kong_cp_helm = new aws_cdk_lib_1.aws_eks.HelmChart(this, 'KongCpHelm', {
            cluster: props.cluster,
            repository: 'https://charts.konghq.com',
            chart: 'kong',
            release: 'kong',
            namespace: props.namespace,
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            wait: true,
            createNamespace: true,
            values: {
                ingressController: {
                    enabled: true,
                    installCRDs: false,
                    image: {
                        repository: 'kong/kubernetes-ingress-controller',
                        tag: '1.3.1-alpine',
                    },
                },
                image: {
                    repository: 'kong/kong-gateway',
                    tag: '2.5.0.0-alpine',
                },
                env: {
                    database: 'postgres',
                    role: 'control_plane',
                    cluster_mtls: 'pki',
                    cluster_cert: '/etc/secrets/kong-cp-cluster-issuer-secret/tls.crt',
                    cluster_cert_key: '/etc/secrets/kong-cp-cluster-issuer-secret/tls.key',
                    cluster_ca_cert: '/etc/secrets/kong-cp-cluster-issuer-secret/ca.crt',
                    pg_user: {
                        valueFrom: {
                            secretKeyRef: {
                                name: 'rds-secrets',
                                key: 'username',
                            },
                        },
                    },
                    pg_password: {
                        valueFrom: {
                            secretKeyRef: {
                                name: 'rds-secrets',
                                key: 'password',
                            },
                        },
                    },
                    pg_database: {
                        valueFrom: {
                            secretKeyRef: {
                                name: 'rds-secrets',
                                key: 'dbname',
                            },
                        },
                    },
                    pg_host: {
                        valueFrom: {
                            secretKeyRef: {
                                name: 'rds-secrets',
                                key: 'host',
                            },
                        },
                    },
                },
                cluster: {
                    enabled: 'true',
                    type: 'LoadBalancer',
                    tls: {
                        enabled: 'true',
                        servicePort: 8005,
                        containerPort: 8005,
                    },
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'external-dns.alpha.kubernetes.io/hostname': 'cluster.kong-cp.internal',
                    },
                },
                clustertelemetry: {
                    enabled: true,
                    type: 'LoadBalancer',
                    tls: {
                        enabled: 'true',
                    },
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'external-dns.alpha.kubernetes.io/hostname': 'telemetry.kong-cp.internal',
                    },
                },
                proxy: {
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'service.beta.kubernetes.io/aws-load-balancer-additional-resource-tags': 'Type=proxy',
                        'external-dns.alpha.kubernetes.io/hostname': 'proxy.kong-cp.internal',
                    },
                },
                admin: {
                    enabled: true,
                    type: 'LoadBalancer',
                    labels: {
                        'enable-metrics': true,
                    },
                    http: {
                        enabled: true,
                    },
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'service.beta.kubernetes.io/aws-load-balancer-additional-resource-tags': 'Type=admin',
                        'external-dns.alpha.kubernetes.io/hostname': 'admin.kong-cp.internal',
                    },
                },
                enterprise: {
                    enabled: true,
                    license_secret: 'kong-license',
                },
                serviceMonitor: {
                    enabled: true,
                },
                manager: {
                    enabled: true,
                    type: 'LoadBalancer',
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'external-dns.alpha.kubernetes.io/hostname': 'manager.kong-cp.internal',
                    },
                },
                postgresql: {
                    enabled: false,
                },
                secretVolumes: [
                    'kong-cp-cluster-issuer-secret',
                ],
            },
        });
        deploy_kong_cp_helm.node.addDependency(secrets_crd_mount, external_dns);
        //TODO : https://github.com/aws/aws-cdk/issues/14933
        // const kong_admin = new KubernetesObjectValue(this, 'KongAdminAttribute', {
        //   cluster: props.cluster,
        //   objectType: 'service',
        //   objectName: 'kong-kong-admin',
        //   jsonPath: '.status.loadBalancer.ingress[0].hostname',
        //   timeout: Duration.minutes(30),
        //   objectNamespace: props.namespace,
        // });
        // const kong_telemetry = new KubernetesObjectValue(this, 'KongTelemetryAttribute', {
        //   cluster: props.cluster,
        //   objectType: 'service',
        //   objectName: 'kong-kong-clustertelemetry',
        //   jsonPath: '.status.loadBalancer.ingress[0].hostname',
        //   timeout: Duration.minutes(30),
        //   objectNamespace: props.namespace,
        // });
        // const kong_cluster = new KubernetesObjectValue(this, 'KongClusterAttribute', {
        //   cluster: props.cluster,
        //   objectType: 'service',
        //   objectName: 'kong-kong-cluster',
        //   jsonPath: '.status.loadBalancer.ingress[0].hostname',
        //   timeout: Duration.minutes(30),
        //   objectNamespace: props.namespace,
        // });
        // // const kong_cluster= props.cluster.getServiceLoadBalancerAddress('kong-kong-cluster', { namespace: props.namespace }) ;
        // // const kong_telemetry = props.cluster.getServiceLoadBalancerAddress('kong-kong-clustertelemetry', { namespace: props.namespace }) ;
        // new CfnOutput(this, 'KongAdminOutput', { value: kong_admin.value, description: 'Kong Admin Endpoint' });
        // new CfnOutput(this, 'KongTelemetryOutput', { value: kong_telemetry.value, description: 'Kong Telemetry Endpoint' });
        // new CfnOutput(this, 'KongClusterOutput', { value: kong_cluster.value, description: 'Kong Cluster Endpoint' });
    }
}
exports.KongControlPlane = KongControlPlane;
//# sourceMappingURL=data:application/json;base64,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